package biss;

import java.awt.Color;
import java.awt.Font;
import java.awt.Rectangle;
import java.io.File;
import java.io.FileInputStream;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * java.util.Property based class to read in Strings from so called
 * "property files" and to convert these Strings to their final
 * representation (Rectangles, Colors, * Fonts etc.).
 * Used by so called "package classes" (like biss.awt.Awt) 
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class PackageProperties
  extends Properties
{
	String Package;
	String PackagePrefix;
	File Source;
	protected static String HomeDir = System.getProperty( "user.home");

public PackageProperties ( String pkgName ) {
	Package = pkgName;
	PackagePrefix = Package + '.';

	String path;
	if ( (path = System.getProperty( "java.resources")) == null ) {
		path = HomeDir;
		if ( path == null ) path = ".java";
		else path += (File.separator + ".java");
	}

	Source = new File( path + File.separator + Package);

	if ( Source.exists() ) {
		try {
			FileInputStream in = new FileInputStream( Source);
			load( in);
			in.close();
		}
		catch ( Exception x ){};
	}
}

public static String expandHome ( String s ) {
	if ( s.charAt(0) == '~' )
		return HomeDir + s.substring( 1, s.length());
	else
		return s;
}

public boolean getBoolean ( String key, boolean def ) {
	String val = getProperty( qualifyKey( key));

	if ( val == null )
		return def;
	else {
		if ( "true".equalsIgnoreCase( val) )       return true;
		else if ( "false".equalsIgnoreCase( val) ) return false;
		else                                       return def;
	}
}

public Color getColor ( String key, Color def ) {
	String val = getProperty( qualifyKey( key));
	if ( val == null ) return def;

	if ( Character.isDigit( val.charAt( 0)) ){
		StringTokenizer tkn = new StringTokenizer( val);
		try {
			int r = Integer.parseInt( tkn.nextToken());
			int g = Integer.parseInt( tkn.nextToken());
			int b = Integer.parseInt( tkn.nextToken());
			return new Color( r,g,b);
		}
		catch ( NumberFormatException x ) {
			return def;
		}
	}

	if (      "black".equalsIgnoreCase( val) )   return Color.black;
	else if ( "blue".equalsIgnoreCase( val) )    return Color.blue; 
	else if ( "cyan".equalsIgnoreCase( val) )    return Color.cyan;
	else if ( "darkGray".equalsIgnoreCase( val) )    return Color.darkGray;
	else if ( "gray".equalsIgnoreCase( val) )    return Color.gray;
	else if ( "green".equalsIgnoreCase( val) )    return Color.green;
	else if ( "lightGray".equalsIgnoreCase( val) )    return Color.lightGray;
	else if ( "magenta".equalsIgnoreCase( val) )    return Color.magenta;
	else if ( "orange".equalsIgnoreCase( val) )    return Color.orange;
	else if ( "pink".equalsIgnoreCase( val) )    return Color.pink;
	else if ( "red".equalsIgnoreCase( val) )    return Color.red;
	else if ( "white".equalsIgnoreCase( val) )    return Color.white;
	else if ( "yellow".equalsIgnoreCase( val) )    return Color.yellow;
	else
		return def;
}

public double getDouble ( String key, double def ) {
	String val = getProperty( qualifyKey( key));
	double i;
	if ( val == null ) return def;
	try {
		i = Double.valueOf( val).doubleValue();
	}
	catch ( NumberFormatException x ) {
		i = def;
	}

	return i;
}

public File getFile ( String key , String defFileName ) {
	String val = getProperty( qualifyKey( key));

	if ( val == null ) {
		if ( defFileName == null )
			return null;
		else
			return new File( defFileName);
	}
	else
		return new File( val);
}

public Font getFont ( String key, String def ) {
	String name, attr;
	int size, style;

	String val = getProperty( qualifyKey( key));
	if ( val == null ) {
		if ( def == null ) return null;
		val = def;
	}

	StringTokenizer tkn = new StringTokenizer( val, " -.\t");
	name = tkn.nextToken();

	attr = tkn.nextToken();
	if ( "PLAIN".equalsIgnoreCase( attr) )       style = Font.PLAIN;
	else if ( "BOLD".equalsIgnoreCase( attr) )   style = Font.BOLD;
	else if ( "ITALIC".equalsIgnoreCase( attr) ) style = Font.ITALIC;
	else style = Font.PLAIN;

	try {
		size = Integer.parseInt( tkn.nextToken());
	}
	catch ( NumberFormatException x ) {
		size = 12;
	}

	return new Font( name, style, size);
}

public int getInt ( String key, int def ) {
	String val = getProperty( qualifyKey( key));
	int i;
	if ( val == null ) return def;
	try {
		i = Integer.parseInt( val);
	}
	catch ( NumberFormatException x ) {
		i = def;
	}

	return i;
}

public int[] getIntArray ( String key, int[] def ) {
	String val = getProperty( qualifyKey( key));
	int    j, n, m;
	int    u[];

	if ( val != null ) {
		try {
			StringTokenizer tkn = new StringTokenizer( val, ": ,\t;");
			m = (def != null) ? def.length : 0;
			n = Math.min( tkn.countTokens(), m);
			u = new int[n];

			for ( j=0; j < n; j++ )
				u[j] = Integer.parseInt( tkn.nextToken());

			for ( ; j<m; j++ )
				u[j] = def[j];

			return u;
		}
		catch ( NumberFormatException x ) {}
	}

	return def;
}

public String[] getPaths ( String key, String defPaths ) {
	String sep = (File.pathSeparatorChar != ':') ? " ;" : " ;:";
	String a[] = getStrings( key, sep, defPaths);

	if ( a != null && File.separatorChar != '/' ) {
		for ( int i=0; i<a.length; i++ )
			a[i] = a[i].replace( '/', File.separatorChar);
	}
	return a;
}

public Rectangle getRectangle ( String key, Rectangle def ) {
	String val = getProperty( qualifyKey( key));
	Rectangle r;
	int[] u = { def.x, def.y, def.width, def.height };

	if ( val != null ) {
		try {
			StringTokenizer tkn = new StringTokenizer( val, ": ,\t;x");
			int n = Math.min( tkn.countTokens(), 4);

			for ( int j=0; j < n; j++ )
				u[j] = Integer.parseInt( tkn.nextToken());
		}
		catch ( NumberFormatException x ) {}
	}

	return new Rectangle( u[0], u[1], u[2], u[3]);
}

public String getString ( String key, String def ) {
	String val = getProperty( qualifyKey( key));

	if ( val == null )
		return def;
	else
		return val;
}

public String[] getStrings ( String key, String separators, String defStrings ) {
	String val = getProperty( qualifyKey( key));

	if ( val == null ) {
		if ( defStrings == null )
			return null;
		else
			val = defStrings;
	}

	StringTokenizer tok = new StringTokenizer( val, separators);
	Vector list = new Vector();
	while ( tok.hasMoreTokens() ) {
		String p = expandHome( tok.nextToken());
		if ( p != null && p.length() > 0 )
			list.addElement( p);
	}

	String[] a = new String[list.size()];
	list.copyInto( a);

	return a;
}

public String qualifyKey ( String key ) {
	if ( key.indexOf( '.' ) >= 0 )
		return key;
	else
		return PackagePrefix + key;
}
}
