/*
                    Joystick caliberation file IO

	Functions:

	char **XJCGetDeviceNamesLinux(char *path, int *total)

	int XJCDoSaveCaliberation(jcw_struct *j, char *path)

	---
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../include/cfgfmt.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "xjc.h"
#include "config.h"


#ifdef __linux__
/*
 *	Get caliberated joystick devices names listing from
 *	the caliberation file specified in path.
 *
 *	The caliberation file format is expected to be
 *	joystick caliberation for Linux.
 */
char **XJCGetDeviceNamesLinux(char *path, int *total)
{
	FILE *fp;
	struct stat stat_buf;
	char *strptr, *strptr2;
        char parm[CFG_PARAMETER_MAX];
        char val[CFG_VALUE_MAX];
	int n, lines_read = 0;

	int strc = 0;
	char **strv = NULL;


	if(path == NULL)
	{
	    if(total != NULL)
		*total = 0;
	    return(NULL);
	}
	if(stat(path, &stat_buf))
	{
            if(total != NULL)
                *total = 0; 
            return(NULL);
        }

	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    if(total != NULL)
                *total = 0;
            return(NULL);
        }


	/* Begin reading. */
        strptr = NULL;

        while(1)
        {   
            /* Free previous line and allocate/read next line. */
            free(strptr); strptr = NULL;
            strptr = FReadNextLineAllocCount(
		fp, UNIXCFG_COMMENT_CHAR, &lines_read
	    );
            if(strptr == NULL) break;

            /* Fetch parameter. */
            strptr2 = StringCfgParseParm(strptr);
            if(strptr2 == NULL) continue;
            strncpy(parm, strptr2, CFG_PARAMETER_MAX);
            parm[CFG_PARAMETER_MAX - 1] = '\0';

            /* Fetch value. */
            strptr2 = StringCfgParseValue(strptr);
            if(strptr2 == NULL) strptr2 = "0";  /* Set it to "0" if NULL. */
            strncpy(val, strptr2, CFG_VALUE_MAX);
            val[CFG_VALUE_MAX - 1] = '\0';

            /* BeginJoystick */
            if(!strcasecmp(parm, "BeginJoystick"))
            {
		n = strc;
		strc++;
		strv = (char **)realloc(
		    strv,
		    strc * sizeof(char *)
		);
		if(strv == NULL)
		{
		    strc = 0;
		    continue;
		}

		strv[n] = StringCopyAlloc(val);
	    }
	}

	/* Close file. */
	fclose(fp);


	if(total != NULL)
	    *total = strc;

	return(strv);
}
#endif	/* __linux__ */


/*
 *	Saves caliberation values in specified jc window to the
 *	file path.
 */
int XJCDoSaveCaliberation(jcw_struct *j, char *path)
{
	char wrote_this_dev = 0;
	int i, n;
	char *strptr;
	char lcalib[PATH_MAX + NAME_MAX];
	char ldevice[PATH_MAX + NAME_MAX];
	char text[PATH_MAX + NAME_MAX + 80];

	int strc = 0;
	char **strv = NULL;
	FILE *fp;

	js_data_struct **jsd;
	int total_jsds;


	if(j == NULL)
	    return(-1);

	if(path == NULL)
	{
	    strptr = getenv("HOME");
	    strptr = PrefixPaths(strptr, DEF_CALIB_FILE);
	    strncpy(
		lcalib,
		((strptr == NULL) ? DEF_CALIB_FILE : strptr),
		PATH_MAX + NAME_MAX
	    );
	    lcalib[PATH_MAX + NAME_MAX - 1] = '\0';
	}
	else
	{
            strncpy(lcalib, path, PATH_MAX + NAME_MAX);
            lcalib[PATH_MAX + NAME_MAX - 1] = '\0';
	}

	if(j->jsd.device_name == NULL)
	    strncpy(ldevice, DEF_DEVICE_NAME, PATH_MAX + NAME_MAX);
	else
	    strncpy(ldevice, j->jsd.device_name, PATH_MAX + NAME_MAX);
	ldevice[PATH_MAX + NAME_MAX - 1] = '\0';

	/* Apply values from axis readout bars to axis on jsd. */
	for(i = 0; i < j->total_axis_bars; i++)
	{
	    if(j->axis_bar[i] == NULL)
		continue;

	    n = PromptGetI(&(j->axis_bar[i]->axis_num_prompt));
	    if(JSIsAxisAllocated(&j->jsd, n))
	    {

		j->jsd.axis[n]->nz = PromptGetI(
		    &(j->axis_bar[i]->null_zone_prompt)
		);
                j->jsd.axis[n]->flip = ((j->axis_bar[i]->flip_tb.state) ?
		    1 : 0
		);
	    }
	}



#ifdef __linux__
	/* Get listing of devices in caliberation file. */
	strv = XJCGetDeviceNamesLinux(lcalib, &strc);

	/* Fetch values for each device. */
	jsd = NULL;
	total_jsds = 0;
	for(i = 0; i < strc; i++)
	{
	    if(strv[i] == NULL)
		continue;

	    n = total_jsds;
	    total_jsds++;
	    jsd = (js_data_struct **)realloc(
		jsd,
		total_jsds * sizeof(js_data_struct *)
	    );
	    if(jsd == NULL)
	    {
		total_jsds = 0;
		continue;
	    }

	    jsd[n] = (js_data_struct *)calloc(
		1,
		sizeof(js_data_struct)
	    );
	    if(jsd[n] == NULL)
		continue;

	    /* Set up jsd file values (this is needed). */
	    jsd[n]->fd = -1;
	    jsd[n]->device_name = StringCopyAlloc(strv[i]);
	    jsd[n]->caliberation_file = StringCopyAlloc(lcalib);

	    JSLoadCaliberationLinux(jsd[n]);
	}

	/* Free device names. */
	StringFreeArray(strv, strc);



	/* Begin writing caliberation. */
	fp = fopen(lcalib, "w");
	if(fp != NULL)
	{
	    for(i = 0; i < total_jsds; i++)
	    {
		if(jsd[i] == NULL)
		    continue;
		if(jsd[i]->device_name == NULL)
		    continue;

		if(!strcmp(jsd[i]->device_name, ldevice))
		{
		    /* Our joystick caliberation block. */
                    fprintf(
                        fp,
                        "BeginJoystick = %s\n",
                        ldevice
                    );

                    /* Write all axises. */
                    for(n = 0; n < j->jsd.total_axises; n++)
                    {
                        if(j->jsd.axis[n] == NULL)
                            continue;

                        fprintf(
                            fp,
                            "    BeginAxis = %i\n",
                            n
                        );
                        fprintf(
                            fp,
                            "        Minimum = %i\n",
                            j->jsd.axis[n]->min
                        );
                        fprintf(
                            fp,
                            "        Center = %i\n",
                            j->jsd.axis[n]->cen
                        );
                        fprintf(
                            fp,
                            "        Maximum = %i\n",
                            j->jsd.axis[n]->max
                        );
                        fprintf(
                            fp,
                            "        NullZone = %i\n",
                            j->jsd.axis[n]->nz
                        );
                        if(j->jsd.axis[n]->flip)
                            fprintf(
                                fp,
                                "        Flip\n"
                            );
			if(j->jsd.axis[n]->is_hat)
                            fprintf(
                                fp,
                                "        IsHat\n"
                            );
                        fprintf(fp, "    EndAxis\n");
                    }
                    fprintf(fp, "EndJoystick\n");

		    wrote_this_dev = 1;
		}
		else
		{
		    /* Some other joystick caliberation block. */
		    fprintf(
		        fp,
		        "BeginJoystick = %s\n",
		        jsd[i]->device_name
		    );

		    /* Write all axises. */
		    for(n = 0; n < jsd[i]->total_axises; n++)
		    {
			if(jsd[i]->axis[n] == NULL)
			    continue;

                        fprintf(
                            fp,
                            "    BeginAxis = %i\n",
                            n
                        );
                        fprintf(
                            fp,
                            "        Minimum = %i\n",
                            jsd[i]->axis[n]->min
                        );
                        fprintf(
                            fp,
                            "        Center = %i\n",
                            jsd[i]->axis[n]->cen
                        );
                        fprintf(
                            fp,
                            "        Maximum = %i\n",
                            jsd[i]->axis[n]->max
                        );
                        fprintf(
                            fp,
                            "        NullZone = %i\n",
                            jsd[i]->axis[n]->nz
                        );
                        if(jsd[i]->axis[n]->flip)
                            fprintf(
                                fp,
                                "        Flip\n"
                            );
                        if(jsd[i]->axis[n]->is_hat)
                            fprintf(
                                fp,
                                "        IsHat\n"
                            );
                        fprintf(fp, "    EndAxis\n");
		    }

                    fprintf(fp, "EndJoystick\n");
		}
	    }	/* for(i = 0; i < total_jsds; i++) */


	    /* If we did not write our device, then append it now. */
	    if(!wrote_this_dev)
	    {
                /* Our device was not prevously written, so append it. */
		fprintf(
                    fp,
                    "BeginJoystick = %s\n",
                    ldevice
                );

                /* Write all axises. */
                for(n = 0; n < j->jsd.total_axises; n++)
                {
                    if(j->jsd.axis[n] == NULL)
                        continue;

                    fprintf(
                        fp,
                        "    BeginAxis = %i\n",
                        n  
                    );
                    fprintf(
                        fp,
                        "        Minimum = %i\n",
                        j->jsd.axis[n]->min
                    );
                    fprintf(
                        fp,
                        "        Center = %i\n",
                        j->jsd.axis[n]->cen
                    ); 
                    fprintf(
                        fp,
                        "        Maximum = %i\n",
                        j->jsd.axis[n]->max
                    ); 
                    fprintf(
                        fp,
                        "        NullZone = %i\n",
                        j->jsd.axis[n]->nz
                    );
		    if(j->jsd.axis[n]->flip)
                        fprintf(
                            fp, 
                            "        Flip\n"
                        );
                    if(j->jsd.axis[n]->is_hat)
                        fprintf(
                            fp,
                            "        IsHat\n"
                        );
                    fprintf(fp, "    EndAxis\n");
                }

                fprintf(fp, "EndJoystick\n");

		wrote_this_dev = 1;
	    }

	    /* Close file. */
	    fclose(fp);
	}


	/* Free all jds's that we allocated. */
	for(i = 0; i < total_jsds; i++)
	{
	    /* Free allocated substructures. */
	    JSClose(jsd[i]);

	    free(jsd[i]);
	}
	free(jsd);
	jsd = NULL;
	total_jsds = 0;



	if(wrote_this_dev)
	{
	    /* Notify of successful save. */
	    dialog.icon_img = xjc_image.info;
	    sprintf(
		text,
		"Successfully saved caliberation to:\n\n    %s",
		lcalib
	    );
	}
	else
	{
            /* Notify of successful save. */
            dialog.icon_img = xjc_image.warning;
            sprintf(
                text,
"Unable to save caliberation for device:\n\n\
    %s\n\n\
into caliberation file:\n\n\
    %s",
		ldevice,
                lcalib
            );
	}
	printdw(&dialog, text);


#endif	/* __linux__ */


	return(0);
}
