/*
	Linux Joystick Driver Wrapper Test and Example Program

	To compile this program, type:

	cc -DJS_SUPPORT jswtest.c -o jswtest -ljsw


	This program is originally distributed in the same
	package as libjsw.

 */


#include <stdio.h>
#include <unistd.h>
#include <signal.h>

#ifdef JS_SUPPORT
/* Include joystick wrapper header file only if JS_SUPPORT is defined.
 * This is so that this program can be compiled without joystick
 * support. Though it would make it useless without joystick support,
 * we show this so that the programs you write can be compiled with
 * or without joystick support and still atleast `run' (and help
 * contrast the portions of joystick code in this source).
 */
# include <jsw.h>
#endif	/* JS_SUPPORT */


/*
 *	Global runlevel code for this program, used for easy start and
 *	exiting of the main while() loop.
 */
int runlevel;



/*
 *	Signal handler, to catch ctrl+c signal.
 */
void SignalHandler(int s)
{
	switch(s)
	{
	  case SIGINT:
	    runlevel = 1;	/* This will cause main while() loop to break. */
	    break;

	  default:
	    runlevel = 1;	/* Interprite all else as interupt. */
	    break;
	}

	return;
}



int main(int argc, char *argv[])
{
	int i, status;
	char *device = "/dev/js0";	/* Device file name for the joystick. */
	char *calib = ".joystick";	/* Caliberation file. */
	char *strv[10];
	int strc;
#ifdef JS_SUPPORT
	js_data_struct jsd;	/* Structure to hold joystick caliberation
				 * and current values.
				 */
#endif	/* JS_SUPPORT */


	/* Catch interupt signal. */
	signal(SIGINT, SignalHandler);


	/* Parse arguments, first argument is the joystick device name
	 * and the second is the caliberation file name.
	 */
	if(argc < 3)
	{
	    /* Print usage and exit. */
	    printf(
		"Usage: jswtest <joystick_device> <calibation_file>\n"
	    );
	    return(0);
	}

	/* Fetch device and caliberation file names from input. */
	device = argv[1];
	calib = argv[2];


	/* Set up four arguments for JSInit() to take. */
	strc = 4;
	strv[0] = "-d";
	strv[1] = device;
	strv[2] = "-f";
	strv[3] = calib;

#ifdef JS_SUPPORT
	/* Initialize the joystick, passing it the pointer to
	 * the jsd (joystick data) structure.
	 */
	status = JSInit(
	    &jsd,
	    strc,
	    strv
	);
	/* Check for error. */
	if(status != JSSuccess)
	{
	    switch(status)
	    {
	      case JSBadValue:
		fprintf(stderr,
"%s: Invalid value encountered while initializing joystick.\n",
		    device
		);
		break;

	      case JSNoAccess:
		fprintf(stderr,
"%s: Cannot access resources, check permissions and file locations.\n",
                    device
                );
		break;

	      case JSNoBuffers:
                fprintf(stderr,
"%s: Insufficient memory to initialize.\n",
                    device
                );
		break;

	      default:	/* JSError */
                fprintf(stderr,
"%s: Error occured while initializing.\n",
                    device
                );
		break;
	    }

	    fprintf(stderr,
"Make sure that:\n\
    1. Your joystick is connected (and connected to the right port).\n\
    2. Your joystick modules are loaded (type lsmod to check).\n\
    3. Your joystick has been caliberated (run xjscal if needed).\n\
\n"
            );

	    /* Close joystick, just in case. */
	    JSClose(&jsd);

	    return(1);
	}
	else
	{
	    printf(
"Reading values from `%s' (press ctrl+c to interrupt).\n",
		device
	    );
	}
#endif	/* JS_SUPPORT */


	/* Set runlevel to 2 and begin main while loop. */
	runlevel = 2;
	while(runlevel >= 2)
	{
#ifdef JS_SUPPORT
	    /* Fetch for event. */
	    if(JSUpdate(&jsd) == JSGotEvent)
	    {
		printf("\r");

		/* Print value of each axis. */
		for(i = 0; i < jsd.total_axises; i++)
		{
		    if(jsd.axis[i] == NULL)
			continue;

		    /* Get axis coefficent and take null zone into consideration. */
		    printf(
			"A%i:%.3lf ",
			i,
			JSGetAxisCoeffNZ(&jsd, i)
		    );
		}

		printf(" ");

		/* Print state of each button. */
		for(i = 0; i < jsd.total_buttons; i++)
		{
		    if(jsd.button[i] == NULL)
			continue;

                    printf(
                        "B%i:%i ",
			i,
                        jsd.button[i]->state
                    );
		}

		fflush(stdout);
	    }
#endif  /* JS_SUPPORT */

	    usleep(16000);	/* Don't hog the cpu. */
	}


#ifdef JS_SUPPORT
	/* Close joystick, jsd may no longer be used in referance
	 * to any other JS*() calls (except JSInit() which would
	 * initialize it again).
	 */
	JSClose(&jsd);
#endif  /* JS_SUPPORT */


	return(0);
}
