/***************************************************************************
                          qbrewcalc.h  -  handles calculations for QBrew
                             -------------------                                         
    begin                : Sun Sep 26 1999                                           
    copyright            : (C) 1999 by David Johnson                         
    email                : arandir@meer.net                                     

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#ifndef QBREWCALC_H
#define QBREWCALC_H

#include <qstringlist.h>

#include "calcresource.h"
#include "grain.h"
#include "hops.h"
#include "miscingredient.h"
#include "qbrewdoc.h"
#include "settings.h"
#include "style.h"

/** a utilization table entry structure */
struct uentry {
	unsigned time;
	unsigned utilization;
};

/**
  *@author David Johnson
  */
class QBrewCalc : public QObject {
	Q_OBJECT
public:
	/** constructor */
	QBrewCalc(Settings *sttngs=0, QBrewDoc *doc=0);
	/** destructor */
	~QBrewCalc();
	
	/** return a list of styles */
	QStringList getStylesList();
	/** return given style from name */
	Style* getStyle(const QString &sname);
	/** return a list of grains */
	QStringList getGrainsList();
	/** return given grain from name */
	Grain* getGrain(const QString &gname);
	/** return a list of hops */
	QStringList getHopsList();
	/** return a list of hop styles (this should be in Hops class) */
	QStringList getFormsList();
	/** return given hop from name */
	Hops* getHop(const QString &hname);
	/** return a list of misc ingredients */
	QStringList getMiscList();
	/** return given misc ingredient from name */
	MiscIngredient* getMisc(const QString &mname);
	
	/** return original gravity */
	double getOG();
	/** return estimated final gravity */
	double getFGEstimate();
	/** return estimated final gravity */
	double getABV();
	/** return alcohol %v */
	double getABW();
	/** return bitterness */
	double getIBU();
	/** return alcohol %w */
	double getSRM();

public slots:
	/** received if stuff needs recalculating */
	void slotRecalc();
	
signals:
	void calcDone();
	
private:	
	/** load calc data */
	bool loadData();
	/** load default data into database*/
	void loadDefaults();
	/** process styles */
	void processStyle(const QString &name, const QString &value);
	/** process grains */
	void processGrain(const QString &name, const QString &value);
	/** process hops */
	void processHops(const QString &name, const QString &value);
	/** process misc ingredient */
	void processMisc(const QString &name, const QString &value);
	/** process utilization table entry */
	void processUtilEntry(const QString &name, const QString &value);
	
	/** calculate original gravity */
	double calcOG();
	/** calculate color */
	double calcSRM();
	/** calculate bitterness */
	double calcIBU();
	/** look up hop utilization for time */
	double utilization(const unsigned &time);
	
private:
	Settings *settings;
	QBrewDoc *document;
	
	double og;
	double ibu;
	double srm;
	
	// the serialization group map
	QMap<QString, int> gmap;
	
	// utilization table
	QValueList<uentry> utable;
	
	// databases
	QMap<QString, Style> styles;
	GrainList grains;
	HopsList hops;
	MiscIngredientList misc;
};

#endif // QBREWCALC_H






























