/***************************************************************************
                          hops.cpp  -  hops entry
                             -------------------                                         
    begin                : Wed Oct 20 1999                                           
    copyright            : (C) 1999 by David Johnson                         
    email                : arandir@meer.net                                     

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#include <math.h>

#include <qstring.h>
#include <qstringlist.h>

#include "hops.h"

Hops::Hops() { name = quantity = alpha = time = 0; form = "pellet"; hbu = 0.0; }

Hops::Hops(const QString &hname, const unsigned &hquant, const QString &hform,
			const unsigned &halpha, const unsigned &htime)
{
	name = hname;
	quantity = hquant;
	form = hform;
	alpha = halpha;
	time = htime;
	hbu = 0.75 * quantity * alpha / 1000.0;
}

Hops::Hops(const Hops &h)
{
	name = h.name;
	quantity = h.quantity;
	form = h.form;
	alpha = h.alpha;
	time = h.time;
	hbu = h.hbu;
}

Hops Hops::operator=(const Hops &h)
{
	name = h.name;
	quantity = h.quantity;
	form = h.form;
	alpha = h.alpha;
	time = h.time;
	hbu = h.hbu;
	return *this;
}

bool Hops::operator==(const Hops &h)
{
	return (name == h.name)
		&& (quantity == h.quantity)
		&& (form == h.form)
		&& (alpha == h.alpha)
		&& (time == h.time);
}

Hops::~Hops() {}

///////////////////////////////////////////////////////////////////////////////
// member access

QString Hops::getName() { return name; }

unsigned Hops::getQuantity() { return quantity; }

QString Hops::getForm() { return form; }

unsigned Hops::getAlpha() { return alpha; }

unsigned Hops::getTime() { return time; }
	
QString Hops::getQuantityString() { return formatValue(quantity, 2); }

QString Hops::getAlphaString() { return formatValue(alpha, 1); }

QString Hops::getTimeString() {return QString::number(time); }

double Hops::getHBU() { return hbu; }

///////////////////////////////////////////////////////////////////////////////
// utility functions

QString Hops::formatValue(const unsigned &value, const unsigned &fix)
{
	unsigned divisor = unsigned(pow(10.0, double(fix)));
	unsigned leftuint = value / divisor;
	QString leftstring = QString::number(leftuint) + ".";
	QString rightstring = QString::number(value - (leftuint * divisor));
	leftstring += rightstring.rightJustify(fix, '0', true);
	return leftstring;
}

///////////////////////////////////////////////////////////////////////////////
// serialization

QString Hops::serializeOut()
{
	QString text;
	text = QString::number(quantity) + ", " + form + ", ";
	text += QString::number(alpha) + ", " + QString::number(time);
	return text;
}

void Hops::serializeIn(const QString &hname, const QString &hvalue)
{
	QString value = hvalue;
	QStringList tokens;
	int pos;
	
	// get tokens
	while (value.length() > 0) {
		pos = value.find(',');
		if (pos > 0) {
			tokens += value.left(pos);
			value.remove(0, pos+1);
			value = value.stripWhiteSpace();
		} else {
			// get last token
			tokens += value;
			value = "";
		}
	}
	// create new style
	name = hname;
	quantity	= tokens[0].toUInt();
	form	= tokens[1];
	alpha	= tokens[2].toUInt();
	time	= tokens[3].toUInt();
	hbu = 0.75 * quantity * alpha / 1000.0;
}

///////////////////////////////////////////////////////////////////////////////
// HopsListItem class

HopsListItem::HopsListItem(QListView * parent, QString a, QString b = QString::null,
			QString c = QString::null, QString d = QString::null, QString e = QString::null,
			QString f = QString::null, QString g = QString::null, QString h = QString::null )
			: QListViewItem(parent, a, b, c, d, e, f, g, h) { hit = NULL; };

HopsListItem::~HopsListItem() {}

HopIterator HopsListItem::getHopIterator() { return hit; }

void HopsListItem::setHopIterator(HopIterator &it) { hit = it; }

void HopsListItem::setToIterator()
{
	setText(0, (*hit).getName());
	setText(1, (*hit).getQuantityString());
	setText(2, (*hit).getForm());
	setText(3, (*hit).getAlphaString());
	setText(4, (*hit).getTimeString());
}



