/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"

#define	APM_FILE	"/proc/apm"

/* ----- see /usr/src/linux/arch/i386/kernel/apm.c ----- */

  /* Defines for ac_line_status
  */
#define	OFF_LINE	0
#define ON_LINE		1
#define	ON_BACKUP	2

  /* Defines for battery_status
  */
#define	HIGH		0
#define	LOW			1
#define CRITICAL	2
#define	CHARGING	3

  /* Defines for battery_flag (only values not covered under battery_status)
  */
#define	NO_BATTERY	0x80


#define	UNKNOWN			0xff
#define	UNKNOWN_LIFE	-1

typedef struct
	{
	gchar		driver_version[16];
	gint		bios_version_hi;
	gint		bios_version_low;
	gint		bios_flags;
	gint		ac_line_status;
	gint		battery_status;
	gint		battery_flag;
	gint		percentage;		/* 0-100 valid, -1 Unknown			*/
	gint		time_units;		/* Remaining battery life in units	*/
	gchar		units[32];		/* min or sec						*/
	gint		minutes;
	}
	ProcApm;

static ProcApm	proc_apm;
static Panel	apm;

static Decal	power_source_decal;
static Decal	minutes_left_decal;

static gint		apm_available,
				apm_warning;


static void
draw_time_left_decal(gint t)
	{
	Decal	*d;
	GdkFont	*font;
	gchar	buf[16];
	gint	x, w;

	d = &minutes_left_decal;
	if (t == d->value)
		return;

	/* Start showing minutes left for last 500 minutes
	*/
	if (t < 0)
		buf[0] = '\0';
	else
		sprintf(buf, "%d %s", t, proc_apm.units);

	font = d->text_style.font;
	w = gdk_string_measure(font, buf);
	x = (d->w - w) / 2;
	if (x < 0)
		x = 0;
	gdk_draw_pixmap(d->pixmap, GK.draw1_GC, apm.background,
			d->x, d->y,  0, 0,  d->w, d->h);
	draw_string(d->pixmap, font, &d->text_style.color,
			d->text_style.effect, x, d->y_baseline, buf);
	d->value = t;
	d->modified = TRUE;
	}

static void
scan_apm_info(gchar *line)
	{
	sscanf(line, "%s %d.%d %x %x %x %x %d%% %d %s\n",
				proc_apm.driver_version,
				&proc_apm.bios_version_hi, &proc_apm.bios_version_low,
				&proc_apm.bios_flags, &proc_apm.ac_line_status,
				&proc_apm.battery_status, &proc_apm.battery_flag,
				&proc_apm.percentage, &proc_apm.time_units,
				proc_apm.units);
	proc_apm.minutes = (strcmp(proc_apm.units, "min") == 0) ? TRUE : FALSE;
	}

static gint
apm_is_available()
	{
	FILE	*f;
	gchar	buf[160];

	apm_available = FALSE;
	if ((f = fopen(APM_FILE, "r")) != NULL)
		{
		fgets(buf, sizeof(buf), f);
		fclose(f);
		scan_apm_info(buf);
		if (   (proc_apm.battery_flag & NO_BATTERY) == 0
			&& proc_apm.battery_status != UNKNOWN
		   )
			apm_available = TRUE;
		}
	return apm_available;
	}

void
update_apm()
	{
	FILE	*f;
	gchar	buf[160];
	gint	t, decal;

	if (apm_available == FALSE)
		return;

	if (GK.two_second_tick || (apm_warning && (GK.timer_ticks % 2) == 0))
		{
		if ((f = fopen(APM_FILE, "r")) != NULL)
			{
			fgets(buf, sizeof(buf), f);
			fclose(f);
			scan_apm_info(buf);
			t = proc_apm.time_units;
			if (t >= 0 && strcmp("sec", proc_apm.units) == 0)
				t /= 60;
			if (   proc_apm.ac_line_status != ON_LINE
				&& (proc_apm.percentage < 25 || (t >= 0 && t < 15))
			   )
				{
				apm_warning = (t >= 0) ? (2 * (t + 1)) : 8;		/* even */
				decal = (GK.timer_ticks % apm_warning)
						? DECAL_BATTERY : DECAL_BATTERY_WARN;
				}
			else
				{
				apm_warning = 0;
				decal = DECAL_BATTERY;
				}

			if (proc_apm.ac_line_status == ON_LINE)
				draw_decal(&apm, &power_source_decal, DECAL_AC);
			else
				draw_decal(&apm, &power_source_decal, decal);

			if (   proc_apm.battery_status == CHARGING
				|| proc_apm.time_units > 999
			   )
				draw_time_left_decal(-1);
			else
				draw_time_left_decal(proc_apm.time_units);

			apm.krell->full_scale = 100;
			apm.krell->previous = 0;
			update_krell(&apm, apm.krell, proc_apm.percentage);

			draw_layers(&apm);
			}
		}

	/* Update LED state, toggle critical at variable rate	*/
	}


static gint
apm_expose_event (GtkWidget *widget, GdkEventExpose *event)
	{
	if (widget == apm.drawing_area)
		{
		gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
				apm.pixmap,
				event->area.x, event->area.y,
				event->area.x, event->area.y,
				event->area.width, event->area.height);
		}
	return FALSE;
	}


void
create_apm(GtkWidget *vbox)
	{
	Style	*style;
	Decal	*d;
	gint	l, r, w, a, dd;

	if (! apm_is_available())
		return;
	if (GK.trace)
		printf("create_apm()\n");

	insert_spacer(vbox);

	style = GK.meter_style[APM_IMAGE];

	d = &power_source_decal;
	apm.decal = d;
	d->x = style->border_panel.left + 2;
	d->y = style->border_panel.top;
	d->value = -1;		/* Force initial draw	*/
	d->w = GK.decal_width;
	d->h = GK.decal_height;
	d->pixmap = GK.decal_pixmap;
	d->mask = GK.decal_mask;
	d->type = DECAL_TRANSPARENCY;

	d->next = &minutes_left_decal;
	d = &minutes_left_decal;
	d->next = NULL;
	d->y = style->border_panel.top;
	default_textstyle(&d->text_style, TEXTSTYLE_TIME);
	d->text_style.font = GK.alt_font;
	gdk_string_extents(GK.label_font, "100", &l, &r, &w, &a, &dd);
	d->y_baseline = a;
	d->h = a + dd + d->text_style.effect;
	d->x = style->border_panel.left + power_source_decal.w + 6;
	d->w = UC.chart_width - d->x - style->border_panel.right;
	d->pixmap = gdk_pixmap_new(top_window->window, d->w, d->h, -1);
	d->value = -2;		/* Force initial draw */
	d->type = DECAL_OPAQUE;

	create_krell("Apm", GK.krell_meter_image[APM_IMAGE], &apm.krell, style);

	configure_panel(&apm, NULL, style);
	create_panel_area(vbox, &apm, GK.bg_meter_image[0]);
	GK.monitor_height += apm.h;

	/* Signals used to handle backing pixmap
	*/
	gtk_signal_connect(GTK_OBJECT (apm.drawing_area), "expose_event",
			(GtkSignalFunc) apm_expose_event, NULL);

	if (GK.trace)
		printf("  <-\n");

	}

