/*
 *
 * (c) Vladi Belperchinov-Shabanski "Cade" <cade@biscom.net> 1998-1999
 *
 * SEE `README',`LICENSE' OR `COPYING' FILE FOR LICENSE AND OTHER DETAILS!
 *
 */
#include <stdlib.h>
#include <string.h>
#include "clusters.h"

#define RETURN(n) { return status = n; }

////////////////////////////////////////////////////////////////////////////
//
//
//

#define flEF(n) (((int32*)(data+(n)*es))[0]) // element `free' field
#define flED(n) ((char*)(data+(n)*es+sizeof(int32))) // element `data' field

#define E_BUSY (-2)
#define E_NULL (-1)
#define E_FREE ( 0)


FLCluster::FLCluster()
{
  base = 0;
  null = -1;

  data = NULL;
  size = 0;
  used = 0;
  ff   = E_NULL;
}

void FLCluster::done()
{
  ASSERT( data );
  if (data) ::free( data );
}

FLCluster::~FLCluster()
{
  done();
}

int FLCluster::create( int32 pcnt, int32 pgrow, int32 pes )
{
  ASSERT( pes > 0 );
  ASSERT( pcnt > -1 );
  ASSERT( pgrow > -1 );
  ASSERT( !(pcnt == 0 && pgrow == 0 ) );

  es = pes + sizeof(int32);
  ds = pes;
  size = 0;
  used = 0;
  growby = pgrow;
  ff   = E_NULL;
  data = NULL;
  return Realloc( pcnt );
}

int32 FLCluster::add( void* pe ) // insert element
{
  ASSERT( pe );
//  if (ff == E_NULL) return -1;
  if (ff == E_NULL)
    {
    int32 res = Realloc( size + growby );
    if (res != CE_OK) return null;
    }

  int32 ret = ff;
  ff = flEF(ret);

  ASSERT( flEF(ret) != E_BUSY );
  flEF(ret) = E_BUSY;

  memcpy( flED(ret), pe, ds );

  return ret + base;
}

int FLCluster::get( int32 pn, void* pe ) // get element
{
  pn -= base;

  ASSERT( pn >= 0 && pn < size );
  ASSERT( pe );
  ASSERT( flEF(pn) == E_BUSY );

  memcpy( pe, flED(pn), ds );

  return CE_OK;
}

char* FLCluster::get( int32 pn ) // get element pointer or NULL if error
{
  pn -= base;

  ASSERT( pn >= 0 && pn < size );
  ASSERT( flEF(pn) == E_BUSY );

  return (char*)(flED(pn));
}

int FLCluster::del( int32 pn ) // get element
{
  pn -= base;

  ASSERT( pn >= 0 && pn < size );
  ASSERT( flEF(pn) == E_BUSY );

  flEF(pn) = ff;
  ff = pn;

  return CE_OK;
}

int FLCluster::is_used( int32 pn ) // 1 if pn element is used
{
  return ( flEF(pn - base) == E_BUSY );
}

int FLCluster::Realloc( int32 pn ) // expand/shrink data list
{
  pn -= base;

  ASSERT( pn >= size );
  if ( pn == size ) return CE_OK;

  char *newdata = (char*)realloc( data, pn*es );
  if (!newdata) return CE_MEM;
  data = newdata;
  memset( data + size*es, 0, (pn - size)*es );

  int32 z;
  int32 base = size > 0;
  for(z = size+base; z < pn; z++) flEF(z) = z - 1;
  flEF(size) = ff;
  ff = pn - 1;
  size = pn;

  return CE_OK;
}

void FLCluster::dump()
{
  int32 z;
  printf("size: %d, ff: %d, used: %d\n", size, ff, used);
  for(z = 0; z < size; z++)
    {
    printf("%2d: nextfree: %2d\n", z, flEF(z));
    }
}

////////////////////////////////////////////////////////////////////////////
//
//
//

  int BaseCluster::create( int p_size, int p_bsize, int p_es )
  {
    cnt = 0;
    es = p_es;
    // size = p_size; this should be set by Realloc
    size = 0;
    bsize = p_bsize;
    RETURN(Realloc( p_size ));
  };

  void BaseCluster::done()
  {
    freeall();
    cnt = 0;
    es = 0;
    bsize = 0;
    if (data) ::free(data);
    data = NULL;
  };

  void BaseCluster::del( int pn )
  {
    ASSERT( pn >= 0 && pn < cnt );
    if (pn < cnt - 1)
      memmove( data + ( pn )*es, data + ( pn + 1 )*es, ( cnt - pn )*es );
    cnt--;
  };

  void BaseCluster::delall()
  {
    int z = cnt;
    while(z)
      del((z--) - 1);
  };

  void BaseCluster::free( int pn )
  {
    ASSERT( pn >= 0 && pn < cnt );
    destroy_element( (void*)(data + (pn)*es) );
    del( pn );
  };

  void BaseCluster::freeall()
  {
    int z = cnt;
    while(z)
      free((z--) - 1);
  }

  int BaseCluster::Realloc( int pn )
  {
    if ( pn == size ) RETURN(CE_OK);
    char* newdata = (char*)malloc( pn*es );
    if (newdata == NULL) RETURN(CE_MEM);
    #ifdef DEBUG
    memset( newdata, '+', pn*es );
    #endif
    if (newdata && data) memcpy( newdata, data, (( pn < size ) ? pn : size)*es );
    if (data) ::free(data);
    data = newdata;
    size = pn;
    RETURN(CE_OK);
  }

////////////////////////////////////////////////////////////////////////////
//
//
//
    int DCluster::add( void* pe )
    {
      RETURN(ins( cnt, pe ));
    };

    int DCluster::ins( int pn, void* pe )
    {
      int res = 0;
      if ( cnt == size )
        if ( (res = Realloc( size + bsize )) != 0 ) RETURN(res);
      if (pn < cnt)
        memmove( data + (pn+1)*es, data + (pn)*es, (cnt - pn)*es );
      memcpy( data + (pn*es), pe, es );
      cnt++;
      RETURN(CE_OK);
    };

    void DCluster::put( int pn, void* pe )
    {
      ASSERT( pn >= 0 && pn < cnt );
      memcpy( data + pn*es, pe, es );
    };

    void DCluster::get( int pn, void* pe )
    {
      ASSERT( pn >= 0 && pn < cnt );
      memcpy( pe, data + pn*es, es );
    };

////////////////////////////////////////////////////////////////////////////
//
//
//

    int PCluster::add( void* pe )
    {
      RETURN(ins( cnt, pe ));
    };

    int PCluster::ins( int pn, void* pe )
    {
      int res = 0;
      if ( cnt == size )
        if ( (res = Realloc( size + bsize )) != 0 ) RETURN(res);
      if (pn < cnt)
        memmove( data + (pn+1)*es, data + (pn)*es, (cnt - pn)*es );
     ((void**)data)[pn] = pe;
      cnt++;
      RETURN(CE_OK);
    };

    void PCluster::put( int pn, void* pe )
    {
      ASSERT( pn >= 0 && pn < cnt );
     ((void**)data)[pn] = pe;
    };

    void* PCluster::get( int pn )
    {
      ASSERT( pn >= 0 && pn < cnt );
      return ((void**)data)[pn];
    };

////////////////////////////////////////////////////////////////////////////
//
// BCluster -- bit cluster
//

    void BSCluster::set1( int pn )
      {
        ASSERT( pn >= 0 && pn < size  );
        data[pn / 8] |= 1 << (pn % 8);
      };

    void BSCluster::set0( int pn )
      {
        ASSERT( pn >= 0 && pn < size  );
        data[pn / 8] &= ~(1 << (pn % 8));
      };

    void BSCluster::set( int pn, int val )
      {
        ASSERT( pn >= 0 && pn < size  );
        if (val)
          data[pn / 8] |= 1 << (pn % 8);
        else
          data[pn / 8] &= ~(1 << (pn % 8));
      };

    int BSCluster::get( int pn )
      {
        ASSERT( pn >= 0 && pn < size  );
        return (data[pn / 8] & (1 << (pn % 8))) != 0;
      };

    int BSCluster::Resize( int p_size )
      {
        ASSERT( p_size > 0 );
        int new_size = p_size;
        int new_datasize = p_size / 8 + (p_size % 8 != 0);
        char *new_data = (char*)malloc( new_datasize );
        if (new_data == NULL) return CE_MEM;
        memset( new_data, 0, new_datasize );
        if (data)
          {
          memcpy( new_data, data, datasize < new_datasize ? datasize : new_datasize );
          free( data );
          data = NULL;
          }
        data = new_data;
        size = new_size;
        datasize = new_datasize;
        return CE_OK;
      }

    void BSCluster::reverse()
     {
       int z;
       for( z = 0; z < datasize; z++ )
         data[z] = ~data[z];
     }

    BSCluster& BSCluster::operator  = ( const BSCluster &b1 )
    {
      Resize( b1.size );
      memcpy( data, b1.data, datasize );
      return *this;
    };

    BSCluster& BSCluster::operator &= ( const BSCluster &b1 )
    {
      ASSERT( datasize == b1.datasize );
      int z;
      for(z = 0; z < (datasize < b1.datasize ? datasize : b1.datasize ); z++)
        data[z] &= b1.data[z];
      return *this;
    };

    BSCluster& BSCluster::operator |= ( const BSCluster &b1 )
    {
      ASSERT( datasize == b1.datasize );
      int z;
      for(z = 0; z < (datasize < b1.datasize ? datasize : b1.datasize ); z++)
        data[z] |= b1.data[z];
      return *this;
    };

    BSCluster BSCluster::operator ~ ()
    {
      BSCluster b1;
      b1 = *this;
      int z;
      for(z = 0; z < b1.datasize; z++)
        b1.data[z] = ~b1.data[z];
      return b1;
    };

    BSCluster operator & ( const BSCluster &b1, const BSCluster &b2 )
    {
      ASSERT( b1.datasize == b2.datasize );
      BSCluster b = b1;
      b &= b2;
      return b;
    };

    BSCluster operator | ( const BSCluster &b1, const BSCluster &b2 )
    {
      ASSERT( b1.datasize == b2.datasize );
      BSCluster b = b1;
      b |= b2;
      return b;
    };

////////////////////////////////////////////////////////////////////////////
//
//
//

    int PSZCluster::add( const char* pe )
    {
      RETURN(ins( cnt, pe ));
    };

    int PSZCluster::ins( int pn, const char* pe )
    {
      size_t slen = strlen( pe );
      if ( slen > max_str_len ) max_str_len = slen;
      if ( slen < min_str_len ) min_str_len = slen;
      char* newstr = new char[slen + 1];
      strcpy( newstr, pe );
      RETURN(PCluster::ins( pn, newstr ));
    };

    void PSZCluster::put( int pn, const char* pe )
    {
      ASSERT( pn >= 0 && pn < cnt );
      free( pn );
      ins( pn, pe );
    };

    void PSZCluster::get( int pn, char* pe )
    {
      ASSERT( pn >= 0 && pn < cnt );
      strcpy( pe, (char*)((void**)data)[pn] );
    };

    char* PSZCluster::get( int pn )
    {
      ASSERT( pn >= 0 && pn < cnt );
      return (char*)((void**)data)[pn];
    };

////////////////////////////////////////////////////////////////////////////
//
// misc support functions
//

int SaveToFile( FILE *f, PSZCluster *szc )
{
  int z;
  for(z = 0; z < szc->count(); z++)
    {
    size_t len = strlen(szc->get(z));
    if (fwrite( szc->get(z), 1, len, f ) != len) return -1;
    if (fwrite( "\n", 1, 1, f ) != 1) return -1;
    }
  return 0;
}

int SaveToFile( const char *fname, PSZCluster *szc )
{
  FILE *f = fopen( fname, "wt" );
  if (!f) return -1;
  int res = SaveToFile( f, szc );
  fclose(f);
  return res;
}

int LoadFromFile( FILE *f, PSZCluster *szc, size_t maxlen )
{
  char *buff = new char[maxlen+1];
  while(fgets( buff, maxlen, f ))
    {
    int slen = strlen( buff );
    if (buff[slen-1] == '\n') buff[slen-1] = 0;
    szc->add( buff );
    }
  return 0;
}

int LoadFromFile( const char *fname, PSZCluster *szc, size_t maxlen )
{
  FILE *f = fopen( fname, "rt" );
  if (!f) return -1;
  int res = LoadFromFile( f, szc, maxlen );
  fclose(f);
  return res;
}

////////////////////////////////////////////////////////////////////////////

// eof
