
#include <cstdlib>
#include <slist>

#include "manager.hh"
#include "data.hh"
#include "language.hh"
#include "config.hh"
#include "suggest.hh"
#include "amanager.hh"
#include "file_exceps.hh"

#include "clone_ptr-t.hh"
#include "copy_ptr-t.hh"

namespace aspell {
  //
  // data_access functions
  //

  Manager::Manager(const Config & override) {
    setup(override);
  }

  Manager::Manager(const string & override) {
    setup(override);
  }

  Manager::Manager(): ignore_repl(true)
  {}

  Manager::~Manager() {
    destroy();
  }

  const char * Manager::lang_name() const {
    return lang_->name();
  }

  void Manager::add_to_personal(const string &word) {
    if (have(personal_id))
      personal_wl().add(word);
  }

  void Manager::add_to_session(const string &word) {
    if (have(session_id))
      session_wl().add(word);
  }

  void Manager::clear_session() {
    if (have(session_id))
      session_wl().clear();
  }

  bool Manager::ignore_repls() const {
    return ignore_repl;
  }

  bool Manager::ignore_repls(bool r) {
    bool temp = ignore_repl;
    if (r) {
      ignore_repl = true;
      config_->replace("ignore_repl","true");
    } else {
      ignore_repl = false;
      config_->replace("ignore_repl","false");
    }
    return temp;
  }

  void Manager::store_repl(const string& mis, const string &cor, bool memory) {
    if (!ignore_repl && have(personal_repl_id) ) {
      string::size_type pos;
      Emulation<const char *> sugels 
	= intr_suggest_->suggest(mis.c_str()).elements();
      const char * first_word = sugels.next();
      if (pos = cor.find(' '), 
	  pos == string::npos 
	  ? check(cor)
	  : (check(cor.substr(0,pos)) && check(cor.substr(pos+1))) )
	{ // cor is a correct spelling

	  if (first_word == 0 || cor != first_word)
	    personal_repl().add(mis,cor);
	  
	  if (memory && prev_cor_repl_ == mis) 
	    store_repl(prev_mis_repl_, cor, false);
	
	} else { // cor is not a correct spelling
	
	  if (memory) {
	    if (prev_cor_repl_ != mis)
	      prev_mis_repl_ = mis;
	    prev_cor_repl_ = cor;
	  }
	}
    }
  }

  //
  // simple functions
  //

  bool Manager::check_notrim(const string &word) const {
    if (check_raw(word)) return true;
    else return check_raw(lang_->to_lower(word));
  }

  bool Manager::check_notrim(const char *word) const {
    if (check_raw(word)) return true; 
    else return check_raw(lang_->to_lower(word));
  }

  bool Manager::check(const string &word) const {
    if (check_notrim(word)) return true;
    return lang_->trim_n_try(*this, word);
  }

  bool Manager::check(const char *word) const {
    if (check_notrim(word)) return true;
    else return lang_->trim_n_try(*this, word);
  }

  SuggestionList & Manager::suggest(const char * word) {
    return suggest_->suggest(word);
  }

  Manager::SpecialId Manager::check_id(const DataSet * wl) const {
    return wls_->locate(wl)->special_id;
  }

  bool Manager::use_to_check(const DataSet * wl) const {
    return wls_->locate(wl)->use_to_check;
  }

  void Manager::use_to_check(const DataSet * wl, bool v) {
    wls_->locate(wl)->use_to_check = v;
  }

  bool Manager::use_to_suggest(const DataSet * wl) const {
    return wls_->locate(wl)->use_to_suggest;
  }

  void Manager::use_to_suggest(const DataSet * wl, bool v) {
    wls_->locate(wl)->use_to_suggest = v;
  }

  bool Manager::save_on_saveall(const DataSet * wl) const {
    return wls_->locate(wl)->save_on_saveall;
  }

  void Manager::save_on_saveall(const DataSet * wl, bool v) {
    wls_->locate(wl)->save_on_saveall = v;
  }

  bool Manager::own(const DataSet * wl) const {
    return wls_->locate(wl)->own;
  }

  void Manager::own(const DataSet * wl, bool v) {
    wls_->locate(wl)->own = v;
  }

  void Manager::destroy() {
    DataSetCollection::Iterator i   = wls_->begin();
    DataSetCollection::Iterator end = wls_->end();
    for (; i != end; ++i) {
      if (i->own && i->word_set)
	delete i->word_set;
    }
  }

  bool Manager::check_raw (const char * w) const {
    const char * x = w;
    while (*x != '\0' && (x-w) < ignore_count) ++x;
    if (*x == '\0') return true;
    DataSetCollection::ConstIterator i   = wls_->begin();
    DataSetCollection::ConstIterator end = wls_->end();
    for (; i != end; ++i) {
      if  (i->use_to_check && 
	   i->word_set->basic_type == DataSet::basic_word_set &&
	   static_cast<const BasicWordSet *>(i->word_set)->operator[](w))
	return true;
    }
    return false;
  };

  bool Manager::check_raw (const string & w) const {
    if (w.size() <= ignore_count) return true;
    DataSetCollection::ConstIterator i   = wls_->begin();
    DataSetCollection::ConstIterator end = wls_->end();
    for (; i != end; ++i) {
      if  (i->use_to_check && 
	   i->word_set->basic_type == DataSet::basic_word_set &&
	   static_cast<const BasicWordSet *>(i->word_set)->operator[](w))
	return true;
    }
    return false;
  };
  
  void Manager::save_all_wls() {
    DataSetCollection::Iterator i   = wls_->begin();
    DataSetCollection::Iterator end = wls_->end();
    WritableDataSet * wl;
    for (; i != end; ++i) {
      if  (i->save_on_saveall && 
	   (wl = dynamic_cast<WritableDataSet *>(i->word_set)))
	wl->synchronize();
    }
  }

  int Manager::num_wordlists() const {
    return wls_->wordlists_.size();
  }

  Manager::WordLists Manager::wordlists() const {
    return WordLists(MakeVirEmulation<DataSetCollection::Parms>
		     (wls_->begin(), DataSetCollection::Parms(wls_->end())));
  }

  bool Manager::have(const DataSet *to_find) const {
    return wls_->locate(to_find) != wls_->end();
  }

  bool Manager::have(Manager::SpecialId to_find) const {
    return wls_->locate(to_find) != wls_->end();
  }

  BasicWordSet & Manager::main_wl() {
    return *static_cast<BasicWordSet *>(wls_->locate(main_id)->word_set);
  }

  const BasicWordSet & Manager::main_wl() const {
    return *static_cast<const BasicWordSet *>(wls_->locate(main_id)->word_set);
  }

  WritableWordSet & Manager::personal_wl() {
    return *static_cast<WritableWordSet *>(wls_->locate(personal_id)->word_set);
  }

  const WritableWordSet & Manager::personal_wl() const {
    return *static_cast<const WritableWordSet *> (wls_->locate(personal_id)->word_set);
  }

  WritableWordSet & Manager::session_wl() {
    return *static_cast<WritableWordSet *>(wls_->locate(session_id)->word_set);
  }

  const WritableWordSet & Manager::session_wl() const {
    return *static_cast<const WritableWordSet *>(wls_->locate(session_id)->word_set);
  }

  WritableReplacementSet & Manager::personal_repl() {
    return *static_cast<WritableReplacementSet *>(wls_->locate(personal_repl_id)->word_set);
  }

  const WritableReplacementSet & Manager::personal_repl() const {
    return *static_cast<const WritableReplacementSet *>(wls_->locate(personal_repl_id)->word_set);
  }

  bool Manager::attach(DataSet *w ) {
    DataSetCollection::Iterator i = wls_->locate(w);
    if (i != wls_->end()) {
      return false;
    } else {
      if (!lang_) 
	{
	  lang_ = new Language(*w->lang());
	  config_->replace("lang", lang_->name());
	}
      w->attach(*lang_);
      wls_->wordlists_.push_front(w);
      return true;
    }
  }

  bool Manager::steal(DataSet *w ) {
    bool ret = attach(w);
    own(w, true);
    return ret;
  }

  bool Manager::detach(const DataSet *w) {
    DataSetCollection::Iterator to_del = wls_->locate(w);
    if (to_del == wls_->wordlists_.end()) return false;
    to_del->word_set->detach();
    wls_->wordlists_.erase(to_del);
    return true;
  }  

  bool Manager::destroy(const DataSet * w) {
    DataSetCollection::Iterator to_del = wls_->locate(w);
    if (to_del == wls_->wordlists_.end()) return false;
    if (!to_del->own) throw DontOwnWordlist(to_del->word_set);
    delete to_del->word_set;
    wls_->wordlists_.erase(to_del);
    return true;
  }

  void Manager::change_id(const DataSet *w , SpecialId id) {
    DataSetCollection::Iterator to_change = wls_->locate(w);

    if (id != none_id && have(id))
      throw AlreadyHaveType(id, to_change->word_set);
    
    switch (id) {
    case main_id:
      if (!dynamic_cast<BasicWordSet *>(to_change->word_set))
	throw DataSetNotCompatible(id, to_change->word_set);
      to_change->use_to_check = true;
      to_change->use_to_suggest = true;
      to_change->save_on_saveall = false;
      break;
    case personal_id:
      if (!dynamic_cast<WritableWordSet *>(to_change->word_set))
	throw DataSetNotCompatible(id, to_change->word_set);
      to_change->use_to_check = true;
      to_change->use_to_suggest = true;
      to_change->save_on_saveall = true;
      break;
    case session_id:
      if (!dynamic_cast<WritableWordSet *>(to_change->word_set))
	throw DataSetNotCompatible(id, to_change->word_set);
      to_change->use_to_check = true;
      to_change->use_to_suggest = true;
      to_change->save_on_saveall = false;
      break;
    case personal_repl_id:
      if (!dynamic_cast<BasicReplacementSet *>(to_change->word_set))
	throw DataSetNotCompatible(id, to_change->word_set);
      to_change->use_to_check = false;
      to_change->use_to_suggest = true;
      to_change->save_on_saveall = true;
      break;
    case none_id:
      break;
    }
    to_change->special_id = id;
  }

  void Manager::setup(const Config & override)
  {
    config_ = new Config(override);
    setup();
  }

  void Manager::setup(const string & override)
  {
    config_ = new Config();
    config_->read_in_string(override);
    setup();
  }

  void Manager::setup() {
    config_->read_in();
    ignore_repl = config_->retrieve_bool("ignore-repl");
    ignore_count = config_->retrieve_int("ignore");
    wls_ = new DataSetCollection();
    BasicWordSet * temp;
    try {
      temp = new_default_readonly_word_set();
      temp->load(config_->retrieve("master-path"), config_.ptr);
      steal(temp);
      change_id(temp, main_id);

      temp = new_default_writable_word_set();
      try {
	temp->load(config_->retrieve("personal-path"),config_.ptr);
      } catch (CantReadFile &) {}
      steal(temp);
      change_id(temp, personal_id);

      temp = new_default_writable_word_set();
      steal(temp);
      change_id(temp, session_id);
      
      BasicReplacementSet * rtemp = new_default_writable_replacement_set();
      try {
	rtemp->load(config_->retrieve("repl-path"),config_.ptr);
      } catch (CantReadFile &) {}
      steal(rtemp);
      change_id(rtemp, personal_repl_id);
      
      suggest_ = new_default_suggest(this);
      intr_suggest_ = new_default_suggest(this);
    } catch (...) {
      destroy();
      throw;
    }
  }
  
  DontOwnWordlist::DontOwnWordlist(const DataSet *ws)
    : Exception("Sorry I don't own that wordlist.  Thus I can't delete it.")
    , word_set(ws) 
  {}

  AlreadyHaveType::AlreadyHaveType(Manager::SpecialId id, DataSet * ws) 
    : ChangeIdException(id, ws, "Sorry, I already have that type")
  {}
  
  DataSetNotCompatible::DataSetNotCompatible
  (Manager::SpecialId id, DataSet * ws)
    : ChangeIdException
  (id, ws, "The current word list is not compatible with the requested id")
  {}

}

namespace autil {
  template class CopyPtr<const aspell::Language>;
}

