
/*
 * The Real SoundTracker - Preferences handling
 *
 * Copyright (C) 1998-1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

#include "i18n.h"
#include "gui-subs.h"
#include "preferences.h"
#include "menubar.h"
#include "scope-group.h"
#include "track-editor.h"
#include "errors.h"

char *
prefs_get_prefsdir (void)
{
    static char xdir[128];
    struct passwd *pw;
    pw = getpwuid(getuid());
    sprintf(xdir, "%s/.soundtracker", pw->pw_dir);
    return(xdir);
}

void
prefs_check_prefs_dir (void)
{
    struct stat st;
    char *dir = prefs_get_prefsdir();
    if(stat(dir, &st) < 0) {
	mkdir(dir, S_IRUSR | S_IWUSR | S_IXUSR);
	error_warning(_("A directory called '.soundtracker' has been created in your\nhome directory to store configuration files.\n"));
    }
}

static gboolean
prefs_get_line (FILE *f,
		const char *key,
		char *destbuf,
		int destbuflen)
{
    int i;
    char readbuf[1024], *p;

    for(i = 0; i < 2; i++) {
	// Linear search
	while(!feof(f)) {
	    fgets(readbuf, 1024, f);
	    p = strchr(readbuf, '=');
	    if(!p || p == readbuf || p[1] == 0) {
		return 0;
	    }
	    p[-1] = 0;
	    if(!strcmp(readbuf, key)) {
		strncpy(destbuf, p + 2, destbuflen - 1);
		destbuf[destbuflen - 1] = 0;

		if(strlen(destbuf) > 0 && destbuf[strlen(destbuf) - 1] == '\n')
		    destbuf[strlen(destbuf) - 1] = 0;

		return 1;
	    }
	}

	// Start from the beginning
	fseek(f, 0, SEEK_SET);
    }

    return 0;
}

gboolean
prefs_get_int (FILE *f,
	       const char *key,
	       int *dest)
{
    char buf[21];

    if(prefs_get_line(f, key, buf, 20)) {
	buf[20] = 0;
	*dest = atoi(buf);
	return 1;
    }

    return 0;
}

gboolean
prefs_get_string (FILE *f,
		  const char *key,
		  char *dest)
{
    char buf[128];

    if(prefs_get_line(f, key, buf, 127)) {
	buf[127] = 0;
	strcpy(dest, buf);
	return 1;
    }

    return 0;
}

void
prefs_put_int (FILE *f,
	       const char *key,
	       int value)
{
    fprintf(f, "%s = %d\n", key, value);
}

void
prefs_put_string (FILE *f,
		  const char *key,
		  const char *value)
{
    fprintf(f, "%s = %s\n", key, value);
}
