#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef	linux
#include <net/if.h>	/* for IFNAMSIZ */
#else
#include <sys/socket.h>
#include <net/if.h>
#ifndef	if_mtu
#include <net/if_var.h>
#endif
#endif

#include <config.h>
#include <support.h>
#include <xcio.h>

#include "option.h"
#include "log.h"
#include "console.h"
#include "env.h"
#include "lcp.h"
#include "timer.h"
#include "frame.h"
#include "fsm.h"
#include "cps.h"
#include "phase.h"
#include "dev/device.h"

#define	LCPCODE_MAX	CPCODE_TIME_REM

#define	DIRECTION_I	0x10
#define	DIRECTION_O	0x01

struct lcpopt_s {
    runmode_t mode;
    time_t k_iv;		/* Kick interval [s] */
    u_int8_t callback_type;
    u_int8_t acfc, pfc;
};

static struct lcpopt_s lcpOpt;

#define	LCPCONF_MRU		1
#define	LCPCONF_ACCM		2
#define	LCPCONF_AUTH		3
#define	LCPCONF_QUALITY		4
#define	LCPCONF_MAGIC		5
#define	LCPCONF_PFC		7
#define	LCPCONF_ACFC		8
#define	LCPCONF_FCSALT		9
#define	LCPCONF_SDESCPAD	10
#define	LCPCONF_NUMMODE		11
#define	LCPCONF_ML_PROC		12
#define	LCPCONF_CALLBACK	13
#define	LCPCONF_CONTIME		14
#define	LCPCONF_COMPFRAMES	15
#define	LCPCONF_NDENCAP		16
#define	LCPCONF_ML_MRRU		17
#define	LCPCONF_ML_SSNHF	18
#define	LCPCONF_ML_ED		19
#define	LCPCONF_PROP		20
#define	LCPCONF_DCEID		21
#define	LCPCONF_ML_PPLUS	22
#define	LCPCONF_BACP_LD		23
#define	LCPCONF_MAX		24

static const char *callbackTypes[]={
    "user", "dial", "id", "E.164", "name", "???", "CBCP", "none"
};

struct lcpreg_s lcprReg, lcplReg;
u_int32_t hbo_remoteAccm=0xffffffff;

static struct lcpreg_s xReg;
static struct timer_s kickTimer;

static int AuthProtoCoder(), QualProtoCoder();
static int CallbackCoder();

static struct fsmreg_s lcpFsm;
static struct cpcodeop_s lcpCop[LCPCODE_MAX];
static struct cptypeop_s lcpTop[]={
    {
	"Maximum-Receive-Unit", TostrD16,
	&xReg.nbo_mru, NULL, 2,
	LCPCONF_MRU
    }, {
	"Async-Control-Character-Map", TostrX32,
	&xReg.nbo_accm, NULL, 4,
	LCPCONF_ACCM
    }, {
	"Authentication-Protocol", NULL,
	NULL, AuthProtoCoder, 2,
	LCPCONF_AUTH
    }, {
	"Quality-Protocol", NULL,
	NULL, QualProtoCoder, 6,
	LCPCONF_QUALITY
    }, {
	"Magic-Number", TostrX32,
	&xReg.nbo_magic, NULL, 4,
	LCPCONF_MAGIC
    }, {
	"Protocol-Field-Compression", NULL,
	&xReg.pfc, NULL, 0,
	LCPCONF_PFC
    }, {
	"Address-and-Control-Field-Compression", NULL,
	&xReg.acfc, NULL, 0,
	LCPCONF_ACFC
    }, {
	"FCS-Alternatives", NULL,
	NULL, NULL, 0,
	LCPCONF_FCSALT
    }, {
	"Self-Describing-Pad", NULL,
	NULL, NULL, 0,
	LCPCONF_SDESCPAD
    }, {
	"Numbered-Mode", NULL,
	NULL, NULL, 0,
	LCPCONF_NUMMODE
    }, {
	"Multi-Link-Procedure", NULL,
	NULL, NULL, 0,
	LCPCONF_ML_PROC
    }, {
	"Callback", NULL,
	NULL, CallbackCoder, 0,
	LCPCONF_CALLBACK
    }, {
	"Connect-Time", NULL,
	NULL, NULL, 0,
	LCPCONF_CONTIME
    }, {
	"Compound-Frames", NULL,
	NULL, NULL, 0,
	LCPCONF_COMPFRAMES
    }, {
	"Nominal-Data-Encapsulation", NULL,
	NULL, NULL, 0,
	LCPCONF_NDENCAP
    }, {
	"Multilink-MRRU", NULL,
	NULL, NULL, 0,
	LCPCONF_ML_MRRU
    }, {
	"Multilink-Short-Sequence-Number-Header-Format", NULL,
	NULL, NULL, 0,
	LCPCONF_ML_SSNHF
    }, {
	"Multilink-Endpoint-Discriminator", NULL,
	NULL, NULL, 0,
	LCPCONF_ML_ED
    }, {
	"Proprietary", NULL,
	NULL, NULL, 0,
	LCPCONF_PROP
    }, {
	"DCE-Identifier", NULL,
	NULL, NULL, 0,
	LCPCONF_DCEID
    }, {
	"Multilink-Protocol-Plus", NULL,
	NULL, NULL, 0,
	LCPCONF_ML_PPLUS
    }, {
	"BACP-Link-Discriminator", NULL,
	NULL, NULL, 0,
	LCPCONF_BACP_LD
    }, {
	"ASCEND type 0", NULL,
	NULL, NULL, 4,
	0
    }
};

static cptype_t remoteOrder[LCPCONF_MAX], localOrder[LCPCONF_MAX];
static u_char *lcprBp[LCPCONF_MAX];

static struct cpstate_s lcpState={
    lcpCop,		/* code oprators */
    lcpTop,		/* type operators */
    (void *)&lcprReg,	/* remote registry */
    (void *)&lcplReg,	/* local registry */
    lcprBp,		/* received buffer pointers */
    NULL,		/* encode order */
    remoteOrder,	/* remote(received) order */
    localOrder,		/* request(local) order */
    {{0}},		/* encode list */
    {{0}},		/* remote(received) list */
    {{0}},		/* request(local) list */
    /* number of configuration types */
    sizeof(lcpTop)/sizeof(struct cptypeop_s),
    NBO_PROTO_LCP,
    0, 0		/* remote/local ID */
};

static int
AuthProtoCoder(u_char *buf, int len)
{
    if (len > 0) {
	/* decode */
	memcpy(&xReg.nbo_auth_type, buf, 2);
	switch(xReg.nbo_auth_type) {
	case NBO_PROTO_CHAP:
	    xReg.a_algorithm = buf[2];
	    break;
	default:
	    xReg.a_algorithm = 0;
	    break;
	}
    } else {
	/* encode */
	memcpy(buf, &xReg.nbo_auth_type, 2);
	switch(xReg.nbo_auth_type) {
	case NBO_PROTO_CHAP:
	    buf[2] = xReg.a_algorithm;
	    len = 3;
	    break;
	default:
	    len = 2;
	}
    }
    if (ISLOG(LOG_CP)) {
	Logf(LOG_CP, "%s", TostrProto(&xReg.nbo_auth_type));
	switch(xReg.nbo_auth_type) {
	case NBO_PROTO_CHAP:
	    Logf(LOG_CP, "/%s", ChapAlgorithmName(xReg.a_algorithm));
	    break;
	}
	Logf(LOG_CP, "\n");
    }
    return(len);
}

static int
QualProtoCoder(u_char *buf, int len)
{
    if (len > 0) {
	/* decode */
	memcpy(&xReg.nbo_qual_type, buf, 2);
	buf += 2;
	memcpy(&xReg.nbo_qual_period, buf, 4);
    } else {
	/* encode */
	memcpy(buf, &xReg.nbo_qual_type, 2);
	buf += 2;
	memcpy(buf, &xReg.nbo_qual_period, 4);
	len = 6;
    }
    if (ISLOG(LOG_CP)) {
	Logf(LOG_CP, "%s %d\n", TostrProto(&xReg.nbo_qual_type),
	     ntohl(xReg.nbo_qual_period));
    }
    return(len);
}

static int
CallbackCoder(u_char *buf, int len)
{
    if (len > 0) {
	/* decode */
	xReg.callback_type = *buf;
    } else {
	/* encode */
	*buf = xReg.callback_type;
	len = 1;
    }
    if (ISLOG(LOG_CP)) {
	Logf(LOG_CP, "%s\n", callbackTypes[xReg.callback_type]);
    }
    return(len);
}

static int
LcpEncodeReg(u_char *buf, struct lcpreg_s *lcr)
{
    if (lcr) memcpy(&xReg, lcr, sizeof(struct lcpreg_s));
    return(CpEncodeReg(&lcpState, buf, FALSE));
}

static int
LcpEncodeRejectReg(u_char *buf, struct lcpreg_s *lcr)
{
    if (lcr) memcpy(&xReg, lcr, sizeof(struct lcpreg_s));
    return(CpEncodeReg(&lcpState, buf, TRUE));
}

/* FSM action functions */

static void
TLD(struct fsmreg_s *fsm)
{
/*
    const char *oldpath;

    oldpath = SetLoadFilePath("ip");
    LoadScriptFile(ipcpOpt.down);
    SetLoadFilePath(oldpath);
*/
}

static void
TLF(struct fsmreg_s *fsm)
{
/*    pppInfo.l_stat &= ~LSTAT_PPP;*/
    phaseShift = PHASE_DOWN;
    if (lcpOpt.mode == RUN_DIRECT
	|| lcpOpt.mode == RUN_GETTY) DevExit();
    FsmInit(fsm);
}

static void
LcpDown(bool_t process)
{
    pppInfo.l_stat &= ~LSTAT_NLINK;
    FsmClose(&lcpFsm);
    if (!process) TLF(&lcpFsm);
    /*    pppInfo.p[PINFO_LCP].st = PIST_DOWN;*/
}

static void
TLU(struct fsmreg_s *fsm)
{
    PhaseUp();
    RegisterDownPhase("LCP", LcpDown);
}

/* FSM event functions */

static int
RCR(u_char *buf, int n)
{
    int i=0;
    cptype_t type;
    b256_t rej, nak;

    memset(&xReg, 0, sizeof(struct lcpreg_s));
    if (CpDecodeReg(&lcpState, buf, n) < 0) {
	FsmRUC(&lcpFsm);
	return(0);
    }
    B256_ZERO(&nak);
    B256_ZERO(&rej);
    lcpFsm.nak_rcr = lcpFsm.rej_rcr = 0;
/*    while ((type = remoteOrder[i++]) < LCPCONF_MAX) {*/
    while ((type = remoteOrder[i++]) < ENDOF_LIST) {
	switch (type) {
	case LCPCONF_MRU:
	    /* check min_mru < mru < max_mru */
	    continue;
	case LCPCONF_ACCM:
	    hbo_remoteAccm = ntohl(xReg.nbo_accm);
	    continue;
	case LCPCONF_AUTH:
	    if (pppOpt.a_server == AUTH_SERVER_NONE) {
		int a;
		u_int16_t hbo_auth;

		hbo_auth = ntohs(xReg.nbo_auth_type);
		for (a = 0; a < AUTH_MAX; a ++)
		    if (hbo_auth == pppOpt.hbo_a_order[a]
			&& xReg.a_algorithm == pppOpt.a_algorithm[a])
			break;
		if (a < AUTH_MAX) {
		    if (!AuthName() || !AuthPasswd()) {
			Logf(LOG_ERROR, "reject %s request because "
			     "name/passwd is not set\n",
			     TostrProto(&xReg.nbo_auth_type));
			goto reject;
		    }
		    continue;
		}
		/* nak */
		xReg.nbo_auth_type = htons(pppOpt.hbo_a_order[0]);
		xReg.a_algorithm = pppOpt.a_algorithm[0];
	    }
	    break;
	case LCPCONF_QUALITY:
	    if (xReg.nbo_qual_type == pppOpt.nbo_qual_type) continue;
	    FrameReject(NBO_PROTO_LQR, 1);
	    goto reject;
	    break;
	case LCPCONF_MAGIC:
	    if (!lcprReg.nbo_magic ||
		xReg.nbo_magic == lcprReg.nbo_magic) continue;
	    if (!lcplReg.nbo_magic &&
		xReg.nbo_magic == lcplReg.nbo_magic) {
		/* loop back !! */
		Logf(LOG_CP, " Loopback detected\n");
		return(0);
	    }
	    break;
	case LCPCONF_CALLBACK:
#if 0
	    goto reject;
#else
	    /* for MN128-SOHO Callback request */
	    if (xReg.callback_type != lcpOpt.callback_type)
		goto reject;
	    continue;
#endif
	case LCPCONF_ACFC:
	    if (!(lcpOpt.acfc & DIRECTION_O)) goto reject;
	    continue;
	case LCPCONF_PFC:
	    if (!(lcpOpt.pfc & DIRECTION_O)) goto reject;
	    continue;
	reject:
	default:
	    /* not supported */
	    lcpFsm.rej_rcr = 1;
	    B256_SET(&rej, type);
	    continue;
	}
	lcpFsm.nak_rcr = 1;
	B256_SET(&nak, type);
    }
    memcpy(&lcprReg, &xReg, sizeof(struct lcpreg_s));
    if (lcpFsm.rej_rcr) B256_CPY(&lcpState.r_list, &rej);
    else if (lcpFsm.nak_rcr) B256_CPY(&lcpState.r_list, &nak);
    return(FsmRCR(&lcpFsm));
}

static int
RCA(u_char *buf, int n)
{
    memset(&xReg, 0, sizeof(struct lcpreg_s));
    CpDecodeReg(&lcpState, buf, n);
/*    lcpFsm.good_rcr = localReject ? 0: 1;*/
    return(FsmRCA(&lcpFsm));
}

static int
RCNJ(u_char *buf, int n, bool_t rcj)
{
    int i=0, a;
    u_int16_t hbo_auth;
    cptype_t type;

    memset(&xReg, 0, sizeof(struct lcpreg_s));
    CpDecodeReg(&lcpState, buf, n);
/*    while ((type = remoteOrder[i++]) < LCPCONF_MAX) {*/
    while ((type = remoteOrder[i++]) < ENDOF_LIST) {
	if (rcj) B256_CLR(&lcpState.l_list, type);
	switch (type) {
	case LCPCONF_MRU:
	    if (!rcj) lcplReg.nbo_mru = xReg.nbo_mru;
	    break;
	case LCPCONF_AUTH:
	    hbo_auth = ntohs(xReg.nbo_auth_type);
	    for (a = 0; a < AUTH_MAX; a ++)
		if (hbo_auth == pppOpt.hbo_a_order[a]
		    && xReg.a_algorithm == pppOpt.a_algorithm[a]) break;
	    if (a < AUTH_MAX) {
		lcplReg.nbo_auth_type = xReg.nbo_auth_type;
		lcplReg.a_algorithm = xReg.a_algorithm;
	    }
	    break;
	case LCPCONF_ACFC:
	    lcplReg.acfc = FALSE;
	    break;
	case LCPCONF_PFC:
	    lcplReg.pfc = FALSE;
	    break;
	case LCPCONF_CALLBACK:
	    lcplReg.callback_type = CALLBACK_MAX;
	    break;
	case LCPCONF_QUALITY:
	    lcplReg.nbo_qual_type = 0;
	    break;
	default:
	    break;
	}
    }
    return(FsmRCN(&lcpFsm));
}

static int
RCN(u_char *buf, int n)
{
    return(RCNJ(buf, n, FALSE));
}

static int
RCJ(u_char *buf, int n)
{
    return(RCNJ(buf, n, TRUE));
}

/*
 * Received Protocol Reject
 */

static int
RXPJ(u_char *buf, int n)
{
    u_int16_t nbo_proto=0;

    memcpy(&nbo_proto, buf, sizeof(u_int16_t));
    if (ISLOG(LOG_CP)) Logf(LOG_CP, " %s\n", TostrProto(&nbo_proto));
    FrameReject(nbo_proto, 1);
    lcpFsm.fatal_rxj = 0;
    return(FsmRXJ(&lcpFsm));
}

static int
RTR(u_char *buf, int n)
{
    if (lcpFsm.state == FSM_CLOSING) {
	/*
	   A Terminate-Request might be ignored.
	   Send Terminate-Ack and wait one timeout.
	 */
	lcpFsm.r_count = 0;
    }
    return(CpRtr(&lcpFsm, buf, n));
}

static int
RTA(u_char *buf, int n)
{
    return(CpRta(&lcpFsm, buf, n));
}

struct echo_s {
    char *buf;
    int len;
};

static struct echo_s echoBuf;

/* Send-Echo-Request */
int
SendEchoRequest(char *dust, int len)
{
    struct echo_s eb;

    eb.buf = dust;
    eb.len = len;
    lcpState.l_id ++;
    CpEnqueueFrame(&lcpFsm, CPCODE_ECHO_REQ, &eb, lcpState.l_id);
    return(lcpState.l_id);
}

/* Send-Echo-Reply */
static void
SER(struct fsmreg_s *fsm)
{
#if 0
    lcpState.l_id ++;
    CpEnqueueFrame(fsm, CPCODE_ECHO_REP, &echoBuf, lcpState.l_id);
#else
    CpEnqueueFrame(fsm, CPCODE_ECHO_REP, &echoBuf, lcpState.r_id);
#endif
}

static int
LcpEncodeEcho(u_char *buf, struct echo_s *ebp)
{
    memcpy(buf, &lcplReg.nbo_magic, sizeof(lcplReg.nbo_magic));
    if (ebp->len)
	memcpy(buf + sizeof(lcplReg.nbo_magic), ebp->buf, ebp->len);
    if (ISLOG(LOG_CP) && ISLOG(LOG_PRIVATE)) {
	Logf(LOG_CP, " Magic-Number: %#x\n", ntohl(lcplReg.nbo_magic));
	Logf(LOG_CP, " Data-Size: %d\n", ebp->len);
    }
    return(ebp->len + sizeof(lcplReg.nbo_magic));
}

/* Receive-Echo-Request */
static int
RXRequest(u_char *buf, int n)
{
    u_int32_t nbo_magic;

    memcpy(&nbo_magic, buf, sizeof(nbo_magic));
    echoBuf.buf = buf + sizeof(lcplReg.nbo_magic);
    echoBuf.len = n - sizeof(lcplReg.nbo_magic);
    if (ISLOG(LOG_CP) && ISLOG(LOG_PRIVATE)) {
	Logf(LOG_CP, " Magic-Number: %#x\n", ntohl(nbo_magic));
	Logf(LOG_CP, " Data-Size: %d\n", echoBuf.len);
    }
    return(FsmRXR(&lcpFsm));
}

/* Receive-Echo-Reply */
static int
RXReply(u_char *buf, int n)
{
    u_int32_t nbo_magic;

    memcpy(&nbo_magic, buf, sizeof(nbo_magic));
    n -= sizeof(nbo_magic);
    buf += sizeof(nbo_magic);
    if (ISLOG(LOG_CP) && ISLOG(LOG_PRIVATE)) {
	Logf(LOG_CP, " Magic-Number: %#x\n", ntohl(nbo_magic));
	Logf(LOG_CP, " Data-Size: %d\n", n);
    }
    return(0);
}

int
LcpEncodePJ(u_char *buf, u_int16_t *nbo_proto)
{
    if (ISLOG(LOG_CP)) Logf(LOG_CP, " %s\n", TostrProto(nbo_proto));
    memcpy(buf, nbo_proto, sizeof(u_int16_t));
    return(sizeof(u_int16_t));
}

void
LcpSPJ(u_int16_t nbo_proto)
{
    CpEnqueueFrame(&lcpFsm, CPCODE_PROT_REJ, &nbo_proto, lcpState.l_id);
}

static bool_t
EnvCallbackType(int argc, char *argv[], char *outs)
{
    int i;

    if (!argc) {
	if (lcpOpt.callback_type < CALLBACK_MAX)
	    sprintf(outs, "%s ", callbackTypes[lcpOpt.callback_type]);
	return FALSE;
    }
    for (i = 0; i < CALLBACK_MAX; i ++)
	if (!strcasecmp(callbackTypes[i], argv[1])) {
	    lcpOpt.callback_type = i;
	    return TRUE;
	}
}

static bool_t
EnvAcfcPfc(int argc, char **argv, char *outs)
{
    int len;
    char *name;
    u_int8_t *ap;

    if ((name = strchr(argv[0], '.')) != NULL) name ++;
    else name = argv[0];
    if (!strcasecmp(name, "ACFC")) ap = &lcpOpt.acfc;
    else ap = &lcpOpt.pfc;

    if (!argc) {
	char *name;

	if ((*ap & DIRECTION_I) && (*ap & DIRECTION_O))
	    name = "yes";
	else if (*ap & DIRECTION_I)
	    name = "in";
	else if (*ap & DIRECTION_O)
	    name = "out";
	else
	    name = "no";
	strcpy(outs, name);
	return(FALSE);
    }

    len = strlen(argv[1]);
    if (!strncasecmp("in", argv[1], len))
	*ap = DIRECTION_I;
    else if (!strncasecmp("out", argv[1], len))
	*ap = DIRECTION_O;
    else if (!strncasecmp("no", argv[1], len))
	*ap = 0;
    else
	*ap = DIRECTION_O|DIRECTION_I;
    return(TRUE);
}

static void
LcpEnable(bool_t sw)
{
    FrameEnable(NBO_PROTO_LCP, sw);
}

static void
KickTimerHandler()
{
    /* from rfc1662:
       7e ff 03 c0 21
       7e ff 7d 23 c0 21
       7e 7d df 7d 23 c0 21
     */
    if (!lcpOpt.k_iv || pppInfo.l_stat & LSTAT_PPP) {
	kickTimer.st_expired = 1;
	return;
    }
    kickTimer.rest = lcpOpt.k_iv;
    if (ISLOG(LOG_CP)) Logf(LOG_CP, "KICK\n");
    FrameEnqueue("\xff", 1, NBO_PROTO_LCP, PRI_KICK);
/*    write(devFd, kick, sizeof(kick));*/
}

static void
KickInit()
{
    if (lcpOpt.k_iv) {
	KickTimerHandler();
	TimerAdd(&kickTimer);
    }
}

void
LcpSetup()
{
    extern bool_t EnvEscMap();
    static struct env_s envlist[]={
	{"TIMEOUT", {&pppOpt.r_to}, ENV_INT32, 0, 0, 0666},
	{"RETRY", {&pppOpt.r_count}, ENV_INT32, 0, 0, 0666},
	{"ACCMAP", {&pppOpt.hbo_accm}, ENV_INT32X, 0, 0, 0666},
	{"ACFC", {EnvAcfcPfc}, ENV_SET, 0, 0, 0666},
	{"PFC", {EnvAcfcPfc}, ENV_SET, 0, 0, 0666},
	{"CALLBACK", {EnvCallbackType}, ENV_SET, 0, 0, 0666},
	{"CHECKID", {&pppOpt.id_check}, ENV_BOOL, 0, 0, 0666},
	{"KICK", {&lcpOpt.k_iv}, ENV_INT32, 0, 0, 0666},
	{"ESC", {EnvEscMap}, ENV_SET, 0, 0, 0666},
	{NULL}
    };

    lcpOpt.callback_type = CALLBACK_MAX;
    lcpFsm.name = "LCP";
    RegisterEnvs(envlist, lcpFsm.name, LcpEnable);
    RegisterCp(&lcpState, &lcpFsm);
    lcpFsm.timer.name = "LCPFSM";
    lcpFsm.timer.arg = &lcpFsm;
    lcpCop[CPCODE_CONF_REQ-1].fsmevent = RCR;
    lcpCop[CPCODE_CONF_REQ-1].encode = LcpEncodeReg;
    lcpCop[CPCODE_CONF_ACK-1].fsmevent = RCA;
    lcpCop[CPCODE_CONF_ACK-1].encode = LcpEncodeReg;
    lcpCop[CPCODE_CONF_NAK-1].fsmevent = RCN;
    lcpCop[CPCODE_CONF_NAK-1].encode = LcpEncodeReg;
    lcpCop[CPCODE_CONF_REJ-1].fsmevent = RCJ;
    lcpCop[CPCODE_CONF_REJ-1].encode = LcpEncodeRejectReg;
/*    lcpCop[CPCODE_CODE_REJ-1].fsmevent = RXCJ;*/
    lcpCop[CPCODE_PROT_REJ-1].fsmevent = RXPJ;
    lcpCop[CPCODE_TERM_REQ-1].fsmevent = RTR;
    lcpCop[CPCODE_TERM_ACK-1].fsmevent = RTA;
    lcpCop[CPCODE_PROT_REJ-1].encode = LcpEncodePJ;

    lcpCop[CPCODE_ECHO_REQ-1].encode = LcpEncodeEcho;
    lcpCop[CPCODE_ECHO_REQ-1].fsmevent = RXRequest;
    lcpCop[CPCODE_ECHO_REP-1].encode = LcpEncodeEcho;
    lcpCop[CPCODE_ECHO_REP-1].fsmevent = RXReply;
    lcpFsm.tlu = TLU;
    lcpFsm.tlf = TLF;
    lcpFsm.tld = TLD;
    lcpFsm.ser = SER;
    /*    strcpy(pppInfo.p[PINFO_LCP].name, lcpFsm.name);*/

    kickTimer.name = "KICK";
    kickTimer.callback = KickTimerHandler;
    kickTimer.f_interval = 1;
}

void
LcpInit()
{
    int i, k;

    FrameResetReject();
    memset(&lcprReg, 0, sizeof(struct lcpreg_s));
    lcpState.r_id = lcpState.l_id = 0;
    B256_ZERO(&lcpState.l_list);
    B256_SET(&lcpState.l_list, LCPCONF_MRU);
    B256_SET(&lcpState.l_list, LCPCONF_ACCM);
    B256_SET(&lcpState.l_list, LCPCONF_MAGIC);
    lcplReg.nbo_mru = htons(pppOpt.hbo_mru);
    lcplReg.nbo_accm = htonl(pppOpt.hbo_accm);
    if (pppOpt.nbo_qual_type) {
	lcplReg.nbo_qual_type = pppOpt.nbo_qual_type;
	lcplReg.nbo_qual_period = htonl(pppOpt.hbo_qual_period * 100);
	B256_SET(&lcpState.l_list, LCPCONF_QUALITY);
    }
    if (lcpOpt.acfc & DIRECTION_I) {
	lcplReg.acfc = TRUE;
	B256_SET(&lcpState.l_list, LCPCONF_ACFC);
    }
    if (lcpOpt.pfc & DIRECTION_I) {
	lcplReg.pfc = TRUE;
	B256_SET(&lcpState.l_list, LCPCONF_PFC);
    }
    if (lcpOpt.callback_type < CALLBACK_MAX) {
	lcplReg.callback_type = lcpOpt.callback_type;
	B256_SET(&lcpState.l_list, LCPCONF_CALLBACK);
    }

    if (pppOpt.a_server != AUTH_SERVER_NONE) {
	lcplReg.nbo_auth_type = htons(pppOpt.hbo_a_order[0]);
	switch(lcplReg.nbo_auth_type) {
	case NBO_PROTO_CHAP:
	    lcplReg.a_algorithm = pppOpt.a_algorithm[0];
	    break;
	default:
	    lcplReg.a_algorithm = 0;
	    break;
	}
	B256_SET(&lcpState.l_list, LCPCONF_AUTH);
    } else if (pppOpt.hbo_a_order[0] && !AuthName()) {
/*	conOp->outf("warning: please set AUTH.ENTRY\n");*/
    }
    lcplReg.nbo_magic = RAND();
    for (k = 0, i = LCPCONF_MRU; i < LCPCONF_MAX; i ++)
	if (B256_ISSET(&lcpState.l_list, i)) localOrder[k++] = i;
    localOrder[k] = ENDOF_LIST;

    lcpFsm.init_r_count = pppOpt.r_count;
    lcpFsm.init_r_to = pppOpt.r_to;
    FsmInit(&lcpFsm);
    FsmUp(&lcpFsm);
    switch(lcpOpt.mode) {
    case RUN_KICK:
	KickInit();
    case RUN_PASSIVE:
	lcpFsm.opt_p = 1;
	break;
    case RUN_DIRECT:
    case RUN_GETTY:
	PhaseQuit();
    case RUN_ACTIVE:
	lcpFsm.opt_p = 0;
	break;
    default:
    }
    FsmOpen(&lcpFsm);
    /*    pppInfo.p[PINFO_LCP].st = PIST_RUN;*/
}

void
LcpMode(runmode_t mode)
{
    lcpOpt.mode = mode;
}

static bool_t
EnvIfMtu(int argc, char **argv, char *outs)
{
    if (!argc) {
	u_int16_t hbo_r_mru, hbo_l_mru;

	if (!(pppInfo.l_stat & LSTAT_PPP)) {
	    sprintf(outs, "%d", pppOpt.hbo_mru);
	    return FALSE;
	}
	hbo_r_mru = ntohs(lcprReg.nbo_mru);
	hbo_l_mru = ntohs(lcplReg.nbo_mru);
	sprintf(outs, "%d", (hbo_r_mru < hbo_l_mru) ?
	    hbo_r_mru: hbo_l_mru);
	return FALSE;
    }
    if (!(pppInfo.l_stat & LSTAT_PPP)) {
	pppOpt.hbo_mru = atoi(argv[1]);
	return TRUE;
    }
    return FALSE;
}

void
IfSetup()
{
    extern char *ifName;
 
    static struct env_s envlist[]={
	{"DEV", {&ifName}, ENV_STRING|ENV_RDONLY, 0, 0, 0444},
	{"MTU", {EnvIfMtu}, ENV_SET, 0, 0, 0666},
	{NULL}
    };
    ifName = (char *)Malloc(IFNAMSIZ);
    RegisterEnvs(envlist, "IF", NULL);
}
