#include <stdio.h>
#include <stdarg.h>
#include <math.h>

#include "fppxp.h"

/*
 * Note tag object
 */

struct notetag_spec {
    char **names;
    int *cols;
    int w, n, h, pos;
};

static int
handle_notetag(FL_OBJECT *obj, int event, FL_Coord mx, FL_Coord my,
	       int key, void *ev)
{
    struct notetag_spec *spec=(struct notetag_spec *)obj->spec;
    int i, x, ret=0;

    if (!spec->names) return(0);
    switch (event) {
    case FL_PUSH:
	x = obj->x;
	if (my > obj->y + spec->h) break;
	for (i = 0; i < spec->n; i ++) {
	    if (x < mx && mx < x + spec->w) {
		spec->pos = i;
		ret = 1;
		break;
	    }
	    x += spec->w;
	}
    case FL_DRAW:
	x = obj->x;
	fl_set_clipping(obj->x, obj->y, obj->w, spec->h);
	for (i = 0; i < spec->n; i ++) {
	    fl_drw_box(obj->boxtype, x, obj->y,
		       spec->w, spec->h + obj->bw + 1,
		       spec->cols[i], -obj->bw);
	    fl_drw_text(FL_ALIGN_INSIDE|FL_ALIGN_CENTER,
			x, obj->y, spec->w, spec->h + obj->bw,
			obj->lcol, obj->lstyle, obj->lsize,
			spec->names[i]);
	    x += spec->w;
	}
	fl_unset_clipping();
	if (event == FL_DRAW) {
	    fl_drw_box(obj->boxtype, obj->x, obj->y + spec->h, obj->w,
		       obj->h - spec->h, spec->cols[spec->pos],
		       -obj->bw);
	    fl_rectf(obj->x + obj->bw + spec->w * spec->pos,
		     obj->y + spec->h - 1, spec->w - (obj->bw << 1),
		     obj->bw + obj->bw, spec->cols[spec->pos]);
	}
    case FL_DRAWLABEL:
	fl_draw_object_label(obj);
	break;
    case FL_FREEMEM:
	for (i = 0; spec->n; i ++)
	    if (spec->names[i]) fl_free(spec->names[i]);
	fl_free(spec->names);
	fl_free(obj->spec);
	break;
    }
    return(ret);
}

FL_OBJECT *
fl_create_notetag(int type, FL_Coord x, FL_Coord y, FL_Coord w,
		  FL_Coord h, const char *label)
{
    FL_OBJECT *obj;
    struct notetag_spec *spec;
    int i, bw;

    obj = fl_make_object(FL_NOTETAG, type, x, y, w, h, label,
			 handle_notetag);
    obj->boxtype = FL_NOTETAG_BOXTYPE;
    obj->col1 = FL_NOTETAG_COL1;
    obj->col2 = FL_NOTETAG_COL2;
    obj->align = FL_NOTETAG_ALIGN;
    obj->lcol = FL_NOTETAG_LCOL;
/*    obj->active = FALSE;*/

    obj->spec = (void *)fl_calloc(1, sizeof(struct notetag_spec));
    spec = (struct notetag_spec *)obj->spec;
    spec->h = fl_get_char_height(obj->lstyle, obj->lsize, 0, 0)
	+ (obj->bw << 1);
    return(obj);
}

FL_OBJECT *
fl_add_notetag(int type, FL_Coord x, FL_Coord y, FL_Coord w,
	       FL_Coord h, const char *label)
{
    FL_OBJECT *obj;

    obj = fl_create_notetag(type, x, y, w, h, label);
    fl_add_object(fl_current_form, obj);
    return(obj);
}

void
fl_setup_notetag(FL_OBJECT *obj, char *names[], int n)
{
    struct notetag_spec *spec=(struct notetag_spec *)obj->spec;
    int i, w;

    spec->n = n;
    spec->names = (char **)fl_calloc(n, sizeof(char *));
    spec->cols = (int *)fl_calloc(n, sizeof(int));
    spec->w = 0;
    for (i = 0; i < n; i ++) {
	spec->names[i] = strdup(names[i]);
	spec->cols[i] = obj->col1;
	w = fl_get_string_width(obj->lstyle, obj->lsize,
				names[i], strlen(names[i]));
	if (spec->w < w) spec->w = w;
    }
    spec->w += (obj->bw << 2);
}

void
fl_set_notetag(FL_OBJECT *obj, int n)
{
    struct notetag_spec *spec=(struct notetag_spec *)obj->spec;
    spec->pos = n;
}

int
fl_get_notetag(FL_OBJECT *obj)
{
    struct notetag_spec *spec=(struct notetag_spec *)obj->spec;
    return(spec->pos);
}

void
fl_set_notetag_color(FL_OBJECT *obj, int n, int col)
{
    struct notetag_spec *spec=(struct notetag_spec *)obj->spec;

    if (!spec->cols) return;
    spec->cols[n] = col;
}

/*
 * IP Input object
 */

struct ipinput_spec {
    unsigned int addr, mbits;
    int pos, n;
    int tabs[5];
    unsigned short new[5];
};

static void
draw_normal_ipinput(FL_OBJECT *obj, struct ipinput_spec *spec)
{
    int i, w, h, x, y, dx;

/*
    w = (obj->w - (obj->bw << 1)) / spec->n - obj->bw;
    h = fl_get_char_height(obj->lstyle, obj->lsize, 0, 0);
    y = obj->y + obj->bw;
    x = obj->x + obj->bw;
    dx = ((w - spec->w) >> 1);
*/
}

static int
key_ipinput(FL_OBJECT *obj, struct ipinput_spec *spec,
	    int key, XKeyEvent *xev)
{
    unsigned short *sp;
    int ret=0;

    switch (key) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
	if (spec->pos < 0) break;
	if (spec->pos < 4) {
	    sp = &spec->new[spec->pos];
	    *sp *= 10;
	    *sp %= 1000;
	    if (*sp > 255) *sp = *sp % 100;
	    *sp += (key - '0');
	} else if (spec->pos == 4) {
	    sp = &spec->new[spec->pos];
	    *sp *= 10;
	    *sp %= 100;
	    if (*sp > 32) *sp = *sp % 10;
	    *sp += (key - '0');
	}
	break;
    case '.':
    case 0x9:
	if (spec->pos < 0) break;
	if (++ spec->pos == spec->n) {
	    spec->pos = -1;
	    ret = FL_RETURN_END;
	}
	break;
    case 0x8:
    case 0x7f:
	if (spec->pos < 0) break;
	if (spec->pos < spec->n) {
	    sp = &spec->new[spec->pos];
	    *sp /= 10;
	}
	break;
    default:
	printf("%x\n", key);
    }
    return(ret);
}

static int
handle_ipinput(FL_OBJECT *obj, int event, FL_Coord mx, FL_Coord my,
	      int key, void *xev)
{
    struct ipinput_spec *spec=(struct ipinput_spec *)obj->spec;
    int i, w, x, h, y, t=0, ret=0;
    unsigned char str[4];

    switch (event) {
    case FL_UNFOCUS:
    case FL_PUSH:
	spec->pos = -1;
	if (event != FL_UNFOCUS) {
	    x = spec->n - 1;
	    for (i = 0; i < x; i ++) {
		if (mx < spec->tabs[i]) {
		    spec->pos = i;
		    break;
		}
	    }
	    if (i == x) spec->pos = x;
	}
	if (!spec->tabs[0]) break;
    case FL_DRAW:
	fl_drw_box(obj->boxtype, obj->x, obj->y, obj->w, obj->h,
		   obj->col1, obj->bw);
	w = ((obj->w - obj->bw) / spec->n);
	x = obj->x;
	h = obj->h;
	y = obj->y;
	for (i = 0; i < 4; i ++) {
	    sprintf(str, "%3d", spec->new[i]);
	    fl_drw_text(FL_ALIGN_RIGHT, x + obj->bw, y, w, h,
			obj->lcol, obj->lstyle, obj->lsize, str);
	    if (i) fl_ovalf(x, y + h - (obj->bw << 1),
			    3, 3, obj->lcol);
	    x += w;
	    spec->tabs[t ++] = x;
	}
	if (obj->type == FL_MASKBITS_IPINPUT) {
	    x += obj->bw * 2;
	    w -= obj->bw * 2;
	    sprintf(str, "%d", spec->new[4]);
	    fl_line(x - obj->bw, y + h - obj->bw,
		    x, y + obj->bw, obj->lcol);
	    fl_drw_text(FL_ALIGN_LEFT, x, y, w, h,
			obj->lcol, obj->lstyle, obj->lsize, str);
	    spec->tabs[t] = x + w;
	}
	if (spec->pos >= 0) {
	    x = spec->tabs[spec->pos] - 1;
	    w = fl_get_linewidth();
	    fl_set_linewidth(2);
	    fl_line(x, y + h - obj->bw, x, y + obj->bw, FL_BLUE);
	    fl_set_linewidth(w);
	}
    case FL_DRAWLABEL:
	fl_draw_object_label(obj);
	break;
    case FL_FREEMEM:
	fl_free(obj->spec);
	break;
    case FL_KEYBOARD:
	ret = key_ipinput(obj, spec, key, xev);
	fl_redraw_object(obj);
/*
    default:
	printf("%x\n", event);
*/
    }
    return(ret);
}

FL_OBJECT *
fl_create_ipinput(int type, FL_Coord x, FL_Coord y, FL_Coord w,
		  FL_Coord h, const char *label)
{
    FL_OBJECT *obj;
    struct ipinput_spec *spec;
    int i, bw;

    obj = fl_make_object(FL_IPINPUT, type, x, y, w, h, label,
			 handle_ipinput);
    obj->boxtype = FL_IPINPUT_BOXTYPE;
    obj->col1 = FL_IPINPUT_COL1;
    obj->col2 = FL_IPINPUT_COL2;
    obj->align = FL_IPINPUT_ALIGN;
    obj->lcol = FL_IPINPUT_LCOL;
    obj->wantkey = FL_KEY_TAB;
/*    obj->wantkey = FL_KEY_ALL;*/
    obj->input = 1;

    obj->spec = (void *)fl_calloc(1, sizeof(struct ipinput_spec));
    spec = (struct ipinput_spec *)obj->spec;
    spec->n = (type == FL_MASKBITS_IPINPUT) ? 5: 4;

    return(obj);
}

FL_OBJECT *
fl_add_ipinput(int type, FL_Coord x, FL_Coord y, FL_Coord w,
	       FL_Coord h, const char *label)
{
    FL_OBJECT *obj;

    obj = fl_create_ipinput(type, x, y, w, h, label);
    fl_add_object(fl_current_form, obj);
    return(obj);
}

void
fl_set_ipinput(FL_OBJECT *obj, unsigned int addr, unsigned int mbits)
{
    struct ipinput_spec *spec=(struct ipinput_spec *)obj->spec;
    int i;
    unsigned char buf[4];

    spec->addr = addr;
    spec->mbits = mbits;
    memcpy(buf, &spec->addr, 4);
    for (i = 0; i < 4; i ++) spec->new[i] = buf[i];
    spec->new[i] = mbits;
    fl_redraw_object(obj);
}

int
fl_apply_ipinput(FL_OBJECT *obj)
{
    struct ipinput_spec *spec=(struct ipinput_spec *)obj->spec;
    int i;
    unsigned int addr, mbits;
    unsigned char buf[4];

    for (i = 0; i < 4; i ++) buf[i] = spec->new[i];
    memcpy(&addr, buf, 4);
    if ((spec->addr != addr) ||
	(obj->type == FL_MASKBITS_IPINPUT
	 && spec->mbits != spec->new[4])) {
	spec->addr = addr;
	spec->mbits = spec->new[4];
	return(1);
    }
    return(0);
}

unsigned int
fl_get_ipinput_addr(FL_OBJECT *obj)
{
    struct ipinput_spec *spec=(struct ipinput_spec *)obj->spec;

    return(spec->addr);
}

unsigned int
fl_get_ipinput_mask(FL_OBJECT *obj)
{
    struct ipinput_spec *spec=(struct ipinput_spec *)obj->spec;

    return(spec->mbits);
}

/*
 * LEDs object
 */

struct leds_spec {
    int *leds;
    char **names;
    int w, h;
    int n;
};

static void
draw_normal_leds(FL_OBJECT *obj, struct leds_spec *spec)
{
    int i, w, h, x, y, dx;

    w = (obj->w - (obj->bw << 1)) / spec->n - obj->bw;
    h = fl_get_char_height(obj->lstyle, obj->lsize, 0, 0);
    y = obj->y + obj->bw;
    x = obj->x + obj->bw;
    dx = ((w - spec->w) >> 1);
    for (i = 0; i < spec->n; i ++) {
	fl_drw_text(FL_ALIGN_CENTER, x, y, w, h,
		    obj->lcol, obj->lstyle, obj->lsize, spec->names[i]);
	fl_rectf(x + dx, y + h, spec->w, spec->h, spec->leds[i]);
	x += w + obj->bw;
    }
}

static int
handle_leds(FL_OBJECT *obj, int event, FL_Coord mx, FL_Coord my,
	      int key, void *ev)
{
    struct leds_spec *spec=(struct leds_spec *)obj->spec;
    int i;

    if (!spec->leds) return(0);
    switch (event) {
    case FL_DRAW:
	fl_drw_box(obj->boxtype, obj->x, obj->y, obj->w, obj->h,
		   obj->col1, obj->bw);
	draw_normal_leds(obj, spec);
    case FL_DRAWLABEL:
	fl_draw_object_label(obj);
	break;
    case FL_FREEMEM:
	for (i = 0; spec->n; i ++)
	    if (spec->names[i]) fl_free(spec->names[i]);
	fl_free(spec->leds);
	fl_free(spec->names);
	fl_free(obj->spec);
	break;
    }
    return(0);
}

FL_OBJECT *
fl_create_leds(int type, FL_Coord x, FL_Coord y, FL_Coord w, FL_Coord h,
	       const char *label)
{
    FL_OBJECT *obj;
    struct leds_spec *spec;
    int i, bw;

    obj = fl_make_object(FL_LEDS, type, x, y, w, h, label, handle_leds);
    obj->boxtype = FL_LEDS_BOXTYPE;
    obj->col1 = FL_LEDS_COL1;
    obj->col2 = FL_LEDS_COL1;
    obj->align = FL_LEDS_ALIGN;
    obj->lcol = FL_LEDS_LCOL;
    obj->active = FALSE;

    obj->spec = (void *)fl_calloc(1, sizeof(struct leds_spec));
    spec = (struct leds_spec *)obj->spec;

    return(obj);
}

FL_OBJECT *
fl_add_leds(int type, FL_Coord x, FL_Coord y, FL_Coord w, FL_Coord h,
	    const char *label)
{
    FL_OBJECT *obj;

    obj = fl_create_leds(type, x, y, w, h, label);
    fl_add_object(fl_current_form, obj);
    return(obj);
}

void
fl_setup_leds(FL_OBJECT *obj, int n, int w, int h)
{
    struct leds_spec *spec=(struct leds_spec *)obj->spec;

    if (spec->leds) fl_free(spec->leds);
    spec->leds = (int *)fl_calloc(n, sizeof(int));
    if (spec->names) fl_free(spec->names);
    spec->names = (char **)fl_calloc(n, sizeof(char *));
    spec->n = n;
    spec->w = w;
    spec->h = h;
}

void
fl_set_led_name(FL_OBJECT *obj, int n, char *name)
{
    struct leds_spec *spec=(struct leds_spec *)obj->spec;

    spec->names[n] = strdup(name);
}

void
fl_set_led(FL_OBJECT *obj, int n, int col)
{
    struct leds_spec *spec=(struct leds_spec *)obj->spec;

    spec->leds[n] = col;
}

/*
 * Bchart object
 */

struct bchart_spec {
    float vmax;
    int scale;
    int ox, oy;	/* origin x, y */
    int w, h;
    char *lname, *rname;
    struct entry_s {
	float *lval, *rval;
	int lcol, rcol, style;	/* LineSolid, LineOnOffDash,
				   LineDoubleDash */
    } entry[FL_BCHART_MAX];
};

static void
draw_normal_bchart(FL_OBJECT *obj, struct bchart_spec *spec)
{
    struct entry_s *entry;
    float vm, max=0.0, min=0.0;
    int w, i, v, r0, l0, r1, l1, y;
    char smax[20];

    r0 = l0 = spec->ox;
    fl_set_clipping(obj->x + 2, obj->y + 2, obj->w - 4, obj->h - 4);
    if (spec->scale == FL_BCHART_LOGSCALE) {
	max = spec->vmax > 0 ? log(spec->vmax): 1;
	vm = (float)(spec->w >> 1)/(max > 0 ? max: 1);
    } else
	vm = (float)(spec->w >> 1)/(spec->vmax > 0 ? spec->vmax: 1);
/*    max = spec->vmax;*/
    fl_line(spec->ox, obj->y + 1, spec->ox, obj->y + obj->h - 2,
	    FL_BLACK);
    for (i = 0; i < FL_BCHART_MAX; i ++) {
	entry = &spec->entry[i];
	if (entry->style < 0) continue;
	fl_linestyle(entry->style);
	for (v = 0; v < spec->h; v ++) {
	    if (entry->lval[v] > max) max = entry->lval[v];
	    if (entry->rval[v] > max) max = entry->rval[v];
	    if (entry->lval[v] < min) min = entry->lval[v];
	    if (entry->rval[v] < min) min = entry->rval[v];
	    if (spec->scale == FL_BCHART_LOGSCALE) {
		l1 = spec->ox - (int)(entry->lval[v] > 0 ?
				      log(entry->lval[v]) * vm: 0);
		r1 = spec->ox + (int)(entry->rval[v] > 0 ?
				      log(entry->rval[v]) * vm: 0);
	    } else {
		l1 = spec->ox - (int)(entry->lval[v] * vm);
		r1 = spec->ox + (int)(entry->rval[v] * vm);
	    }
	    if (v) {
		y = spec->oy + v;
		fl_line(l0, y, l1, y, entry->lcol);
		fl_line(r0, y, r1, y, entry->rcol);
	    } else {
		min = entry->lval[0] < entry->rval[0] ?
		    entry->lval[0]: entry->rval[0];
	    }
	    l0 = l1;
	    r0 = r1;
	}
    }
    if (spec->scale == FL_BCHART_MIDSCALE)
	spec->vmax = max + min;
    else if (spec->scale != FL_BCHART_LOGSCALE)
	spec->vmax = max;
    fl_unset_clipping();
    fl_drw_text(FL_ALIGN_LEFT|FL_ALIGN_TOP,
		obj->x, obj->y - obj->bw, 0, 0,
		obj->lcol, obj->lstyle, obj->lsize, spec->lname);
    fl_drw_text(FL_ALIGN_RIGHT|FL_ALIGN_TOP,
		obj->x + obj->w, obj->y - obj->bw, 0, 0,
		obj->lcol, obj->lstyle, obj->lsize, spec->rname);
    if (spec->scale != FL_BCHART_LOGSCALE) {
	sprintf(smax, "%g", spec->vmax);
	fl_rectf(obj->x, obj->y + obj->h + 1, obj->w,
		 fl_get_char_height(obj->lstyle, obj->lsize, 0, 0),
		 obj->col2);
	fl_drw_text_beside(FL_ALIGN_CENTER|FL_ALIGN_BOTTOM,
			   obj->x, obj->y, obj->w, obj->h,
			   obj->lcol, obj->lstyle, obj->lsize, smax);
    }
}

static int
handle_bchart(FL_OBJECT *obj, int event, FL_Coord mx, FL_Coord my,
	      int key, void *ev)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;
    int i;

    switch (event) {
    case FL_DRAW:
	fl_drw_box(obj->boxtype, obj->x, obj->y, obj->w, obj->h,
		   obj->col1, obj->bw);
	switch (obj->type) {
	case FL_NORMAL_BCHART:
	    draw_normal_bchart(obj, spec);
	    break;
	}
    case FL_DRAWLABEL:
	fl_draw_object_label(obj);
	break;
    case FL_FREEMEM:
	for (i = 0; i < FL_BCHART_MAX; i ++) {
	    fl_free(spec->entry[i].lval);
	    fl_free(spec->entry[i].rval);
	}
	if (spec->lname) fl_free(spec->lname);
	if (spec->rname) fl_free(spec->rname);
	fl_free(obj->spec);
	break;
    }
    return(0);
}

FL_OBJECT *
fl_create_bchart(int type, FL_Coord x, FL_Coord y, FL_Coord w, FL_Coord h,
	       const char *label)
{
    FL_OBJECT *obj;
    struct bchart_spec *spec;
    int i, bw;

    obj = fl_make_object(FL_BCHART, type, x, y, w, h, label, handle_bchart);
    obj->boxtype = FL_BCHART_BOXTYPE;
    obj->col1 = FL_BCHART_COL1;
    obj->col2 = FL_BCHART_COL2;
    obj->align = FL_BCHART_ALIGN;
    obj->lcol = FL_BCHART_LCOL;
    obj->active = FALSE;

    obj->spec = (void *)fl_calloc(1, sizeof(struct bchart_spec));
    spec = (struct bchart_spec *)obj->spec;
    spec->w = obj->w - (obj->bw << 1);
    spec->h = obj->h - (obj->bw << 1);
    spec->vmax = (float)spec->w;
    bw = (obj->w - spec->w) >> 1;
    spec->ox = obj->x + (obj->w >> 1);
    spec->oy = obj->y + obj->bw;

    for (i = 0; i < FL_BCHART_MAX; i ++) {
	spec->entry[i].lval = (float *)
	    fl_calloc(spec->h, sizeof(float));
	spec->entry[i].rval =  (float *)
	    fl_calloc(spec->h, sizeof(float));
	spec->entry[i].style = -1;
    }
    return(obj);
}

FL_OBJECT *
fl_add_bchart(int type, FL_Coord x, FL_Coord y, FL_Coord w, FL_Coord h,
	    const char *label)
{
    FL_OBJECT *obj;

    obj = fl_create_bchart(type, x, y, w, h, label);
    fl_add_object(fl_current_form, obj);
    return(obj);
}

void
fl_set_bchart_value(FL_OBJECT *obj, int e, double lv, double rv)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;

    memcpy(spec->entry[e].lval, spec->entry[e].lval + 1,
	   sizeof(float) * (spec->h - 1));
    memcpy(spec->entry[e].rval, spec->entry[e].rval + 1,
	   sizeof(float) * (spec->h - 1));
    spec->entry[e].lval[spec->h - 1] = lv;
    spec->entry[e].rval[spec->h - 1] = rv;
}

void
fl_set_bchart_name(FL_OBJECT *obj, char *ln, char *rn)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;

    if (spec->lname) fl_free(spec->lname);
    if (spec->rname) fl_free(spec->rname);
    spec->lname = strdup(ln);
    spec->rname = strdup(rn);
}

void
fl_set_bchart_color(FL_OBJECT *obj, int e, int lc, int rc)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;

    spec->entry[e].lcol = lc;
    spec->entry[e].rcol = rc;
}

void
fl_set_bchart_scale(FL_OBJECT *obj, int scale)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;

    spec->scale = scale;
}

void
fl_set_bchart_style(FL_OBJECT *obj, int e, int style)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;

    spec->entry[e].style = style;
}

void
fl_set_bchart_max(FL_OBJECT *obj, double max)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;

    spec->vmax = max;
}

double
fl_get_bchart_max(FL_OBJECT *obj)
{
    struct bchart_spec *spec=(struct bchart_spec *)obj->spec;

    return(spec->vmax);
}
