/*
 * passes.c  -  Compress kernel image and concatenate it with a boot loader
 *
 * Copyright (C) 1997,1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"
#include "nblib.h"
#include <memory.h>
#include "makerom.h"



/*
 * Definitions local for this module
 */
#define PKTHEADER	4		/* Length of packet driver header */
#define SIZEOFS		2		/* Offset to word with size of pktdrv */
#define EXESIG		0x5A4D		/* Signature of DOS EXE files */
#define ROMSIG		0xaa55		/* ROM signature */
#define COMSIG		0x4B47		/* Signature for packet driver header */

#define FILEHEADER	0x001A		/* Req size to determine type of file */

#define EXESECTSIZE	512		/* Size of one sector in EXE file */
#define EXELASTSECT	0x0002		/* Number of bytes in last EXE sector */
#define EXEFSIZE	0x0004		/* Number of sectors in EXE file */
#define EXEHEADSIZE	0x0008		/* Offset to size of EXE file header */
#define EXEMINPARA	0x000A		/* Required number of paragraphs */
#define EXEMINHSIZE	0x0090		/* Minimum size of EXE file header */
#define EXEMAXSIZE	0xFFFE		/* Maximum memory size of EXE file */

#define ROMVECT18	0x18		/* Rom interrupt vector 18h */
#define ROMVECT19	0x19		/* Rom interrupt vector 19h */



/*
 * Check amount of memory required by EXE file
 */
static void execheck(buf, drivername)
unsigned char *buf;
char *drivername;
{
  long size, l;

  l = ttoh(getval(*((__u16 *)(&buf[EXELASTSECT]))));
  size  = (ttoh(getval(*((__u16 *)(&buf[EXEFSIZE])))) - 1) * EXESECTSIZE;
  size += ttoh(getval(*((__u16 *)(&buf[EXEMINPARA])))) * 16;
  size += (l == 0 ? EXESECTSIZE : l);
  if (size > EXEMAXSIZE) {
	fprintf(stderr, "%s: Program %s requires too much memory\n",
							progname, drivername);
	exit(EXIT_MAKEROM_INVEXE);
  }
}



/*
 * Pass 1 - concatenate the kernel image with the driver programs
 */
void pass1(bp, tempfile)
struct bootdef *bp;
int tempfile;
{
  unsigned long argofs[MAXDRIVERS];
  unsigned long pktofs[MAXDRIVERS];
  unsigned long pktlen[MAXDRIVERS];
  unsigned long tabofs;
  unsigned long writecnt;
  unsigned char inbuf[BLKSIZE];
  unsigned long krnlength;
  unsigned int len;
  int i, infile;
  __u16 *u16p;

  /*
   * Open the input file, read the first block, and check type of file.
   * If the input file is a boot floppy image, copy the first block directly
   * to the output file, because it is the floppy boot sector.
   */
  len = BLKSIZE;
  if ((infile = open(bp->kernelname, O_RDONLY | O_BINARY)) < 0) {
	fprintf(stderr, "%s: unable to open kernel file %s\n",
						progname, bp->kernelname);
	exit(EXIT_MAKEROM_OPENKERN);
  }

  /*
   * Read the first block and determine the actual size of the rom kernel
   * without the BSS segment. Also check the version number.
   */
  if (nbread(inbuf, len, infile) != len) {
	fprintf(stderr, "%s: unexpected end of file %s\n",
						progname, bp->kernelname);
	exit(EXIT_MAKEROM_READKERN);
  }
  if (*((__u8 *)(&inbuf[(int)(KRNMAJOROFS)])) != VER_MAJOR ||
      *((__u8 *)(&inbuf[(int)(KRNMINOROFS)])) != VER_MINOR) {
	fprintf(stderr, "%s: invalid kernel version number\n", progname);
	exit(EXIT_MAKEROM_INVKERN);
  }
  krnlength  = (ttoh(getval(*((__u16 *)(&inbuf[(int)(KRNTEXTOFS)])))) + 15) & 0xfff0;
  krnlength += (ttoh(getval(*((__u16 *)(&inbuf[(int)(KRNDATAOFS)])))) + 15) & 0xfff0;

  /*
   * Now copy the rom image file into the output file. While writing to
   * the output file, count the number of bytes written, so that the offset
   * to the packet driver can be determined. Also only copy the relevant
   * bytes excluding the BSS segment, which has only zero bytes anyway.
   */
  writecnt = 0;
  krnlength -= len;
  while (TRUE) {
	writecnt += nbwrite(inbuf, len, tempfile);
	if (krnlength <= 0) break;
	if ((len = nbread(inbuf, BLKSIZE, infile)) == 0) break;
	if (len > krnlength)
		len = krnlength;
	krnlength -= len;
  }
  close(infile);
  if (verbose > 2)
	printf("End position of kernel:         %ld\n", writecnt);

  /*
   * Next write a dummy pointer table to the output file. This table gets
   * later filled with the pointers to all driver arguments and program
   * images.
   */
  tabofs = writecnt;
  len = (bp->drivernum + 1) * 4;
  memset(inbuf, 0, len);
  writecnt += nbwrite(inbuf, len, tempfile);
  if (verbose > 2)
	printf("End position of pointer table:  %ld\n", writecnt);

  /*
   * Next write all driver argument lines to the output file. The argument
   * strings contain a leading byte which gives the number of characters
   * in the string.
   */
  for (i = 0; i < bp->drivernum; i++)
	if (bp->drivernames[i] != NULL) {
		argofs[i] = writecnt;
		len = strlen(bp->driverargs[i]);
		inbuf[0] = len & 0xFF;
		strncpy((char *)(&inbuf[1]), bp->driverargs[i], len);
		inbuf[len+1] = 0;
		len += 2;
		writecnt += nbwrite(inbuf, len, tempfile);
	}
  if (verbose > 2)
	printf("End position of prog arguments: %ld\n", writecnt);

  /*
   * Now copy the packet driver into the output file, but retain some
   * space for the packet driver header.
   */
  for (i = 0; i < bp->drivernum; i++)
	if (bp->drivernames[i] != NULL) {
		if (bp->drivertypes[i] != TYPE_DOS &&
		    bp->drivertypes[i] != TYPE_PD) {
			fprintf(stderr, "%s: unsupported network driver type\n",
								progname);
			exit(EXIT_MAKEROM_INVDRV);
		}
		pktofs[i] = writecnt;
		pktlen[i] = 0;
		if ((infile = open(bp->drivernames[i], O_RDONLY | O_BINARY)) < 0) {
			fprintf(stderr, "%s: unable to open driver file %s\n",
						progname, bp->drivernames[i]);
			exit(EXIT_MAKEROM_OPENDRV);
		}
		memset(inbuf, 0, PKTHEADER);
		len = nbread(&inbuf[PKTHEADER], FILEHEADER, infile) + PKTHEADER;

		/* Write optional EXE program loader */
		if (ttoh(getval(*((__u16 *)(&inbuf[PKTHEADER])))) == EXESIG &&
		    len >= FILEHEADER + PKTHEADER) {
			unsigned char *exebuf;
			long exehlength;
			size_t newlen;

			/* Write packet driver header and EXE loader */
			pktlen[i] += nbwrite(inbuf, PKTHEADER, tempfile);
			pktlen[i] += nbwrite(execom_data, execom_data_size,
								tempfile);
			len -= PKTHEADER;
			exehlength = ttoh(getval(*((__u16 *)(&inbuf[PKTHEADER +
							EXEHEADSIZE])))) * 16;
			if (exehlength < len) {
				fprintf(stderr, "%s: invalid EXE header in file %s\n",
						progname, bp->drivernames[i]);
				exit(EXIT_MAKEROM_INVEXE);
			}

			/* Pad EXE header to minimum size */
			newlen = (exehlength < EXEMINHSIZE ?
						EXEMINHSIZE : exehlength);
			exebuf = nbmalloc(newlen);
			memcpy(exebuf, &inbuf[PKTHEADER], len);
			u16p = (__u16 *)(&exebuf[EXEHEADSIZE]);
			assign(*u16p, htot(newlen / 16));

			/* Copy remaining part of EXE file header */
			if (nbread(&exebuf[len], exehlength - len, infile) <
							exehlength - len) {
				fprintf(stderr, "%s: unexpected end of program %s\n",
						progname, bp->drivernames[i]);
				exit(EXIT_MAKEROM_EXEEOF);
			}
			execheck(exebuf, bp->drivernames[i]);
			pktlen[i] += nbwrite(exebuf, newlen, tempfile);
			free(exebuf);
			len = nbread(inbuf, BLKSIZE, infile);
		}

		/* Finally write the rest of the executable */
		while (TRUE) {
			pktlen[i] += nbwrite(inbuf, len, tempfile);
			if ((len = nbread(inbuf, BLKSIZE, infile)) == 0)
				break;
		}
		close(infile);
		writecnt += pktlen[i];
		pktlen[i] -= PKTHEADER;
		if (verbose > 2)
			printf("End position of prog %d:         %ld\n",
								i, writecnt);
	}

  /*
   * Finally write the driver table offset, the driver command line
   * and image pointers and the driver header information into the
   * destination file.
   */
  if (writecnt > 65535L) {
	fprintf(stderr, "%s: size of output file >64kB\n", progname);
	exit(EXIT_MAKEROM_SIZE64);
  }

  if (lseek(tempfile, (long)DATAOFS, 0) < 0) {
	fprintf(stderr, "%s: unable to seek to data area in output file\n",
								progname);
	exit(EXIT_SEEK);
  }
  u16p = (__u16 *)(&inbuf[(int)(KRNPRGOFS - DATAOFS)]);
  assign(*u16p, htot(tabofs));
  (void)nbwrite(inbuf, sizeof(__u16), tempfile);

  if (lseek(tempfile, (long)tabofs, 0) < 0) {
	fprintf(stderr, "%s: unable to seek to driver table in output file\n",
								progname);
	exit(EXIT_SEEK);
  }
  u16p = (__u16 *)inbuf;
  for (i = 0; i < bp->drivernum; i++)
	if (bp->drivernames[i] != NULL) {
		assign(*u16p, htot(pktofs[i]));
		u16p++;
		assign(*u16p, htot(argofs[i]));
		u16p++;
	}
  (void)nbwrite(inbuf, (unsigned int)((unsigned char *)u16p - inbuf), tempfile);

  for (i = 0; i < bp->drivernum; i++)
	if (bp->drivernames[i] != NULL) {
		if (lseek(tempfile, (long)pktofs[i], 0) < 0) {
			fprintf(stderr, "%s: unable to seek to beginning of "
						"driver in output file\n",
						progname);
			exit(EXIT_SEEK);
		}
		u16p = (__u16 *)(&inbuf[0]);
		assign(*u16p, htot(COMSIG));
		u16p = (__u16 *)(&inbuf[SIZEOFS]);
		assign(*u16p, htot((pktlen[i] + 15) >> 4));
		nbwrite(inbuf, PKTHEADER, tempfile);
	}

  /* Rewind to beginning of output file for the following pass 2 */
  if (lseek(tempfile, 0L, 0) < 0) {
	fprintf(stderr, "%s: unable to seek to beginning of output file\n",
								progname);
	exit(EXIT_SEEK);
  }
  if (verbose > 2)
	printf("\n\n");
}



/*
 * Pass 2 - compress kernel image and concatenate it with the loader
 */
void pass2(bp, kernfile, loader)
struct bootdef *bp;
int kernfile;
int loader;
{
  unsigned long romlength;		/* Size of bootrom kernel image */
  unsigned long writecnt;		/* number of bytes in output file */
  unsigned char inbuf[BLKSIZE];		/* Input file buffer */
  unsigned int len;
  char *tmpfname;
  char *ldrname = bp->loadernames[loader];
  char *outname = bp->outnames[loader];
  int outtype = bp->outtypes[loader];
  int outfile;
  int infile;
  int dorom;
  int i;
  __u16 *u16p;

  /* Always have to set the checksum to zero before starting pass 2 */
  write_chksum = 0;

  /*
   * Check for the ROM signature to determine whether we are going to build
   * a ROM or a floppy image.
   */
  if ((infile = open(ldrname, O_RDONLY | O_BINARY)) < 0) {
	fprintf(stderr, "%s: unable to open loader file %s\n",
							progname, ldrname);
	exit(EXIT_MAKEROM_OPENLDR);
  }
  len = nbread(inbuf, BLKSIZE, infile);
  dorom = (ttoh(getval(*((__u16 *)(&inbuf[0])))) == ROMSIG);
  if (!dorom && outtype == OUT_FLASH) {
	/* Don't produce a flash header for floppy images */
	fprintf(stderr, "%s: unable to to produce floppy image with flash header\n",
								progname);
	exit(EXIT_MAKEROM_INVLDR);
  }

  /*
   * Open the output file. If we have to produce anything different than raw
   * binary we have to initially use a temporary file.
   */
  if (outtype != OUT_BINARY) {
	if ((tmpfname = tempnam(NULL, "mkrom")) == NULL) {
		fprintf(stderr, "%s: unable to generate temporary file name\n",
								progname);
		exit(EXIT_TEMPNAME);
	}
	if ((outfile = open(tmpfname, O_CREAT | O_RDWR | O_TRUNC | O_BINARY,
								0644)) < 0) {
		fprintf(stderr, "%s: unable to open temporary file %s\n",
							progname, tmpfname);
		exit(EXIT_CREATE);
	}
	if (verbose < 2)
		unlink(tmpfname);
  } else {
	if ((outfile = open(outname, O_CREAT | O_WRONLY | O_TRUNC | O_BINARY,
								0644)) < 0) {
		fprintf(stderr, "%s: unable to open output file %s\n",
							progname, outname);
		exit(EXIT_CREATE);
	}
  }

  /* Now copy the specified bootrom loader into the output file */
  writecnt = 0;
  while (TRUE) {
	writecnt += nbwrite(inbuf, len, outfile);
	if ((len = nbread(inbuf, BLKSIZE, infile)) == 0) break;
  }
  close(infile);
  if ((writecnt & 0x0f) != 0) {
	/* The kernel image has to start at a paragraph boundary */
	memset(inbuf, 0, 16);
	writecnt += nbwrite(inbuf, 16 - (writecnt & 0x0f), outfile);
  }

  /* Next copy the kernel image into the output file */
  romlength = freeze(kernfile, outfile);
  writecnt += romlength;
  if (writecnt > 65534L) {
	fprintf(stderr, "%s: bootrom code cannot be >64kB\n", progname);
	exit(EXIT_MAKEROM_SIZE64);
  }

  /*
   * If we got a real ROM here, determine the next available ROM size and
   * fill the output file up to that amount with 0 bytes. The last byte
   * has to remain for the checksum. Then update the size of the physical
   * ROM at the beginning of the output file, and write the size of the
   * bootrom kernel image.
   */
  if (dorom) {
	unsigned long serno;
	unsigned long romvector;	/* Rom interrupt vector */
	unsigned long romsize = 8192;	/* Physical size of ROM, 8kB is min. */
	unsigned char romsizchar;

	/* Compute the smallest possible ROM size */
	for (; romsize < writecnt + 1; )
		romsize <<= 1;

	/* Compute the serial number from the checksum and the current time */
	serno = (unsigned long)(time(NULL) & 0xffff) + write_chksum;

	/* Compute the ROM interrupt vector */
	romvector = ((bp->use_int18 ? ROMVECT18 : ROMVECT19) * 4) & 0xffff;

	/* Compute the missing size values and the final checksum */
	romsizchar = romsize >> 9;	/* Divide by 512 */
	write_chksum += romsizchar;
	for (i = 0; i < sizeof(romsize); i++)
		write_chksum += ((unsigned char *)&romlength)[i];
	write_chksum += (unsigned char)(serno & 0xff) +
				(unsigned char)((serno >> 8) & 0xff) +
				(unsigned char)(romvector & 0xff) +
				(unsigned char)((romvector >> 8) & 0xff);

	/* Copy 0 byte buffer and checksum to output file */
	len = BLKSIZE;
	memset(inbuf, 0, BLKSIZE);
	while (writecnt < romsize) {
		/* Check if this is the last block */
		if (romsize - writecnt <= len) {
			len = (int)(romsize - writecnt);
			inbuf[len - 1] = (unsigned char)
						(0 - (write_chksum & 0xff));
		}
		writecnt += nbwrite(inbuf, len, outfile);
	}

	/* Put the physical ROM length into the output file */
	if (lseek(outfile, (long)ROMLENOFS, 0) < 0) {
		fprintf(stderr, "%s: unable to seek %s\n", progname, outname);
		exit(EXIT_SEEK);
	}
	(void)nbwrite((__u8 *)&romsizchar, sizeof(__u8), outfile);

	/* Put the serial number into the output file */
	if (lseek(outfile, (long)ROMSEROFS, 0) < 0) {
		fprintf(stderr, "%s: unable to seek %s\n", progname, outname);
		exit(EXIT_SEEK);
	}
	u16p = (__u16 *)(&inbuf[0]);
	assign(*u16p, htot(serno));
	(void)nbwrite(inbuf, sizeof(__u16), outfile);

	/* Put the interrupt vector into the output file */
	if (lseek(outfile, (long)ROMVECTOFS, 0) < 0) {
		fprintf(stderr, "%s: unable to seek %s\n", progname, outname);
		exit(EXIT_SEEK);
	}
	u16p = (__u16 *)(&inbuf[0]);
	assign(*u16p, htot(romvector));
	(void)nbwrite(inbuf, sizeof(__u16), outfile);
  }

  /* The length of the kernel image has to go at the beginning of the file */
  if (lseek(outfile, (long)ROMSIZEOFS, 0) < 0) {
	fprintf(stderr, "%s: unable to seek %s\n", progname, outname);
	exit(EXIT_SEEK);
  }
  u16p = (__u16 *)(&inbuf[0]);
  assign(*u16p, htot(romlength));
  (void)nbwrite(inbuf, sizeof(__u16), outfile);

  /* Rewind to beginning of kernel file for another pass 2 */
  if (lseek(kernfile, 0L, 0) < 0) {
	fprintf(stderr, "%s: unable to seek to beginning of kernel file\n",
								progname);
	exit(EXIT_SEEK);
  }

  /* Generate the final output file format */
  if (outtype != OUT_BINARY) {
	if (lseek(outfile, 0L, 0) < 0) {
		fprintf(stderr, "%s: unable to seek to beginning of temporary file\n",
								progname);
		exit(EXIT_SEEK);
	}
	switch (outtype) {
		case OUT_FLASH:
			makeflash(outname, outfile);
			break;
		case OUT_IHEX:
			makeihex(outname, outfile);
			break;
		case OUT_MHEX:
			makemhex(outname, outfile);
			break;
		case OUT_THEX:
			makethex(outname, outfile);
			break;
		default:
			fprintf(stderr, "%s: out type %d not supported\n",
							progname, outtype);
			exit(EXIT_INTERNAL);
	}
  }
  close(outfile);
}

