/*
 *  nwrevoke.c
 *
 *  Remove trustee rights from file or directory
 *
 *  Copyright (C) 1996 by Volker Lendecke
 *
 *  Netware 3.11 and above support by Petr Vandrovec, 1999.
 *
 */

#include "config.h"

#include <ncp/nwcalls.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <libintl.h>
#define _(X) gettext(X)

static char *progname;

static void
usage(void)
{
	fprintf(stderr, _("usage: %s [options]\n"), progname);
}

static void
help(void)
{
	printf(_("\n"
		 "usage: %s [options] file/directory\n"), progname);
	printf(_("\n"
	       "-h             Print this help text\n"
	       "-S server      Server name to be used\n"
	       "-U username    Username sent to server\n"
	       "-P password    Use this password\n"
	       "-n             Do not use any password\n"
	       "-C             Don't convert password to uppercase\n"
	       "\n"
	       "-o object_name Name of object removed as trustee\n"
	       "-t type        Object type (decimal value)\n"
	       "\n"
	       "file/directory\n"
	       "\n"));
}

int
main(int argc, char *argv[])
{
	struct ncp_conn *conn;
	char *object_name = NULL;
	int object_type = -1;
	struct ncp_bindery_object o;
	char *path = NULL;
	long err;
	int result = 1;
	int opt;
	char volume[1000];
	unsigned char encpath[2000];
	int enclen;
	TRUSTEE_INFO tstinfo;
	int useConn = 0;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
	
	progname = argv[0];

	if ((conn = ncp_initialize_2(&argc, argv, 1, NCP_BINDERY_USER, &err, 0)) == NULL)
	{
		useConn = 1;
	}
	while ((opt = getopt(argc, argv, "h?o:t:")) != EOF)
	{
		switch (opt)
		{
		case 'o':
			object_name = optarg;
			str_upper(object_name);
			break;
		case 't':
			object_type = atoi(optarg);
			break;
		case 'h':
		case '?':
			help();
			goto finished;
		default:
			usage();
			goto finished;
		}
	}

	if (object_type < 0)
	{
		fprintf(stderr, _("%s: You must specify an object type\n"),
			argv[0]);
		goto finished;
	}
	if (object_name == NULL)
	{
		fprintf(stderr, _("%s: You must specify an object name\n"),
			argv[0]);
		goto finished;
	}
	if (!useConn) {
		if (optind != argc - 1)
		{
			fprintf(stderr, _("%s: You must specify a directory\n"),
				progname);
			goto finished;
		}
		path = argv[optind];
	} else {
		char directory[1000];

		if (optind < argc)
			path = argv[optind];
		else
			path = ".";
		
		if (NWParsePath(path, NULL, &conn, volume, directory) || !conn) {
			fprintf(stderr, _("%s: Could not find directory %s\n"),
					progname, path);
			goto finished;
		}
		strcat(volume, ":");
		strcat(volume, directory);
		path = volume;
	}
	if (ncp_get_bindery_object_id(conn, object_type, object_name, &o) != 0)
	{
		fprintf(stderr, _("%s: Could not find object %s\n"),
			progname, object_name);
		goto finished;
	}
	enclen = ncp_path_to_NW_format(path, encpath, sizeof(encpath));
	if (enclen < 0) {
		fprintf(stderr, _("%s: Invalid path: %s\n"), strerror(-enclen));
		goto finished;
	}
	tstinfo.objectID = o.object_id;
	tstinfo.objectRights = 0;
	if (ncp_ns_trustee_del(conn, NW_NS_DOS, 0xFF, 0, 0,
			encpath, enclen, &tstinfo, 1) != 0)
	{
		fprintf(stderr, _("%s: Could not remove trustee rights\n"),
			progname);
		goto finished;
	}
	result = 0;

finished:
	ncp_close(conn);
	return result;
}
