
/*********************************************************************
 *                
 * Filename:      tpctl.c
 * Description:   ThinkPad control utility
 * Status:        beta
 * Author:        Thomas Hood <jdthood@mail.com>
 *                Bill Mair <bill.mair@bm-soft.com>
 * Created:       19 July 1999 
 *
 * Please report bugs to the author ASAP.
 * 
 *     Copyright (c) 1999 J.D. Thomas Hood, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *
 ********************************************************************/

/*
 * A note for those parsing tpctl output:
 * The colon character can be counted on to be the last
 * character of the label for a line of output.
 */

#include <linux/unistd.h>
#include <stdlib.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <stdio.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <string.h>
#include <term.h>
#include "thinkpad_common.h"
#include "thinkpad.h"
#include "smapi.h"
#include "superio.h"
#include "smapidev.h"


/****** defines ******/

#define SZ_TPCTL_VERSION "0.8.1"

#define SZ_DEFAULT_DEVICE_NAME "/dev/thinkpad"
#define SZ_TPCTL_NAME "tpctl"

/*** tpctl error codes ***/
#define ERR_TPCTL ((int) 0x2000 )

/*
 * define ALLOW_POWEROFF to enable power-off command. 
 * For masochists only.
 */

/*
 * define VIAGRA to enable hardware access to I/O ports.  Without this,
 * tpctl will be restricted to the safe SMAPI BIOS softare interface to 
 * ThinkPad internals.  Hard access is slightly risky if your machine has
 * not already been tested with tpctl because tpctl (or more specifically
 * the superio module) makes assumptions about the presence of registers at
 * certain I/O addresses.  If your ThinkPad has a different architecture
 * from that of the 380Z, 600 and 770X (for examples) then unpleasant
 * consequences are conceivable.
 */

#ifndef VIAGRA
#undef DEBUG_REGS
#endif

/*** Highlighting ***/
#define HI(sz) pchHiOn, (sz), pchHiOff


/****** variables ******/

char szTpctlName[] = SZ_TPCTL_NAME;
char szTpctlVersion[] = SZ_TPCTL_VERSION;
char *pchHiOn, *pchHiOff;
char fHighlight, fVerbose;


/****** main functions ******/

int print_version( int intFiledesc )
{
	smapidev_info_t infoThe;
	smapidev_moduleinfo_t moduleinfoThe;
	int intRtn;

	intRtn = smapidev_GetInfo( &infoThe );
	if ( intRtn ) return intRtn;

	intRtn = smapidev_GetModuleInfo( intFiledesc, &moduleinfoThe );
	if ( intRtn ) return intRtn;

	printf(
		"%s %s accessing thinkpad module %s via smapidev %s\n",
		szTpctlName,
		szTpctlVersion,
		moduleinfoThe.szVersion,
		infoThe.szVersion
	);

	return 0;
}



int info_print_bios( int intFiledesc )
{
    smapidev_biosinfo_t biosinfoThe;
	smapidev_slaveinfo_t slaveinfoThe;
    int intRtn;

    biosinfoThe.sizeStruct = sizeof( biosinfoThe );

    intRtn = smapidev_GetBiosInfo( intFiledesc, &biosinfoThe );
    if ( intRtn ) return intRtn;

    printf(
        "%s%s%snformation about %s%s%sios and system:\n"
        "   system ID:                           %d\n"
        "   country code:                        %d\n"
        "   system BIOS revision:                %d.%02d\n"
        "   system management BIOS revision:     %d.%02d\n"
        "   SMAPI BIOS interface revision:       %d.%02d\n"
        "   video BIOS revision:                 %d.%02d\n",
		HI("i"), HI("b"),
        biosinfoThe.wSysId,
        biosinfoThe.wCountryCode,
        biosinfoThe.wSysBiosRevMajor,     biosinfoThe.wSysBiosRevMinor,
        biosinfoThe.wSysMgmtBiosRevMajor, biosinfoThe.wSysMgmtBiosRevMinor,
        biosinfoThe.wSmapiBiosIfaceRevMajor, biosinfoThe.wSmapiBiosIfaceRevMinor,
        biosinfoThe.wVideoBiosRevMajor,   biosinfoThe.wVideoBiosRevMinor
    );

	slaveinfoThe.sizeStruct = sizeof( slaveinfoThe );
	intRtn = smapidev_GetSlaveControllerInfo( intFiledesc, &slaveinfoThe );
	if ( intRtn ) return intRtn;

	if ( slaveinfoThe.fAscii ) {
		/* result is in ASCII format */
		printf(
			"   slave controller revision:           %s\n",
			slaveinfoThe.szID
		);
	} else {
		/* result is in non-ASCII format */
		printf(
			"   slave controller revision:           %d.%d %s\n",
			slaveinfoThe.wIDMajor,
			slaveinfoThe.wIDMinor,
			slaveinfoThe.wIDMajor == 0xFFFF ? "(i.e., not valid)" : ""
		);
	}

    return 0;
}


int info_print_processor( int intFiledesc )
{
	smapidev_cpuinfo_t cpuinfoThe;
	int intRtn;
	char *pchManufacturer;

	cpuinfoThe.sizeStruct = sizeof( cpuinfoThe );
	intRtn = smapidev_GetCpuInfo( intFiledesc, &cpuinfoThe );
	if ( intRtn ) return intRtn;

	switch ( cpuinfoThe.wManufacturer ) {
		case 1: pchManufacturer = "Intel"; break;
		case 2: pchManufacturer = "AMD"; break;
		default: pchManufacturer = "(manufacturer code not recognized)"; break;
	}

	printf(
		"%s%s%snformation about %s%s%srocessor:\n"
		"   CPU manufacturer:                    %s\n"
		"   microprocessor type:                 %d\n"
		"   microprocessor stepping level:       %d\n"
		"   CPU clock:                           %d MHz\n"
		"   internal clock:                      %d MHz\n",
		HI("i"), HI("p"),
		pchManufacturer,
		cpuinfoThe.wType,
		cpuinfoThe.wStepping,
		cpuinfoThe.wClock,
		cpuinfoThe.wInternalClock
	);

	return 0;
}


int info_print_display( int intFiledesc )
{
	smapidev_displayinfo_t displayinfoThe;
	int intRtn;
	int fMonDDC1, fMonDDC2;
	char * pchPanelType;
	char * pchPanelDim;
	char * pchMonType;

	displayinfoThe.sizeStruct = sizeof( displayinfoThe );
	intRtn = smapidev_GetDisplayInfo( intFiledesc, &displayinfoThe );
	if ( intRtn ) return intRtn;

	switch ( displayinfoThe.bPanelType ) {
		case 0: pchPanelType = "monochrome STN LCD"; break;
		case 1: pchPanelType = "monochrome TFT LCD"; break;
		case 2: pchPanelType = "color STN LCD"; break;
		case 3: pchPanelType = "color TFT LCD"; break;
		case 0xFF: pchPanelType = "(unknown)"; break;
		default: pchPanelType = "(panel type code not recognized)"; break;
	}

	switch ( displayinfoThe.bPanelDim ) {
		case 0: pchPanelDim = "640 x 480"; break;
		case 1: pchPanelDim = "800 x 600"; break;
		case 2: pchPanelDim = "1024 x 768"; break;
		case 3: pchPanelDim = "1280 x 1024"; break; /* TP560Z Tech Ref has "1024 x 1024" -- wrong! */
		case 0xFF: pchPanelDim = "(unknown)"; break;
		default: pchPanelDim = "(panel dimension code not recognized)"; break;
	}

	switch ( displayinfoThe.bCrtType ) {
		case 0x00: pchMonType = "(no CRT attached)"; break;
		case 0x10: pchMonType = "color CRT"; break;
		case 0x20: pchMonType = "monochrome CRT"; break;
		case 0xFF: pchMonType = "(unknown)"; break;
		default: pchMonType = "(monitor type code not recognized)"; break;
	}

	fMonDDC1 = displayinfoThe.bCrtFeatures & 1;
	fMonDDC2 = (displayinfoThe.bCrtFeatures & 2) ? 1 : 0;

	printf(
		"%s%s%snformation about %s%s%sisplay:\n"
		"   panel type:                          %d = %s\n"
		"   panel dimension:                     %d = %s\n"
		"   monitor type:                        %d = %s\n"
		"   monitor has DDC1 capability?:        %c\n"
		"   monitor has DDC2 capability?:        %c\n",
		HI("i"), HI("d"),
		displayinfoThe.bPanelType, pchPanelType,
		displayinfoThe.bPanelDim, pchPanelDim,
		displayinfoThe.bCrtType, pchMonType,
		fMonDDC1 ? 'Y' : 'N',
		fMonDDC2 ? 'Y' : 'N'
	);

	return 0;
}


int info_print_docking_station( int intFiledesc )
{
	smapidev_dockinginfo_t dockinginfoThe;
	char szDockID[32];
	int intRtn;

	dockinginfoThe.sizeStruct = sizeof( dockinginfoThe );
	intRtn = smapidev_GetDockingInfo( intFiledesc, &dockinginfoThe );
	if ( intRtn ) return intRtn;

	if ( dockinginfoThe.fDocked ) 
		sprintf( szDockID, "%d", dockinginfoThe.wID );
	else
		strcpy( szDockID, "(not docked)" );

	printf(
		"%s%s%snformation about d%s%s%scking station:\n"
		"   docking station ID:                  %s\n"
		"   security key unlocked?:              %c\n"
		"   bus connected?:                      %c\n",
		HI("i"), HI("o"),
		szDockID,
		dockinginfoThe.fDocked ? (dockinginfoThe.fKeyUnlocked ? 'Y' : 'N') : '-',
		dockinginfoThe.fDocked ? (dockinginfoThe.fBusConnected ? 'Y' : 'N') : '-'
	);

	return 0;
}


int info_print_UltraBay_II( int intFiledesc )
{
	smapidev_ultrabayinfo_t ultrabayinfoThe;
	int intRtn;
	char * pchDevType;
	char * pchDevID;

	ultrabayinfoThe.sizeStruct = sizeof( ultrabayinfoThe );
	intRtn = smapidev_GetUltrabayInfo( intFiledesc, &ultrabayinfoThe );
	if ( intRtn ) return intRtn;

	switch ( ultrabayinfoThe.bType ) {
		case 0x00: pchDevType = "FDD"; break;
		case 0x01: pchDevType = "serial device"; break;
		case 0x02: pchDevType = "TV tuner"; break;
		case 0x10: pchDevType = "IDE device"; break;
		case 0x20: pchDevType = "PCMCIA adapter"; break;
		case 0x30: pchDevType = "battery"; break;
		case 0x40: pchDevType = "AC adapter"; break;
		case 0xFE: pchDevType = "(no UltraBay exists)"; break;
		case 0xFF: pchDevType = "(unknown)"; break;
		default: pchDevType = "(UltraBay device type code not recognized)"; break;
	}

	switch ( ultrabayinfoThe.bID ) {
		case 0x00: pchDevID = "FDD"; break;
		case 0x01: pchDevID = "cellular"; break;
		case 0x02: pchDevID = "TV tuner"; break;
		case 0x10: pchDevID = "CD-ROM"; break;
		case 0x11: pchDevID = "IDE HDD"; break;
		case 0x12: pchDevID = "DVD"; break;
		case 0x13: pchDevID = "ZIP"; break;
		case 0xFF: pchDevID = "(ID not available)"; break;
		default: pchDevID = "(ID code not recognized)"; break;
	}

	printf(
		"%s%s%snformation about %s%s%sltraBay II:\n"
		"   device type:                         0x%x = %s\n"
		"   device ID:                           0x%x = %s\n",
		HI("i"), HI("U"),
		ultrabayinfoThe.bType, pchDevType,
		ultrabayinfoThe.bID, pchDevID
	);

	return 0;
}



int info_print_sensors( int intFiledesc )
{
	smapidev_sensorinfo_t sensorinfoThe;
	int intRtn;

	sensorinfoThe.sizeStruct = sizeof( sensorinfoThe );
	intRtn = smapidev_GetSensorInfo( intFiledesc, &sensorinfoThe );
	if ( intRtn ) return intRtn;

	printf(
		"%s%s%s:\n"
		"   the lid closed?:                     %c\n"
		"   the keyboard open?:                  %c\n"
		"   the AC adapter attached?:            %c\n",
		HI("is"),
		sensorinfoThe.fLidClosed ? 'Y' : 'N',
		sensorinfoThe.fKeyboardOpen ? 'Y' : 'N',
		sensorinfoThe.fACAdapterAttached ? 'Y' : 'N'
	);

	return 0;
}

/*
 * See the file VGA_MODES for info about the encoding of wMode
 */
int info_print_refresh( int intFiledesc, word wMode )
{
	smapidev_screenrefreshinfo_t screenrefreshinfoThe;
	int intRtn;

	screenrefreshinfoThe.sizeStruct = sizeof(screenrefreshinfoThe);

	intRtn = smapidev_GetScreenRefreshInfo( intFiledesc, wMode, &screenrefreshinfoThe );
	if ( intRtn ) return intRtn;

	printf(
		"%s%s%snformation: %s%s%sefresh rate capabilities for VGA mode 0x%x:\n"
		"   43 Hz (I) available?:                %c\n"
		"   48 Hz (I) available?:                %c\n"
		"   56 Hz available?:                    %c\n"
		"   60 Hz available?:                    %c\n"
		"   70 Hz available?:                    %c\n"
		"   72 Hz available?:                    %c\n"
		"   75 Hz available?:                    %c\n"
		"   85 Hz available?:                    %c\n",
		HI("i"), HI("r"),
		wMode,
		screenrefreshinfoThe.f43i ? 'Y' : 'N',
		screenrefreshinfoThe.f48i ? 'Y' : 'N',
		screenrefreshinfoThe.f56 ? 'Y' : 'N',
		screenrefreshinfoThe.f60 ? 'Y' : 'N',
		screenrefreshinfoThe.f70 ? 'Y' : 'N',
		screenrefreshinfoThe.f72 ? 'Y' : 'N',
		screenrefreshinfoThe.f75 ? 'Y' : 'N',
		screenrefreshinfoThe.f85 ? 'Y' : 'N'
	);

	return 0;
}


int info_dump_cmos( int intFiledesc )
{
	rtcmosram_ioparm_t ioparmMy;
	int i;
	int intRtn;

	ioparmMy.in.wFunc = RTCMOSRAM_FUNC_GETDATA;
	intRtn = ioctl_rtcmosram( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	printf( "CMOS data\n" );
	for ( i=0; i<sizeof(rtcmosram_data_t); i+=16 )
	printf(
		"%2x: %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x\n",
		i,
		*(((byte *)&ioparmMy)+i),
		*(((byte *)&ioparmMy)+i+1),
		*(((byte *)&ioparmMy)+i+2),
		*(((byte *)&ioparmMy)+i+3),
		*(((byte *)&ioparmMy)+i+4),
		*(((byte *)&ioparmMy)+i+5),
		*(((byte *)&ioparmMy)+i+6),
		*(((byte *)&ioparmMy)+i+7),
		*(((byte *)&ioparmMy)+i+8),
		*(((byte *)&ioparmMy)+i+9),
		*(((byte *)&ioparmMy)+i+0xA),
		*(((byte *)&ioparmMy)+i+0xB),
		*(((byte *)&ioparmMy)+i+0xC),
		*(((byte *)&ioparmMy)+i+0xD),
		*(((byte *)&ioparmMy)+i+0xE),
		*(((byte *)&ioparmMy)+i+0xF)
	);

	return 0;
}


int setupcmos_print_daylightsavingtime_state( int intFiledesc )
{
	rtcmosram_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = RTCMOSRAM_FUNC_DAYLIGHTSAVINGTIME_GET;
	intRtn = ioctl_rtcmosram( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	printf(
		"   daylight %s%s%saving %s%s%sime:                %s\n",
		HI("s"), HI("t"),
		(flag_t)ioparmMy.out.dwParm1 ? "enable" : "disable"
	);

	return 0;
}


int setupcmos_set_daylightsavingtime_state( int intFiledesc, flag_t fEnable )
{
	rtcmosram_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = RTCMOSRAM_FUNC_DAYLIGHTSAVINGTIME_ABLIFY;
	ioparmMy.in.dwParm1 = (word) fEnable;
	intRtn = ioctl_rtcmosram( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	printf( "%s: floppy boot has been %s.\n", szTpctlName, fEnable ? "enabled" : "disabled" );

	return 0;
}


int info_print_cmos( int intFiledesc )
{
	rtcmosram_ioparm_t ioparmMy;
	byte bHDD0Type, bHDD1Type, bHDD2Type, bHDD3Type;
	byte bDisplayOperatingMode;
	byte bDisketteDriveType;
	int intRtn;

	ioparmMy.in.wFunc = RTCMOSRAM_FUNC_GETDATA;
	intRtn = ioctl_rtcmosram( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bDisketteDriveType = ioparmMy.data.bDisketteDriveType >> 4;

	bHDD0Type = ioparmMy.data.bHardDiskDrive01Type >> 4;
	bHDD1Type = ioparmMy.data.bHardDiskDrive01Type & 0xF;
	bHDD2Type = ioparmMy.data.bHardDiskDrive23Type >> 4;
	if ( bHDD2Type == 0xF ) bHDD2Type = ioparmMy.data.bHardDiskDrive2Type;
	bHDD3Type = ioparmMy.data.bHardDiskDrive23Type & 0xF;
	if ( bHDD3Type == 0xF ) bHDD3Type = ioparmMy.data.bHardDiskDrive3Type;

	bDisplayOperatingMode = (ioparmMy.data.bEquipment >> 4) & 3;

	printf(
		"%s%s%snfo from %s%s%smos RAM:\n"
		"   CMOS RAM has power?:                 %c\n"
		"   CMOS RAM has lost power?:            %c\n"
		"   CMOS RAM has bad checksum?:          %c\n"
		"   CMOS RAM has bad equipment info?:    %c\n"
		"   CMOS RAM has bad mem size info?:     %c\n"
		"   CMOS RAM has bad time info?:         %c\n"
		"   hard disk failed initialization?:    %c\n",
		HI("i"), HI("c"),
		ioparmMy.data.bStatusD & 0x80 ? 'Y' : 'N',
		ioparmMy.data.bStatusDiag & 0x80 ? 'Y' : 'N',
		ioparmMy.data.bStatusDiag & 0x40 ? 'Y' : 'N',
		ioparmMy.data.bStatusDiag & 0x20 ? 'Y' : 'N',
		ioparmMy.data.bStatusDiag & 0x10 ? 'Y' : 'N',
		ioparmMy.data.bStatusDiag & 0x04 ? 'Y' : 'N',
		ioparmMy.data.bStatusDiag & 0x08 ? 'Y' : 'N'
	);
	printf(
		"   shutdown status:                     0x%x\n"
		"   display operating mode:              %s\n"
		"   coprocessor present?:                %c\n",
		ioparmMy.data.bStatusShutdown,
		(
			bDisplayOperatingMode == 3 ? "monochrome" :
			bDisplayOperatingMode == 2 ? "80-column" :
			bDisplayOperatingMode == 1 ? "40-column" :
			                             "(code is reserved)"
		),
		(ioparmMy.data.bEquipment & 2) ? 'Y' : 'N'
	);
	printf(
		"   diskette drive type:                 %d == %s\n"
		"   number of diskette drives:           %c\n"
		"   diskette drive 0 present?:           %c\n"
		"   hard disk drive 0 type:              0x%x%s\n"
		"   hard disk drive 1 type:              0x%x%s\n"
		"   hard disk drive 2 type:              0x%x%s\n"
		"   hard disk drive 3 type:              0x%x%s\n",
		bDisketteDriveType,
		(
			bDisketteDriveType == 4 ? "1.44 MB" :
			bDisketteDriveType == 6 ? "2.88 MB" :
			                          "(code is reserved)"
		),
		((ioparmMy.data.bEquipment >> 6) & 3) == 0 ? '1' : '?',
		(ioparmMy.data.bEquipment & 1) ? 'Y' : 'N',
		bHDD0Type, (bHDD0Type == 0) ? " (not installed)" : "",
		bHDD1Type, (bHDD1Type == 0) ? " (not installed)" : "",
		bHDD2Type, (bHDD2Type == 0) ? " (not installed)" : "",
		bHDD3Type, (bHDD3Type == 0) ? " (not installed)" : ""
	);
	printf(
		"   conventional memory below 640KB:     %d KB\n"
		"   expansion memory above 1MB:          %d KB\n"
		"   usable contiguous memory:            %d KB\n",
		ioparmMy.data.bSizeBaseMemLow + (word)(ioparmMy.data.bSizeBaseMemHigh << 8),
		ioparmMy.data.bSizeExpMemLow + (word)(ioparmMy.data.bSizeExpMemHigh << 8),
		ioparmMy.data.wSizeUsableMem
	);

	return 0;
}


int setupsmapi_print_display_state( int intFiledesc, char chDevice )
{
	smapidev_ablestate_t ablestatePanelCMOS, ablestateCRTCMOS, ablestateTVCMOS;
	smapidev_ablestate_t ablestatePanelCurr, ablestateCRTCurr, ablestateTVCurr;
	smapidev_ablestate_t ablestateCRTDetectionIgnoreCMOS;
	smapidev_ablestate_t ablestateCRTDetectionIgnoreCurr;
	smapidev_ablestate_t ablestateDualCMOS;
	smapidev_ablestate_t ablestateDualCurr;
	smapidev_ablestate_t ablestateSelectTVCMOS;
	smapidev_ablestate_t ablestateSelectTVCurr;
	smapidev_displaycap_t displaycapCMOS;
	smapidev_displaycap_t displaycapCurr;
	int intRtn;
	char *pchTVCMOS, *pchTVCurr;

	displaycapCMOS.sizeStruct = sizeof( displaycapCMOS );
	intRtn = smapidev_GetDisplayCapability(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS,
		&displaycapCMOS
	);
	if ( intRtn ) return intRtn;
	displaycapCurr.sizeStruct = sizeof( displaycapCurr );
	intRtn = smapidev_GetDisplayCapability(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CURR,
		&displaycapCurr
	);
	if ( intRtn ) return intRtn;

	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS, SMAPIDEV_DISPMODE_INTERNAL, &ablestatePanelCMOS
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CURR, SMAPIDEV_DISPMODE_INTERNAL, &ablestatePanelCurr
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS, SMAPIDEV_DISPMODE_CRT, &ablestateCRTCMOS
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CURR, SMAPIDEV_DISPMODE_CRT, &ablestateCRTCurr
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS, SMAPIDEV_DISPMODE_TV, &ablestateTVCMOS
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CURR, SMAPIDEV_DISPMODE_TV, &ablestateTVCurr
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS, SMAPIDEV_DISPMODE_CRT_DETECTION_IGNORE, &ablestateCRTDetectionIgnoreCMOS
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CURR, SMAPIDEV_DISPMODE_CRT_DETECTION_IGNORE, &ablestateCRTDetectionIgnoreCurr
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS, SMAPIDEV_DISPMODE_DUAL, &ablestateDualCMOS
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CURR, SMAPIDEV_DISPMODE_DUAL, &ablestateDualCurr
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS, SMAPIDEV_DISPMODE_SELECT_TV, &ablestateSelectTVCMOS
	);
	if ( intRtn ) return intRtn;
	intRtn = smapidev_GetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CURR, SMAPIDEV_DISPMODE_SELECT_TV, &ablestateSelectTVCurr
	);
	if ( intRtn ) return intRtn;

	switch ( displaycapCMOS.tv ) {
		case SMAPIDEV_DISPLAYCAPTV_NONE: pchTVCMOS = "no TV"; break;
		case SMAPIDEV_DISPLAYCAPTV_NONSIMULTANEOUS: pchTVCMOS = "nonsimultaneous-with-CRT"; break;
		case SMAPIDEV_DISPLAYCAPTV_OTHER: pchTVCMOS = "other"; break;
		default:
			printf( "%s: displaycapCMOS value not understood -- error in program.\n", szTpctlName );
			return ERR_TPCTL;
	}
	switch ( displaycapCurr.tv ) {
		case SMAPIDEV_DISPLAYCAPTV_NONE: pchTVCurr = "no TV"; break;
		case SMAPIDEV_DISPLAYCAPTV_NONSIMULTANEOUS: pchTVCurr = "nonsimultaneous-with-CRT"; break;
		case SMAPIDEV_DISPLAYCAPTV_OTHER: pchTVCurr = "other"; break;
		default:
			printf( "%s: displaycapCMOS value not understood -- error in program.\n", szTpctlName );
	}

	if ( chDevice == 'x' ) printf(
		"   display function supported?:         %c             %c\n"
		"   TV:                    %26s  %-16s\n",
		displaycapCMOS.fSupported ? 'Y' : 'N',
		displaycapCurr.fSupported ? 'Y' : 'N',
		pchTVCMOS,
		pchTVCurr
	);
	/* Below we rely on the state type being encoded like a flag */
	if ( chDevice == 'x' || chDevice == 'i' ) printf(
		"   %s%s%snternal display enabled?:           %-12s  %s\n",
		HI("i"),
		ablestatePanelCMOS ? "enable" : "disable",
		ablestatePanelCurr ? "enable" : "disable"
	);
	if ( chDevice == 'x' || chDevice == 'c' ) printf(
		"   %s%s%srt display:                         %-12s  %s\n",
		HI("c"),
		ablestateCRTCMOS ? "enable" : "disable",
		ablestateCRTCurr ? "enable" : "disable"
	);
	if ( chDevice == 'x' || chDevice == 't' ) printf(
		"   %s%s%sv display:                          %-12s  %s\n",
		HI("t"),
		ablestateTVCMOS ? "enable" : "disable",
		ablestateTVCurr ? "enable" : "disable"
	);
	if ( chDevice == 'x' || chDevice == '2' ) printf(
		"   %s%s%sual display:                        %-12s  %s\n",
		HI("2"),
		ablestateDualCMOS ? "enable" : "disable",
		ablestateDualCurr ? "enable" : "disable"
	);
	if ( chDevice == 'x' || chDevice == 'T' ) printf(
		"   %s%s%sV display selected?:                %c             %c\n",
		HI("T"),
		ablestateSelectTVCMOS ? 'Y' : 'N',
		ablestateSelectTVCurr ? 'Y' : 'N'
	);
	if ( chDevice == 'x' || chDevice == 'm' ) printf(
		"   %s%s%sonitor detection ignored?:          %c             %c\n",
		HI("m"),
		ablestateCRTDetectionIgnoreCMOS ? 'Y' : 'N',
		ablestateCRTDetectionIgnoreCurr ? 'Y' : 'N'
	);

	return 0;
}

/*
 * NOTE!  The ThinkPad 600 Tech Ref is confusing on the formatting
 *        of the parameters.  See the ThinkPad 560Z Tech Ref.
 * NOTE.  The TV-selection bit (low bit of parm 4) seems to have no
 *        effect on my ThinkPad 600 21U.  The bit is not documented
 *        in the 560Z Tech Ref.
 */

int setupsmapi_set_display_state( int intFiledesc, char chDisplayModeToChange, flag_t fEnable )
{
	int intRtn;

	smapidev_dispmode_t dispmode = -1;

	switch( chDisplayModeToChange )
	{
		case 'i' : dispmode = SMAPIDEV_DISPMODE_INTERNAL;          break;
		case 'c' : dispmode = SMAPIDEV_DISPMODE_CRT;               break;
		case 't' : dispmode = SMAPIDEV_DISPMODE_TV;                break;
		case 'T' : dispmode = SMAPIDEV_DISPMODE_SELECT_TV;         break;
		case '2' : dispmode = SMAPIDEV_DISPMODE_DUAL;              break;
		case 'm' : dispmode = SMAPIDEV_DISPMODE_CRT_DETECTION_IGNORE; break;
		default:
			return ERR_TPCTL;
	}
	intRtn = smapidev_SetDisplayState(
		intFiledesc,
		SMAPIDEV_STATEPLACE_CMOS_AND_CURR,
		dispmode, fEnable?SMAPIDEV_ABLESTATE_ENABLED:SMAPIDEV_ABLESTATE_DISABLED
	);
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Display state %c has been %s\n", szTpctlName, chDisplayModeToChange, fEnable ? "enabled" : "disabled" );

	return 0;
}


/*
 * NOTE!  The ThinkPad 600 Technical Reference and the ThinkPad 560Z
 *        Technical Reference disagree on which parameter is used
 *        to return the information, parameter 1 (according to the
 *        former) or parameter 2 (according to the latter).  Generally
 *        the 560Z spec is more reliable and seems to be a revision of
 *        the 600 spec.  And experiment proves the 560Z correct.
 */
int setupsmapi_print_pointing_device_state( int intFiledesc, smapidev_ternality_t ternalityThe )
{
	smapi_ioparm_t ioparmMy;
	char *pchActModeIntCMOS, *pchActModeExtCMOS;
	char *pchActModeIntCurr, *pchActModeExtCurr;
	byte bStateCMOS, bCapabilityCMOS;
	byte bStateCurr, bCapabilityCurr;
	int intRtn;

	/*** Get configuration recorded in CMOS ***/
	ioparmMy.in.bFunc = (byte) 0x11;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0x100;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	DEBUG_PRINT_OUTPARMS( ioparmMy );
	if ( intRtn ) return intRtn;

	/* Note that the significant bits for this info are not adjacent */
	bCapabilityCMOS = (byte)(ioparmMy.out.wParm2 & 0xFF);
	bStateCMOS = (byte)(ioparmMy.out.wParm2 >> 8);
	switch (bStateCMOS & 5) {
		case 0: pchActModeIntCMOS = "disable"; break;
		case 1: pchActModeIntCMOS = "enable"; break;
		case 4: pchActModeIntCMOS = "auto-disable"; break;
		default:
		case 5: pchActModeIntCMOS = "(mode code not recognized)"; break;
	}
	pchActModeExtCMOS = (bStateCMOS & 2) ? "enable" : "disable";

	/*** Get current configuration ***/
	ioparmMy.in.bFunc = (byte) 0x11;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	DEBUG_PRINT_OUTPARMS( ioparmMy );
	if ( intRtn ) return intRtn;

	/* Note that the significant bits for this info are not adjacent */
	bCapabilityCurr = (byte)(ioparmMy.out.wParm2 & 0xFF);
	bStateCurr = (byte)(ioparmMy.out.wParm2 >> 8);
	switch (bStateCurr & 5) {
		case 0: pchActModeIntCurr = "disable"; break;
		case 1: pchActModeIntCurr = "enable"; break;
		case 4: pchActModeIntCurr = "auto-disable"; break;
		default:
		case 5: pchActModeIntCurr = "(mode code not recognized)"; break;
	}
	pchActModeExtCurr = (bStateCurr & 2) ? "enable" : "disable";

	if ( ternalityThe == SMAPIDEV_TERNALITY_IN ) { printf(
		"   %s%s%snternal pointing device\n"
		"      controllable state?:              %c             %c\n"
		"      auto control supported?:          %c             %c\n"
		"      activation mode:                  %-12s  %s\n",
		HI("i"),
		( bCapabilityCMOS       & 1) ? 'Y' : 'N',
		( bCapabilityCurr       & 1) ? 'Y' : 'N',
		((bCapabilityCMOS >> 2) & 1) ? 'Y' : 'N',
		((bCapabilityCurr >> 2) & 1) ? 'Y' : 'N',
		pchActModeIntCMOS,
		pchActModeIntCurr
	); } else { printf (
		"   %s%s%sxternal pointing device\n"
		"      controllable state?:              %c             %c\n"
		"      activation mode:                  %-12s  %s\n",
		HI("e"),
		((bCapabilityCMOS >> 1) & 1) ? 'Y' : 'N',
		((bCapabilityCurr >> 1) & 1) ? 'Y' : 'N',
		pchActModeExtCMOS,
		pchActModeExtCurr
	); }

	return 0;
}


/*
 * NOTE!  The ThinkPad 600 Technical Reference and the ThinkPad 560Z
 *        Technical Reference disagree on which parameter is used
 *        to send the information, parameter 2 (according to the
 *        former) or parameter 1 (according to the latter).  Generally
 *        the 560Z spec is more reliable and seems to be a revision of
 *        the 600 spec.  And experiment proves the 560Z correct.
 */
/*
 * NOTE!  The ThinkPad 600 Technical Reference and the ThinkPad 560Z
 *        Technical Reference disagree on how to select the type of
 *        request.  In the 600 spec, bits 7-0 (of input parameter 2)
 *        are a number, where 0 indicates a current-hardware request
 *        and 1 indicates a CMOS request.  In the 560Z spec, bits
 *        7-0 (of parameter 1) are flags, where bit 0
 *        indicates that current hardware should be updated and bit 1
 *        indicates that CMOS should be updated.  Generally the 560Z
 *        spec is more reliable and seems to be a revision of the 
 *        600 spec.  Experiment on a 600 21U seems to bear out the
 *        560Z spec.
 */
/* 
 * bState encoding:
 *    0 : disable
 *    1 : enable
 *    2 : auto-disable
 */
int setupsmapi_set_pointing_device_state(
	int intFiledesc,
	flag_t fInternal,
	flag_t fExternal,
	byte bState )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	if ( bState > 2 ) {
		printf( "%s: pointing device state not recognized\n", szTpctlName );
		return ERR_TPCTL;
	}
	if ( fExternal && bState > 1 ) {
		printf( "%s: pointing device mode not valid for external pointing device \n", szTpctlName );
		return ERR_TPCTL;
	}

	/*** Set configuration in CMOS ***/

	/*** get ***/
	ioparmMy.in.bFunc = (byte) 0x11;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0x100;
	DEBUG_PRINT_INPARMS( ioparmMy ); 
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	/*** set ***/
	ioparmMy.in.bFunc = (byte) 0x11;
	ioparmMy.in.bSubFunc = (byte) 3;
	ioparmMy.in.wParm1 = ioparmMy.out.wParm2 & 0x700;

	if ( fInternal ) {
		ioparmMy.in.wParm1 &= ~0x500;
		ioparmMy.in.wParm1 |= (((word)bState & 2) << 9) | (((word)bState & 1) << 8);
	}
	if ( fExternal ) {
		ioparmMy.in.wParm1 &= ~0x200;
		ioparmMy.in.wParm1 |= (((word)bState & 1) << 9);
	}

	ioparmMy.in.wParm1 |= 2; /* set in CMOS */
	DEBUG_PRINT_INPARMS( ioparmMy ); 
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fInternal ) printf( "%s: Internal pointing device state has been set in CMOS to %s\n", szTpctlName, bState == 2 ? "auto-disable" : bState == 1 ? "enable" : "disable" );
	if ( fExternal ) printf( "%s: External pointing device state has been set in CMOS to %s\n", szTpctlName, bState == 2 ? "auto-disable" : bState == 1 ? "enable" : "disable" );

	/*** Set current configuration ***/

	/*** get ***/
	ioparmMy.in.bFunc = (byte) 0x11;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0;
	DEBUG_PRINT_INPARMS( ioparmMy ); 
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	/*** set ***/
	ioparmMy.in.bFunc = (byte) 0x11;
	ioparmMy.in.bSubFunc = (byte) 3;
	ioparmMy.in.wParm1 = ioparmMy.out.wParm2 & 0x700;

	if ( fInternal ) {
		ioparmMy.in.wParm1 &= ~0x500;
		ioparmMy.in.wParm1 |= (((word)bState & 2) << 9) | (((word)bState & 1) << 8);
	}
	if ( fExternal ) {
		ioparmMy.in.wParm1 &= ~0x200;
		ioparmMy.in.wParm1 |= (((word)bState & 1) << 9);
	}

	ioparmMy.in.wParm1 |= 1; /* set current */
	DEBUG_PRINT_INPARMS( ioparmMy ); 
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	DEBUG_PRINT_OUTPARMS( ioparmMy ); 
	if ( intRtn && (intRtn != 0x81) ) {
			return intRtn;
	}

	/* intRtn == 0 or intRtn == 0x81 */

	if ( fVerbose ) {
		if ( fInternal ) {
			if ( intRtn ) {
				printf( "%s: Internal pointing device state (current) could not be changed.\n", szTpctlName );	
			} else {
				printf( "%s: Internal pointing device state (current) has been set to %s\n", szTpctlName, bState == 2 ? "auto-disable" : bState == 1 ? "enable" : "disable" );
			}
		}
	}
	if ( fVerbose )	{
		if ( fExternal ) {
			if ( intRtn ) {
				printf( "%s: External pointing device state (current) could not be changed.\n", szTpctlName );	
			} else {
				printf( "%s: External pointing device state (current) has been set to %s\n", szTpctlName, bState == 2 ? "auto-disable" : bState == 1 ? "enable" : "disable" );
			}
		}
	}

	return 0;
}


int setupsmapi_print_hotkey_state( int intFiledesc )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x13;
	ioparmMy.in.bSubFunc = (byte) 2;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	printf(
		"   sticky Fn key supported?:                          %c\n"
		"   sticky & locked Fn key supported?:                 %c\n"
		"   %s%s%sn hotkey state:                                   %s\n",
		((ioparmMy.out.wParm2 >> 8) & 1) ? 'Y' : 'N',
		((ioparmMy.out.wParm2 >> 9) & 1) ? 'Y' : 'N',
		HI("f"),
		(ioparmMy.out.wParm2 & 3) == 3 ? "sticky & locked" : (
			(ioparmMy.out.wParm2 & 3) == 1 ? "sticky" : (
				(ioparmMy.out.wParm2 & 3) == 0 ? "nonsticky" : (
						"(stickitivity code not recognized)"
				)
			)
		)
	);

	return 0;
}


/*
 * Coding of bState:
 *    0 : nonsticky
 *    1 : sticky
 *    3 : sticky & locked
 */

int setupsmapi_set_hotkey_state( int intFiledesc, byte bState )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	if ( bState > 3 || bState == 2 ) { 
		printf ( "%s: hotkey state not recognized\n", szTpctlName );
		return ERR_TPCTL;
	}

	ioparmMy.in.bFunc = (byte) 0x13;
	ioparmMy.in.bSubFunc = (byte) 3;
	ioparmMy.in.wParm1 = (word) bState;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf(
		"%s: Hotkey state has been set to %s\n",
		szTpctlName,
		bState == 3 ? "stick & locked" : bState == 1 ? "sticky" : "nonsticky"
	);

	return 0;
}


int pm_print_expenditure_mode( int intFiledesc, smapidev_powersrc_t powersrcThe  )
{
	smapidev_powermode_t powermodeThe;

	smapidev_GetPowerExpenditureMode( intFiledesc, powersrcThe, &powermodeThe );

	if ( powersrcThe == SMAPIDEV_POWERSRC_AC ) {
		printf( "   %s%s%sc power expenditure:                              ", HI("a") );
	} else {
		printf( "   %s%s%sattery power expenditure:                         ", HI("b") );
	}
	printf (
		"%s\n",
		(	(powermodeThe == SMAPIDEV_POWERMODE_HIGH) ? "high" :
			(powermodeThe == SMAPIDEV_POWERMODE_AUTO) ? "auto" :
			(powermodeThe == SMAPIDEV_POWERMODE_MANUAL) ? "manual" :
			"(power management code not recognized)"
		)
	);

	return 0;
}


int pm_set_expenditure_mode(
	int intFiledesc,
	int fAC, int fBattery,
	smapidev_powermode_t powermodeThe
) {
	char *pchMode, *pchPowerSource;

	if ( fAC ) smapidev_SetPowerExpenditureMode( intFiledesc, SMAPIDEV_POWERSRC_AC, powermodeThe );
	if ( fBattery ) smapidev_SetPowerExpenditureMode( intFiledesc, SMAPIDEV_POWERSRC_BATTERY, powermodeThe );

	switch ( powermodeThe ) {
		case SMAPIDEV_POWERMODE_HIGH: pchMode = "high"; break;
		case SMAPIDEV_POWERMODE_AUTO: pchMode = "auto"; break;
		case SMAPIDEV_POWERMODE_MANUAL: pchMode = "manual"; break;
		default: pchMode = "(not recognized)"; break;
	}

	if ( fBattery && fAC ) pchPowerSource = "battery and AC";
	else if ( fBattery && !fAC ) pchPowerSource = "battery";
	else if ( !fBattery && fAC ) pchPowerSource = "AC";

	if ( fVerbose ) printf(
		"%s: Power management mode for %s power has been set to %s\n",
		szTpctlName,
		pchPowerSource,
		pchMode
	);

	return 0;
}


int pm_print_redisafe_mode( int intFiledesc )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x30;
	ioparmMy.in.bSubFunc = (byte) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	printf(
		"   RediSafe global mode overrides non-global mode?:   %c\n"
		"   %s%s%sediSafe globally enabled?:                        %c\n",
		(ioparmMy.out.wParm2 >> 8) & 1 ? 'Y' : 'N',
		HI("R"),
		(ioparmMy.out.wParm2     ) & 1 ? 'Y' : 'N'
	);

	return 0;
}


int pm_set_redisafe_mode( int intFiledesc, int fEnable )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	fEnable &= 1;

	ioparmMy.in.bFunc = (byte) 0x30;
	ioparmMy.in.bSubFunc = (byte) 1;
	ioparmMy.in.wParm1 = fEnable;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Safe suspend has been %s\n", szTpctlName, fEnable ? "enabled" : "disabled" );

	return 0;
}


int pm_print_timer_mode( int intFiledesc, char chTimer )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x22;
	ioparmMy.in.bSubFunc = (byte) 2;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( chTimer == 'x' || chTimer == 'B' ) printf(
		"   %s%s%slank-internal-display timer:        %c             %c\n",
		HI("B"),
		(ioparmMy.out.wParm2 >> 10) & 1 ? 'Y' : 'N',
		(ioparmMy.out.wParm2 >>  2) & 1 ? 'Y' : 'N'
	);
	if ( chTimer == 'x' || chTimer == 'S' ) printf(
		"   %s%s%standby timer:                       %c             %c\n",
		HI("S"),
		(ioparmMy.out.wParm2 >>  9) & 1 ? 'Y' : 'N',
		(ioparmMy.out.wParm2 >>  1) & 1 ? 'Y' : 'N'
	);
	if ( chTimer == 'x' || chTimer == 'Z' ) printf(
		"   %s%s%suspend-or-hibernation timer:        %c             %c\n",
		HI("Z"),
		(ioparmMy.out.wParm2 >>  8) & 1 ? 'Y' : 'N',
		(ioparmMy.out.wParm2      ) & 1 ? 'Y' : 'N'
	);
	if ( chTimer == 'x' || chTimer == 'd' ) printf(
		"   %s%s%srive power-down timer:              %c             %c\n",
		HI("d"),
		(ioparmMy.out.wParm2 >> 11) & 1 ? 'Y' : 'N',
		(ioparmMy.out.wParm2 >>  3) & 1 ? 'Y' : 'N'
	);

	return 0;
}


int pm_set_timer_mode( int intFiledesc, char chTimer, int fEnable )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x22;
	ioparmMy.in.bSubFunc = (byte) 2;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	ioparmMy.in.bFunc = (byte) 0x22;
	ioparmMy.in.bSubFunc = (byte) 3;
	ioparmMy.in.wParm1 = ioparmMy.out.wParm2 & 0x0F;

	switch ( chTimer ) {
		case 'S': 
			ioparmMy.in.wParm1 &= ~2;
			ioparmMy.in.wParm1 |= fEnable ? 2 : 0;
			break;
		case 'Z': 
			ioparmMy.in.wParm1 &= ~1;
			ioparmMy.in.wParm1 |= fEnable ? 1 : 0;
			break;
		case 'B': 
			ioparmMy.in.wParm1 &= ~4;
			ioparmMy.in.wParm1 |= fEnable ? 4 : 0;
			break;
		case 'd': 
			ioparmMy.in.wParm1 &= ~8;
			ioparmMy.in.wParm1 |= fEnable ? 8 : 0;
			break;
		default:
			printf( "%s: timer type not recognized; error in program\n", szTpctlName );
			return ERR_TPCTL;
	}
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Timer %c has been %s\n", szTpctlName, chTimer, fEnable ? "enabled" : "disabled" );

	return 0;
}

/*
 * I don't quite understand what the "specifiable in each power mode"
 * bit really means.  Does it mean "specifiable in all power modes"
 * (as the latter description would seem to imply) or does it mean
 * "specifiable in this power mode" (as one might think, given that
 * the bit is returned for each mode inquiry)?  On my machine the
 * bit returned is 1 in every case, which leads me to think that
 * the first interpretation is correct.  However I print out the
 * result for each mode, just in case the second is correct and
 * there exist some thinkpad that have delays specifiable in some modes
 * but not others.
 */
int pm_print_suspend_or_hibernate_delay( int intFiledesc, char chMode )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0xF300;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
#ifndef TP570_KLUDGE
	if ( intRtn ) return intRtn;
#endif

	if ( chMode == 'x' || chMode == 'h' ) printf(
		"   \"%s%s%sigh\" power expenditure mode:       %c             %d %s\n",
		HI("h"),
		ioparmMy.out.wParm1 & 0x100 ? 'Y' : 'N',
		ioparmMy.out.wParm2 & 0xFF,
		(ioparmMy.out.wParm2 & 0xFF) ? "minutes" : "(i.e., disabled)"
	);

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0xF400;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
#ifndef TP570_KLUDGE
	if ( intRtn ) return intRtn;
#endif

	if ( chMode == 'x' || chMode == 'u' ) printf(
		"   \"a%s%s%sto\" power expenditure mode:       %c             %d %s\n",
		HI("u"),
		ioparmMy.out.wParm1 & 0x100 ? 'Y' : 'N',
		ioparmMy.out.wParm2 & 0xFF,
		(ioparmMy.out.wParm2 & 0xFF) ? "minutes" : "(i.e., disabled)"
	);

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0x100;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
#ifndef TP570_KLUDGE
	if ( intRtn ) return intRtn;
#endif

	if ( chMode == 'x' || chMode == 'a' ) printf(
		"   %s%s%sc power \"manual\" expenditure mode:  %c             %d %s\n",
		HI("a"),
		ioparmMy.out.wParm1 & 0x100 ? 'Y' : 'N',
		ioparmMy.out.wParm2 & 0xFF,
		(ioparmMy.out.wParm2 & 0xFF) ? "minutes" : "(i.e., disabled)"
	);

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0x200;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
#ifndef TP570_KLUDGE
	if ( intRtn ) return intRtn;
#endif

	if ( chMode == 'x' || chMode == 'b' ) printf(
		"   %s%s%sattery \"manual\" expenditure mode:   %c             %d %s\n",
		HI("b"),
		ioparmMy.out.wParm1 & 0x100 ? 'Y' : 'N',
		ioparmMy.out.wParm2 & 0xFF,
		(ioparmMy.out.wParm2 & 0xFF) ? "minutes" : "(i.e., disabled)"
	);

	return 0;
}


int pm_set_suspend_or_hibernate_delay( int intFiledesc, char chMode, byte bDuration )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 3;
	ioparmMy.in.wParm1 = (word) bDuration;

	switch ( chMode ) {
		case 'a': ioparmMy.in.wParm1 |= 0x100; break;
		case 'b': ioparmMy.in.wParm1 |= 0x200; break;
		case 'h': break;
		case 'u': break;
		case 'x': break;
		default:
			printf( "%s: Timer type not recognized; error in program.\n", szTpctlName );
			return ERR_TPCTL;
	}
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Delay for mode %c has been set to %d\n", szTpctlName, chMode, bDuration );

	return 0;
}


int pm_print_hibernate_from_suspend_delay( int intFiledesc )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 6;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	printf(
		"%s%s%sower management %s%s%selay of %s%s%sibernate-from-suspend:     %d %s\n",
		HI("p"), HI("d"), HI("H"),
		ioparmMy.out.wParm2 & 0xFF,
		(ioparmMy.out.wParm2 & 0xFF) ? "minutes" : "(i.e., disabled)"
	);

	return 0;
}

int pm_set_hibernate_from_suspend_delay( int intFiledesc, byte bDelay )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 7;
	ioparmMy.in.wParm1 = (word) bDelay;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf(
		"%s: Power management hibernate-from-suspend delay has been set to %d minutes.\n",
		szTpctlName,
		bDelay
	);

	return 0;
}


/*
 * NOTE!  The ThinkPad 600 Technical Reference and the ThinkPad 560Z
 *        Technical Reference disagree on which parameter is used
 *        to receive the information from function 32:0, parameter 1
 *        (according to the former) or parameter 2 (according to the latter).
 *        Generally the 560Z spec is more reliable and seems to be a revision
 *        of the 600 spec.
 */

#define EVENTCHARS( b )  (b>>3)&1?'S':'-', (b>>4)&1?'Z':'-', (b>>5)&1?'R':'-', (b>>6)&1?'H':'-', (b>>7)&1?'O':'-'

/*
 * chEvent is a character selecting a type of power management event
 */
int pm_print_sedation_actions( int intFiledesc, char chEvent )
{
	smapi_ioparm_t ioparmMy;
	byte bHardCap, bHardCur, bPowerCap, bPowerCur, bLidCap, bLidCur;
	byte bSysTimerCap, bSysTimerCur, bStandbyTimerCap, bStandbyTimerCur, bHibTimerCap, bHibTimerCur;
	byte bBatLowCap, bBatLowCur, bEnvExhaustedCap, bEnvExhaustedCur;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x32;
	ioparmMy.in.bSubFunc = (byte) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bSysTimerCap      = (byte)(ioparmMy.out.wParm2 >> 8);
	bSysTimerCur     = (byte)(ioparmMy.out.wParm2 & 0xFF);
	bStandbyTimerCap  = (byte)(ioparmMy.out.dwParm4 >> 8);
	bStandbyTimerCur = (byte)(ioparmMy.out.dwParm4 & 0xFF);
	bHibTimerCap      = (byte)(ioparmMy.out.dwParm5 >> 8);
	bHibTimerCur     = (byte)(ioparmMy.out.dwParm5 & 0xFF);

	if ( chEvent == 'x' || chEvent == 'S' ) printf(
		"   %s%s%standby-timer-initiated:             %c%c%c%c%c         %c%c%c%c%c\n",
		HI("S"),
		EVENTCHARS( bStandbyTimerCap ), EVENTCHARS( bStandbyTimerCur )
	);
	if ( chEvent == 'x' || chEvent == 'Z' ) printf(
		"   %s%s%suspend-or-hibern.-timer-initiated:  %c%c%c%c%c         %c%c%c%c%c\n",
		HI("Z"),
		EVENTCHARS( bSysTimerCap ), EVENTCHARS( bSysTimerCur )
	);
	if ( chEvent == 'x' || chEvent == 'H' ) printf(
		"   %s%s%sibernate-from-susp.-timer-init'd:   %c%c%c%c%c         %c%c%c%c%c\n",
		HI("H"),
		EVENTCHARS( bHibTimerCap ), EVENTCHARS( bHibTimerCur )
	);

	ioparmMy.in.bFunc = (byte) 0x33;
	ioparmMy.in.bSubFunc = (byte) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bBatLowCap    = (byte)(ioparmMy.out.wParm2 >> 8);
	bBatLowCur   = (byte)(ioparmMy.out.wParm2 & 0xFF);
	bEnvExhaustedCap  = (byte)(ioparmMy.out.dwParm4 >> 8);
	bEnvExhaustedCur = (byte)(ioparmMy.out.dwParm4 & 0xFF);

	if ( chEvent == 'x' || chEvent == 'b' ) printf(
		"   %s%s%sattery-low-initiated:               %c%c%c%c%c         %c%c%c%c%c\n",
		HI("b"),
		EVENTCHARS( bBatLowCap ), EVENTCHARS( bBatLowCur )
	);
	if ( chEvent == 'x' || chEvent == 'e' ) printf(
		"   %s%s%snv'mt-exhausted-initiated:          %c%c%c%c%c         %c%c%c%c%c\n",
		HI("e"),
		EVENTCHARS( bEnvExhaustedCap ), EVENTCHARS( bEnvExhaustedCur )
	);

	ioparmMy.in.bFunc = (byte) 0x31;
	ioparmMy.in.bSubFunc = (byte) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bHardCap   = (byte)(ioparmMy.out.wParm2 >> 8);
	bHardCur  = (byte)(ioparmMy.out.wParm2 & 0xFF);
	bLidCap    = (byte)(ioparmMy.out.dwParm5 >> 8);
	bLidCur   = (byte)(ioparmMy.out.dwParm5 & 0xFF);
	bPowerCap  = (byte)(ioparmMy.out.dwParm4 >> 8);
	bPowerCur = (byte)(ioparmMy.out.dwParm4 & 0xFF);

	if ( chEvent == 'x' || chEvent == 'h' ) printf(
		"   %s%s%sardware-or-software-initiated:      %c%c%c%c%c         %c%c%c%c%c\n",
		HI("h"),
		EVENTCHARS( bHardCap ), EVENTCHARS( bHardCur )
	);
	if ( chEvent == 'x' || chEvent == 'l' ) printf(
		"   %s%s%sid-closure-initiated:               %c%c%c%c%c         %c%c%c%c%c\n",
		HI("l"),
		EVENTCHARS( bLidCap ), EVENTCHARS( bLidCur )
	);
	if ( chEvent == 'x' || chEvent == 'p' ) printf(
		"   %s%s%sower-switch-initiated:              %c%c%c%c%c         %c%c%c%c%c\n",
		HI("p"),
		EVENTCHARS( bPowerCap ), EVENTCHARS( bPowerCur )
	);

	return 0;
}


/*
 * chEvent is a character selecting a type of power management event
 * chAction is a character that selects the action to perform when that
 *    event occurs
 */
int pm_set_sedation_action( int intFiledesc, char chEvent, char chAction )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;
	byte bSedativeAction;
	
	switch ( chAction ) {
		case '0':
		case '-': bSedativeAction = 0; break;
		case 'S': bSedativeAction = 0x8; break;
		case 'Z': bSedativeAction = 0x10; break;
		case 'R': bSedativeAction = 0x20; break;
		case 'H': bSedativeAction = 0x40; break;
		case 'O': bSedativeAction = 0x80; break;
		default:
			printf( "%s: Sedation action not recognized.\n", szTpctlName );
			return ERR_TPCTL;
	}

	switch ( chEvent ) {
		case 'h':
			ioparmMy.in.bFunc = (byte) 0x31;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x31;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = (word)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: Hardware/software sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		case 'p':
			ioparmMy.in.bFunc = (byte) 0x31;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x31;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = ioparmMy.out.wParm2;
			ioparmMy.in.dwParm4 = (dword)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: Power-switch sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		case 'l':
			ioparmMy.in.bFunc = (byte) 0x31;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x31;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = ioparmMy.out.wParm2;
			ioparmMy.in.dwParm5 = (dword)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: Lid-closure sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		case 'Z':
			ioparmMy.in.bFunc = (byte) 0x32;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x32;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = (word)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: Zystem timer sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		case 'S':
			ioparmMy.in.bFunc = (byte) 0x32;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x32;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = ioparmMy.out.wParm2;
			ioparmMy.in.dwParm4 = (dword)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: Standby timer sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		case 'H':
			ioparmMy.in.bFunc = (byte) 0x32;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x32;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = ioparmMy.out.wParm2;
			ioparmMy.in.dwParm5 = (dword)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: Hibernation timer sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		case 'b':
			ioparmMy.in.bFunc = (byte) 0x33;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x33;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = (word)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: battery-low sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		case 'e':
			ioparmMy.in.bFunc = (byte) 0x33;
			ioparmMy.in.bSubFunc = (byte) 0;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			ioparmMy.in.bFunc = (byte) 0x33;
			ioparmMy.in.bSubFunc = (byte) 1;
			ioparmMy.in.wParm1 = ioparmMy.out.wParm2;
			ioparmMy.in.dwParm4 = (dword)bSedativeAction;
			intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
			if ( intRtn ) return intRtn;
			if ( fVerbose ) printf( "%s: environment-exhausted sedative action has been set to %c\n", szTpctlName, chAction );
			break;
		default:
			printf( "%s: Sedation event not recognized.  Error in program\n", szTpctlName );
			return ERR_TPCTL;
	}

	return 0;
}


/*
 * chEvent is a character selecting a type of resumptive event
 */
int pm_print_resume_event( int intFiledesc, char chEvent )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;
	int fAppointmentEnabled, fDaily;

	ioparmMy.in.bFunc = (byte) 0x34;
	ioparmMy.in.bSubFunc = (byte) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	fAppointmentEnabled = (ioparmMy.out.wParm2 >> 2) & 1;

	if ( chEvent == 'x' || chEvent == 'h' ) printf(
		"   %s%s%sardware-initiated?:                 %c             %c\n",
		HI("h"),
		( ioparmMy.out.wParm3       & 1) ? 'Y' : 'N',
		( ioparmMy.out.wParm2       & 1) ? 'Y' : 'N'
	);
	if ( chEvent == 'x' || chEvent == 'l' ) printf(
		"   %s%s%sid-opening-initiated?:              %c             %c\n",
		HI("l"),
		((ioparmMy.out.wParm3 >> 1) & 1) ? 'Y' : 'N',
		((ioparmMy.out.wParm2 >> 1) & 1) ? 'Y' : 'N'
	);
	if ( chEvent == 'x' || chEvent == 's' ) printf(
		"   %s%s%serial-RI-initiated?:                %c             %c\n",
		HI("s"),
		((ioparmMy.out.wParm3 >> 3) & 1) ? 'Y' : 'N',
		((ioparmMy.out.wParm2 >> 3) & 1) ? 'Y' : 'N'
	);
	if ( chEvent == 'x' || chEvent == 'a' ) {
	printf(
		"   %s%s%sppointment-r.t.c.-initiated?:       %c             %c ",
		HI("a"),
		((ioparmMy.out.wParm3 >> 2) & 1) ? 'Y' : 'N',
		fAppointmentEnabled ? 'Y' : 'N'
	);

	ioparmMy.in.bFunc = (byte) 0x34;
	ioparmMy.in.bSubFunc = (byte) 2;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	fDaily = (ioparmMy.out.dwParm5 & 0x80000000) ? 1 : 0;
	
	if ( !fAppointmentEnabled ) printf ( "(" );

	if ( fDaily ) {
		printf(
			"daily:%02d:%02d:%02d",
			byte_of_bcd8( (bcd8_t)( (ioparmMy.out.dwParm4 & 0xFF0000) >> 16 ) ),
			byte_of_bcd8( (bcd8_t)( (ioparmMy.out.dwParm4 & 0x00FF00) >> 8  ) ),
			byte_of_bcd8( (bcd8_t)(  ioparmMy.out.dwParm4 & 0x0000FF        ) )
		);
	} else {
		printf(
			"%02d:%02d:%02d:%02d:%02d:%02d",
			byte_of_bcd8( (bcd8_t)( (ioparmMy.out.dwParm5 & 0xFF0000) >> 16 ) ),
			byte_of_bcd8( (bcd8_t)( (ioparmMy.out.dwParm5 & 0x00FF00) >> 8  ) ),
			byte_of_bcd8( (bcd8_t)(  ioparmMy.out.dwParm5 & 0x0000FF        ) ),
			byte_of_bcd8( (bcd8_t)( (ioparmMy.out.dwParm4 & 0xFF0000) >> 16 ) ),
			byte_of_bcd8( (bcd8_t)( (ioparmMy.out.dwParm4 & 0x00FF00) >> 8  ) ),
			byte_of_bcd8( (bcd8_t)(  ioparmMy.out.dwParm4 & 0x0000FF        ) )
		);
	}

	if ( !fAppointmentEnabled ) 
		printf ( ")\n" );
	else
		printf( "\n" );
	}

	return 0;
}


/*
 * chEvent is a character selecting a type of resumptive event
 */
int pm_set_resume_event( int intFiledesc, char chEvent, int fEnable )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x34;
	ioparmMy.in.bSubFunc = (byte) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	ioparmMy.in.bFunc = (byte) 0x34;
	ioparmMy.in.bSubFunc = (byte) 1;
	ioparmMy.in.wParm1 = ioparmMy.out.wParm2; 
	switch ( chEvent ) {
		case 'h':
			ioparmMy.in.wParm1 &= ~1;
			ioparmMy.in.wParm1 |= fEnable ? 1 : 0;
			break;
		case 'l':
			ioparmMy.in.wParm1 &= ~2;
			ioparmMy.in.wParm1 |= fEnable ? 2 : 0;
			break;
		case 'a':
			ioparmMy.in.wParm1 &= ~4;
			ioparmMy.in.wParm1 |= fEnable ? 4 : 0;
			break;
		case 's':
			ioparmMy.in.wParm1 &= ~8;
			ioparmMy.in.wParm1 |= fEnable ? 8 : 0;
			break;
		default:
			printf( "%s: event type not recognized; error in program.\n", szTpctlName );
			return ERR_TPCTL;
	}
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Resume on event %c has been %s\n", szTpctlName, chEvent, fEnable ? "enabled" : "disabled" );

	return 0;
}


int pm_set_resume_appointment( int intFiledesc, short sintYear, short sintMonth, short sintDay, short sintHour, short sintMinute, short sintSecond, int fDaily )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

#if 0
Do this if we change the code to allow changing parts of the date
	ioparmMy.in.bFunc = (byte) 0x34;
	ioparmMy.in.bSubFunc = (byte) 2;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;
#endif

	ioparmMy.in.bFunc = (byte) 0x34;
	ioparmMy.in.bSubFunc = (byte) 3;

	ioparmMy.in.dwParm4 =
		  ( (dword) bcd8_of_byte((byte)sintHour)   ) << 16
		| ( (dword) bcd8_of_byte((byte)sintMinute) ) << 8
		|   (dword) bcd8_of_byte((byte)sintSecond) ;

	if ( fDaily )
		ioparmMy.in.dwParm5 = 0x80990230;
	else
		ioparmMy.in.dwParm5 =
			  ( (dword) bcd8_of_byte((byte)sintYear)  ) << 16
			| ( (dword) bcd8_of_byte((byte)sintMonth) ) << 8
			|   (dword) bcd8_of_byte((byte)sintDay)   ;
	DEBUG_PRINT_INPARMS( ioparmMy );
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) {
		if ( fDaily ) 
			printf( "%s: Resume appointment made daily at %02d:%02d:%02d.\n", szTpctlName, sintHour, sintMinute, sintSecond );
		else
			printf( "%s: Resume appointment made at %02d:%02d:%02d:%02d:%02d:%02d.\n", szTpctlName, sintYear, sintMonth, sintDay, sintHour, sintMinute, sintSecond );
	}

	return 0;
}


int pm_request_standby( int intFiledesc )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x70;
	ioparmMy.in.bSubFunc = (byte) 0;
	ioparmMy.in.wParm1 = (word) 0;
	ioparmMy.in.wParm2 = (word) 0;
	ioparmMy.in.wParm3 = (word) 0;
	ioparmMy.in.dwParm4 = (dword) 0;
	ioparmMy.in.dwParm5 = (dword) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Standby requested.\n", szTpctlName );

	return 0;
}


int pm_request_suspend( int intFiledesc )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x70;
	ioparmMy.in.bSubFunc = (byte) 1;
	ioparmMy.in.wParm1 = (word) 0;
	ioparmMy.in.wParm2 = (word) 0;
	ioparmMy.in.wParm3 = (word) 0;
	ioparmMy.in.dwParm4 = (dword) 0;
	ioparmMy.in.dwParm5 = (dword) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Suspend requested.\n", szTpctlName );

	return 0;
}


int pm_request_hibernate( int intFiledesc )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x70;
	ioparmMy.in.bSubFunc = (byte) 2;
	ioparmMy.in.wParm1 = (word) 0;
	ioparmMy.in.wParm2 = (word) 0;
	ioparmMy.in.wParm3 = (word) 0;
	ioparmMy.in.dwParm4 = (dword) 0;
	ioparmMy.in.dwParm5 = (dword) 0;
	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Hibernation requested.\n", szTpctlName );

	return 0;
}


/*
 * I include this function for completeness only.
 * To enable it, define ALLOW_POWEROFF.
 * I did try it out, and it does switch the power off without further ado.
 * I synced the filesystem first, of course, and so should you.
 */

int pm_request_poweroff( int intFiledesc )
{
	smapi_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.bFunc = (byte) 0x70;
	ioparmMy.in.bSubFunc = (byte) 3;
	ioparmMy.in.wParm1 = (word) 0;
	ioparmMy.in.wParm2 = (word) 0;
	ioparmMy.in.wParm3 = (word) 0;
	ioparmMy.in.dwParm4 = (dword) 0;
	ioparmMy.in.dwParm5 = (dword) 0;

	if ( fVerbose ) printf( "%s: I'll now request that the power be switched off.\n", szTpctlName );

	intRtn = ioctl_smapi( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: Power-off requested....   Aaiiiiiieeeeee.....!\n", szTpctlName );

	return 0;
}


int res_dump_regs( int intFiledesc )
{
	superio_ioparm_t ioparmMy;
	byte bID, bRev;
	byte bFunctionEnableRegister;
	byte bFunctionAddressRegister;
	byte bPowerAndTestRegister;
	byte bFunctionControlRegister;
	byte bPrinterControlRegister;
	byte bUnused;
	byte bPowerManagementControlRegister;
	byte bTapeSerParConfigRegister;
	dword dwPnpRegs;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_LEG_REGS_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bID = ioparmMy.out.wParm0 >> 8;
	bRev = ioparmMy.out.wParm0 & 0xFF;
	bFunctionEnableRegister =   ioparmMy.out.dwParm1        & 0xFF;
	bFunctionAddressRegister = (ioparmMy.out.dwParm1 >>  8) & 0xFF;
	bPowerAndTestRegister =    (ioparmMy.out.dwParm1 >> 16) & 0xFF;
	bFunctionControlRegister = (ioparmMy.out.dwParm1 >> 24) & 0xFF;
	bPrinterControlRegister =          ioparmMy.out.dwParm2        & 0xFF;
	bUnused =	                      (ioparmMy.out.dwParm2 >>  8) & 0xFF;
	bPowerManagementControlRegister = (ioparmMy.out.dwParm2 >> 16) & 0xFF;
	bTapeSerParConfigRegister =       (ioparmMy.out.dwParm2 >> 24) & 0xFF;

	ioparmMy.in.wFunc = SUPERIO_FUNC_PNP_REGS_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	dwPnpRegs = ioparmMy.out.dwParm1;

	printf(
		"Super I/O regs:\n"
		"   chip ID:                             0x%x%s\n"
		"   chip revision:                       %d\n"
		"   function enable register:            0x%x\n"
		"   function address register:           0x%x\n"
		"   power and test mode register:        0x%x\n"
		"   function control register:           0x%x\n"
		"   printer control register:            0x%x\n"
		"   (unused):                            0x%x\n"
		"   power management control register:   0x%x\n"
		"   tape, ser, par config register:      0x%x\n"
		"   pnp registers:                       0x%lx\n",
		bID, (bID == 0xB0) ? " (correct)" : " (incorrect!!!)",
		bRev,
		bFunctionEnableRegister,
		bFunctionAddressRegister,
		bPowerAndTestRegister,
		bFunctionControlRegister,
		bPrinterControlRegister,
		bUnused,
		bPowerManagementControlRegister,
		bTapeSerParConfigRegister,
		(unsigned long) dwPnpRegs
	);

	return 0;
}


int res_set_lowerlegregs( int intFiledesc, dword dwThe )
{
	superio_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_LEG_REGS_SET;
	ioparmMy.in.dwParm1 = dwThe;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: first four super-IO registers have been set to 0x%lx\n", szTpctlName, (unsigned long) dwThe );

	return 0;
}


int res_print( int intFiledesc, char chRes )
{
	superio_ioparm_t ioparmMy;
	byte bID, bRev;
	byte bFunctionEnableRegister;
	byte bFunctionAddressRegister;
	byte bPowerAndTestRegister;
	byte bFunctionControlRegister;
	byte bPrinterControlRegister;
	byte bUnused;
	byte bPowerManagementControlRegister;
	byte bTapeSerParConfigRegister;
	byte bSer1IRQ, bSer2IRQ;
	word wSer1Base, wSer2Base;
	byte bParIRQ1, bParIRQ2;
	word wParBase;
	byte bFddIRQ;
	word wFddBase;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_LEG_REGS_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bID = ioparmMy.out.wParm0 >> 8;
	bRev = ioparmMy.out.wParm0 & 0xFF;
	bFunctionEnableRegister =   ioparmMy.out.dwParm1        & 0xFF;
	bFunctionAddressRegister = (ioparmMy.out.dwParm1 >>  8) & 0xFF;
	bPowerAndTestRegister =    (ioparmMy.out.dwParm1 >> 16) & 0xFF;
	bFunctionControlRegister = (ioparmMy.out.dwParm1 >> 24) & 0xFF;
	bPrinterControlRegister =          ioparmMy.out.dwParm2        & 0xFF;
	bUnused =	                      (ioparmMy.out.dwParm2 >>  8) & 0xFF;
	bPowerManagementControlRegister = (ioparmMy.out.dwParm2 >> 16) & 0xFF;
	bTapeSerParConfigRegister =       (ioparmMy.out.dwParm2 >> 24) & 0xFF;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_SER_IRQ_GET;
	ioparmMy.in.wParm0 = (word)1;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bSer1IRQ = (byte)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_SER_IRQ_GET;
	ioparmMy.in.wParm0 = (word)2;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bSer2IRQ = (byte)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_SER_BASE_GET;
	ioparmMy.in.wParm0 = (word)1;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	wSer1Base = (word)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_SER_BASE_GET;
	ioparmMy.in.wParm0 = (word)2;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	wSer2Base = (word)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_PAR_IRQ_GET_1;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bParIRQ1 = (byte)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_PAR_IRQ_GET_2;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bParIRQ2 = (byte)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_PAR_BASE_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	wParBase = (word)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_FDD_IRQ_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	bFddIRQ = (byte)ioparmMy.out.dwParm1;

	ioparmMy.in.wFunc  = SUPERIO_FUNC_FDD_BASE_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	wFddBase = (word)ioparmMy.out.dwParm1;

	if ( chRes == 'x' || chRes == 'f' ) printf(
		"   %s%s%sloppy controller:                                 %s FDD  0x%-3x IRQ%d\n",
		HI("f"),
		(bFunctionEnableRegister & 8) ? "enable " : "disable",
		wFddBase,
		bFddIRQ
	);
	if ( chRes == 'x' || chRes == 'p' ) printf(
		"   %s%s%sarallel port:                                     %s %s 0x%-3x IRQ%d/%d\n",
		HI("p"),
		(bFunctionEnableRegister & 1) ? "enable " : "disable",
		(
			(bFunctionAddressRegister & 3) == 0 ? "LPT2" /* 0x378 */ :
			(bFunctionAddressRegister & 3) == 1 ? "LPT1" /* 0x3BC */ :
			(bFunctionAddressRegister & 3) == 2 ? "LPT3" /* 0x278 */ :
			                                      "tri-state"
		),
		wParBase,
		bParIRQ1,
		bParIRQ2
	);
	if ( chRes == 'x' || chRes == '1' ) printf(
		"   %s%s%serial port %s%s%s:                                     %s %s 0x%-3x IRQ%d\n",
		HI("s"), HI("1"),
		(bFunctionEnableRegister & 2) ? "enable " : "disable",
		(
			((bFunctionAddressRegister >> 2) & 3) == 0 ? "COM1" :
			((bFunctionAddressRegister >> 2) & 3) == 1 ? "COM2" :
			((bFunctionAddressRegister >> 2) & 3) == 2 ? "COM3" :
			                                             "COM4"
		),
		wSer1Base,
		bSer1IRQ
	);
	if ( chRes == 'x' || chRes == '2' ) printf(
		"   %s%s%serial port %s%s%s:                                     %s %s 0x%-3x IRQ%d\n",
		HI("s"), HI("2"),
		(bFunctionEnableRegister & 4) ? "enable " : "disable",
		(
			((bFunctionAddressRegister >> 4) & 3) == 0 ? "COM1" :
			((bFunctionAddressRegister >> 4) & 3) == 1 ? "COM2" :
			((bFunctionAddressRegister >> 4) & 3) == 2 ? "COM3" :
			                                             "COM4"
		),
		wSer2Base,
		bSer2IRQ
	);

	return 0;
}


int res_set_parallel_LPT( int intFiledesc, byte bLPT )
{
	superio_ioparm_t ioparmMy;
	int intRtn;
	byte bMode;

	if ( bLPT == 0 || bLPT > 3 ) {
		printf( "%s: invalid LPT number: %d\n", szTpctlName, bLPT );
		return ERR_TPCTL;
	}

	bMode = (bLPT == 1) ? 1 : 
	        (bLPT == 2) ? 0 :
	                      2 ;

	ioparmMy.in.wFunc = SUPERIO_FUNC_LEG_REGS_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_LEG_REGS_SET;
	ioparmMy.in.dwParm1 &= ~(((dword)3) << 8);
	ioparmMy.in.dwParm1 |= (((dword)bMode) << 8);
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf(
		"%s: parallel port has been set to LPT%d\n",
		szTpctlName,
		bLPT
	);

	return 0;
}


int res_ablify_par( int intFiledesc, flag_t fEnable )
{
	superio_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_PAR_ABLIFY;
	ioparmMy.in.dwParm1 = fEnable;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: parallel port has been %s\n", szTpctlName, fEnable ? "enabled" : "disabled" );

	return 0;
}


int res_set_par_irq( int intFiledesc, byte bIRQ )
{
	superio_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_PAR_IRQ_SET;
	ioparmMy.in.dwParm1 = (dword)bIRQ;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: parallel port interrupt has been set to IRQ%d\n", szTpctlName, bIRQ );

	return 0;
}


int res_set_par_base( int intFiledesc, word wBase )
{
	superio_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_PAR_BASE_SET;
	ioparmMy.in.dwParm1 = (dword)wBase;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: parallel port base address has been set to 0x%x\n", szTpctlName, wBase );

	return 0;
}


int res_set_serial_COM( int intFiledesc, byte bSer, byte bCOM )
{
	superio_ioparm_t ioparmMy;
	int intRtn;
	byte bMode;

	if ( bCOM == 0 || bCOM > 4 ) {
		printf( "%s: invalid COM number: %d\n", szTpctlName, bCOM );
		return ERR_TPCTL;
	}

	bMode = bCOM - 1;

	ioparmMy.in.wFunc = SUPERIO_FUNC_LEG_REGS_GET;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_LEG_REGS_SET;

	switch (  bSer ) {
		case 1:
			ioparmMy.in.dwParm1 &= ~(((dword)3) << 10);
			ioparmMy.in.dwParm1 |= (((dword)bMode) << 10);
			break;
		case 2:
			ioparmMy.in.dwParm1 &= ~(((dword)3) << 12);
			ioparmMy.in.dwParm1 |= (((dword)bMode) << 12);
			break;
		default:
			printf( "%s: invalid serial port number: %d\n", szTpctlName, bSer );
			return ERR_TPCTL;
	}
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf(
		"%s: serial port %d has been set to COM%d\n",
		szTpctlName, bSer, bCOM
	);

	return 0;
}


int res_ablify_ser( int intFiledesc, byte bWhich, flag_t fEnable )
{
	superio_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_SER_ABLIFY;
	ioparmMy.in.wParm0 = bWhich;
	ioparmMy.in.dwParm1 = fEnable;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: serial port %d has been %s\n", szTpctlName, bWhich, fEnable ? "enabled" : "disabled" );

	return 0;
}


int res_set_ser_irq( int intFiledesc, byte bWhich, byte bIRQ )
{
	superio_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_SER_IRQ_SET;
	ioparmMy.in.wParm0 = bWhich;
	ioparmMy.in.dwParm1 = (dword)bIRQ;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: serial port %d interrupt has been set to IRQ%d\n", szTpctlName, bWhich, bIRQ );

	return 0;
}


int res_set_ser_base( int intFiledesc, byte bWhich, word wBase )
{
	superio_ioparm_t ioparmMy;
	int intRtn;

	ioparmMy.in.wFunc = SUPERIO_FUNC_SER_BASE_SET;
	ioparmMy.in.wParm0 = bWhich;
	ioparmMy.in.dwParm1 = (dword)wBase;
	intRtn = ioctl_superio( intFiledesc, &ioparmMy );
	if ( intRtn ) return intRtn;

	if ( fVerbose ) printf( "%s: serial port %d base address has been set to 0x%x\n", szTpctlName, bWhich, wBase );

	return 0;
}


/*** flags for options ****/
/* They have to be int's in order to work with getopt() */

int fdumpregs;

int fhelp=0, fV=0, fdevice=0, fdull=0, fquiet;

int fS=0, fZ=0, fH=0, fO=0;

int fx,
		fix=0,
			fib=0,
			fic=0,
			fid=0,
			fio=0,
			fiU=0,
			fis=0,
			fir=0,
			fip=0,
		fsx=0,
			fsdx=0,
				fsdi=0,
				fsdc=0,
				fsdt=0,
				fsdT=0,
				fsdm=0,
				fsd2=0,
			fsf=0,
			fspx=0,
				fspi=0,
				fspe=0,
		fsst=0,
		fpx=0,
			fpmx=0,
				fpma=0,
				fpmb=0,
				fpmR=0,
			fprx=0,
				fpra=0,
				fprh=0,
				fprl=0,
				fprs=0,
			fpsx=0,
				fpsh=0,
				fpsp=0,
				fpsl=0,
				fpsS=0,
				fpsZ=0,
				fpsH=0,
				fpsb=0,
				fpse=0,
			fptx=0,
				fptS=0,
				fptZ=0,
				fptB=0,
				fptd=0,
			fpdx=0,
				fpdH=0,
				fpdZx=0,
					fpdZa=0,
					fpdZb=0,
					fpdZh,
					fpdZu,
		frx=0,
			frf=0,
			frp=0,
			frsx=0,
				frs1=0,
				frs2=0;

/*** other variables ***/

int opterr=1; /* enable option parsing error messages */
int optind;   /* set by the getopt function */
char *optarg; /* set by the getopt function */

/*
 * N.B.
 * Options without arguments should require only read privileges on the device
 * Options with arguments should require read/write privileges on the device
 */

/*
 * N.B. This is where features are enabled/disabled at compile time
 * Debugging features do not appear in the Usage.
 */

/*
 * Array of my long options (of type "struct option")
 * To avoid ambiguity, do not use option "-in" since 
 * this is like the beginning of "--info"
 */
struct option asoMyLongopts[] = {

	{ "h", no_argument, &fhelp, 1 },
	{ "help", no_argument, &fhelp, 1 },

	{ "V", no_argument, &fV, 1 },
	{ "version", no_argument, &fV, 1 },

	{ "device", required_argument, &fdevice, 1 },

	{ "dull", no_argument, &fdull, 1 },
	{ "quiet", no_argument, &fquiet, 1 },

	{ "S", no_argument, &fS, 1 },
	{ "standby", no_argument, &fS, 1 },
	{ "Z", no_argument, &fZ, 1 },
	{ "suspend", no_argument, &fZ, 1 },
	{ "H", no_argument, &fH, 1 },
	{ "hibernate", no_argument, &fH, 1 },
#ifdef ALLOW_POWEROFF
	{ "O", required_argument, &fO, 1 },
	{ "poweroff", required_argument, &fO, 1 },
#endif

	{ "x", no_argument, &fx, 1 },
	{ "all", no_argument, &fx, 1 },
		{ "ix", no_argument, &fix, 1 },
		{ "info-all", no_argument, &fix, 1 },
			{ "ib", no_argument, &fib, 1 },
			{ "info-bios", no_argument, &fib, 1 },
#ifdef VIAGRA
			{ "ic", no_argument, &fic, 1 },
			{ "info-CMOS", no_argument, &fic, 1 },
#endif
			{ "id", no_argument, &fid, 1 },
			{ "info-display", no_argument, &fid, 1 },
			{ "io", no_argument, &fio, 1 },
			{ "info-docking", no_argument, &fio, 1 },
			{ "ip", no_argument, &fip, 1 },
			{ "info-processor", no_argument, &fip, 1 },
			{ "ir", required_argument, &fir, 1 },
			{ "info-refresh", required_argument, &fir, 1 },
			{ "is", no_argument, &fis, 1 },
			{ "info-sensor", no_argument, &fis, 1 },
			{ "iU", no_argument, &fiU, 1 },
			{ "info-ultrabay", no_argument, &fiU, 1 },
			{ "info-UltraBay", no_argument, &fiU, 1 },
		{ "sx", no_argument, &fsx, 1 },
		{ "setup-all", no_argument, &fsx, 1 },
			{ "sdx", no_argument, &fsdx, 1 },
			{ "setup-display-all", no_argument, &fsdx, 1 },
				{ "sdi", optional_argument, &fsdi, 1 },
				{ "setup-display-internal", optional_argument, &fsdi, 1 },
				{ "sdc", optional_argument, &fsdc, 1 },
				{ "setup-display-CRT", optional_argument, &fsdc, 1 },
				{ "sdt", optional_argument, &fsdt, 1 },
				{ "setup-display-TV", optional_argument, &fsdt, 1 },
				{ "sdT", optional_argument, &fsdT, 1 },
				{ "setup-display-TV-select", optional_argument, &fsdT, 1 },
				{ "sdm", optional_argument, &fsdm, 1 },
				{ "setup-display-monitor-detection-ignore", optional_argument, &fsdm, 1 },
				{ "sd2", optional_argument, &fsd2, 1 },
				{ "setup-display-dual", optional_argument, &fsd2, 1 },
			{ "sf", optional_argument, &fsf, 1 },
			{ "setup-fn-hotkey", optional_argument, &fsf, 1 },
			{ "setup-Fn-hotkey", optional_argument, &fsf, 1 },
			{ "spx", no_argument, &fspx, 1 },
			{ "setup-pointing-device-all", no_argument, &fspx, 1 },
				{ "spi", optional_argument, &fspi, 1 },
				{ "setup-pointing-device-internal", optional_argument, &fspi, 1 },
				{ "spe", optional_argument, &fspe, 1 },
				{ "setup-pointing-device-external", optional_argument, &fspe, 1 },
#ifdef VIAGRA
#if 0
/* removed until I fix rtcmosram so that it corrects the checksum */
			{ "sst", optional_argument, &fsst, 1 },
			{ "setup-daylight-saving-time", optional_argument, &fsst, 1 },
#else
			{ "sst", no_argument, &fsst, 1 },
			{ "setup-daylight-saving-time", no_argument, &fsst, 1 },
#endif
#ifdef DEBUG_REGS
			{ "dumpregs", no_argument, &fdumpregs, 1 },
#endif
#endif
		{ "px", no_argument, &fpx, 1 },
		{ "pm-all", no_argument, &fpx, 1 },
			{ "pmx", no_argument, &fpmx, 1 },
			{ "pm-mode-all", no_argument, &fpmx, 1 },
				{ "pma", optional_argument, &fpma, 1 },
				{ "pm-mode-AC", optional_argument, &fpma, 1 },
				{ "pmb", optional_argument, &fpmb, 1 },
				{ "pm-mode-battery", optional_argument, &fpmb, 1 },
				{ "pmR", optional_argument, &fpmR, 1 },
				{ "pm-mode-safe-suspend", optional_argument, &fpmR, 1 },
				{ "pm-mode-RediSafe", optional_argument, &fpmR, 1 },
			{ "prx", no_argument, &fprx, 1 },
			{ "pm-resume-all", no_argument, &fprx, 1 },
				{ "pra", optional_argument, &fpra, 1 },
				{ "pm-resume-appointment", optional_argument, &fpra, 1 },
				{ "prh", optional_argument, &fprh, 1 },
				{ "pm-resume-hardware", optional_argument, &fprh, 1 },
				{ "prl", optional_argument, &fprl, 1 },
				{ "pm-resume-lid-opening", optional_argument, &fprl, 1 },
				{ "prs", optional_argument, &fprs, 1 },
				{ "pm-resume-serial-RI", optional_argument, &fprs, 1 },
			{ "psx", no_argument, &fpsx, 1 },
			{ "pm-sedation-all", no_argument, &fpsx, 1 },
				{ "psS", optional_argument, &fpsS, 1 },
				{ "pm-sedation-standby-timer", optional_argument, &fpsS, 1 },
				{ "psZ", optional_argument, &fpsZ, 1 },
				{ "pm-sedation-suspend-or-hibernate-timer", optional_argument, &fpsZ, 1 },
				{ "psH", optional_argument, &fpsH, 1 },
				{ "pm-sedation-hibernate-from-suspend-timer", optional_argument, &fpsH, 1 },
				{ "psb", optional_argument, &fpsb, 1 },
				{ "pm-sedation-battery-low", optional_argument, &fpsb, 1 },
				{ "pse", optional_argument, &fpse, 1 },
				{ "pm-sedation-environment-exhausted", optional_argument, &fpse, 1 },
				{ "psh", optional_argument, &fpsh, 1 },
				{ "pm-sedation-hardware-or-software", optional_argument, &fpsh, 1 },
				{ "psl", optional_argument, &fpsl, 1 },
				{ "pm-sedation-lid-closure", optional_argument, &fpsl, 1 },
				{ "psp", optional_argument, &fpsp, 1 },
				{ "pm-sedation-power-switch", optional_argument, &fpsp, 1 },
			{ "ptx", no_argument, &fptx, 1 },
			{ "pm-timer-mode-all", no_argument, &fptx, 1 },
				{ "ptB", optional_argument, &fptB, 1 },
				{ "pm-timer-mode-blank-display", optional_argument, &fptB, 1 },
				{ "ptS", optional_argument, &fptS, 1 },
				{ "pm-timer-mode-standby", optional_argument, &fptS, 1 },
				{ "ptZ", optional_argument, &fptZ, 1 },
				{ "pm-timer-mode-suspend-or-hibernate", optional_argument, &fptZ, 1 },
				{ "ptd", optional_argument, &fptd, 1 },
				{ "pm-timer-mode-drive-powerdown", optional_argument, &fptd, 1 },
			{ "pdx", no_argument, &fpdx, 1 },
			{ "pm-delay-all", no_argument, &fpdx, 1 },
				{ "pdH", optional_argument, &fpdH, 1 },
				{ "pm-delay-hibernate-from-suspend", optional_argument, &fpdH, 1 },
				{ "pdZx", no_argument, &fpdZx, 1 },
				{ "pm-delay-suspend-or-hibernate-all", no_argument, &fpdZx, 1 },
					{ "pdZa", optional_argument, &fpdZa, 1 },
					{ "pm-delay-suspend-or-hibernate-AC", optional_argument, &fpdZa, 1 },
					{ "pdZb", optional_argument, &fpdZb, 1 },
					{ "pm-delay-suspend-or-hibernate-battery", optional_argument, &fpdZb, 1 },
					{ "pdZh", no_argument, &fpdZh, 1 },
					{ "pm-delay-suspend-or-hibernate-high", no_argument, &fpdZh, 1 },
					{ "pdZu", no_argument, &fpdZu, 1 },
					{ "pm-delay-suspend-or-hibernate-auto", no_argument, &fpdZu, 1 },
#ifdef VIAGRA
#ifdef DEBUG_REGS
		{ "rx", optional_argument, &frx, 1 },
		{ "resource-all", optional_argument, &frx, 1 },
#else
		{ "rx", no_argument, &frx, 1 },
		{ "resource-all", no_argument, &frx, 1 },
#endif
			{ "rf", no_argument, &frf, 1 },
			{ "resource-floppy", no_argument, &frf, 1 },
			{ "rp", optional_argument, &frp, 1 },
			{ "resource-parallel", optional_argument, &frp, 1 },
			{ "rsx", no_argument, &frsx, 1 },
			{ "resource-serial-all", no_argument, &frsx, 1 },
				{ "rs1", optional_argument, &frs1, 1 },
				{ "resource-serial-1", optional_argument, &frs1, 1 },
				{ "rs2", optional_argument, &frs2, 1 },
				{ "resource-serial-2", optional_argument, &frs2, 1 },
#endif

	{ 0, 0, 0, 0 }
};

char szUsage[] = "Usage: %s [--h|--help] [--V|--version]\n"
	"   [--device=<name>] [--dull] [--quiet]\n"
#ifdef ALLOW_POWEROFF
	"   [--S|--standby|--Z|--suspend|--H|--hibernate|--O|--poweroff]\n"
#else
	"   [--S|--standby|--Z|--suspend|--H|--hibernate]\n"
#endif
	"   [--x|--all]\n"
	"   [--ix|--info-all]\n"
#ifdef VIAGRA
	"   [--ib|--info-bios] [--ic|--info-CMOS-RAM] [--id|--info-display]\n"
#else
	"   [--ib|--info-bios] [--id|--info-display]\n"
#endif
	"   [--io|--info-docking] [--ip|--info-processor]\n"
	"   [--ir|--info-refresh=<VGA-mode-#>]\n"
	"   [--is|--info-sensor] [--iU|--info-UltraBay]\n"
	"   [--sx|--setup-all]\n"
	"   [--sdx|--setup-display-all]\n"
	"   [--sdi|--setup-display-internal[=(disable|enable)]]\n"
	"   [--sdc|--setup-display-CRT[=(disable|enable)]]\n"
	"   [--sdt|--setup-display-TV[=(disable|enable)]]\n"
	"   [--sdT|--setup-display-TV-select[=(disable|enable)]]\n"
	"   [--sdm|--setup-display-monitor-detection-ignore[=(disable|enable)]]\n"
	"   [--sd2|--setup-display-dual[=(disable|enable)]]\n"
	"   [--sf |--setup-Fn-hotkey[=(nonsticky|sticky|locked)]]\n"
	"   [--spx|--setup-pointing-device-all]\n"
	"   [--spi|--setup-pointing-device-internal[=(disable|enable|auto)]]\n"
	"   [--spe|--setup-pointing-device-external[=(disable|enable|auto)]]\n"
#ifdef VIAGRA
#if 0
	"   [--sst|--setup-daylight-saving-time[=(enable|disable)]]\n"
#else
	"   [--sst|--setup-daylight-saving-time]\n"
#endif
#endif
	"   [--px|--pm-all]\n"
	"   [--pmx|--pm-mode-all]\n"
	"   [--pma|--pm-mode-AC[=(high|auto|manual)]]\n"
	"   [--pmb|--pm-mode-battery[=(high|auto|manual)]]\n"
	"   [--pmR|--pm-mode-RediSafe|--pm-mode-safe-suspend[=(disable|enable)]]\n"
	"   [--prx|--pm-resume-all]\n"
	"   [--pra|--pm-resume-appointment[=(enable|disable|(daily|<yy>:<mm>:<dd>):<hh>:<mm>:<ss>)]]\n"
	"   [--prh|--pm-resume-hardware[=(disable|enable)]]\n"
	"   [--prl|--pm-resume-lid[=(disable|enable)]]\n"
	"   [--prs|--pm-resume-serial-RI[=(disable|enable)]]\n"
	"   [--psx|--pm-sedation-all]\n"
	"   [--psS|--pm-sedation-standby-timer[=(-|S|Z|R|H|O)]]\n"
	"   [--psZ|--pm-sedation-suspend-or-hibernate-timer[=(-|S|Z|R|H|O)]]\n"
	"   [--psH|--pm-sedation-hibernate-from-suspend-timer[=(-|S|Z|R|H|O)]]\n"
	"   [--psb|--pm-sedation-battery-low[=(-|S|Z|R|H|O)]]\n"
	"   [--pse|--pm-sedation-environment-exhausted[=(-|S|Z|R|H|O)]]\n"
	"   [--psh|--pm-sedation-hardware-or-software[=(-|S|Z|R|H|O)]]\n"
	"   [--psl|--pm-sedation-lid-closure[=(-|S|Z|R|H|O)]]\n"
	"   [--psp|--pm-sedation-power-switch[=(-|S|Z|R|H|O)]]\n"
	"   [--ptx|--pm-timer-mode-all]\n"
	"   [--ptB|--pm-timer-mode-blank-display[=(disable|enable)]]\n"
	"   [--ptS|--pm-timer-mode-standby[=(disable|enable)]]\n"
	"   [--ptZ|--pm-timer-mode-suspend-or-hibernate[=(disable|enable)]]\n"
	"   [--ptd|--pm-timer-mode-drive-powerdown[=(disable|enable)]]\n"
	"   [--pdx|--pm-delay-all]\n"
	"   [--pdH|--pm-delay-hibernate-from-suspend[=<minutes>]]\n"
	"   [--pdZx|--pm-delay-suspend-or-hibernate-all]\n"
	"   [--pdZa|--pm-delay-suspend-or-hibernate-AC-manual[=<minutes>]]\n"
	"   [--pdZb|--pm-delay-suspend-or-hibernate-battery-manual[=<minutes>]]\n"
	"   [--pdZh|--pm-delay-suspend-or-hibernate-high]]\n"
	"   [--pdZu|--pm-delay-suspend-or-hibernate-auto]]\n"
#ifdef VIAGRA
	"   [--rx|--resource-all]\n"
	"   [--rf|--resource-floppy]\n"
	"   [--rp|--resource-parallel[=(<port>|LPT<#>|IRQ<#>|enable|disable)]]\n"
	"   [--rsx|--resource-serial-all]\n"
	"   [--rs1|--resource-serial-1[=(<port>|COM<#>|IRQ<#>|enable|disable)]]\n"
	"   [--rs2|--resource-serial-2[=(<port>|COM<#>|IRQ<#>|enable|disable)]]\n"
#endif
;


/*
 * Compares two strings up to the length
 * of the shorter of them
 */
int strcmp_arg( char * pch1, char * pch2 )
{
	size_t size1, size2, sizeLesser;

	size1 = strlen( pch1 );
	size2 = strlen( pch2 );

	if ( size1 < size2 ) sizeLesser = size1; else sizeLesser = size2;

	if ( sizeLesser == 0 ) return ERR_TPCTL;  /* Reject args of length 0 */

	return strncasecmp( pch1, pch2, sizeLesser );
}


smapidev_powermode_t powermode_named( char * pchArg )
{
	if ( !strcmp_arg(pchArg, "high") ) return SMAPIDEV_POWERMODE_HIGH;
	if ( !strcmp_arg(pchArg, "auto") ) return SMAPIDEV_POWERMODE_AUTO;
	if ( !strcmp_arg(pchArg, "manual") ) return SMAPIDEV_POWERMODE_MANUAL;

	return SMAPIDEV_POWERMODE_UNRECOGNIZED;
}


int int_pointing_device_state_named( char * pchArg )
{
	if ( !strcmp_arg(pchArg, "disable") ) return 0;
	if ( !strcmp_arg(pchArg, "enable") ) return 1;
	if ( !strcmp_arg(pchArg, "auto") ) return 2;

	return -1;
}


int int_Fn_hotkey_state_named( char * pchArg )
{
	if ( !strcmp_arg(pchArg, "nonsticky") ) return 0;
	if ( !strcmp_arg(pchArg, "sticky") ) return 1;
	if ( !strcmp_arg(pchArg, "locked") ) return 3;

	return -1;
}


int int_flagval_named( char * pchArg )
{
	if ( !strcmp_arg(pchArg, "disable") ) return 0;
	if ( !strcmp_arg(pchArg, "enable") ) return 1;

	return -1;
}


/****** main ******/


int main( int argc, char * argv[] )
{
	char *pchOpts=""; /* no short opts */
	int iasoCur; /* index of current opt in the array of struct options */
	int intFiledesc;
	int intOpenFlags;
	int intRtn;
	char chRtn;
	int fErr;
	flag_t fdevice_arg=0;
	char *pchDeviceName = SZ_DEFAULT_DEVICE_NAME;
	flag_t fO_arg=0;
	flag_t fir_arg=0;
	word wVGAMode;
	flag_t fsdi_arg=0, fsdc_arg=0, fsdt_arg=0, fsdm_arg=0, fsd2_arg=0, fsdT_arg=0;
	flag_t fDisplayEnableInternal, fDisplayEnableCRT, fDisplayEnableTV, fDisplayEnableMonitorDetectionIgnore, fDisplayEnableDual, fDisplayEnableTVSelected;
	flag_t fsf_arg=0;
	byte bFnHotkeyState;
	flag_t fspi_arg=0, fspe_arg=0;
	byte bPointingDeviceStateInt, bPointingDeviceStateExt;
	flag_t fsst_arg=0;
	flag_t fDaylightsavingtimeEnable;
	flag_t fpma_arg=0, fpmb_arg=0, fpmR_arg=0;
	smapidev_powermode_t powermodeAC, powermodeBattery;
	flag_t fSafeSuspendEnable;
	flag_t fpra_date_arg=0, fpra_endis_arg=0;
	short sintResumeApptYear, sintResumeApptMonth, sintResumeApptDay, sintResumeApptHour, sintResumeApptMinute, sintResumeApptSecond;
	flag_t fResumeApptDaily;
	flag_t fResumeEnableAppointment;
	flag_t fprh_arg=0, fprl_arg=0, fprs_arg=0;
	flag_t fResumeEnableHardware, fResumeEnableLid, fResumeEnableSerial;
	flag_t fpsh_arg=0, fpsp_arg=0, fpsl_arg=0, fpsS_arg=0, fpsZ_arg=0, fpsH_arg=0, fpsb_arg=0, fpse_arg=0;
	char ch_h_Sedation, ch_p_Sedation, ch_l_Sedation, ch_S_Sedation, ch_Z_Sedation, ch_H_Sedation, ch_b_Sedation, ch_e_Sedation;
	flag_t fptS_arg=0, fptZ_arg=0, fptB_arg=0, fptd_arg=0;
	flag_t fTimerEnableStandby, fTimerEnableSuspendHibernate, fTimerEnableBlank, fTimerEnableDrivePowerdown;
	flag_t fpdZa_arg=0, fpdZb_arg=0, fpdH_arg=0;
	byte bDelaySuspendOrHibernateAC, bDelaySuspendOrHibernateBattery, bDelaySuspendToHibernate;
	flag_t frx_arg=0;
	dword dwSuperioRegisters;
	flag_t frp_ablify_arg=0;
	flag_t fParEn;
	flag_t frp_lpt_arg=0;
	byte bParLPT;
	flag_t frp_base_arg=0;
	word wParBase;
	flag_t frp_irq_arg=0;
	byte bParIRQ;
	flag_t frs1_ablify_arg=0, frs2_ablify_arg=0;
	flag_t fSer1En, fSer2En;
	flag_t frs1_com_arg=0, frs2_com_arg=0;
	byte bSer1COM, bSer2COM;
	flag_t frs1_base_arg=0, frs2_base_arg=0;
	word wSer1Base, wSer2Base;
	flag_t frs1_irq_arg=0, frs2_irq_arg=0;
	byte bSer1IRQ, bSer2IRQ;

	fErr = 0;
	fHighlight = 1;
	fVerbose = 1;

	while(
		(chRtn = getopt_long_only(argc, argv, pchOpts, asoMyLongopts, &iasoCur )) != EOF
	) {
		if ( chRtn == '?' || chRtn == '-' ) {
			/* unrecognized option or missing argument */
			fErr = 1;
			break; /* out of while loop */
		}
		/*** Process option arguments ***/
		if ( optarg == NULL ) continue; /* with while loop */
		/* optarg is not NULL */
		if ( asoMyLongopts[iasoCur].flag == &fdevice ) {
			sscanf( optarg, "%as", &pchDeviceName );
			fdevice_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fO ) {
			/* For safety, we require an argument for the --O option */
			if ( strcasecmp ( optarg, "really" ) == 0 ) {
				fO_arg = 1;
			} else {
				printf( "%s: If you really want to switch the power off, add the argument 'really'.\n", szTpctlName );
				fO = 0;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fir ) {
			sscanf( optarg, "%hi", &wVGAMode );
			fir_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fsdi ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Display device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsdi = 0;
			} else {
				fDisplayEnableInternal = intFlag;
				fsdi_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fsdc ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Display device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsdc = 0;
			} else {
				fDisplayEnableCRT = intFlag;
				fsdc_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fsdt ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Display device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsdt = 0;
			} else {
				fDisplayEnableTV = intFlag;
				fsdt_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fsdm ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Display device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsdm = 0;
			} else {
				fDisplayEnableMonitorDetectionIgnore = intFlag;
				fsdm_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fsd2 ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Display device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsd2 = 0;
			} else {
				fDisplayEnableDual = intFlag;
				fsd2_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fsdT ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Display device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsdT = 0;
			} else {
				fDisplayEnableTVSelected = intFlag;
				fsdT_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fsf ) {
			int intFnHotkeyState;
			intFnHotkeyState = int_Fn_hotkey_state_named( optarg );
			if ( intFnHotkeyState == -1 ) { /* not recognized */
				printf( "%s: Fn hotkey mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsf = 0;
			} else {
				bFnHotkeyState = intFnHotkeyState;
				fsf_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fspi ) {
			int intPointingDeviceState;
			intPointingDeviceState = int_pointing_device_state_named( optarg );
			if ( intPointingDeviceState == -1 ) { /* not recognized */
				printf( "%s: Pointing device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fspi = 0;
			} else {
				bPointingDeviceStateInt = intPointingDeviceState;
				fspi_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fspe ) {
			int intPointingDeviceState;
			intPointingDeviceState = int_pointing_device_state_named( optarg );
			if ( intPointingDeviceState == -1 ) { /* not recognized */
				printf( "%s: Pointing device mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fspe = 0;
			} else {
				bPointingDeviceStateExt = intPointingDeviceState;
				fspe_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fsst ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: daylight saving time mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fsst = 0;
			} else {
				fDaylightsavingtimeEnable = intFlag;
				fsst_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fpma ) {
			powermodeAC = powermode_named( optarg );
			if ( powermodeAC == SMAPIDEV_POWERMODE_UNRECOGNIZED ) {
				printf( "%s: AC pm mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fpma = 0;
			} else {
				fpma_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fpmb ) {
			powermodeBattery = powermode_named( optarg );
			if ( powermodeBattery == SMAPIDEV_POWERMODE_UNRECOGNIZED ) {
				printf( "%s: Battery pm mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fpmb = 0;
			} else {
				fpmb_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fpmR ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) {
				printf( "%s: Safe suspend mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fpmR = 0;
			} else {
				fSafeSuspendEnable = intFlag;
				fpmR_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fpra ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == 0 || intFlag == 1 ) { /* recognized as flag */
				fResumeEnableAppointment = intFlag;
				fpra_endis_arg = 1;
			} else {
				char *pchOptarg;
				pchOptarg = optarg;
				if ( !strncasecmp( optarg, "d", 1 ) ) {
					fResumeApptDaily = 1;
					sscanf(
						pchOptarg, "%*[daily:DAILY]%2hi:%2hi:%2hi",
						&sintResumeApptHour, &sintResumeApptMinute, &sintResumeApptSecond
					);
				} else {
					fResumeApptDaily = 0;
					sscanf(
						pchOptarg, "%2hi:%2hi:%2hi:%2hi:%2hi:%2hi",
						&sintResumeApptYear, &sintResumeApptMonth, &sintResumeApptDay, &sintResumeApptHour, &sintResumeApptMinute, &sintResumeApptSecond
					);
				}
				fpra_date_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fprh ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Resume mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fprh = 0;
			} else {
				fResumeEnableHardware = intFlag;
				fprh_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fprl ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Resume mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fprl = 0;
			} else {
				fResumeEnableLid = intFlag;
				fprl_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fprs ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Resume mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fprs = 0;
			} else {
				fResumeEnableSerial = intFlag;
				fprs_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fpsh ) {
			sscanf( optarg, "%c", &ch_h_Sedation );
			fpsh_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpsp ) {
			sscanf( optarg, "%c", &ch_p_Sedation );
			fpsp_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpsl ) {
			sscanf( optarg, "%c", &ch_l_Sedation );
			fpsl_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpsS ) {
			sscanf( optarg, "%c", &ch_S_Sedation );
			fpsS_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpsZ ) {
			sscanf( optarg, "%c", &ch_Z_Sedation );
			fpsZ_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpsH ) {
			sscanf( optarg, "%c", &ch_H_Sedation );
			fpsH_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpsb ) {
			sscanf( optarg, "%c", &ch_b_Sedation );
			fpsb_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpse ) {
			sscanf( optarg, "%c", &ch_e_Sedation );
			fpse_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fptS ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Timer mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fptS = 0;
			} else {
				fTimerEnableStandby = intFlag;
				fptS_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fptZ ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Timer mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fptZ = 0;
			} else {
				fTimerEnableSuspendHibernate = intFlag;
				fptZ_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fptB ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Timer mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fptB = 0;
			} else {
				fTimerEnableBlank = intFlag;
				fptB_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fptd ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag == -1 ) { /* not recognized */
				printf( "%s: Timer mode '%s' not recognized -- skipping option.\n", szTpctlName, optarg );
				fptd = 0;
			} else {
				fTimerEnableDrivePowerdown = intFlag;
				fptd_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &fpdZa ) {
			word wDuration;
			sscanf( optarg, "%hi", &wDuration );
			bDelaySuspendOrHibernateAC = (byte) wDuration;
			fpdZa_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpdZb ) {
			word wDuration;
			sscanf( optarg, "%hi", &wDuration );
			bDelaySuspendOrHibernateBattery = (byte) wDuration;
			fpdZb_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &fpdH ) {
			word wDuration;
			sscanf( optarg, "%hi", &wDuration );
			bDelaySuspendToHibernate = (byte) wDuration;
			fpdH_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &frx ) {
			sscanf( optarg, "%i", &dwSuperioRegisters );
			frx_arg = 1;
		} else if ( asoMyLongopts[iasoCur].flag == &frp ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag != -1 ) { /* is a flag */
				fParEn = (flag_t)intFlag;
				frp_ablify_arg = 1;
			} else if ( strncasecmp( optarg, "IRQ", 3 )==0 ) { /* is an IRQ */
					bParIRQ = optarg[3] - '0';
					frp_irq_arg = 1;
			} else if ( strncasecmp( optarg, "LPT", 3 )==0 ) { /* is an LPT */
					bParLPT = optarg[3] - '0';
					frp_lpt_arg = 1;
			} else { /* assume that it's a number */
				sscanf( optarg, "%hi", &wParBase );
				frp_base_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &frs1 ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag != -1 ) { /* is a flag */
				fSer1En = (flag_t)intFlag;
				frs1_ablify_arg = 1;
			} else if ( strncasecmp( optarg, "IRQ", 3 )==0 ) { /* is an IRQ */
					bSer1IRQ = optarg[3] - '0';
					frs1_irq_arg = 1;
			} else if ( strncasecmp( optarg, "COM", 3 )==0 ) { /* is a COM */
					bSer1COM = optarg[3] - '0';
					frs1_com_arg = 1;
			} else { /* assume that it's a number */
				sscanf( optarg, "%hi", &wSer1Base );
				frs1_base_arg = 1;
			}
		} else if ( asoMyLongopts[iasoCur].flag == &frs2 ) {
			int intFlag;
			intFlag = int_flagval_named( optarg );
			if ( intFlag != -1 ) { /* is a flag */
				fSer2En = (flag_t)intFlag;
				frs2_ablify_arg = 1;
			} else if ( strncasecmp( optarg, "IRQ", 3 )==0 ) { /* is an IRQ */
				bSer2IRQ = optarg[3] - '0';
				frs2_irq_arg = 1;
			} else if ( strncasecmp( optarg, "COM", 3 )==0 ) { /* is a COM */
				bSer2COM = optarg[3] - '0';
				frs2_com_arg = 1;
			} else { /* assume that it's a number */
				sscanf( optarg, "%hi", &wSer2Base );
				frs2_base_arg = 1;
			}
		}
	} /* while */

	if ( argc <= 1 ) {
		/* No options were given: give help */
		fErr = 0; fhelp = 1;
	}

	if ( optind != argc ) {
		/* at least one non-option argument was given */
		fErr = 1;
	}

	if ( fErr ) {
		fprintf( stderr, "%s: Error scanning options.  Try '%s --help'.\n", szTpctlName, szTpctlName );
		exit( EXIT_FAILURE );
	} else if ( fhelp ) {
		printf( szUsage, szTpctlName );
		exit( EXIT_SUCCESS );
	}

	if ( fdull ) fHighlight = 0;
	if ( fquiet ) fVerbose = 0;

	if ( fx ) fix = fsx = fpx = frx = 1;
		if ( fix ) fib = fid = fio = fip = fis = fiU = 1;
#ifdef VIAGRA
		if ( fix ) fic = 1;
#endif
		if ( fsx ) fsdx = fsf = fsst = fspx = 1;
			if ( fsdx ) fsdi = fsdc = fsdt = fsdT = fsdm = fsd2 = 1;
			if ( fspx ) fspi = fspe = 1;
		if ( fpx ) fpmx = fprx = fpsx = fptx = fpdx = 1;
			if ( fpmx ) fpma = fpmb = fpmR = 1;
			if ( fprx ) fpra = fprh = fprl = fprs = 1;
			if ( fpsx ) fpsS = fpsZ = fpsH = fpsb = fpse = fpsh = fpsl = fpsp = 1;
			if ( fptx ) fptB = fptS = fptZ = fptd = 1;
			if ( fpdx ) fpdH = fpdZx = 1;
				if ( fpdZx ) fpdZa = fpdZb = fpdZh = fpdZu = 1;
		if ( frx ) frf = frp = frsx = 1;
			if ( frsx ) frs1 = frs2 = 1;

	/*** Perform only one type of sedation request ***/
	if ( fS ) fZ = fH = fO = 0; 
	if ( fZ ) fS = fH = fO = 0; 
	if ( fH ) fS = fZ = fO = 0; 
	if ( fO ) fS = fZ = fH = 0;


	if ( fS || fZ || fH || fO || fsdi_arg || fsdc_arg || fsdt_arg || fsdm_arg || fsd2_arg || fsdT_arg || fsf_arg || fspi_arg || fspe_arg || fsst_arg || fpma_arg || fpmb_arg || fpmR_arg || fpra_date_arg || fpra_endis_arg || fprh_arg || fprl_arg || fprs_arg || fpsh_arg || fpsp_arg || fpsl_arg || fpsS_arg || fpsZ_arg || fpsH_arg || fpsb_arg || fpse_arg || fptS_arg || fptZ_arg || fptB_arg || fptd_arg || fpdZa_arg || fpdZb_arg || fpdH_arg || frx_arg || frp_lpt_arg || frp_ablify_arg || frp_irq_arg || frp_base_arg || frs1_com_arg || frs1_ablify_arg || frs1_irq_arg || frs1_base_arg || frs2_com_arg || frs2_irq_arg || frs2_ablify_arg || frs2_base_arg ) {
		intOpenFlags = O_RDWR;
	} else {
		intOpenFlags = O_RDONLY;
	}

	/*** Open the device ***/
#	ifdef DEBUG_FILEOPS
	printf( "%s: Opening device file %s with flags 0x%x\n", szTpctlName, pchDeviceName, intOpenFlags );
#	endif
	intFiledesc = open( pchDeviceName, intOpenFlags );
	if ( intFiledesc < 0 ) {
		printf( "%s: Can't open device file %s with flags 0x%x.\n", szTpctlName, pchDeviceName, intOpenFlags );
		perror( SZ_TPCTL_NAME ": System error message is" );
		exit( EXIT_FAILURE );
	}

	/*** Get highlighting characters ***/
	if ( fHighlight ) {
		/* For more info: man curs_terminfo */
		setupterm( (char*)NULL, 1 /* stdout */, (int*)NULL );
		pchHiOn = tigetstr( "bold" );
		pchHiOff = tigetstr( "sgr0" );
		if ( pchHiOn == (char*)NULL || pchHiOn == (char*)-1 || pchHiOff == (char*)NULL || pchHiOff == (char*)-1 ) {
			pchHiOn = "";
			pchHiOff = "";
		}
	} else {
		pchHiOn = "";
		pchHiOff = "";
	}

	/*** Do the stuff ***/
	intRtn = 0;

	if ( !intRtn && fV ) intRtn = print_version( intFiledesc );

	if ( !intRtn && fib ) intRtn = info_print_bios( intFiledesc );
	if ( !intRtn && fic ) intRtn = info_print_cmos( intFiledesc );
	if ( !intRtn && fid ) intRtn = info_print_display( intFiledesc );
	if ( !intRtn && fio ) intRtn = info_print_docking_station( intFiledesc );
	if ( !intRtn && fip ) intRtn = info_print_processor( intFiledesc );
	if ( !intRtn && fis ) intRtn = info_print_sensors( intFiledesc );
	if ( !intRtn && fiU ) intRtn = info_print_UltraBay_II( intFiledesc );
	if ( !intRtn && fir_arg ) intRtn = info_print_refresh( intFiledesc, wVGAMode );

	if ( !intRtn && fsdi_arg ) intRtn = setupsmapi_set_display_state( intFiledesc, 'i', fDisplayEnableInternal );
	if ( !intRtn && fsdc_arg ) intRtn = setupsmapi_set_display_state( intFiledesc, 'c', fDisplayEnableCRT );
	if ( !intRtn && fsdt_arg ) intRtn = setupsmapi_set_display_state( intFiledesc, 't', fDisplayEnableTV );
	if ( !intRtn && fsdT_arg ) intRtn = setupsmapi_set_display_state( intFiledesc, 'T', fDisplayEnableTVSelected );
	if ( !intRtn && fsdm_arg ) intRtn = setupsmapi_set_display_state( intFiledesc, 'm', fDisplayEnableMonitorDetectionIgnore );
	if ( !intRtn && fsd2_arg ) intRtn = setupsmapi_set_display_state( intFiledesc, '2', fDisplayEnableDual );

	if ( !intRtn && fsf_arg ) intRtn = setupsmapi_set_hotkey_state( intFiledesc, bFnHotkeyState );

	if ( !intRtn && fspi_arg ) intRtn = setupsmapi_set_pointing_device_state( intFiledesc, 1, 0, bPointingDeviceStateInt );
	if ( !intRtn && fspe_arg ) intRtn = setupsmapi_set_pointing_device_state( intFiledesc, 0, 1, bPointingDeviceStateExt );
	if ( !intRtn && fsst_arg ) intRtn = setupcmos_set_daylightsavingtime_state( intFiledesc, fDaylightsavingtimeEnable );

	if ( !intRtn && ( fsdi || fsdc || fsdt || fsdT || fsdm || fsd2 ) ) printf(
		"%s%s%setup of %s%s%sisplay:                       CMOS          current\n",
		HI("s"), HI("d")
	);
	if ( !intRtn && fsdi ) intRtn = setupsmapi_print_display_state( intFiledesc, 'i' );
	if ( !intRtn && fsdc ) intRtn = setupsmapi_print_display_state( intFiledesc, 'c' );
	if ( !intRtn && fsdt ) intRtn = setupsmapi_print_display_state( intFiledesc, 't' );
	if ( !intRtn && fsd2 ) intRtn = setupsmapi_print_display_state( intFiledesc, '2' );
	if ( !intRtn && fsdT ) intRtn = setupsmapi_print_display_state( intFiledesc, 'T' );
	if ( !intRtn && fsdm ) intRtn = setupsmapi_print_display_state( intFiledesc, 'm' );

	if ( !intRtn && fsf ) printf(
		"%s%s%setup of fn hotkey:\n",
		HI("s") 
	);
	if ( !intRtn && fsf ) intRtn = setupsmapi_print_hotkey_state( intFiledesc );

	if ( !intRtn && ( fspi || fspe ) ) printf(
		"%s%s%setup of %s%s%sointing device:               CMOS          current\n",
		HI("s"), HI("p")
	);
	if ( !intRtn && fspi ) intRtn = setupsmapi_print_pointing_device_state( intFiledesc, SMAPIDEV_TERNALITY_IN );
	if ( !intRtn && fspe ) intRtn = setupsmapi_print_pointing_device_state( intFiledesc, SMAPIDEV_TERNALITY_EX );

	if ( !intRtn && ( fsst ) ) printf(
		"%s%s%setup of:                               CMOS\n",
		HI("s")
	);
	if ( !intRtn && fsst ) intRtn = setupcmos_print_daylightsavingtime_state( intFiledesc );

	/*** pm section ***/

	if ( !intRtn && fpma_arg ) intRtn = pm_set_expenditure_mode( intFiledesc, 1, 0, powermodeAC );
	if ( !intRtn && fpmb_arg ) intRtn = pm_set_expenditure_mode( intFiledesc, 0, 1, powermodeBattery );
	if ( !intRtn && fpmR_arg ) intRtn = pm_set_redisafe_mode( intFiledesc, fSafeSuspendEnable );

	if ( !intRtn && fpra_endis_arg ) {
		intRtn = pm_set_resume_event(
			intFiledesc, 'a', fResumeEnableAppointment
		);
	}
	if ( !intRtn && fpra_date_arg ) {
		intRtn = pm_set_resume_appointment(
			intFiledesc,
			sintResumeApptYear, sintResumeApptMonth, sintResumeApptDay,
			sintResumeApptHour, sintResumeApptMinute, sintResumeApptSecond,
			fResumeApptDaily
		);
	}
	if ( !intRtn && fprh_arg ) intRtn = pm_set_resume_event( intFiledesc, 'h', fResumeEnableHardware );
	if ( !intRtn && fprl_arg ) intRtn = pm_set_resume_event( intFiledesc, 'l', fResumeEnableLid );
	if ( !intRtn && fprs_arg ) intRtn = pm_set_resume_event( intFiledesc, 's', fResumeEnableSerial );

	if ( !intRtn && fpsh_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'h', ch_h_Sedation );
	if ( !intRtn && fpsp_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'p', ch_p_Sedation );
	if ( !intRtn && fpsl_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'l', ch_l_Sedation );
	if ( !intRtn && fpsZ_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'Z', ch_Z_Sedation );
	if ( !intRtn && fpsS_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'S', ch_S_Sedation );
	if ( !intRtn && fpsH_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'H', ch_H_Sedation );
	if ( !intRtn && fpsb_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'b', ch_b_Sedation );
	if ( !intRtn && fpse_arg ) intRtn = pm_set_sedation_action( intFiledesc, 'e', ch_e_Sedation );

	if ( !intRtn && fptS_arg ) intRtn = pm_set_timer_mode( intFiledesc, 'S', fTimerEnableStandby );
	if ( !intRtn && fptZ_arg ) intRtn = pm_set_timer_mode( intFiledesc, 'Z', fTimerEnableSuspendHibernate );
	if ( !intRtn && fptB_arg ) intRtn = pm_set_timer_mode( intFiledesc, 'B', fTimerEnableBlank );
	if ( !intRtn && fptd_arg ) intRtn = pm_set_timer_mode( intFiledesc, 'd', fTimerEnableDrivePowerdown );

	if ( !intRtn && fpdH_arg ) intRtn = pm_set_hibernate_from_suspend_delay( intFiledesc, bDelaySuspendToHibernate );
	if ( !intRtn && fpdZa_arg ) intRtn = pm_set_suspend_or_hibernate_delay( intFiledesc, 'a', bDelaySuspendOrHibernateAC );
	if ( !intRtn && fpdZb_arg ) intRtn = pm_set_suspend_or_hibernate_delay( intFiledesc, 'b', bDelaySuspendOrHibernateBattery );

	if ( !intRtn && (fpma || fpmb || fpmR) ) printf(
		"%s%s%sower management %s%s%sodes:\n",
		HI("p"), HI("m")
	);
	if ( !intRtn && fpma ) intRtn = pm_print_expenditure_mode( intFiledesc, SMAPIDEV_POWERSRC_AC );
	if ( !intRtn && fpmb ) intRtn = pm_print_expenditure_mode( intFiledesc, SMAPIDEV_POWERSRC_BATTERY );
	if ( !intRtn && fpmR ) intRtn = pm_print_redisafe_mode( intFiledesc );

	if ( !intRtn && (fpra || fprh || fprl || fprs) ) printf(
		"%s%s%sower management %s%s%sesume events:         capability    current\n",
		HI("p"), HI("r")
	);
	if ( !intRtn && fpra ) intRtn = pm_print_resume_event( intFiledesc, 'a' );
	if ( !intRtn && fprh ) intRtn = pm_print_resume_event( intFiledesc, 'h' );
	if ( !intRtn && fprl ) intRtn = pm_print_resume_event( intFiledesc, 'l' );
	if ( !intRtn && fprs ) intRtn = pm_print_resume_event( intFiledesc, 's' );

	if ( !intRtn && (fpsh || fpsp || fpsl || fpsS || fpsZ || fpsH || fpsb || fpse) ) printf(
		"%s%s%sower management %s%s%sedative events:       capability    current\n"
		"                                       (%c%c%c%c%c)       (%c%c%c%c%c)\n",
		HI("p"), HI("s"),
		EVENTCHARS( 0xFF ), EVENTCHARS( 0xFF )
	);
	if ( !intRtn && fpsh ) intRtn = pm_print_sedation_actions( intFiledesc, 'h' );
	if ( !intRtn && fpsp ) intRtn = pm_print_sedation_actions( intFiledesc, 'p' );
	if ( !intRtn && fpsl ) intRtn = pm_print_sedation_actions( intFiledesc, 'l' );
	if ( !intRtn && fpsS ) intRtn = pm_print_sedation_actions( intFiledesc, 'S' );
	if ( !intRtn && fpsZ ) intRtn = pm_print_sedation_actions( intFiledesc, 'Z' );
	if ( !intRtn && fpsH ) intRtn = pm_print_sedation_actions( intFiledesc, 'H' );
	if ( !intRtn && fpsb ) intRtn = pm_print_sedation_actions( intFiledesc, 'b' );
	if ( !intRtn && fpse ) intRtn = pm_print_sedation_actions( intFiledesc, 'e' );

	if ( !intRtn && (fptS || fptZ || fptB || fptd) ) printf(
		"%s%s%sower management %s%s%simer modes:           capability    current\n",
		HI("p"), HI("t")
	);
	if ( !intRtn && fptS ) intRtn = pm_print_timer_mode( intFiledesc, 'S' );
	if ( !intRtn && fptZ ) intRtn = pm_print_timer_mode( intFiledesc, 'Z' );
	if ( !intRtn && fptB ) intRtn = pm_print_timer_mode( intFiledesc, 'B' );
	if ( !intRtn && fptd ) intRtn = pm_print_timer_mode( intFiledesc, 'd' );

	/* No title needed */
	if ( !intRtn && fpdH ) intRtn = pm_print_hibernate_from_suspend_delay( intFiledesc );

	if ( !intRtn && (fpdZa || fpdZb || fpdZh || fpdZu) ) printf(
		"%s%s%sower management %s%s%selays of %s%s%suspend or hibernate:      current\n"
		"            (specifiable in each power mode)?\n",
		HI("p"), HI("d"), HI("Z")
	);
	if ( !intRtn && fpdZa ) intRtn = pm_print_suspend_or_hibernate_delay( intFiledesc, 'a' );
	if ( !intRtn && fpdZb ) intRtn = pm_print_suspend_or_hibernate_delay( intFiledesc, 'b' );
	if ( !intRtn && fpdZh ) intRtn = pm_print_suspend_or_hibernate_delay( intFiledesc, 'h' );
	if ( !intRtn && fpdZu ) intRtn = pm_print_suspend_or_hibernate_delay( intFiledesc, 'u' );

	/*** resources section ***/
	if ( !intRtn && frp_lpt_arg ) intRtn = res_set_parallel_LPT( intFiledesc, bParLPT );
	if ( !intRtn && frp_ablify_arg ) intRtn = res_ablify_par( intFiledesc, fParEn );
	if ( !intRtn && frp_irq_arg ) intRtn = res_set_par_irq( intFiledesc, bParIRQ );
	if ( !intRtn && frp_base_arg ) intRtn = res_set_par_base( intFiledesc, wParBase );
	if ( !intRtn && frs1_com_arg ) intRtn = res_set_serial_COM( intFiledesc, 1, bSer1COM );
	if ( !intRtn && frs1_ablify_arg ) intRtn = res_ablify_ser( intFiledesc, 1, fSer1En );
	if ( !intRtn && frs1_irq_arg ) intRtn = res_set_ser_irq( intFiledesc, 1, bSer1IRQ );
	if ( !intRtn && frs1_base_arg ) intRtn = res_set_ser_base( intFiledesc, 1, wSer1Base );
	if ( !intRtn && frs2_com_arg ) intRtn = res_set_serial_COM( intFiledesc, 2, bSer2COM );
	if ( !intRtn && frs2_ablify_arg ) intRtn = res_ablify_ser( intFiledesc, 2, fSer2En );
	if ( !intRtn && frs2_irq_arg ) intRtn = res_set_ser_irq( intFiledesc, 2, bSer2IRQ );
	if ( !intRtn && frs2_base_arg ) intRtn = res_set_ser_base( intFiledesc, 2, wSer2Base );
	if ( !intRtn && frx_arg ) intRtn = res_set_lowerlegregs( intFiledesc, dwSuperioRegisters );

	if ( !intRtn && (frf || frp || frs1 || frs2) ) printf(
		"%s%s%sesource state:                                       current\n",
		HI("r") 
	);
	if ( !intRtn && frf ) intRtn = res_print( intFiledesc, 'f' );
	if ( !intRtn && frp ) intRtn = res_print( intFiledesc, 'p' );
	if ( !intRtn && frs1 ) intRtn = res_print( intFiledesc, '1' );
	if ( !intRtn && frs2 ) intRtn = res_print( intFiledesc, '2' );

	/*** debugging options ***/
	if ( !intRtn && fdumpregs ) intRtn = info_dump_cmos( intFiledesc );

	/*** Do the sedation last ***/
	if ( !intRtn && fS ) intRtn = pm_request_standby( intFiledesc );
	if ( !intRtn && fZ ) intRtn = pm_request_suspend( intFiledesc );
	if ( !intRtn && fH ) intRtn = pm_request_hibernate( intFiledesc );
	if ( !intRtn && fO && fO_arg ) intRtn = pm_request_poweroff( intFiledesc );

	/*** Check for error codes ***/
	if ( intRtn == 0 ) {
		exit( EXIT_SUCCESS );
	} else if ( intRtn < 0 ) {
		/* device error */
		switch ( intRtn ) {
			case -ETHINKPAD_PROGRAMMING:
				printf( "%s: bug in program!\n", szTpctlName );
				break;
			case -ETHINKPAD_HW_NOT_FOUND:
				printf( "%s: module could not find what it is supposed to control.\n", szTpctlName );
				break;
			case -ETHINKPAD_EXECUTION:
				printf( "%s: execution error\n", szTpctlName );
				break;
			case -ETHINKPAD_MODULE_DISABLED:
				printf( "%s: module required for request is not enabled.\n", szTpctlName );
				break;
			case -ETHINKPAD_MODULE_NOT_FOUND:
				printf( "%s: module required for request is not loaded.\n", szTpctlName );
				break;
			case -EBUSY:
				printf( "%s: device is busy.\n", szTpctlName );
				break;
			case -EINVAL:
				printf( "%s: unrecognized function code.\n", szTpctlName );
				break;
			case -ENOTTY:
				printf( "%s: unrecognized ioctl code.\n", szTpctlName );
				break;
			case -EACCES:
				printf( "%s: permission denied.\n", szTpctlName );
				break;
			default:
				printf( "%s: ioctl error whose standard meaning is \"%s\".\n", szTpctlName, strerror( -intRtn ) );
		}
		exit( EXIT_FAILURE );
	} else if ( intRtn == ERR_TPCTL ) {
		printf( "%s: error -- exiting.\n", szTpctlName );
		exit( EXIT_FAILURE );
	} else if ( intRtn >= ERR_SMB_MIN && intRtn <= ERR_SMB_MAX ) {
		char * pchMessage;
		switch ( intRtn ) {
			case ERR_SMB_FUNC_NOT_AVAIL:
				pchMessage = "SMAPI function is not available"; break;
			case ERR_SMB_FUNC_NOT_SUPPORTED:
				pchMessage = "function is not supported"; break;
			case ERR_SMB_SYSTEM_ERROR:
				pchMessage = "system error"; break;
			case ERR_SMB_SYSTEM_INVALID:
				pchMessage = "system is invalid"; break;
			case ERR_SMB_SYSTEM_BUSY:
				pchMessage = "system is busy"; break;
			case ERR_SMB_DEVICE_ERROR:
				pchMessage = "device error (disk read error)"; break;
			case ERR_SMB_DEVICE_BUSY:
				pchMessage = "device is busy"; break;
			case ERR_SMB_DEVICE_NOT_ATTACHED:
				pchMessage = "device is not attached"; break;
			case ERR_SMB_DEVICE_DISABLED:
				pchMessage = "device is disabled"; break;
			case ERR_SMB_PARM_INVALID:
				pchMessage = "invalid parameter"; break;
			case ERR_SMB_PARM_OUT_OF_RANGE:
				pchMessage = "request parameter is out of range"; break;
			case ERR_SMB_PARM_NOT_ACCEPTED:
				pchMessage = "request parameter is not accepted"; break;
			default:
				pchMessage = "(error code not recognized)"; break;
		}
		printf( "%s: SMAPI BIOS error 0x%x (\"%s\") -- exiting.\n", szTpctlName, intRtn, pchMessage );
		exit( EXIT_FAILURE );
	} else if ( intRtn >= ERR_SMAPIDEV_MIN && intRtn <= ERR_SMAPIDEV_MAX ) {
		/* smapidev error */
		char * pchMessage;
		switch ( intRtn ) {
			case ERR_SMAPIDEV_STRUCTURE_SIZE_INVALID:
				pchMessage = "structure size invalid"; break;
			case ERR_SMAPIDEV_PARM_INVALID:
				pchMessage = "invalid parameter"; break;
			case ERR_SMAPIDEV_SMAPI_RESULT_NOT_UNDERSTOOD:
				pchMessage = "SMAPI result not understood"; break;
			default:
				pchMessage = "(error code not recognized)"; break;
		}
		printf( "%s: smapidev error 0x%x (\"%s\").\n", szTpctlName, intRtn, pchMessage );
		exit( EXIT_FAILURE );
	} else {
		printf( "%s: unrecognized error 0x%x -- exiting.\n", szTpctlName, intRtn );
		exit( EXIT_FAILURE );
	}

} /* main */




