/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)xyz2grd.c	2.84  11/04/99
 *
 *	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/gmt
 *--------------------------------------------------------------------*/
/*
 * xyz2grd reads a xyz file from standard input and creates the
 * corresponding grd-file. The input file does not have to be
 * sorted. xyz2grd will report if some nodes are missing. nx and
 * ny are computed as:
 *	nx = rint((east-west)/dx) + 1, ny = rint((north-south)/dy) + 1
 *
 * Author:	Paul Wessel
 * Date:	3-JAN-1991-1999
 * Version:	2.0	Based on old v1.x
 * Version:	3.1	Based on old 3.0
 * Version:	3.2	Added -S option for swapping only
 *		3.3.1	-S option need to set GMT_fopen modes to rb an wb
 */
 
#include "gmt.h"

int *flag;
float *a;

main (int argc, char **argv)
{
	BOOLEAN error = FALSE, got_input = FALSE, pixel = FALSE, z_only = FALSE;
	BOOLEAN xy_units = TRUE, skip, nodata_set = FALSE, b_only = FALSE, just_swap = FALSE;

	int i, ij, gmt_ij, ii, jj, x, y, z, nm, n_read = 0, n_filled = 0, n_used = 0, one_or_zero;
	int n_empty = 0, n_stuffed = 0, n_bad = 0, entry, n_expected_fields, n_fields, n_confused = 0;
	
	double w, e, s, n, dx = 0.0, dy = 0.0, *in, d_value, off, idx, idy, no_data_d;

	float no_data_f;
	
	char *grdfile, *zfile, *dfile, line[BUFSIZ], input[BUFSIZ], *ptr;
	
	FILE *fp = NULL, *fps = NULL;
	
	struct GRD_HEADER grd;

	struct GMT_Z_IO r;
	
	argc = GMT_begin (argc, argv);

	grdfile = zfile = dfile = CNULL;
	input[0] = 0;
	w = e = s = n = 0.0;
	no_data_f = GMT_f_NaN;
	no_data_d = GMT_d_NaN;

	GMT_init_z_io (&r, TRUE);
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'H':
				case 'R':
				case 'V':
				case ':':
				case '\0':
					error += GMT_get_common_args (argv[i], &w, &e, &s, &n);
					break;

				case 'b':
					error += GMT_io_selection (&argv[i][2]);
					b_only = TRUE;
					break;
				case 'D':
					strcpy (input, &argv[i][2]);
					got_input = TRUE;
					break;
				case 'F':
					pixel = TRUE;
					break;
				case 'G':
					grdfile = &argv[i][2];
					break;
				case 'I':
					GMT_getinc (&argv[i][2], &dx, &dy);
					break;
				case 'L':
					xy_units = FALSE;
					break;
				case 'N':
					if (!argv[i][2]) {
						fprintf (stderr, "%s: GMT SYNTAX ERROR -N option:  Must specify value or NaN\n", GMT_program);
						error++;
					}
					else {
						no_data_d = (argv[i][2] == 'N' || argv[i][2] == 'n') ? GMT_d_NaN : atof (&argv[i][2]);
						no_data_f = (float)no_data_d;
						nodata_set = TRUE;
					}
					break;
				case 'S':
					just_swap = TRUE;
					if (argv[i][2]) zfile = &argv[i][2];
					break;
				case 'Z':
					error += GMT_parse_z_io (&argv[i][2], &r, TRUE);
					z_only = TRUE;
					break;
				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else
			dfile = argv[i];
	}
		
	if (argc == 1 || GMT_quick) {
		fprintf (stderr, "xyz2grd %s - Converting a [xy]z-file to a GMT grdfile\n\n", GMT_VERSION);
		fprintf (stderr, "usage: xyz2grd [<[xy]zfile>] -G<grdfile> -I<dx[m|c]>[/<dy[m|c]>] -R<west/east/south/north>\n");
		fprintf (stderr, "	[-D<xunit/yunit/zunit/scale/offset/title/remark>] [-F] [-H[<nrec>]] [-L] [-N<nodata>] [-S[<zfile]] [-V] [-Z[<flags>]] [-:] [-bi[s][<n>]]\n");
		
		if (GMT_quick) exit (EXIT_FAILURE);
		
		fprintf (stderr, "	xyzfile is a 1- or 3-column file [or standard input] with [x,y,]z values\n");
		fprintf (stderr, "	-G to name the output grdfile.\n");
		fprintf (stderr, "	-I specifies grid size(s).  Append m [or c] to <dx> and/or <dy> for minutes [or seconds]\n");
		GMT_explain_option ('R');
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "	-D to enter header information.  Specify '=' to get default value\n");
		fprintf (stderr, "	-F will force pixel registration [Default is grid registration]\n");
		GMT_explain_option ('H');
		fprintf (stderr, "	-L means that x is longitude, i.e. assumed to be periodic in 360\n");
		fprintf (stderr, "	-N set value for nodes without input xyz triplet [Default is NaN]\n");
		fprintf (stderr, "	   Z-table entries that equal <nodata> are replaced by NaN\n");
		fprintf (stderr, "	-S Requires -Z and will swap the byte-order of the input data and write\n");
		fprintf (stderr, "	   the result to <zfile> (or stdout if no file given).  No grdfile created!\n");
		GMT_explain_option ('V');
		fprintf (stderr, "	-Z sets exact specification of incoming 1-column z-table\n");
		fprintf (stderr, "	   If data is in row format, state if first row is at T(op) or B(ottom)\n");
		fprintf (stderr, "	     Then, append L or R to indicate starting point in row\n");
		fprintf (stderr, "	   If data is in column format, state if first columns is L(left) or R(ight)\n");
		fprintf (stderr, "	     Then, append T or B to indicate starting point in column\n");
		fprintf (stderr, "	   To skip a header of size <n> bytes, append s<n> [<n> = 0]\n");
		fprintf (stderr, "	   To swap byte-order in 2-byte words, append w\n");
		fprintf (stderr, "	   Append x if gridline-registered, periodic data in x without repeating column at xmax\n");
		fprintf (stderr, "	   Append y if gridline-registered, periodic data in y without repeating row at ymax\n");
		fprintf (stderr, "	   Specify one of the following data types (all binary except a):\n");
		fprintf (stderr, "	     a  Ascii\n");
		fprintf (stderr, "	     c  signed 1-byte character\n");
		fprintf (stderr, "	     u  unsigned 1-byte character\n");
		fprintf (stderr, "	     h  signed short 2-byte integer\n");
		fprintf (stderr, "	     H  unsigned short 2-byte integer\n");
		fprintf (stderr, "	     i  signed 4-byte integer\n");
		fprintf (stderr, "	     I  unsigned 4-byte integer\n");
		fprintf (stderr, "	     l  signed long (4- or 8-byte) integer\n");
		fprintf (stderr, "	     f  4-byte floating point single precision\n");
		fprintf (stderr, "	     d  8-byte floating point double precision\n");
		fprintf (stderr, "	   [Default format is scanline orientation in ASCII representation: -ZTLa]\n");
		fprintf (stderr, "	   This option assumes all nodes have data values.\n");
		GMT_explain_option (':');
		GMT_explain_option ('i');
		GMT_explain_option ('n');
		fprintf (stderr, "	   Default is 3 input columns\n");
		GMT_explain_option ('.');
		exit (EXIT_FAILURE);
	}

	if (just_swap) {	/* Reading and writing binary files */
		strcpy (GMT_io.r_mode, "rb");
		strcpy (GMT_io.w_mode, "wb");
	}

	if (dfile && (fp = GMT_fopen (dfile, GMT_io.r_mode)) == NULL) {
		fprintf (stderr, "%s: Cannot find input file %s\n", GMT_program, dfile);
		exit (EXIT_FAILURE);
	}

	if (b_only && z_only) GMT_io.binary[0] = FALSE;

	if (!just_swap) {
		if (!project_info.region_supplied) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", GMT_program);
			error++;
		}
		if (dx <= 0.0 || dy <= 0.0) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR -I option.  Must specify positive increment(s)\n", GMT_program);
			error++;
		}
		if (!grdfile) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR option -G:  Must specify output file\n", GMT_program);
			error++;
		}
	}
	if (just_swap && !z_only) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR option -S:  Must also specify -Z\n", GMT_program);
		error++;
	}	

	if ((GMT_io.binary[0] || r.binary) && gmtdefs.io_header) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", GMT_program);
		error++;
	}
	if (GMT_io.binary[0] && GMT_io.ncol[0] == 0) GMT_io.ncol[0] = 3;
	if (GMT_io.binary[0] && GMT_io.ncol[0] < 3) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data (-bi) must have at least 3 columns\n", GMT_program);
		error++;
	}
				
	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands */

	if (b_only && z_only) fprintf (stderr, "%s: GMT Warning.  -Z overrides -bi\n", GMT_program);

	if (GMT_io.binary[0] && gmtdefs.verbose) {
		char *type[2] = {"double", "single"};
		fprintf (stderr, "%s: Expects %d-column %s-precision binary data\n", GMT_program, GMT_io.ncol[0], type[GMT_io.single_precision[0]]);
	}

	if (fp == NULL) {
		fp = GMT_stdin;
#ifdef SET_IO_MODE
		GMT_setmode (0);
#endif
	}

	GMT_grd_init (&grd, argc, argv, FALSE);
	
	/* if (!project_info.region) d_swap (s, e); */  /* Got w/s/e/n, make into w/e/s/n */

	/* Decode grd information given, if any */
	
	if (got_input) {
		for (i = 0; input[i]; i++) if (input[i] == '_') input[i] = ' ';
		ptr = strtok (input, "/");
		entry = 0;
		while (ptr) {
			if (ptr[0] != '=') {
				switch (entry) {
					case 0:
						memset ((void *)grd.x_units, 0, (size_t)80);
						strcpy (grd.x_units, ptr);
						break;
					case 1:
						memset ((void *)grd.y_units, 0, (size_t)80);
						strcpy (grd.y_units, ptr);
						break;
					case 2:
						memset ((void *)grd.z_units, 0, (size_t)80);
						strcpy (grd.z_units, ptr);
						break;
					case 3:
						grd.z_scale_factor = atof (ptr);
						break;
					case 4:
						grd.z_add_offset = atof (ptr);
						break;
					case 5:
						strcpy (grd.title, ptr);
						break;
					case 6:
						strcpy (grd.remark, ptr);
						break;
					default:
						break;
				}
			}
			ptr = strtok (CNULL, "/");
			entry++;
		}
	}
	
	if (pixel) {
		grd.node_offset = 1;
		one_or_zero = 0;
		off = 0.0;
	}
	else {
		grd.node_offset = 0;
		one_or_zero = 1;
		off = 0.5;
	}

	if (just_swap)
		r.swab = TRUE;
	else {
		grd.nx = irint ((e-w)/dx) + one_or_zero;
		grd.ny = irint ((n-s)/dy) + one_or_zero;
		grd.x_min = w;	grd.x_max = e;
		grd.y_min = s;	grd.y_max = n;
		grd.x_inc = dx;	grd.y_inc = dy;
	
		GMT_grd_RI_verify (&grd);
	}

	GMT_set_z_io (&r, &grd);

	if (just_swap) {	/* Just swap data and exit */

		if (gmtdefs.verbose) fprintf (stderr, "%s: Swapping data bytes only\n", GMT_program);

		if (!zfile) {
			fps = GMT_stdout;
#ifdef SET_IO_MODE
			GMT_setmode (1);
#endif
		}
		else if ((fps = GMT_fopen (zfile, GMT_io.w_mode)) == NULL) {
			fprintf (stderr, "%s: Cannot create file %s\n", GMT_program, zfile);
			exit (EXIT_FAILURE);
		}

		if (r.skip) fseek (fp, (long)r.skip, SEEK_CUR);
		while ((r.read_item) (fp, &d_value)) (r.write_item) (fps, d_value);

		if (fp != GMT_stdin) GMT_fclose (fp);
		if (fps != GMT_stdout) GMT_fclose (fps);

		exit (EXIT_SUCCESS);
	}
		
	if (gmtdefs.verbose) fprintf (stderr, "%s: nx = %d  ny = %d\n", GMT_program, grd.nx, grd.ny);
	
	nm = grd.nx * grd.ny;
	
	a = (float *) GMT_memory (VNULL, (size_t)nm, sizeof (float), GMT_program);
	flag = (int *) GMT_memory (VNULL, (size_t)nm, sizeof (int), GMT_program);

	x = (gmtdefs.xy_toggle) ? 1 : 0;	y = 1 - x;		/* Set up which columns have x and y */

	if (gmtdefs.io_header) for (i = 0; i < gmtdefs.n_header_recs; i++) fgets (line, BUFSIZ, fp);
	
	idx = 1.0 / dx;	idy = 1.0 / dy;
	n_expected_fields = (z_only) ? 1 : 3;
	z = (z_only) ? 0 : 2;
	ij = -1;	/* Will be incremented to 0 or set first time around */
	
	if (z_only) {	/* Read separately because of all the possible formats */
		if (nodata_set && GMT_is_dnan (no_data_d)) nodata_set = FALSE;	/* No point testing */
		if (r.skip) fseek (fp, (long)r.skip, SEEK_CUR);
		while ((r.read_item) (fp, &d_value)) {
			ij++;
			(r.get_gmt_ij) (&r, ij, &gmt_ij);
			if (nodata_set && d_value == no_data_d)
				a[gmt_ij] = GMT_f_NaN;
			else
				a[gmt_ij] = (float)d_value;
		}
		ij++;
		if (fp != GMT_stdin) GMT_fclose (fp);
		if (ij != r.n_expected) {
			fprintf (stderr, "%s: Read %d items, but %d was expected!\n", GMT_program, ij, r.n_expected);
			exit (EXIT_FAILURE);
		}
		GMT_check_z_io (&r, a);
	}
	else {	/* Get x, y, z */
		n_fields = GMT_input (fp, &n_expected_fields, &in);
		
		while (! (GMT_io.status & GMT_IO_EOF)) {	/* Not yet EOF */

			skip = FALSE;
			if (GMT_io.status & GMT_IO_MISMATCH) {
				fprintf (stderr, "%s: Mismatch between actual (%d) and expected (%d) fields near line %d\n", GMT_program, n_fields, n_expected_fields, n_read);
				exit (EXIT_FAILURE);
			}
			n_read++;
		
			if (in[y] >= s && in[y] <= n) {	/* y ok, check x */
				if (xy_units && (in[x] < w || in[x] > e))
					skip = TRUE;
				else {
					while (in[x] < w) in[x] += 360.0;
					if (in[x] > e) skip = TRUE;
				}
			}
			else
				skip = TRUE;
		
			if (!skip) {
				ii = (int)floor ((in[x] - w) * idx + off);
				if (ii == grd.nx) ii--, n_confused++;
				jj = (int)floor ((n - in[y]) * idy + off);
				if (jj == grd.ny) jj--, n_confused++;
				ij = jj * grd.nx + ii;
				a[ij] += (float)in[z];	/* Add up incase we must average */
				flag[ij]++;
				n_used++;
			}
		
			n_fields = GMT_input (fp, &n_expected_fields, &in);
		}
		if (fp != GMT_stdin) GMT_fclose (fp);

		for (ij = 0; ij < nm; ij++) {	/* Check if all nodes got one value only */
			if (flag[ij] == 1)
				n_filled++;
			else if (flag[ij] == 0) {
				n_empty++;
				a[ij] = no_data_f;
			}
			else {	/* More than 1 value went to this node */
				a[ij] /= flag[ij];
				n_stuffed++;
			}
		}
		if (!xy_units && !grd.node_offset) {	/* Make sure longitudes got replicated */
			int j, ij_left, ij_right;

			for (j = 0; j < grd.ny; j++) {
				ij_left = j * grd.nx;
				ij_right = ij_left + grd.nx - 1;
				if (flag[ij_left] && !flag[ij_right]) {
					a[ij_right] = a[ij_left];
					n_empty--;
				}
				else if (flag[ij_right] && !flag[ij_left]) {
					a[ij_left] = a[ij_right];
					n_empty--;
				}
			}
		}

		if (n_read != n_used || n_filled != nm || n_empty > 0) gmtdefs.verbose = TRUE;
		if (gmtdefs.verbose) {
			sprintf (line, "%s\n\0", gmtdefs.d_format);
			fprintf (stderr, "%s:  n_read: %d  n_used: %d  n_filled: %d  n_empty: %d set to ", GMT_program,
				n_read, n_used, n_filled, n_empty);
			(GMT_is_dnan (no_data_d)) ? fprintf (stderr, "NaN\n") : fprintf (stderr, line, no_data_d);
		}
		if (n_bad) fprintf (stderr, "%s: %d records unreadable\n", GMT_program, n_bad);
		if (n_stuffed) fprintf (stderr, "%s: Warning - %d nodes had multiple entries that were averaged\n", GMT_program, n_stuffed);
		if (n_confused) fprintf (stderr, "%s: Warning - %d values gave bad indeces: Pixel vs gridline confusion?\n", GMT_program, n_confused);
	}

	if (GMT_write_grd (grdfile, &grd, a, 0.0, 0.0, 0.0, 0.0, GMT_pad, FALSE)) {
		fprintf (stderr, "%s: Error writing file %s\n", GMT_program, grdfile);
		exit (EXIT_FAILURE);
	}
	
	GMT_free ((void *)a);
	GMT_free ((void *)flag);
	
	GMT_end (argc, argv);
}
