/*
 * Do Serial Port settings given parameters and a FILE pointer.
 * (or file descriptor)
 * A. Maitland Bottoms
 *
 * Updated 29 September 1999, from original file created circa 1996.
 * Thanks to Ron Luse, kd9kx, for showing me how to use termios calls.
 *
 */

#include <termios.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

int fserial_port_setup(FILE *fp,char *settings);
int serial_port_setup(int fd,char *settings);

int
fserial_port_setup(FILE *fp,char *settings)
{
  int fd;

  if (*settings == 0) return(0); /* Nothing to do */
  if (fp == 0) return(0); /* Nothing to do */
  fd = fileno(fp);
  return(serial_port_setup(fd,settings));
}

int
serial_port_setup(int fd,char *settings)
{
  struct termios ts;
  int parity=0, bits=CS8, stops=0, speed=B9600, s;
  int count;
  char sargs[5][80];

  if (*settings == 0) return(0); /* Nothing to do */
  if (!isatty(fd)) return(0); /* Nothing we can do */ 
  if (tcgetattr(fd,&ts)==-1) return(-1);

  count = sscanf(settings, "%[^, 	\n]%*[ 	,]"
		 "%[^, 	\n]%*[ 	,]"
		 "%[^, 	\n]%*[ 	,]"
		 "%[^, 	\n]%*[ 	,]"
		 "%[^, 	\n]%*[ 	,]",
		 sargs[0], sargs[1], sargs[2], sargs[3], sargs[4]);

  if ((count>4)||(count<0)) 
    return(2); /* Can't parse settings string */
  else switch(count) {
  case 4:
    switch (atoi(sargs[3])) {
    case 2:
	    stops = CSTOPB; break;
    case 1:
    default:
	    stops = 0; break;
    }
  case 3:
    switch (sargs[2][0]) {
    case 'O':
    case 'o':
	    parity = PARENB | PARODD; break;
    case 'E':
    case 'e':
	    parity = PARENB; break;
    case 'N':
    case 'n':
    default:
	    parity = 0; break;
    }
  case 2:
    switch (atoi(sargs[1])) {
    case 5:
	    bits = CS5; break;
    case 6:
	    bits = CS6; break;
    case 7:
	    bits = CS7; break;
    case 8:
    default:
	    bits = CS8; break;
    }  
  case 1:
    s = atoi(sargs[0]);
    if (s>=230400)
      speed = B230400;
    else if (s>=115200)
      speed = B115200;
    else if (s>=57600)
      speed = B57600;
    else if (s>=38400)
      speed = B38400;
    else if (s>=19200)
      speed = B19200;
    else if (s>=9600)
      speed = B9600;
    else if (s>=4800)
      speed = B4800;
    else if (s>=2400)
      speed = B2400;
    else if (s>=1800)
      speed = B1800;
    else if (s>=1200)
      speed = B1200;
    else if (s>=600)
      speed = B600;
    else if (s>=300)
      speed = B300;
    else speed = B1200;
  case 0:
  default:
    break;
  }

  ts.c_iflag = IGNBRK; /* | IGNCR; */
  ts.c_oflag = 0;
  ts.c_cflag = parity | bits | stops | CREAD | CLOCAL;
  ts.c_lflag = 0;
  if (cfsetospeed(&ts, speed)==-1) {
    perror("output speed");
    return(-1);
  }
  if (cfsetispeed(&ts, speed)==-1) {
    perror("input speed");
    return(-1);
  }
  if (tcsetattr(fd,TCSANOW,&ts)==-1) {
    perror("serial_port_setup");
    return(-1);
  }
  return(0);
}

#ifdef STANDALONE
int
main(int argc,char *argv[])
{
  FILE *serialport;

  if (argc==3) {
    serialport = fopen(argv[1],"r");
    if (!serialport) { perror(argv[1]); return(1); }
    fserial_port_setup(serialport,argv[2]);
    fclose(serialport);
  } else {
    printf("Usage: serport /dev/ttyS1 19200,8,N,1\n");
  }
  return(0);
}
#endif
