;;;
;;; mmelmo.el -- mm-backend by ELMO.
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

;;; mmelmo.
(require 'mime)
(require 'mime-parse)
(require 'elmo-vars)
(eval-when-compile
  (require 'std11))

(defvar mmelmo-force-reload nil)
(defvar mmelmo-sort-field-list nil)

(defun mmelmo-original-mode ()
  (setq major-mode 'mmelmo-original-mode)
  (setq buffer-read-only t)
  (elmo-set-buffer-multibyte nil)
  (setq mode-name "MMELMO-Original"))

(defun mmelmo-get-original-buffer ()
  (let (ret-val)
    (if (setq ret-val 
	      (get-buffer (concat mmelmo-entity-buffer-name "0")))
	ret-val
      (set-buffer (setq ret-val
			(get-buffer-create 
			 (concat mmelmo-entity-buffer-name "0"))))
      (mmelmo-original-mode)
      ret-val)))

(defun mmelmo-cleanup-entity-buffers ()
  "Cleanup entity buffers of mmelmo."
  (mapcar (lambda (x)
	    (if (string-match mmelmo-entity-buffer-name x)
		(kill-buffer x)))
	  (mapcar 'buffer-name (buffer-list))))

(require 'mmbuffer)

;;; mmelmo: Only the initialization method is different from mmbuffer.
(mm-define-backend elmo (buffer))

(mm-define-method initialize-instance ((entity elmo))
  (mime-entity-set-buffer-internal 
   entity
   (get-buffer-create (concat mmelmo-entity-buffer-name "0")))
  (let* ((location (mime-entity-location-internal entity)))
    (save-excursion
      (set-buffer (mime-entity-buffer-internal entity))
      (mmelmo-original-mode)
      (setq mime-message-structure entity)
      (elmo-read-msg (nth 0 location)
		     (nth 1 location)
		     (current-buffer)
		     (nth 2 location)
		     mmelmo-force-reload)
      (let ((header-start (point-min))
	    header-end
	    body-start
	    (body-end (point-max)))
	(goto-char header-start)
	(if (re-search-forward "^$" nil t)
	    (setq header-end (match-end 0)
		  body-start (if (= header-end body-end)
				 body-end
			       (1+ header-end)))
	  (setq header-end (point-min)
		body-start (point-min)))
	(save-restriction
	  (narrow-to-region header-start header-end)
	  (mime-entity-set-content-type-internal
	   entity
	   (let ((str (std11-fetch-field "Content-Type")))
	     (if str
		 (mime-parse-Content-Type str)
	       )))
	  )
	(mime-entity-set-header-start-internal entity header-start)
	(mime-entity-set-header-end-internal entity header-end)
	(mime-entity-set-body-start-internal entity body-start)
	(mime-entity-set-body-end-internal entity body-end)
	))))

(defun mmelmo-mime-insert-header-from-buffer (buffer 
					      start end
					      &optional invisible-fields
					      visible-fields)
  (let ((the-buf (current-buffer))
	(mode-obj (mime-find-field-presentation-method 'wide))
	field-decoder
	f-b p f-e field-name len field field-body
        vf-alist (sl mmelmo-sort-field-list))
    (save-excursion
      (set-buffer buffer)
      (save-restriction
	(narrow-to-region start end)
	(goto-char start)
	(while (re-search-forward std11-field-head-regexp nil t)
	  (setq f-b (match-beginning 0)
		p (match-end 0)
		field-name (buffer-substring f-b p)
		len (string-width field-name)
		f-e (std11-field-end))
	  (when (mime-visible-field-p field-name
				      visible-fields invisible-fields)
	    (setq field (intern
			 (capitalize (buffer-substring f-b (1- p))))
		  field-body (buffer-substring p f-e)
		  field-decoder (inline (mime-find-field-decoder-internal
					 field mode-obj)))
            (setq vf-alist (append (list
                                    (cons field-name
                                          (list field-body field-decoder)))
                                   vf-alist))))
        (and vf-alist
             (setq vf-alist
                   (sort vf-alist
                         (function (lambda (s d)
                                     (let ((n 0) re
                                           (sf (car s))
                                           (df (car d)))
                                       (catch 'done
                                         (while (setq re (nth n sl))
                                           (setq n (1+ n))
                                           (and (string-match re sf)
                                                (throw 'done t))
                                           (and (string-match re df)
                                                (throw 'done nil)))
                                         t)))))))
        (with-current-buffer the-buf
          (while vf-alist
            (let* ((vf (car vf-alist))
                   (field-name (car vf))
                   (field-body (car (cdr vf)))
                   (field-decoder (car (cdr (cdr vf)))))
              (insert field-name)
	      (insert (if field-decoder
			  (funcall field-decoder field-body
                                   (string-width field-name))
			;; Don't decode
			field-body))
              (insert "\n"))
            (setq vf-alist (cdr vf-alist))))))))

(mm-define-method insert-header ((entity elmo)
				 &optional invisible-fields visible-fields)
  (mmelmo-mime-insert-header-from-buffer
   (mime-entity-buffer entity)
   (mime-entity-header-start-internal entity)
   (mime-entity-header-end-internal entity)
   invisible-fields visible-fields)
  )


(provide 'mmelmo)

