static char rcsid[] = "@(#)$Id: mime_decode.c,v 1.7.2.5 1999/10/10 15:56:57 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.7.2.5 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "headers.h"
#include "melib.h"
#include "s_me.h"

static int index_hex[128] = {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     0, 1, 2, 3,  4, 5, 6, 7,  8, 9,-1,-1, -1,-1,-1,-1,
    -1,10,11,12, 13,14,15,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,10,11,12, 13,14,15,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1
};

static int index_64[128] = {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
    52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,-1,-1,-1,
    -1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
    15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
    -1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
    41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1
};

#define base64(c) ((((c) > 0) && ((c) < 127)) ? index_64[ (c) ] : -1)
#define hex(c) ((((c) > 0) && ((c) < 127)) ? index_hex[ (c) ] : -1)

/* Prototype */
static void Xbit_decode	P_((in_state_t *, out_state_t *, int, int)); 

static void Xbit_decode (s_in, s_out, len, is_text)
     in_state_t *s_in;
     out_state_t *s_out;
     int len;
     int is_text;
{
    char buf[VERY_LONG_STRING];
    int offset = 0, length;
    
    if (s_out->prefix) {
	strfcpy (buf, s_out->prefix, sizeof buf);
	offset = strlen (buf);
    }
    while (len > 0) {
	if (0 ==
	    (length = state_getl (buf + offset, sizeof buf - offset, s_in)))
	    break;
	len -= length;
	
	/* Take care of CRLF => LF conversion */
	if (is_text > 0 &&
	    length > 1 &&
	    buf[offset + length - 1] == '\n' &&
	    buf[offset + length - 2] == '\r') {
	    buf[offset + length - 2] = '\n';
	    buf[offset + length - 1] = '\0';
	    length--;
	}

	state_put(buf,offset+length,s_out);
    }
}

void base64_decode (s_in, s_out, length, astext)
     in_state_t  *s_in;
     out_state_t *s_out;
     int length;
     int astext;
{
  /* Notes:  this routine assumes that strlen(line length) % 4 == 0.
     Most intelligent software will do this, so I haven't worried
     about being *completely* compliant with the standard.  I've
     not run into any encoded data where this was not true.  */
  
  char buf[STRING], *p;
  int
    bytes = 0, /* the total number of bytes read so far */
    terminate = FALSE,
    c1, c2, c3, c4;
  unsigned char
    ch,
    store_ch = 0; /* for astext */
  int corrupted = FALSE;

  dprint(10,(debugfile,
           "base64_decode: length=%d, prefix=%s, astext=%d\n",
           length,
           s_out->prefix ? s_out->prefix : "<NULL>",
           astext));

  /* Make sure to quote the first line! */
  state_add_prefix(s_out);

  while (! terminate) {
    int len;

    if (bytes >= length)
      break;

    if ((len = state_getl (buf, sizeof buf, s_in)) <= 0)
	break;

    bytes += len;

    p = buf;
    while (*p) {
      c1 = *p++;

      if (base64(c1) == -1) {
        p--;
        /* partial fix -- K E H   <hurtta@dionysos.FMi.FI>  */

        while (*p == ' ' || *p == '\t' || *p == '\r')
          p++;

        if (*p != '\n')
          corrupted = TRUE;  /* Perhaps corrupted */

        if (*p == 0)
          break;

        c1 = *p++;
      }

      /* End of the line. */
      if (c1 == '\n')
	continue;

      c2 = *p++;
      
      c1 = base64(c1);
      if (c1 == -1)
        corrupted = TRUE;
      c2 = base64(c2);
      if (c2 == -1)
        corrupted = TRUE;
      ch = (c1 << 2) | (c2 >> 4);
      
      if (store_ch && ch != '\n') /* for astext */
	state_putc(store_ch,s_out);

      store_ch = 0;
      if (astext && ch == '\r')
        store_ch = ch;
      else {
	state_putc(ch,s_out);
	if (ch == '\n')
	  state_add_prefix(s_out);
      }      
    
      c3 = *p++;
      if (c3 == '=') {
	terminate = TRUE;
	break;
      }
      c3 = base64(c3);
      if (c3 == -1)
	corrupted = TRUE;
      ch = ((c2 & 0xf) << 4) | (c3 >> 2);

      if (store_ch && ch != '\n')
	state_putc(store_ch,s_out);

      store_ch = 0;

      if (astext && ch == '\r')
	store_ch = ch;
      else {
	state_putc(ch,s_out);
	if (ch == '\n')
	  state_add_prefix(s_out);
      }
  
      c4 = *p++;
      if (c4 == '=') {
	terminate = TRUE;
	break;
      }
      
      c4 = base64(c4);
      if (c4 == -1)
	corrupted = TRUE;
      ch = ((c3 & 0x3) << 6) | c4;
      
      if (store_ch && ch != '\n')
	state_putc(store_ch,s_out);

      store_ch = 0;

      if (astext && ch == '\r') {
	store_ch = ch;
      }
      else {
	state_putc(ch,s_out);
	if (ch=='\n')
	  state_add_prefix(s_out);
      }
    }
  }

  /* Make sure to flush anything left in the internal buffer. */
  if (store_ch)  /* for astext */
    state_putc(store_ch,s_out);

  if (corrupted) {
    if (s_out->displaying) {
      state_puts("\n[",s_out);
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeBASE64Corrupt,
			 "BASE64 data was corrupt!"),s_out);
      state_puts("]\n",s_out);
    }
    lib_error (CATGETS(elm_msg_cat, MeSet, MeDecodeBASE64Corrupt,
		       "BASE64 data was corrupt!"));
    sleep_message();
  }

  dprint(10,(debugfile,"base64_decode: readed=%d bytes, corrupted=%d.\n",
	     bytes,corrupted));   

  return;
}
  
void quoted_printable_decode (s_in, s_out, length, astext)
     in_state_t  *s_in;
     out_state_t *s_out;
     int length;
     int astext;
{
  int bytes = 0; /* number of bytes read */
  int nl = TRUE; /* flag to indicate hitting a new line */
  char *p;
  int c1, c2;
  unsigned char ch, store_ch = 0;
  char buf[VERY_LONG_STRING];
  int corrupted = 0;

  dprint(10,(debugfile,
	     "quoted_printable_decode: length=%d, prefix=%s, astext=%d\n",
	     length,
	     s_out->prefix ? s_out->prefix : "<NULL>",
	     astext));

  for (;;) {
    int len;

    if (bytes >= length)
      break;

    if ((len=state_getl (buf, sizeof buf, s_in)) <= 0)
      break;
    bytes += len;

    p = buf;
    while (*p) {

      /* If there is a prefix and this is the beginning of a new line... */
      if (nl) {
	state_add_prefix(s_out);
	nl = FALSE;
      }

      if (store_ch)
	state_putc(store_ch,s_out);
      
      if (*p == '=') {
	p++;
	/* Ignore spaces in end of line   -- see MIME */
	if (*p == '\r' || *p == ' ' || *p == '\t') {
	  char *t = p;
	  while (*t && (*t == '\r' || *t == ' ' || *t == '\t'))
	    t++;
	  if (*t && *t == '\n')
	    p = t;
	}

	if (*p == '\n') { /* soft linebreak */
	  if (length <= 0)
	    break;
	  p++;
	}
	else {
	  c1 = hex(*p);
	  if (c1 == -1)
	    corrupted = TRUE;
	  p++;
	  c2 = hex(*p);
	  if (c2 == -1)
	    corrupted = TRUE;
	  p++;
	  ch = (c1 << 4) | c2;

	  /* We not need here CR LF -> LF removing, because
	   * CRLF's which presents end of line should NOT be encoded.
	   *                             - K E H <hurtta@dionysos.FMI.FI> */

	  state_putc(ch,s_out);
	}
      }
      else {
       if (astext && *p == '\r')
         store_ch = *p;
       else
	 state_putc(*p,s_out);
	 
       if (*p == '\n') {
	 nl = TRUE;
	 if (length <= 0)
	   break;
       }

       p++;
      }  
    }
  }

  /* Flush anything left in the buffer */
  if (store_ch) /* for astext */
    state_putc(store_ch,s_out);

  if (corrupted) {
    if (s_out -> displaying) {
      state_puts("\n[",s_out);
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeQPCorrupt,
			 "Seems that QUOTED-PRINTABLE data was corrupted."),
		 s_out);
      state_puts("]\n",s_out);
    }
    lib_error (CATGETS(elm_msg_cat, MeSet, MeDecodeQPCorrupt,
		       "Seems that QUOTED-PRINTABLE data was corrupted."));
    sleep_message();
  }
  
  dprint(10,(debugfile,
	     "quoted_printable_decode: readed=%d bytes, corrupted=%d.\n",
	     bytes,corrupted));

  return;
}

/* Prototype */
static void multipart_decode  P_((mime_t *, 
				  in_state_t *, out_state_t *)); 

static void multipart_decode (att, state_in, state_out)
     mime_t *att; /* The list of attachments for this MULTIPART data */
     in_state_t   *state_in;
     out_state_t  *state_out;
{
  int nattach = 0;
  char buf[VERY_LONG_STRING];

  dprint(12,(debugfile,
	     "multipart_decode -> START\n"));

  if (att->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"multipart_decode",
	       "Bad magic number");

  if (!in_state_seekable(state_in)) {
    state_putc('[',state_out);
    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeMultipartUnsupportedIS,
		       "multipart_decode: unsupported input state"),
	       state_out);
    state_puts("]\n",state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartUnsupportedIS,
		  "multipart_decode: unsupported input state"));
    sleep_message();
    return;
  }

  while (att) {

    if (att->magic != MIME_magic)
      mime_panic(__FILE__,__LINE__,"multipart_decode",
		 "Bad magic number (next -chain)");

    nattach++;

    dprint(12,(debugfile,
	       "multipart_decode: [%d]: Type: %.15s/%.30s, Encoding: %d, Size: %d\n",
	       nattach, 
	       mime_types[att->type], att->subtype,
	       att->encoding, att->length));

    if (in_state_fseek(state_in,att->offset) != 0) {
      state_putc('[',state_out);
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeMultipartSeekFail,
			"multipart_decode: seek failed"),
		 state_out);
      state_puts("]\n",state_out);
      lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartSeekFail,
			"multipart_decode: seek failed"));
      sleep_message();
      return;
    }

    if (state_out->displaying) {
      char Buffer[70], tmp[40];
      char *Encoding = "???";
       
      Buffer[0] = '\0';
      tmp[0] = '\0';
      if (att->description) {
	elm_sfprintf(Buffer,sizeof Buffer,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeDesc,
			     "\"%.60s\"]\n["),
		     att->description);
      } else if (att->disposition_opts &&
		 mime_get_param ("filename", tmp, att->disposition_opts, sizeof(tmp))) {
	elm_sfprintf(Buffer,sizeof Buffer,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeDescFilename,
			     "Filename: %.60s]\n["),
		     tmp);
      }

      Encoding = ENCODING(att->encoding);
      
      /* First print out a nice display line about this attachment */
      if (att->disposition != DISP_INLINE)
	elm_sfprintf (buf, sizeof buf,
		      CATGETS(elm_msg_cat, MeSet, MeDecodeAttach,
			      "%s[Attach #%d: %sType: %.15s/%.30s, Encoding: %s, Size: %d]\n"),
		      nattach > 1 ? "\n" : "",
		      nattach, Buffer,
		      mime_types[att->type], att->subtype,
		      Encoding, att->length);
      else
	elm_sfprintf (buf, sizeof buf,
		      CATGETS(elm_msg_cat, MeSet, MeDecodePart,
			      "%s[Part #%d: %sType: %.15s/%.30s, Encoding: %s, Size: %d]\n"),
		      nattach > 1 ? "\n" : "",
		      nattach, Buffer,
		      mime_types[att->type], att->subtype,
		      Encoding, att->length);
    }
    else {
      buf[0] = '\0';
      if (nattach > 1)
        strfcpy (buf, "\n", sizeof buf);
      if (att->description && (strlen(att->description) < sizeof(buf) -30)) {
	elm_sfprintf (buf, sizeof buf,
		      CATGETS(elm_msg_cat, MeSet, MeDecodeContentDesc,
			      "Content-Description: %.300s\n\n"),
		      att->description, sizeof buf);
      }
    }
      
    state_puts(buf,state_out);
    if (state_out->displaying)
      state_putc('\n', state_out);
    mime_decode (att, state_in, state_out);
    att = att->next;
  }
  dprint(12,(debugfile,
	     "multipart_decode <- END\n"));
}

static void null_SG_decoder(body,sign,state_in,state_out,micalg)
     mime_t *body; 
     mime_t *sign;
     in_state_t   *state_in;
     out_state_t  *state_out;
     char *micalg;
{
    if (state_out->displaying) {	
	char * buf = elm_message(CATGETS(elm_msg_cat, MeSet, 
					 MeDecodeUnsupportedSig,
					 "Signature %s/%.30s is unsupported, not checked"), 
				 TYPE(sign->type), sign->subtype);
	
	state_puts("\n[",state_out);
	state_puts(buf,state_out);
	state_puts("]\n",state_out);
	free(buf);
    }
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedSig,
		      "Signature %s/%.30s is unsupported, not checked"),
	      TYPE(sign->type), sign->subtype);
    sleep_message();

    mime_decode(body,state_in,state_out);
}

static void null_EC_decoder(init,data,state_in,state_out)
     mime_t *init;
     mime_t *data;
     in_state_t   *state_in;
     out_state_t  *state_out;
{
    char * buf = elm_message(CATGETS(elm_msg_cat, MeSet, 
				     MeDecodeUnsupportedEnc,
				     "Encryption %s/%.30s is unsupported"), 
			     TYPE(init->type), init->subtype);
	
    state_puts("\n[",state_out);
    state_puts(buf,state_out);
    state_puts("]\n",state_out);
    free(buf);
    lib_error(CATGETS(elm_msg_cat, MeSet, 
		      MeDecodeUnsupportedEnc,
		      "Encryption %s/%.30s is unsupported"), 
	      TYPE(init->type), init->subtype);
}

typedef void signed_decoder P_((mime_t *body, mime_t *sign,
				in_state_t   *state_in,
				out_state_t  *state_out,
				char *micalg));
typedef signed_decoder * SG_decoder_t;

static SG_decoder_t select_SG_decoder P_((char *protocol));
static SG_decoder_t select_SG_decoder(protocol)
     char *protocol;
{
#ifdef USE_PGP
    if (0 == strcasecmp(protocol,"application/pgp-signature")) {
	dprint(12,(debugfile,
		   "select_SG_decoder [%s] = pgp_SG_decoder\n",
		   protocol));
	return pgp_SG_decoder;
    }
#endif

    dprint(12,(debugfile,
	       "select_SG_decoder [%s] = null_SG_decoder\n",
	       protocol));
    return null_SG_decoder;
}

/* Prototype */
static void signed_decode P_((mime_t *, in_state_t *, out_state_t *)); 

static void signed_decode (ptr, state_in, state_out)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
    char protocol[100];
    char micalg[100];
    SG_decoder_t decode;

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"signed_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeSignedUnsupportedIS,
			   "signed_decode: unsupported input state"),
		   state_out);
	state_puts("]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeSignedUnsupportedIS,
			  "signed_decode: unsupported input state"));
	sleep_message();
	return;
    }
    
    if (! ptr->parts || !ptr->parts->next) {
	state_putc('[',state_out);
	state_puts (catgets(elm_msg_cat, MeSet, MeDecodeSignedNosubtypes,
			    "Content-Type: multipart/signed, no subtypes (parts=null)"), 
		    state_out);
	state_puts("]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeSignedNosubtypes,
			  "Content-Type: multipart/signed, no subtypes (parts=null)"));
	sleep_message();
	return;
    }
    
    if (!mime_get_param ("protocol", protocol, ptr->type_opts, 
			 sizeof protocol)) {
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeNoProtocol,
			    "'protocol' paramater is missing from Multipart/Signed -type!"), 
		    state_out);
	state_puts("]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocol,
			  "'protocol' paramater is missing from Multipart/Signed -type!"));
	sleep_message();
	return;
    }

    if (!mime_get_param ("micalg", micalg, ptr->type_opts, 
			 sizeof micalg)) {
	micalg[0] = '\0';
    }


    decode = select_SG_decoder(protocol);

    decode(ptr->parts,ptr->parts->next,state_in,state_out,micalg);
}


typedef void encrypted_decoder P_((mime_t *init, mime_t *data,
				   in_state_t   *state_in,
				   out_state_t  *state_out));

typedef encrypted_decoder * EC_decoder_t;


static EC_decoder_t select_EC_decoder P_((char *protocol));
static EC_decoder_t select_EC_decoder(protocol)
     char *protocol;
{
#ifdef USE_PGP
    if (0 == strcasecmp(protocol,"application/pgp-encrypted")) {
	dprint(12,(debugfile,
		   "select_EC_decoder [%s] = pgp_EC_decoder\n",
		   protocol));
	return pgp_EC_decoder;
    }
#endif

    dprint(12,(debugfile,
	       "select_EC_decoder [%s] = null_EC_decoder\n",
	       protocol));
    return null_EC_decoder;
}

/* Prototype */
static void encrypted_decode P_((mime_t *, in_state_t *, out_state_t *)); 

static void encrypted_decode (ptr, state_in, state_out)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
    char protocol[100];
    EC_decoder_t decode;

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"encrypted_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeEncryptedUnsupportedIS,
			   "encrypted_decode: unsupported input state"),
		   state_out);
	state_puts("]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeEncryptedUnsupportedIS,
			  "encrypted_decode: unsupported input state"));
	sleep_message();
	return;
    }
    
    if (! ptr->parts || !ptr->parts->next) {
	state_putc('[',state_out);
	state_puts (catgets(elm_msg_cat, MeSet, MeDecodeEncryptedNosubtypes,
			    "Content-Type: multipart/encrypted, no subtypes (parts=null)"), 
		    state_out);
	state_puts("]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeEncryptedNosubtypes,
			  "Content-Type: multipart/encrypted, no subtypes (parts=null)"));
	sleep_message();
	return;
    }
    
    if (!mime_get_param ("protocol", protocol, ptr->type_opts, 
			 sizeof protocol)) {
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeNoProtocolEnc,
			   "'protocol' paramater is missing from multipart/encrypted -type!"), 
		   state_out);
	state_puts("]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocolEnc,
			  "'protocol' paramater is missing from multipart/encrypted -type!"));
	sleep_message();
	return;
    }

    decode = select_EC_decoder(protocol);

    decode(ptr->parts,ptr->parts->next,state_in,state_out);
}


static mime_t * best_alternative P_((mime_t *att));
/* Prototype */
static void alternative_decode P_((mime_t *, 
				   in_state_t *, out_state_t *)); 

static void alternative_decode (ptr, state_in, state_out)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
  mime_t *best;
  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"alternative_decode",
	       "Bad magic number");

  if (!in_state_seekable(state_in)) {
    state_putc('[',state_out);
    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeAlternativeUnsupportedIS,
		       "alternative_decode: unsupported input state"),
	       state_out);
    state_puts("]\n",state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeUnsupportedIS,
		      "alternative_decode: unsupported input state"));
    sleep_message();
    return;
  }

  if (! ptr->parts) {
    state_putc('[',state_out);
    state_puts (catgets(elm_msg_cat, MeSet, MeDecodeAlternativeNosubtypes,
			"Content-Type: multipart/alternative, no subtypes (parts=null)"), 
		state_out);
    state_puts("]\n",state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeNosubtypes,
		      "Content-Type: multipart/alternative, no subtypes (parts=null)"));
    sleep_message();
    return;
  }

  best = best_alternative(ptr->parts);

  if (best) {
      char * buf = NULL;
      dprint(12,(debugfile,
		 "multipart_alternative: Type: %.15s/%.30s, Encoding: %d, Size: %d\n",
		 mime_types[best->type], best->subtype,
		 best->encoding, best->length));

      if (in_state_fseek(state_in,best->offset) != 0) {
	  state_putc('[',state_out);
	  state_puts(catgets(elm_msg_cat, MeSet, MeDecodeAlternativeSeekFailed,
			     "multipart_alternative: seek failed"),
		     state_out);
	  state_puts("]\n",state_out);
	  lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeSeekFailed,
			    "multipart_alternative: seek failed"));
	  sleep_message();
	  return;
      }

    if (state_out->displaying) {
	char tmp[40];
	char *Encoding = "???";
	char * Buffer = NULL;
	
	tmp[0] = '\0';
	if (best->description) {
	    Buffer = elm_message(CATGETS(elm_msg_cat, MeSet, MeDecodeDesc,
					 "\"%.60s\"]\n["),
				 best->description);
	} else if (best->disposition_opts &&
		   mime_get_param ("filename", tmp, 
				   best->disposition_opts, sizeof(tmp))) {
	    Buffer = elm_message(CATGETS(elm_msg_cat, MeSet, 
					 MeDecodeDescFilename,
					 "Filename: %.60s]\n["),
				 tmp);
	}
	
	Encoding = ENCODING(best->encoding);
      
	/* First print out a nice display line about this part */
	if (best->disposition != DISP_INLINE)
	    buf = elm_message(CATGETS(elm_msg_cat, MeSet, MeDecodeSelAttach,
				  
				      "[Selecting attach: %sType: %.15s/%.30s, Encoding: %s, Size: %d]\n"),
			      Buffer ? Buffer : "",
			      mime_types[best->type], best->subtype,
			      Encoding, best->length);
	else
	    buf = elm_message(CATGETS(elm_msg_cat, MeSet, MeDecodeSelPart,
				      "[Selecting part: %sType: %.15s/%.30s, Encoding: %s, Size: %d]\n"),
			      Buffer ? Buffer : "",
			      mime_types[best->type], best->subtype,
			      Encoding, best->length);
	if (Buffer)
	    free(Buffer);
    }
    else {
	if (best->description && (strlen(best->description) < sizeof(buf) -30)) {
	    buf = elm_message(CATGETS(elm_msg_cat, MeSet, MeDecodeContentDesc,
				      "Content-Description: %.300s\n\n"),
			      best->description, sizeof buf);
	}
    }
    
    if (buf) {
	state_puts(buf,state_out);
	free(buf);
    }
    if (state_out->displaying)
	state_putc('\n', state_out);
    mime_decode (best, state_in, state_out);

  } else {
    state_puts("[", state_out);
    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeCtNoAlternative,
		       "Content-Type: multipart/alternative, alternative not found"),
	       state_out);
    
    state_puts("]\n", state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeCtNoAlternative,
		      "Content-Type: multipart/alternative, alternative not found"));
    sleep_message();
  }
}

/* Prototype */
static void multipart_0_decode P_((mime_t *, 
				   in_state_t *, out_state_t *)); 

static void multipart_0_decode (ptr, state_in, state_out)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"multipart_0_decode",
	       "Bad magic number");

  if (ptr->parts)
    multipart_decode (ptr->parts, state_in, state_out);
  else {
    state_putc('[',state_out);
    state_puts (catgets(elm_msg_cat, MeSet, MeDecodeMultipartNoparts,
			"Content-Type: multipart/*, no subtypes (parts = NULL)"),
 		state_out);
    state_puts("]\n",state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartNoparts,
		      "Content-Type: multipart/*, no subtypes (parts = NULL)"));
    sleep_message();
  }
}

/* Prototype */
static void rfc822_decode P_((mime_t *, 
			      in_state_t *, out_state_t *)); 

static int rfc822_header_filter     P_((header_list_ptr, int));
static void rfc822_header_converter P_((header_list_ptr, int, char *, int));

static int rfc822_header_filter(hdr,flag)
     header_list_ptr hdr;
     int flag;                  /* elm_filter */
{
  char buf[80];

  strfcpy(buf,hdr->header_name->header,78);
  strfcat(buf,": ", sizeof buf);
  
  if (flag) {
    if (matches_weedlist (buf)) 
      return 0;
    else
      return 1;
  } else
    return 1;
}

static void rfc822_header_converter (hdr,flag,buffer,size)
     header_list_ptr hdr;
     int flag;
     char *buffer;
     int size;
{
  int class = hdr->header_name->flag;

  if (!hdr->body) {
    buffer[0] = '\0';
    return;
  }

  dprint(12,(debugfile,"rfc822_header_converter: header=%s,class=%d,rest=%s",
	     hdr->header_name->header,class,hdr->body));

  flag++;      /* So that flag is used */

  strfcpy(buffer,hdr->body,size);

  if (class & HDR_TEXT) {
    if (is_rfc1522 (buffer))
      rfc1522_decode (buffer, size);
  } else if (class & HDR_STRUCTURED) {
    rfc1522_decode_structured(class,buffer,size);
  }

  dprint(12,(debugfile,"rfc822_header_converter: decoded rest=%s",buffer));

}

static void rfc822_decode (mt, state_in, state_out)
     mime_t *mt;
     in_state_t *state_in;
     out_state_t *state_out;
{
  /* This looks a lot like multipart_decode() except that we want to print
   * the headers for the message.  NOTE: "mt" should be a pointer to the
   * RFC822 attachment, NOT its subpart, since we need the offset in order
   * to print out the headers.
   */

  header_list_ptr headers = NULL;
  header_list_ptr From_header, mime_version, osv;
  int decode_headers = 1;

  if (mt->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"rfc822_decode",
	       "Bad magic number");
  
  if (!in_state_seekable(state_in)) {
    state_putc('[',state_out);
    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeRFC822UnsupportedIS,
		       "rfc822_decode: unsupported input state"),
	       state_out);
    state_puts("]\n",state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeRFC822UnsupportedIS,
		      "rfc822_decode: unsupported input state"));
    sleep_message();
    return;
  }

  if (in_state_fseek(state_in,mt->offset) != 0) {
    state_putc('[',state_out);
    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeRFC822SeekFailed,
		       "rfc822_decode: seek failed"),
	       state_out);
    state_puts("]\n",state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeRFC822SeekFailed,
		      "rfc822_decode: seek failed"));
    sleep_message();
    return;
  }

  headers = state_read_headers(state_in, RHL_MARK_FOLDING);

  mime_version = locate_header_by_name(headers,"MIME-Version");
  osv          = locate_header_by_name(headers,"X-ELM-OSV");

  if (!mime_version && !req_mime_hdrencoding && !req_mime_hdrencoding)
    decode_headers = 0;
  if (osv && osv->body) {
    char value[20];
    if (mime_get_param("no-hdr-encoding",value,osv->body,
		       sizeof value) && atoi(value) > 0)
      decode_headers = 0;
  }

  From_header  = locate_header_by_name(headers,"From");

  if (From_header) {
      char buffer[1024+1];
      char * buffer2 = NULL;
      strfcpy(buffer,From_header->body,sizeof buffer);
      if (decode_headers && is_rfc1522 (buffer))
	  rfc1522_decode_structured (From_header->header_name->flag,
				     buffer, sizeof buffer);
      buffer2 = elm_message(CATGETS(elm_msg_cat, MeSet, MeDecodeStartMail,
				    "-- Start of included mail From: %s\n"),
			    buffer);
      state_puts(buffer2,state_out);
      free(buffer2);
  }
  else
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeStartMail1,
			 "-- Start of included mail.\n"),
		 state_out);

  state_write_headers(state_out,headers,
		      rfc822_header_filter,
		      decode_headers ? rfc822_header_converter :
		      NULL_header_converter,
		      elm_filter);

  delete_headers(headers); headers = NULL;

  state_putc('\n',state_out);

  if (mt->parts == NULL) {
    state_putc('[',state_out);
    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeNoRFC822,
		       "rfc822_decode: no body of RFC 822 mail (parts = NULL)"),
	       state_out);
    state_puts("]\n",state_out);
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoRFC822,
		  "rfc822_decode: no body of RFC 822 mail (parts = NULL)"));
    sleep_message();
    return;
  }

  if (mt->parts->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"rfc822_decode",
	       "Bad magic number (parts)");

  mime_decode (mt->parts, state_in, state_out);
  state_puts(catgets(elm_msg_cat, MeSet, MeDecodeEndMail,
		     "-- End of included mail.\n"),
	     state_out);
  return;
}

static int ASCII_filter(c,st)
     char c;
     struct out_state * st;
{
  int res = (unsigned char) c;
  st++;

  if (res & 128)
    res = '_';
  return res;
}

/* Prototype */
static int run_decoder P_((mime_t *, in_state_t *, out_state_t *)); 

static int run_decoder (ptr, state_in, state_out) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
  int is_text;
  dprint(12,(debugfile,
	     "run_decoder -> state: offset=%d, length=%d\n",
	     ptr -> offset, ptr -> length));

  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"run_decoder",
	       "Bad magic number");


  if (in_state_seekable(state_in)) {
    dprint(12,(debugfile,
	       "              (file): ftell=%ld\n",in_state_ftell(state_in))); 
  } else {
    dprint(12,(debugfile,
	       "              (\?\?\?\?): magic=%d\n",state_in->magic));
  }

  is_text = is_text_type (mime_types[ptr->type], ptr->subtype, 
			  ptr->encoding);

  dprint(12,(debugfile,
	     "run_decoder: is_text=%d; type=%s (%d)/%s; encoding =%s (%d)\n",
	     is_text, mime_types[ptr->type], ptr->type, ptr->subtype,
	     ENCODING(ptr->encoding),
	     ptr->encoding));


  if (ptr->encoding == ENCODING_BASE64)
    base64_decode (state_in, state_out, ptr->length, is_text);
  else if (ptr->encoding == ENCODING_QUOTED)
    quoted_printable_decode (state_in, state_out, ptr->length, is_text);
  else if (ptr->encoding != ENCODING_NONE && 
	   ptr->encoding != ENCODING_7BIT &&
	   ptr->encoding != ENCODING_8BIT &&
	   ptr->encoding != ENCODING_BINARY) {

    dprint(12,(debugfile,
	       "run_decoder=0 <- END; \n"));
    if (in_state_seekable(state_in)) {
      dprint(12,(debugfile,
		 "              (file); ftell=%ld\n",
		 in_state_ftell(state_in))); 
    }
    return 0;
  }
  else
    Xbit_decode (state_in, state_out, ptr->length, is_text);

  dprint(12,(debugfile,
	     "run_decoder=1 <- END; \n"));
  if (in_state_seekable(state_in)) {
    dprint(12,(debugfile,
	       "              (file); ftell=%ld\n",
	       in_state_ftell(state_in))); 
  }
  
  return 1;
}

int set_filter (ptr, state) 
     mime_t *ptr;
     out_state_t *state;
{
    char tmp[SLEN];
    int code;
    
    dprint(12,(debugfile,
	       "set_filter -> state: offset=%d, length=%d\n",
	       ptr -> offset, ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"set_filter",
		   "Bad magic number");
    
    state -> filter = NULL_filter;
    
    dprint(12,(debugfile,
	       "set_filter: filter=%p (NULL_filter)\n",
	       (state -> filter)));
    
    if (0 == (code = mime_get_charset (tmp, ptr->type_opts, sizeof(tmp)))) {
    	/* Don't show this part (or copy to reply buffer) ... */	  
	char * buf = elm_message(CATGETS(elm_msg_cat, MeSet, 
					 MeDecodeCharsetSkipping,
					 "[Charset %.15s unsupported, skipping...]\n"), 
			  tmp);
	state_puts(buf,state);
	free(buf);
	
	if (state->displaying)
	    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeUseVtoView,
			       "[Use 'v' to view or save this part.]\n"),
		       state);
		
	dprint(12,(debugfile,
		   "set_filter=0 <- END\n"));
	return 0;
    } else {
	if (code < 0) {
	    char * buf = elm_message(CATGETS(elm_msg_cat, MeSet, 
					     MeDecodeCharsetFiltering,
					     "[Charset %.15s unsupported, filtering to ASCII...]\n"),
				     tmp);
	    state_puts(buf,state);
	    free(buf);
	    
	    if (state->displaying)	
		state_puts(catgets(elm_msg_cat, MeSet, MeDecodeUseAlsoV,
				   "[You can also use 'v' to view or save this part.]\n\n"),
			   state);
	    
	    state -> filter = ASCII_filter;
	    
	    dprint(12,(debugfile,
		       "set_filter: filter=%p (ASCII_filter)\n",
		       (state -> filter)));
	}
	dprint(12,(debugfile,
		   "set_filter=1 <- END\n"));
	return 1;
    }
}

/* Prototype */
static void text_decode	P_((mime_t *, in_state_t *, out_state_t *)); 

static void text_decode (ptr, state_in, state_out) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
  dprint(12,(debugfile,
	     "text_decode -> state: offset=%d, length=%d \n",
	     ptr -> offset, ptr -> length));

  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"text_decode",
	       "Bad magic number");

  if (in_state_seekable(state_in)) {
    dprint(12,(debugfile,
	       "              (file): ftell=%ld\n",
	       in_state_ftell(state_in))); 
  }

  if (set_filter(ptr,state_out)) {
    if (!run_decoder(ptr,state_in, state_out)) {
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeEncodingSkipping,
			 "[Unsupported encoding, skipping...]\n"),
		 state_out);
      if (state_out->displaying)	
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeUseVEncodedSave,
			   "[Use 'v' to save this part in encoded form.]\n"),
		   state_out);
    }
  }

  state_out -> filter = NULL_filter;

  dprint(12,(debugfile,
	     "text_decode <- END; \n"));
  if (in_state_seekable(state_in)) {
    dprint(12,(debugfile,
	       "          (file); ftell=%ld\n",
	       in_state_ftell(state_in))); 
  }
}

FILE * arrange_binary(ptr,state_in,state_out,newstate2, name)
     mime_t *ptr;
     in_state_t  *state_in;
     out_state_t *state_out;
     in_state_t *newstate2;
     char **name;
{
    FILE * tmpfp = NULL;
    int result = 1,len;
    char *fname = NULL;
    
    dprint(12,(debugfile,
	       "arrange_binary -> state: offset=%d, length=%d \n",
	       ptr -> offset, ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"arrange_binary",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	dprint(12,(debugfile,
		   "                 (file): ftell=%ld\n",
		   in_state_ftell(state_in))); 
    }
  
    fname = elm_message(FRM("%selmdecode-%d"), 
			temp_dir, getpid ());
    if (NULL == (tmpfp = safeopen_rdwr(fname))) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			  "Failed to create file for decoding."));
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			   "Failed to create file for decoding."),
		   state_out);
	state_puts("]\n",state_out);
	sleep_message();
	result = 0;
    } else { /* Tempfile opened */
	char buf[VERY_LONG_STRING];
	int c = EOF, prev = EOF;
	int is_binary = ptr->encoding == ENCODING_BINARY;
	long pos;

	if (name)
	    *name = fname;
	else
	    unlink(fname);  /* We can unlink it now ... */

	dprint(12,(debugfile,"                 : is_binary = %d\n",is_binary));

	if (in_state_fseek(state_in,ptr->begin_offset) != 0) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeArrangeSeekFail,
			      "arrange_binary: seek failed"));
	    sleep_message();
	    return NULL;
	}

	while ((pos = in_state_ftell(state_in)) < 
	       ptr -> offset + ptr ->length &&
	       EOF != (c = state_getc(state_in))) {

	    if (pos < ptr -> offset) {   /*  --- HEADER PART -- */
		if ('\n' == c && '\r' != prev) {
		    if (is_binary) {
			dprint(12,(debugfile,
				   "arrange_binary -- (header) not really binary input!\n")); 
			is_binary = 0;
		    }
		    fputc('\r',tmpfp);
		}		
	    } else if (!is_binary && '\n' == c && '\r' != prev) {
		fputc('\r',tmpfp);
	    }
	    fputc(c,tmpfp);
	    prev = c;
	}
	rewind(tmpfp); /* Rewind it for reading */	    
	set_in_state_file(tmpfp,newstate2);
    }

    if (!name)
	free(fname);
    return tmpfp;
}

FILE * arrange_decoded(ptr,state_in,state_out,newstate2,name)
     mime_t *ptr;
     in_state_t  *state_in;
     out_state_t *state_out;
     in_state_t *newstate2;
     char **name;
{
    FILE * tmpfp = NULL;
    int result = 1,len;
    char *fname = NULL;
    out_state_t newstate;
    
    out_state_clear(&newstate,STATE_out_file);
    
    dprint(12,(debugfile,
	       "arrange_decoded -> state: offset=%d, length=%d \n",
	       ptr -> offset, ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"arrange_decoded",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	dprint(12,(debugfile,
		   "                 (file): ftell=%ld\n",
		   in_state_ftell(state_in))); 
    }
  
    fname = elm_message(FRM("%selmdecode.%d"), 
			temp_dir, getpid ());
    if (NULL == (tmpfp = safeopen_rdwr(fname))) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			  "Failed to create file for decoding."));
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			   "Failed to create file for decoding."),
		   state_out);
	state_puts("]\n",state_out);
	sleep_message();
	result = 0;
    } else { /* Tempfile opened */
	int crypted = OFF;
	int bytes = 0;

	if (name)
	    *name = fname;
	else
	    unlink(fname);  /* We can unlink it now ... */
    
	newstate.displaying = state_out->displaying;
	
	set_out_state_file(tmpfp, &newstate);
	newstate.prefix     = NULL;             /* No prefix in that pass */
	newstate.filter     = state_out->filter;
	
	if (run_decoder(ptr,state_in,&newstate)) { 
	    int len;
	    
	    if (EOF == fflush(tmpfp)) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeErrorFlush,
				  "Error when flushing temporary file."));
		state_putc('[',state_out);
		state_puts(catgets(elm_msg_cat, MeSet, MeDecodeErrorFlush,
				   "Error when flushing temporary file."),
			   state_out);
		state_puts("]\n",state_out);
		sleep_message();
		result = 0;
	    }
	    rewind(tmpfp); /* Rewind it for reading */
	    
	    set_in_state_file(tmpfp,newstate2);
	} else { /* Run decoder failed */
	    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeEncodingSkipping,
			       "[Unsupported encoding, skipping...]\n"),
		       state_out);
	    if (state_out->displaying)	
		state_puts(catgets(elm_msg_cat, MeSet, MeDecodeUseVEncodedSave,
				   "[Use 'v' to save this part in encoded form.]\n"),
			   state_out);
	    result = 0;
	}
    }
    out_state_destroy(&newstate);

    dprint(12,(debugfile,"arrange_decoded; result=%d\n",result));
    
    if (!result && tmpfp) {
	fclose(tmpfp);
	tmpfp = NULL;
	if (name)
	    *name = NULL;
	free(fname);
    }

    if (!name)
	free(fname);
    return tmpfp;
}

/* Prototype */
static void elm_decode	P_((mime_t *, in_state_t *, out_state_t *)); 

static void elm_decode (ptr, state_in, state_out) 
     mime_t *ptr;
     in_state_t  *state_in;
     out_state_t *state_out;
{  
  in_state_t newstate2;

  FILE *tmpfp;
  char buffer[LONG_STRING];
  
  in_state_clear(&newstate2,STATE_in_file);
  
  dprint(12,(debugfile,
	     "elm_decode -> state: offset=%d, length=%d; \n",
	     ptr -> offset, ptr -> length));

  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"elm_decode",
	       "Bad magic number");

  if (in_state_seekable(state_in)) {
    dprint(12,(debugfile,
	       "             (file): ftell=%ld\n",
	       in_state_ftell(state_in))); 
  }

  if (tmpfp = arrange_decoded(ptr,state_in,state_out,&newstate2,NULL)) {
    if (set_filter(ptr,state_out)) {
      int len, crypted = 0;
      int count = 0;
      getkey(OFF);
      
      while((len = state_getl(buffer,sizeof(buffer),&newstate2)) > 0) {
	count += len;
	  
	if (!strncmp(buffer, START_ENCODE, strlen(START_ENCODE))) {
	  state_puts(catgets(elm_msg_cat, MeSet, MeDecodeStartElmEncoded,
			     "-- Start of (Elm) encoded section.\n"),
		     state_out);
	  crypted = ON;
	  continue;
	} else if (!strncmp(buffer, END_ENCODE, strlen(END_ENCODE))) {
	  crypted = OFF;
	  state_puts(catgets(elm_msg_cat, MeSet, MeDecodeEndElmEncoded,
			     "-- End of (Elm) encoded section.\n"),
		     state_out);
	  continue;
	} else if (crypted) {
	  no_ret(buffer);
	  encode(buffer);      
	  strfcat(buffer, "\n", sizeof buffer);
	}
	state_add_prefix(state_out);
	state_puts(buffer,state_out);
      }

      dprint(12,(debugfile,
		 "elm_decode: Readed %d bytes from temp file\n",count));
    }
    fclose(tmpfp);
  }

  dprint(12,(debugfile,
	     "elm_decode <- END; \n"));
  if (in_state_seekable(state_in)) {
    dprint(12,(debugfile,
	       "           (file); ftell=%ld\n",
	       in_state_ftell(state_in))); 
  }

  in_state_destroy(&newstate2);
}
 /* Prototype */
static void text_unsupported_decode P_((mime_t *, 
					in_state_t *, out_state_t *));

static void text_unsupported_decode (ptr, state_in, state_out) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
    char *buf = NULL;

    dprint(12,(debugfile,
	       "text_unsupported_decode -> state: offset=%d, length=%d; \n",
	       ptr -> offset, ptr -> length));

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"text_unsupported_decode",
		   "Bad magic number");

    if (in_state_seekable(state_in)) {
	dprint(12,(debugfile,
		   "                          (file): ftell=%ld\n",
		   in_state_ftell(state_in))); 
    }

    buf = elm_message(CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedText,
			      "[%s/%.30s is unsupported, treating like TEXT/PLAIN]\n\n"), 
		      TYPE(ptr->type), ptr->subtype);
    state_puts (buf,state_out);
    text_decode (ptr, state_in, state_out);

    dprint(12,(debugfile,
	       "text_unsupported_decode <- END; \n"));
    if (in_state_seekable(state_in)) {
	dprint(12,(debugfile,
		   "                        (file); ftell=%ld\n",
		   in_state_ftell(state_in))); 
    }
    free(buf);
}

void null_decode (ptr, state_in, state_out) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
    char *buf = NULL;

    dprint(12,(debugfile,
	       "null_decode <-> state: offset=%d, length=%d; \n",
	       ptr -> offset, ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"null_decode",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	dprint(12,(debugfile,
		   "               (file): ftell=%ld\n",
		   in_state_ftell(state_in))); 
    }

    buf = elm_message(CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedSkip,
			      "[%.15s/%.30s is not supported, skipping...]\n"),
		      TYPE(ptr->type),ptr->subtype);
    state_puts (buf,state_out);
    if (state_out->displaying)
	state_puts(catgets(elm_msg_cat, MeSet, MeDecodeUseVtosave,
			   "[Use 'v' to view or save this part.]\n"),
		   state_out);
    free(buf);
}

CT_decoder_t select_CT_decoder (ptr) 
     mime_t *ptr;
{
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"select_CT_decoder",
		   "Bad magic number");

    if (ptr->type == MIME_TYPE_MULTIPART) {
	if (ptr->flags & MIME_ALTERNATIVE) {
	    dprint(12,(debugfile,
		       "select_CT_decoder [%s/%s] = alternative_decode\n",
		       mime_types[ptr->type], ptr->subtype
		       ));
	    return alternative_decode;
	} else if (ptr->flags & MIME_SIGNED) {
	    dprint(12,(debugfile,
		       "select_CT_decoder [%s/%s] = signed_decode\n",
		       mime_types[ptr->type], ptr->subtype
		       ));
	    return signed_decode;
	} else if (ptr->flags & MIME_ENCRYPTED) {
	    dprint(12,(debugfile,
		       "select_CT_decoder [%s/%s] = encrypted_decode\n",
		       mime_types[ptr->type], ptr->subtype
		       ));
	    return encrypted_decode;
	} else {
	    dprint(12,(debugfile,
		       "select_CT_decoder [%s/%s] = multipart_0_decode\n",
		       mime_types[ptr->type], ptr->subtype));
	    return multipart_0_decode;
	}
    } else if (ptr->flags & MIME_RFC822) {
	dprint(12,(debugfile,
		   "select_CT_decoder [%s/%s] = rfc822_decode\n",
		   mime_types[ptr->type], ptr->subtype));
	return rfc822_decode;
    }
    else if (ptr->type == MIME_TYPE_MESSAGE &&
	     (0 == istrcmp(ptr->subtype, "delivery-status") ||
	      0 == strincmp(ptr->subtype,"X-delivery-status-",18))) {
	dprint(12,(debugfile,
		   "select_CT_decoder [%s/%s] = text_decode\n",
		   mime_types[ptr->type], ptr->subtype));
	return text_decode;
    }
    else if (ptr->type == MIME_TYPE_TEXT) {
	if (0 == istrcmp(ptr->subtype, "rfc822-headers") ||
	    0 == istrcmp(ptr->subtype, "plain")) {
	    dprint(12,(debugfile,
		       "select_CT_decoder [%s/%s] = text_decode\n",
		       mime_types[ptr->type], ptr->subtype));
	    return text_decode;
	}
#ifdef USE_PGP
	if (0 == istrcmp(ptr->subtype, "x-pgp")) {
	    dprint(12,(debugfile,
		       "select_CT_decoder [%s/%s] = pgp_decode\n",
		       mime_types[ptr->type], ptr->subtype));
	    return pgp_decode;
	}
#endif
	dprint(12,(debugfile,
		   "select_CT_decoder [%s/%s] = text_unsupported_decode\n",
		   mime_types[ptr->type], ptr->subtype));
	return text_unsupported_decode; /* For that type we try use
					   metamail if available! */
    }
#ifdef USE_PGP
    else if (ptr->type == MIME_TYPE_APPLICATION &&
	     (istrcmp(ptr->subtype, "pgp") == 0)) {
	dprint(12,(debugfile,
		   "select_CT_decoder [%s/%s] = pgp_decode\n",
		   mime_types[ptr->type], ptr->subtype));
	return pgp_decode;
    }
    else if (ptr->type == MIME_TYPE_APPLICATION &&
	     (strincmp(ptr->subtype,"x-pgp",5) == 0)) {
	dprint(12,(debugfile,
		   "select_CT_decoder [%s/%s] = pgp_decode\n",
		   mime_types[ptr->type], ptr->subtype));
	return pgp_decode;
    }
#endif
    else if (ptr->type == MIME_TYPE_APPLICATION &&
	     (istrcmp(ptr->subtype, "X-ELM-encode") == 0)) {
	dprint(12,(debugfile,
		   "select_CT_decoder [%s/%s] = elm_decode\n",
		   mime_types[ptr->type], ptr->subtype));
	return elm_decode;
    }
  
    dprint(12,(debugfile,
	       "select_CT_decoder [%s/%s] = null_decode\n",
	       mime_types[ptr->type], ptr->subtype));
    return null_decode;
}


void mime_decode (ptr, state_in, state_out)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
  /* This routine calls the appropriate routine to decode the data
   * described in "ptr".
   */

  dprint(12,(debugfile,"mime_decode -> state: offset=%d, length=%d\n",
	     ptr -> offset, ptr -> length));

  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"mime_decode",
	       "Bad magic number");

  if (!in_state_seekable(state_in)) {
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMimeUnsupportedIS,
		      "mime_decode: unsupported input state"));
    state_puts("\n[",state_out);
    state_puts(catgets(elm_msg_cat, MeSet, MeDecodeMimeUnsupportedIS,
		       "mime_decode: unsupported input state"), 
	       state_out);
    state_puts("]\n",state_out);
    sleep_message();
    return;
  }

  if (ptr->disposition == DISP_INLINE) {
    CT_decoder_t decoder = select_CT_decoder(ptr);

    dprint(12,(debugfile,"mime_decode: decoder=%p\n",decoder));

    if (in_state_fseek(state_in,ptr->offset) != 0) {
      state_putc('[',state_out);
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeMimeSeekFailed,
			"mime_decode: seek failed"),
		 state_out);
      state_puts("]\n",state_out);
      lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMimeSeekFailed,
			"mime_decode: seek failed"));
      sleep_message();
      return;
    }

    decoder (ptr, state_in, state_out);    
  }
  else { 
    if (state_out->displaying)      
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeAttachUseV,
			 "[Attachment, skipping...  Use 'v' to view this part.]\n"),
		 state_out);
    else
      state_puts(catgets(elm_msg_cat, MeSet, MeDecodeAttachSkipping,
			 "[Attachment, skipping...]\n"),
		 state_out);
  }

  dprint(12,(debugfile,"mime_decode <- END\n"));
}

static int rfc1522_decode_word (ptr, state)
     char *ptr;
     out_state_t *state;
{
  char *charset = NULL;
  char *encoded = NULL;
  char *buffer = NULL;
  char *c;
  int encoding = ENCODING_ILLEGAL;
  int count;
  int result = -1; /* Default to ERROR */
 
  count = 0;

  /* Make a copy so the original is not destroyed */
  c = buffer = strmcpy (buffer, ptr+2);

  while ((c = strtok (c, "?")) != NULL) {
    if (count == 0)
      /* get the charset */
      charset = c;
    else if (count == 1) {
      /* Check the encoding */
      switch(*c) {
      case 'q': case 'Q':
	encoding = ENCODING_QUOTED;
	break;
      case 'b': case 'B':
	encoding = ENCODING_BASE64;
	break;
      }
    }
    else if (count == 2) 
      encoded = c;
    count++;
    c = NULL;
  }

  if (charset && encoding != ENCODING_ILLEGAL && encoded) {
    int code = class_charset(charset);
    result = 0; /* OK */

    dprint (14, (debugfile, 
		"rfc1522_decode_word: OK: ptr=%s, code=%d\n",ptr,code));

    if (code != 0) {
      in_state_t state_in;

      in_state_clear (&state_in, STATE_in_string);

      state->filter = NULL_filter;

      if (code < 0) {
	dprint (14,(debugfile, 
		   "rfc1522_decode_word: unsupported charset: %s -- filtering\n",
		   charset));
	state->filter = ASCII_filter;
	state_puts("[",state);
      }


      if (encoding == ENCODING_QUOTED) { /* Hack */
	char * ptr;
	for (ptr = encoded; *ptr != '\0'; ptr++)
	  if (*ptr == '_')
	    *ptr = ' ';
      }

      set_in_state_buffer (encoded, &state_in);
      
      if (encoding == ENCODING_QUOTED)
	quoted_printable_decode (&state_in, state, strlen (encoded), TRUE);
      else if (encoding == ENCODING_BASE64)
	base64_decode (&state_in,state, strlen (encoded), TRUE);
      if (code < 0) 
	state_puts("]",state);

      state->filter = NULL_filter;

      in_state_destroy(&state_in);
    } else {
      dprint (14, (debugfile, 
		  "rfc1522_decode_word: unsupported charset: %s -- skipping\n",
		  charset));
      state_puts("[?",state);
      state_puts(charset,state);
      state_puts("?]",state); 
    }
  }

  if (result == -1) {
    
    dprint (14, (debugfile, 
		"rfc1522_decode_word: FAILED: ptr=%s\n",ptr));
    dprint (14, (debugfile, 
		"rfc1522_decode_word: -> charset=%s, encoding=%d, encoded=%s\n",
		NONULL(charset),encoding,NONULL(encoded)));
    
  }
  
  free (buffer);
  buffer = NULL;

  return result;
}

static int decode_encoded_atom P_((char *trg,int trg_len,const char * src));

static int decode_encoded_atom (trg,trg_len,src) 
  char *trg;
  int trg_len; 
  CONST char * src;
{
  int len = strlen(src);
  out_state_t state;
  int is_encoded = 0;

  out_state_clear (&state, STATE_out_string);
  set_out_state_buffer(trg,trg_len,&state);

  /* Check if current atom is valid encoded word */
  if (len > 9 && src[0] == '=' && src[1] == '?' &&
      src[len-1] == '=' && src[len-2] == '?') {
    CONST char *ptr4;
    int count = 0;
    for (ptr4 = src; ptr4 - src < len; ptr4++)
      if (*ptr4 == '?')
	count++;
    if (4 == count)
      is_encoded = 1;
  }

  if (is_encoded) {
    if (rfc1522_decode_word(src,&state) == -1)
      is_encoded = 0;
  }

  out_state_destroy (&state);  

  return is_encoded;
}


static int rfc1522_decode_real (p, state)
     char *p;
     out_state_t *state;
{
  /* This routine decodes RFC1522 compliant headers */

  char *buffer = strmcpy(NULL,p);
  char *ptr = buffer;
  unsigned char last_char = 0;
  int result = 0;

  while (ptr && *ptr) {
    char *ptr2 = strpbrk(ptr," \t\r\n");
    unsigned char last_char2 = 0;
    int len;
    int is_encoded = 0;

    if (ptr2) {
      last_char2 = *ptr2;
      len = ptr2 - ptr;
      *ptr2 = '\0';
    } else {
      len = strlen(ptr);
    }

    /* Check if current word is valid encoded word */
    if (len > 9 && ptr[0] == '=' && ptr[1] == '?' &&
	ptr[len-1] == '=' && ptr[len-2] == '?') {
      char *ptr4;
      int count = 0;
      for (ptr4 = ptr; ptr4 - ptr < len; ptr4++)
	if (*ptr4 == '?')
	  count++;
      if (4 == count)
	is_encoded = 1;
    }

    dprint (14, (debugfile, 
		"rfc1522_decode_real: word = %s, len=%d, is_encoded=%d\n",
		ptr,len,is_encoded));

    /* Space between encoded chars must delete! --
     * This allows splitting more than 75 character long
     * encoded words to two encoded word and using of
     * continuation lines!
     */
    if (!is_encoded && last_char) {
      state_putc (last_char, state);
    }

    if (is_encoded) {
      if (-1 == rfc1522_decode_word(ptr,state)) {
	dprint (14, (debugfile, 
		    "rfc1522_decode_real: Error in rfc1522_decode_word ...\n"));
	result = -1;
	break;
      }
	
    } else {
      state_put(ptr,len,state);
    }

    if (!is_encoded && last_char2) {
      state_putc (last_char2, state);
      last_char2 = 0;
    }

    ptr = ptr2; /* Next word if any */
    if (ptr)
      ptr++;
    last_char = last_char2;
  }

  free(buffer);
  return 0;
}

int is_rfc1522 (s)
     char *s;
{
  /* Returns TRUE if the string contains RFC 1522 format encoded data */

  while ((s = strchr (s, '=')) != NULL) {
    s++;
    if (*s == '?')
      return TRUE;
  }
  return FALSE;
}

void rfc1522_decode (ptr, len)
     char *ptr;
     int len; /* size of "ptr" */
{
  /* Support for RFC 1522, which allows for encoding of non-ASCII characters
   * in RFC 822 headers.
   */
  char *tmp = safe_malloc (len);
  out_state_t state;

  /* Make a working copy of the header */
  strfcpy (tmp, ptr, len);

  /* The output from the decoding should be put back into the memory area
   * which was passed into this routine.
   */
  out_state_clear (&state, STATE_out_string);
  set_out_state_buffer(ptr,len,&state);
  
  if (rfc1522_decode_real (tmp, &state) == -1) {
    /* An error occured.  Restore the original value. */    
    
    strfcpy (ptr, tmp, len);
  }

  out_state_destroy (&state);  

  free (tmp);
}

char * blstrpbrk (string, set) 
  char *string;
  CONST char *set;
{
  char *p;

  for (p = string; *p; p++) 
    if ('\\' == *p) {
      p++;
      if ('\0' == *p)
	return NULL;
    } else if (NULL != strchr(set,*p))
      return p;
  return NULL;
}

static void rfc1522_decode_comment P_((char *ptr,
				       int size));

static void rfc1522_decode_comment (ptr, size)
     char *ptr;
     int size; /* size of "ptr" */
{
  char *tmp = safe_strdup(ptr);
  char *walk, *ptr2=ptr;
  int len;
  int skip_space = 0;
  char add_space = '\0';
  
  dprint (22, (debugfile, 
	       "rfc1522_decode_comment: ptr=\"%s\", size=%d\n",
	       ptr,size));
	  
  size--;  /* place for \0 */
  
  for (walk=tmp; *walk && size > 0; walk += len) {
    char *ptr1 = blstrpbrk(walk," \t\n()");  
    char c;

    if (NULL == ptr1)      len = strlen(walk);
    else if (walk == ptr1) { 
      len = 1;
      
      if (skip_space && NULL != strchr(" \t\n",*ptr1)) {
	dprint (22, (debugfile, 
		     "rfc1522_decode_comment: \"%c\" -> add space\n",
		     *ptr1));
	
	add_space = ' ';
      } else {
	dprint (22, (debugfile, 
		     "rfc1522_decode_comment: \"%c\" -> \"%c\"\n",
		     *ptr1,*ptr1));
	
	skip_space = 0;
	*ptr2++ = *ptr1;
	size--;
      }
      continue;
    } else {
      len = ptr1 - walk;
      c = *ptr1;
      *ptr1 = '\0';

      dprint (23, (debugfile, 
		   "rfc1522_decode_comment: \"%s\", len=%d c=%c\n",
		   walk,len,c));
    }
    
    if (decode_encoded_atom(ptr2, size,walk)) {
      int l = strlen(ptr2);
      
      dprint (22, (debugfile, 
		   "rfc1522_decode_comment: \"%s\" -> \"%.*s\"\n",
		   walk,l,ptr2));
      
      ptr2 += l;
      size -= l;
      add_space  = '\0';
      skip_space = 1;
    } else {
      int l;
      skip_space = 0;
      if (add_space) {
	dprint (22, (debugfile, 
		     "rfc1522_decode_comment: add space -> \"%c\"\n",
		     add_space));
	
	
	*ptr2++ = add_space;
	size--;
	add_space  = '\0';
	if (!size)
	  break;
      }
      strfcpy(ptr2,walk,size);
      l = strlen(ptr2);
      
      dprint (22, (debugfile, 
		   "rfc1522_decode_comment: \"%s\" -> \"%.*s\"\n",
		   walk,l,ptr2));
      
      ptr2 += l;
      size -= l;
    }
    if (ptr1)
      *ptr1 = c;
  }
  
  free(tmp);
  *ptr2 = '\0';
  dprint (22, (debugfile, 
	       "rfc1522_decode_comment = \"%s\"\n",
	       ptr));
  
}

void rfc1522_decode_structured (class, ptr, size)
     int class;
     char *ptr;
     int size; /* size of "ptr" */
{
  /* Decodes comments from structured fields 
   * if class & HDR_PHRASE decodes also prases (or all words outside of < >)
   *   -- not exactly correct for address lines, but otherwise we need full
   *      STD 11 (RFC 822) parser ....
   */
  char *walk, *walk2;
  char **tokenized;
  char *was_space = NULL;
  int skip_space = 0;
  int i;
  int q = 0;

  dprint (14, (debugfile, 
	      "rfc1522_decode_structured -> class=%d,size=%d,ptr=%s\n",
	      class,size,ptr));

  switch(class) {
  case HDR_COMMENT:
    rfc1522_decode_comment(ptr,size);
    return;
  case HDR_TEXT:
    rfc1522_decode(ptr,size);
    return;
  }

  tokenized = rfc822_tokenize(ptr);
  ptr[0] = '\0';   /* In place */

  for (i = 0; tokenized[i]; i++) {
    int copy = 1;
    if ('<' == tokenized[i][0]) 
      q++;
    
    if (!q && '(' != tokenized[i][0] && 0 != (class & HDR_PHRASE)) {
      int maybe_addr = 0;
      int j;
      
      for (j = i; tokenized[j]; j++) {
	if ('@' == tokenized[j][0]) {
	  maybe_addr = 1;
	  break;
	}
	if (',' == tokenized[j][0] ||
	    '<' == tokenized[j][0] ||
	    ':' == tokenized[j][0])
	  break;
      }

      if (!maybe_addr) {
	if (whitespace(tokenized[i][0]) || '\n' == tokenized[i][0]) {
	  if (skip_space) {
	    was_space = tokenized[i];
	    copy = 0;
	    dprint (22, (debugfile, 
			 "rfc1522_decode_structured: [%d]=\"%s\" \t(space)\n",
			 i, tokenized[i]));
	  }
	} else {
	  char temp[STRING];
	  if (decode_encoded_atom(temp, sizeof temp,tokenized[i])) {
	    skip_space = 1;
	    copy = 0;
	    strfcat(ptr,temp,size);
	    dprint (22, (debugfile, 
			 "rfc1522_decode_structured: [%d]=\"%s\" \t=> \"%s\"\n",
			 i, tokenized[i], temp));
	  } else if (was_space) {
	    strfcat(ptr,was_space,size);
	    dprint (22, (debugfile, 
			 "rfc1522_decode_structured: ADD SPACE \t=> \"%s\"\n",
		   was_space));      	    
	    was_space  = NULL;
	    skip_space = 0;
	  }
	}
      }
    } else if (was_space) {
      strfcat(ptr,was_space,size);
      dprint (22, (debugfile, 
		   "rfc1522_decode_structured: ADD SPACE \t=> \"%s\"\n",
		   was_space));      
      was_space  = NULL;
      skip_space = 0;
    }

    if ('>' == tokenized[i][0]) 
      q--;

    if ('(' == tokenized[i][0]) {
      char temp[STRING];
      int l;
      
      strfcpy(temp,tokenized[i]+1,sizeof temp);
      l = strlen(temp);
      if (l > 0 && ')' == temp[l-1])
	temp[l-1] = '\0';

      rfc1522_decode_comment(temp,sizeof temp);

      strfcat(ptr,"(",size);
      strfcat(ptr,temp,size);
      strfcat(ptr,")",size);
      copy = 0;
      dprint (22, (debugfile, 
		   "rfc1522_decode_structured: [%d]=\"%s\" \t=> \"(%s)\"\n",
		   i, tokenized[i], temp));
    }

    if (copy) {
      strfcat(ptr,tokenized[i],size);
      dprint (22, (debugfile, 
		   "rfc1522_decode_structured: [%d]=\"%s\" \tCOPY\n",
		   i, tokenized[i]));      
    }
  }

  free_rfc822tokenized(tokenized);

  dprint (14, (debugfile, "rfc1522_decode_structured <- ptr=%s\n",ptr));
}

static mime_t * best_alternative(att)     
     mime_t *att;
{
  mime_t *ret = att;
  if (att->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"best_alternative",
	       "Bad magic number");

  while (att) {
    if (!mime_notplain(att)) {
	dprint(9,(debugfile,"-- best_alternative, found: %s/%s\n",
		  mime_types[att->type], att->subtype));
      ret = att;
    }
    att = att->next;
  }

  if (ret) {
    dprint(9,(debugfile,"best_alternative - returns: %s/%s\n",
	      mime_types[ret->type], ret->subtype));
  } else {
    dprint(9,(debugfile,"best_alternative=NULL"));
  }

  return ret;
}

static int can_handle P_((mime_t *att));
static int can_handle(att)     
     mime_t *att;
{
  if (att->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"can_handle",
	       "Bad magic number");

  while (att) {
    if (att->magic != MIME_magic)
      mime_panic(__FILE__,__LINE__,"can_handle",
		 "Bad magic number (next -chain)");

    if (att->disposition == DISP_INLINE) {      
      if (mime_notplain(att)) {
	dprint(9,(debugfile,"can_handle=NO -- Failed: %s/%s\n",
		  mime_types[att->type], att->subtype));
	return FALSE;
      }
      dprint(9,(debugfile,  "can_handle:     Succeed: %s/%s\n",
		mime_types[att->type], att->subtype));
    } else {
      dprint(9,(debugfile,  "can_handle:  Attachment: %s/%s\n",
		mime_types[att->type], att->subtype));
    }
    att = att->next;
  }
  dprint(9,(debugfile,"can_handle=TRUE\n"));
  return TRUE;
}

int mime_notplain (p)
     mime_t *p;
{
    CT_decoder_t decoder = select_CT_decoder(p);
    mime_t *z;
    char buf[STRING];
    
    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_not_plain",
		   "Bad magic number");
    
    if (p->encoding < ENCODING_NONE ||
	p->encoding >= ENCODING_EXPERIMENTAL) {
	dprint(9,(debugfile,
		  "mime_notplain=TRUE -- type: %s/%s; encoding=%s (%d)\n",
		  mime_types[p->type], p->subtype,
		  ENCODING(p->encoding),p->encoding));
    return TRUE;      
  }

    if (decoder == rfc822_decode && p->parts && can_handle(p->parts)) {
	dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s\n",
		  mime_types[p->type], p->subtype));
	return FALSE;      
    }
    if (decoder == rfc822_decode) {
	dprint(9,(debugfile,"mime_notplain=TRUE -- type: %s/%s\n",
		  mime_types[p->type], p->subtype));
	return TRUE;
    }
 
    /* Notice: if (decoder == text_unsupported_decode)
     *         we return TRUE, because we want call metamail
     *         if it is available
     */
 
    if (decoder == text_decode || decoder == elm_decode) {
	/* If mime_get_charsets < 0, we want call metamail
	 * (instead of filtering to US-ASCII)
	 */
	if (mime_get_charset(buf,p->type_opts,sizeof(buf)) > 0) {
	    dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s; charset=%s\n",
		      mime_types[p->type], p->subtype,buf));
	    return FALSE;
	}
    }
    else if (decoder == multipart_0_decode &&
	     ((p->flags & MIME_MIXED) || (p->flags & MIME_DIGEST)) &&
	     p->parts && can_handle(p->parts)) {
	dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s, flags=%d\n",
		  mime_types[p->type], p->subtype, p->flags));
	return FALSE;	   
    }
    else if (decoder == alternative_decode &&
	     p->parts && ( z= best_alternative(p->parts)) && 
	     (pagealternative ? !mime_notplain(z) : can_handle(p->parts))) {
	dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s, "
		  "best_alternative: %s/%s\n",
		  mime_types[p->type], p->subtype, 
		  mime_types[z->type], z->subtype));
	return FALSE;
    } else if (decoder == signed_decode && 
	       p->parts && !mime_notplain(p->parts)) {
	SG_decoder_t sg_decode;
	if (!mime_get_param ("protocol", buf, p->type_opts, 
			     sizeof buf)) {
	    dprint(9,(debugfile,"mime_notplain=TRUE -- type: %s/%s, flags=%d -- no 'protocol'\n",
		      mime_types[p->type], p->subtype, p->flags));
	    return TRUE;	   	    
	}
	sg_decode = select_SG_decoder(buf);

	if (sg_decode != null_SG_decoder) {
	    dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s, flags=%d, protocol=%s\n",
		      mime_types[p->type], p->subtype, p->flags, buf));
	    return FALSE;
	} else if (pagesigned) {
	    dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s, flags=%d, protocol=%s, pagesigned=TRUE\n",
		      mime_types[p->type], p->subtype, p->flags, buf));
	    return FALSE;
	}
	
	dprint(9,(debugfile,"mime_notplain=TRUE -- type: %s/%s, flags=%d, protocol=%s\n",
		  mime_types[p->type], p->subtype, p->flags, buf));
	return TRUE;	
    } else if (decoder == encrypted_decode && p->parts) {
	EC_decoder_t ec_decode;
	if (!mime_get_param ("protocol", buf, p->type_opts, 
			     sizeof buf)) {
	    dprint(9,(debugfile,"mime_notplain=TRUE -- type: %s/%s, flags=%d -- no 'protocol'\n",
		      mime_types[p->type], p->subtype, p->flags));
	    return TRUE;	   	    
	}
	ec_decode = select_EC_decoder(buf);

	if (ec_decode != null_EC_decoder) {
	    dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s, flags=%d, protocol=%s\n",
		      mime_types[p->type], p->subtype, p->flags, buf));
	    return FALSE;
	} 

	dprint(9,(debugfile,"mime_notplain=TRUE -- type: %s/%s, flags=%d, protocol=%s\n",
		  mime_types[p->type], p->subtype, p->flags, buf));
	return TRUE;	
    } else if (p->type == MIME_TYPE_MULTIPART && pagemultipart) {
	dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s "
		  "(pagemultipart=TRUE)\n",
		  mime_types[p->type], p->subtype));
	return FALSE;
    }
#ifdef USE_PGP
    else if (decoder == pgp_decode) {
	dprint(9,(debugfile,"mime_notplain=FALSE -- type: %s/%s\n",
		  mime_types[p->type], p->subtype));
	return FALSE;
    }
#endif
    dprint(9,(debugfile,"mime_notplain=TRUE -- type: %s/%s\n",
	    mime_types[p->type], p->subtype));
    return TRUE;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
