/*
	Raw audio file format library
 */

#ifndef RAW_H
#define RAW_H

#include <db.h>
#include <sys/stat.h>
#include <sys/types.h>


/*
 *	Master version number:
 */
#define RawLibraryVersionMajor		1
#define RawLibraryVersionMinor		0


/*
 *	Raw library function error codes:
 */
#define RawSuccess              0
#define RawErrorNoAccess        1
#define RawErrorNotRaw		2
#define RawErrorNoBuffers	3
#define RawErrorCorrupt		4
#define RawErrorUnsupported	5
#define RawErrorBadValue	6
#define RawErrorNoData		7
#define RawErrorEndOfData	8


/*
 *	Byte value shift options for RawReadPartialData():
 */
#define RawReadUnsigned8	0
#define RawReadSigned8		1

/*
 *	Raw data structure
 */
typedef struct {

	char *filename;

	int sample_size;
        int channels;
        int sample_rate;
        int block_align;    

	off_t data_offset;	/*   Number of bytes to skip
				 *   untill actual data.
				 */

	off_t data_length;	/* Length of audio data in bytes. */


	/* Data loaded by RawReadPartialData(). */
	char *data;		/* The actual audio data. */
	off_t data_len;

} raw_data_struct;


/*
 *   Checks if givin file name is a valid RAW file, returns
 *   RawSuccess if it is, RawErrorNotRaw if it is not a valid
 *   RAW file, or appropriate error if an error occured.
 */
extern int RawIsFileRaw(char *filename);


/*
 *   Reads the RAW header information from RAW file char *filename  
 *   and loads the data into the raw_data_struct *rd structure.
 *
 *   All values will be read from the header, however no audio data
 *   will be loaded.
 *
 *   WARNING: raw_data_struct *rd must be bzero()'ed or sent to
 *   RawDestroyData() before passing to this function!!!
 */
extern int RawReadHeader(char *filename, raw_data_struct *rd);


/*
 *   Reads a segment of data from the RAW file specified by
 *   the filename member of the givin raw_data_struct *rd structure.
 *
 *   The raw_data_struct *rd structure MUST HAVE VALID VALUES FROM
 *   A PRIOR SUCCESSFUL CALL TO RawReadHeader().
 * 
 *   offset is the relative byte position FROM THE START OF THE AUDIO
 *   DATA AS A COMPLETE TRACK and not a position in the file.
 *   This value must be positive.
 *
 *   max_chunk_size is the number of bytes that you want to read from
 *   offset.   This value must be positive and greater than 0.
 *
 *   read_opt is not used currently.
 *   
 *   The loaded data will be placed into the member data of the
 *   raw_data_struct *rd with the member data_len set appropriately.
 *   If member data is not NULL, then the old data will be first free()ed
 *   automatically before loading is performed.
 */
extern int RawReadPartialData(
        raw_data_struct *rd,
        long offset,            /* In bytes. */
        long max_chunk_size,    /* In bytes. */
        int read_opt
);



/*
 *   Frees all allocated data and resets all values of the 
 *   raw_data_struct *rd members.
 *
 *   If raw_data_struct *rd is NULL then no action will be taken. 
 */
extern void RawDestroyData(raw_data_struct *rd);



#endif	/* RAW_H */

