#
# $Id: Scrollbar.w,v 1.1 1996-09-25 09:24:08+02 mho Exp $
#
# Purpose: Scrollbar widget based on the Slider widget
#
# Authors: Markus Holzem
#
# Copyright: (C) 1996, GNU (Markus)
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Additionally everyone using this library has to announce it with:
#
#   This software uses the wxWindows-Xt GUI library
#   (C) Markus Holzem, available via
#       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
#

@class XfwfScrollbar (XfwfSlider) @file = Scrollbar

@ The scrollbar widget helps the user to view data that is too large
to be displayed all at once.  They are often used through a
ScrolledWindow (see there), where they are put next to and/or below
the widget that contains the data. The scrollbar controls which part
is visible. By manipulating the scrollbar the user can move (`scroll')
the data within the other widget.

A scrollbar consists of two arrows placed at each end of a rectangle,
either at the top and bottom or at the left and right. The former is
called a vertical scrollbar, the latter a horizontal one.  A smaller
rectangle, called the thumb or slider, is placed within the larger
one. It can move up and down (vertical scrollbar) or left/right
(horizontal scrollbar). Clicking an arrow moves the data in that
direction. Pressing the mouse button on an arrow and holding it, moves
the data by small increments as long as the mouse button is down.
Dragging the slider moves the data proportionally with the slider,
either along with the movement of the mouse, or all at once when the
mouse button is released. Pressing the mouse button onthe rectangle
outside the slider moves the data in larger increments.

The ratio of the slider size to the scroll region size typically
corresponds to the relationship between the size of the visible data
and the total size of the data.  For example, if 10 percent of the
data is visible, the slider typically occupies 10 percent of the
scroll region.  This provides the user with a visual clue to the size
of the invisible data.

@public

@ The orientation of the scrollbar is set with the |vertical|
resource. |True| means vertical, |False| horizontal. This resource may
only be set during initialization. Any changes via |XtVaSetValues|
result in a warning and are then ignored.

	@var Boolean vertical = True

@ The user can interact with the scrollbar in many ways, but there is
only a single callback list: |scrollCallback|. The callback uses the
|call_data| parameter to pass a pointer to an |XfwfScrollInfo|
structure: |struct {XfwfSReason reason; XfwfSFlags flags; float vpos,
vsize, hpos, hsize;}| The |flags| field is a bitwise combination of
|XFWF_VPOS|, |XFWF_HPOS|, |XFWF_VSIZE| and |XFWF_HSIZE|.  The
structure contains a field |reason|, which can have the following
values (there exist other values, but they are not used by the
scrollbar):

\item{-} |XfwfSUp|: if the user clicked (or holds) the up arrow.
\item{-} |XfwfSLeft|: ditto left arrow.
\item{-} |XfwfSDown|: ditto down arrow.
\item{-} |XfwfSRight|: ditto right arrow.
\item{-} |XfwfSPageUp|: ditto area above thumb.
\item{-} |XfwfSPageDown|: ditto area below thumb.
\item{-} |XfwfSPageLeft|: ditto area left of thumb.
\item{-} |XfwfSPageRight|: ditto area right of thumb.
\item{-} |XfwfSDrag|: if the user drags the thumb.
\item{-} |XfwfSMove|: if the user stops dragging the thumb.
\item{-} |XfwfSTop|: if the user clicked the up arrow with control down.
\item{-} |XfwfSLeftSide|: ditto left arrow.
\item{-} |XfwfSBottom|: ditto down arrow.
\item{-} |XfwfSRightSide|: ditto right arrow.

@ When the user presses and then holds the mouse button on an arrow or
on an area outside the thumb, the scrollbar waits some milliseconds
before it starts repeating the callbacks.

	@var Cardinal initialDelay = 500

@ Between repeated calls to the callback routines, the scrollbar will
wait a few milliseconds.

	@var Cardinal repeatDelay = 50

@ Some widgets may be simple enough that they can scroll a fixed
amount whenever the user clicks on one of the arrow buttons. If that
is the case, they can let the scrollbar compute the new position. It
will be passed in the |XfwfScrollInfo| structure as the suggested new
position. Any receiver is free to ignore the suggestion, however. The
default is to add or subtract 0.05.

	@var float increment = <String> "0.05"

@private

@ The repeating callback is implemented with a time out routine. The
timer is a private variable of the widget.

	@var XtIntervalId timer

@ For the repeating the widget has to keep the current scroll mode
|reason| to repeat the correct function.

	@var XfwfSReason reason

	@var int currentOffset

@exports

@ The |XfwfSetScrollbar| convenience function can be used to set the
position and size of a scrollbar. The two arguments must be between
0.0 and 1.0 (inclusive).

@proc XfwfSetScrollbar($, double pos, double size)
{
    if (! XtIsSubclass($, xfwfScrollbarWidgetClass))
	XtError("XfwfSetScrollbar called with incorrect widget type");
    if (pos < 0.0 || pos > 1.0 || size < 0.0 || size > 1.0)
	XtError("XfwfSetScrollbar called with incorrect arguments");
    if ($vertical) {
	XfwfResizeThumb($, 1.0, size);
	XfwfMoveThumb($, 0.0, pos);
    } else {
	XfwfResizeThumb($, size, 1.0);
	XfwfMoveThumb($, pos, 0.0);
    }
}

@methods

@ The |initialize| method creates the three widgets that make up the
scrollbar: two arrows and a slider. It sets the resources of these
widgets and redirects the callbacks.

@proc initialize
{
    Position x, y; Dimension w, h; $compute_thumb($, &x, &y, &w, &h);
    /* adjust offsets for arrows */
    $innerVOffset = $innerHOffset = 0;
    if ($vertical) $innerVOffset = w;
    else	   $innerHOffset = h;
    /* init private variables */
    $timer = 0;
    $currentOffset = 0;
}

@ |destroy| has to stop a running timer.

@proc destroy
{
    if ($timer)
	XtRemoveTimeOut($timer);
    $timer = 0;
}

@ |resize| leaves enough space for the arrows

@proc resize
{
    Position x, y; Dimension w, h; $compute_thumb($, &x, &y, &w, &h);
    /* adjust offsets for arrows */
    if ($vertical) $innerVOffset = w;
    else	   $innerHOffset = h;
}

@ The |expose| method of the superclass is called to draw the slider.
Only the arrows are drawn here.

@proc _expose
{
    /* draw arrows */
    paint_arrows($, False, False);
    /* draw borders and thumb */
    #_expose($, event, region);
}

@ The |set_values| does not much but to forbid the setting of a few
resources which should not change after creation.

@proc set_values
{
    if ($old$vertical != $vertical) {
	XtWarning("Cannot change the \"vertical\" resource of a scrollbar\n");
	$vertical = $old$vertical;
    }
    if ($old$innerHOffset != $innerHOffset || $old$innerVOffset != $innerVOffset) {
	XtWarning("Cannot change the inner offsets of a scrollbar\n");
	$innerHOffset = $old$innerHOffset;
	$innerVOffset = $old$innerVOffset;
    }
    return False;
}

@utilities

@ |paint_arrows| draws the arrows of the scrollbar and fills the background with
|$sunkengc| so that the arrow background appears in the same colour as the background
of the slider.

@proc paint_arrows($, Boolean toppressed, Boolean botpressed)
{
    /* get inner space of scrollbar */
    Position x, y; Dimension w, h;
    xfwfLabelClassRec.xfwfCommon_class.compute_inside(self, &x, &y, &w, &h); /* same as ##compute_inside */

    if ($vertical) {
	XFillRectangle(XtDisplay($), XtWindow($), $sunkengc, x, y - $innerVOffset, w, w);
	XfwfDrawArrow ($, XtWindow($), $thumblightgc, $thumbdarkgc, $sunkengc, $thumbgc,
		       x, y - $innerVOffset, w, w, $thumbFrameWidth, XfwfArrowUp, toppressed);
	XFillRectangle(XtDisplay($), XtWindow($), $sunkengc, x, y + h, w, w);
	XfwfDrawArrow ($, XtWindow($), $thumblightgc, $thumbdarkgc, $sunkengc, $thumbgc,
		       x, y + h, w, w, $thumbFrameWidth, XfwfArrowDown, botpressed);
    } else {
	XFillRectangle(XtDisplay($), XtWindow($), $sunkengc, x - $innerHOffset, y, h, h);
	XfwfDrawArrow ($, XtWindow($), $thumblightgc, $thumbdarkgc, $sunkengc, $thumbgc,
		       x - $innerHOffset, y, h, h, $thumbFrameWidth, XfwfArrowLeft, toppressed);
	XFillRectangle(XtDisplay($), XtWindow($), $sunkengc, x + w, y, h, h);
	XfwfDrawArrow ($, XtWindow($), $thumblightgc, $thumbdarkgc, $sunkengc, $thumbgc,
		       x + w, y, h, h, $thumbFrameWidth, XfwfArrowRight, botpressed);
    }
}

@ The |repeat_notify| function is a timeout function to repeat the stepwise scrolling
(arrow up/down and page up/down).

@proc repeat_notify(XtPointer client_data, XtIntervalId *timer)
{
    XfwfScrollInfo info; Widget $ = (Widget)client_data;

    info.reason = $reason;
    if ($vertical) {
	info.flags = XFWF_VPOS;
	info.vpos = max($thumb_y + $currentOffset, 0.0);
	info.vpos = min(info.vpos, 1.0);
    } else {
	info.flags = XFWF_HPOS;
	info.hpos = max($thumb_x + $currentOffset, 0.0);
	info.hpos = min(info.hpos, 1.0);
    }
    XtCallCallbackList($, $scrollCallback, &info);

    $timer = XtAppAddTimeOut(XtWidgetToApplicationContext($),
			     $repeatDelay, repeat_notify, $);
}

@translations

@ The |start| and |jump| actions initiate the handling of the scrollbar.
|drag| handles the draggin of the thumb. |finish| removes pending time outs,
redraws arrows and finishes dragging. |jump| and |drag| are inherited from
the slider widget. |start| and |finish| are extentions to the actions of the
slider widget.

	@trans <Btn1Down>:   start()
	@trans <Btn2Down>:   jump()
	@trans <Btn1Motion>: drag()
	@trans <Btn2Motion>: drag()
	@trans <Btn1Up>:     finish()
	@trans <Btn2Up>:     finish()

@actions

@ The |start| action checks the position of the mouse to find out on which
action should be started. The arrows initiate up/down or left/right actions.
The areas outside the thumb initiate pageup/pagedown or pageleft/pageright
actions. If the mouse is on the thumb, the moue position is recorded.

@proc start
{
    Position fx, fy; Dimension fw, fh;
    Position tx, ty; Dimension tw, th;
    XfwfScrollInfo info;
    Boolean outside = False;
    Boolean ctrl = (event->xbutton.state & ControlMask) || (event->xbutton.button == Button2);

    if (event->type != ButtonPress && event->type != ButtonRelease && event->type != MotionNotify)
	XtError("The start action must be bound to a mouse event");

    /* get sizes of areas */
    $compute_thumb($, &tx, &ty, &tw, &th);
    xfwfLabelClassRec.xfwfCommon_class.compute_inside(self, &fx, &fy, &fw, &fh);
    /* find the active arean */
    if ($vertical) {
	int y = event->xbutton.y;
	if (y < fy) {				/* top arrow */
	    info.reason = $reason = (ctrl ? XfwfSTop : XfwfSUp);
	    $currentOffset        = (ctrl ? -1.0 : -$increment);
	    outside = True;
	    paint_arrows($, True, False);
	} else if (y < ty) {			/* above thumb */
	    info.reason = $reason = XfwfSPageUp;
	    $currentOffset = -$thumb_ht;
	    outside = True;
	} else if (y >= fy + fh) {		/* down arrow */
	    info.reason = $reason = (ctrl ? XfwfSBottom : XfwfSDown);
	    $currentOffset        = (ctrl ? 1.0 : $increment);
	    outside = True;
	    paint_arrows($, False, True);
	} else if (y >= ty + th) {		/* below thumb */ 
	    info.reason = $reason = XfwfSPageDown;
	    $currentOffset = $thumb_ht;
	    outside = True;
	}
	info.flags = XFWF_VPOS;
	info.vpos  = max($thumb_y + $currentOffset, 0.0);
	info.vpos  = min(info.vpos, 1.0);
    } else {
	int x = event->xbutton.x;
	if (x < fx) {				/* left arrow */
	    info.reason = $reason = (ctrl ? XfwfSLeftSide : XfwfSLeft);
	    $currentOffset        = (ctrl ? -1.0 : -$increment);
	    outside = True;
	    paint_arrows($, True, False);
	} else if (x < tx) {			/* left of thumb */
	    info.reason = $reason = XfwfSPageLeft;
	    $currentOffset = -$thumb_wd;

	    outside = True;
	} else if (x >= fx + fw) {		/* right arrow */
	    info.reason = $reason = (ctrl ? XfwfSRightSide : XfwfSRight);
	    $currentOffset        = (ctrl ? 1.0 : $increment);
	    outside = True;
	    paint_arrows($, False, True);
	} else if (x >= tx + tw) {		/* right of thumb */ 
	    info.reason = $reason = XfwfSPageRight;
	    $currentOffset = $thumb_wd;
	    outside = True;
	}
	info.flags = XFWF_HPOS;
	info.hpos  = max($thumb_x + $currentOffset, 0.0);
	info.hpos  = min(info.hpos, 1.0);
    }
    if (outside) {
	/* info is set correctly -> call callback */
	XtCallCallbackList($, $scrollCallback, &info);
	if (XfwfSUp <= $reason && $reason <= XfwfSPageRight)
	    /* add time out for (page) up, down, left or right */
	    $timer = XtAppAddTimeOut(XtWidgetToApplicationContext($),
				     $initialDelay, repeat_notify, $);
    } else {
	/* inside thumb -> start dragging */
	$drag_in_progress = True;
	$m_delta_x = tx - event->xbutton.x;
	$m_delta_y = ty - event->xbutton.y;
    }
}

@ The |finish| action removes pending time outs. The callbacks for events
outside the thumb have already been called. If this is the end of a drag
action, the flag |drag_in_process| set to False and a last drop callback is
called.

@proc finish
{
    if ($timer) {
	XtRemoveTimeOut($timer);
	$timer = 0;
    }
    paint_arrows($, False, False);
    if ($drag_in_progress) {
	XfwfScrollInfo info;

	$drag_in_progress = False;
	info.reason = XfwfSMove;
	info.flags = XFWF_VPOS | XFWF_HPOS;
	info.hpos = $thumb_x;
	info.vpos = $thumb_y;
	XtCallCallbackList($, $scrollCallback, &info);
    }
}

@ The following action is not used by default, but it is defined here,
because someone might want to bind it to keys. For example, the
subclasses |XfwfVScrollbar| and |XfwfHScrollbar| do that.

@proc Scroll
{
    XfwfScrollInfo info;

    XfwfGetThumb($, &info);
    info.reason = XfwfCvtStringToScrollReason(params[0]);
    switch (info.reason) {
    case XfwfSUp: 
	info.flags = XFWF_VPOS;
	info.vpos = max(0.0, info.vpos - $increment);
	break;
    case XfwfSDown:
	info.flags = XFWF_VPOS;
	info.vpos = min(1.0, info.vpos + $increment);
	break;
    case XfwfSLeft:
	info.flags = XFWF_HPOS;
	info.hpos = max(0.0, info.hpos - $increment);
	break;
    case XfwfSRight:
	info.flags = XFWF_HPOS;
	info.hpos = min(1.0, info.hpos + $increment);
	break;
    case XfwfSPageUp:
	info.flags = XFWF_VPOS;
	info.vpos = max(0.0, info.vpos - info.vsize);
	break;
    case XfwfSPageDown:
	info.flags = XFWF_VPOS;
	info.vpos = min(1.0, info.vpos + info.vsize);
	break;
    case XfwfSPageLeft:
	info.flags = XFWF_HPOS;
	info.hpos = max(0.0, info.hpos - info.hsize);
	break;
    case XfwfSPageRight:
	info.flags = XFWF_HPOS;
	info.hpos = min(1.0, info.hpos + info.hsize);
	break;
    case XfwfSTop:
	info.flags = XFWF_VPOS;
	info.vpos = 0.0;
	break;
    case XfwfSBottom:
	info.flags = XFWF_VPOS;
	info.vpos = 1.0;
	break;
    case XfwfSLeftSide:
	info.flags = XFWF_HPOS;
	info.hpos = 0.0;
	break;
    case XfwfSRightSide:
	info.flags = XFWF_HPOS;
	info.hpos = 1.0;
	break;
    default: break;				/* Not understood */
    }
    XtCallCallbackList($, $scrollCallback, &info);
}


@imports

@incl <stdio.h>
