/* gtkfilelist - gtkfilelist widget for gtk+
 * Copyright 1999  Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Some code and icons borrwed from gtktalog and gtkfilebrowser */

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <fnmatch.h>
#include <vdk/gtkfilelist.h>
#include "gtkfileicons.c"


static void gtk_file_list_class_init          (GtkFileListClass *klass);
static void gtk_file_list_init                (GtkFileList      *file_list);
static void gtk_file_list_destroy             (GtkObject      *object);
static void gtk_file_list_realize             (GtkWidget *widget);
static gint sort_list			      (gpointer a, gpointer b);

static GtkIconListClass *parent_class = NULL;

static GtkFileListType default_types[] = {
                              {"*.cpp",GTK_FILE_LIST_CPP},
                              {"*.cc",GTK_FILE_LIST_CPP},
                              {"*.c",GTK_FILE_LIST_C},
                              {"*.h",GTK_FILE_LIST_H},
                              {"*.for",GTK_FILE_LIST_F},
                              {"*.f",GTK_FILE_LIST_F},
                              {"*.f77",GTK_FILE_LIST_F},
                              {"*.f90",GTK_FILE_LIST_F},
                              {"*.jar",GTK_FILE_LIST_JAVA},
                              {"*.java",GTK_FILE_LIST_JAVA},
                              {"*.j",GTK_FILE_LIST_JAVA},
                              {"*.mp3",GTK_FILE_LIST_SOUND},
                              {"*.wav",GTK_FILE_LIST_SOUND},
                              {"*.ps",GTK_FILE_LIST_PS},
                              {"*.pdf",GTK_FILE_LIST_PDF},
                              {"*.doc",GTK_FILE_LIST_DOC},
                              {"*.txt",GTK_FILE_LIST_TEXT},
                              {"README",GTK_FILE_LIST_TEXT},
                              {"TODO",GTK_FILE_LIST_TEXT},
                              {"NEWS",GTK_FILE_LIST_TEXT},
                              {"AUTHORS",GTK_FILE_LIST_TEXT},
                              {"COPYING",GTK_FILE_LIST_TEXT},
                              {"INSTALL",GTK_FILE_LIST_TEXT},
                              {"BUGS",GTK_FILE_LIST_TEXT},
                              {"*.xpm",GTK_FILE_LIST_IMG},
                              {"*.jpg",GTK_FILE_LIST_IMG},
                              {"*.jpeg",GTK_FILE_LIST_IMG},
                              {"*.gif",GTK_FILE_LIST_IMG},
                              {"*.tif",GTK_FILE_LIST_IMG},
                              {"*.tiff",GTK_FILE_LIST_IMG},
                              {"*.png",GTK_FILE_LIST_IMG},
                              {"*.tar",GTK_FILE_LIST_ARCH},
                              {"*.tar.bz2",GTK_FILE_LIST_ARCH},
                              {"*.bz2",GTK_FILE_LIST_ARCH},
                              {"*.zip",GTK_FILE_LIST_ARCH},
                              {"*.tar.gz",GTK_FILE_LIST_PKG},
                              {"*.tgz",GTK_FILE_LIST_PKG},
                              {"*.rpm",GTK_FILE_LIST_RPM},
                              {"*.deb",GTK_FILE_LIST_DEB},
                              {"*.htm",GTK_FILE_LIST_HTML},
                              {"*.html",GTK_FILE_LIST_HTML},
                              {"*.mpeg",GTK_FILE_LIST_MOVIE},
                              {"*.mpeg1",GTK_FILE_LIST_MOVIE},
                              {"*.mpeg2",GTK_FILE_LIST_MOVIE},
                              {"*.mpg",GTK_FILE_LIST_MOVIE},
                              {"*.mpg1",GTK_FILE_LIST_MOVIE},
                              {"*.mpg2",GTK_FILE_LIST_MOVIE},
                              {"*.avi",GTK_FILE_LIST_MOVIE},
                              {"*.mov",GTK_FILE_LIST_MOVIE},
                              {"core",GTK_FILE_LIST_CORE},
                              {NULL, 0},
                             };


GtkType
gtk_file_list_get_type (void)
{
  static GtkType file_list_type = 0;

  if (!file_list_type)
    {
      GtkTypeInfo file_list_info =
      {
	"GtkFileList",
	sizeof (GtkFileList),
	sizeof (GtkFileListClass),
	(GtkClassInitFunc) gtk_file_list_class_init,
	(GtkObjectInitFunc) gtk_file_list_init,
	/* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      file_list_type = gtk_type_unique (gtk_icon_list_get_type(), &file_list_info);
    }

  return file_list_type;
}

GtkWidget*
gtk_file_list_new (gint icon_width, gint mode, gchar *path)
{
  GtkWidget *widget;
  GtkFileList *file_list;
  GtkIconList *icon_list;

  widget = gtk_type_new (gtk_file_list_get_type());

  file_list = GTK_FILE_LIST(widget);
  icon_list = GTK_ICON_LIST(widget);

  icon_list->mode = mode;
  icon_list->icon_width = icon_width;

  if(path)
     file_list->path = g_strdup(path);
  else
     file_list->path = g_strdup("/");

  return widget;
}

static void
gtk_file_list_class_init (GtkFileListClass *class)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;

  widget_class = (GtkWidgetClass*) class;
  object_class = (GtkObjectClass*) class;
  parent_class = gtk_type_class (gtk_icon_list_get_type ());

  widget_class->realize = gtk_file_list_realize;
  object_class->destroy = gtk_file_list_destroy;
}

static void
gtk_file_list_destroy(GtkObject *object)
{
  GtkIconList *icon_list;
  GtkIconListItem *item;
  GtkFileListItem *file_item;
  GList *files;

  icon_list = GTK_ICON_LIST(object);

  files = icon_list->icons;
  while(files){
    item = (GtkIconListItem *)files->data;
    file_item = (GtkFileListItem *)item->link;
    g_free(file_item->file_name);
    g_free(item->link);
    item->link = NULL;
    files = files->next;
  }

  g_free(GTK_FILE_LIST(object)->path);
  g_free(GTK_FILE_LIST(object)->filter);
}

static void
gtk_file_list_init (GtkFileList *file_list)
{
  GtkFileListType *types;
  gchar **pixmap_data[GTK_FILE_LIST_CORE+1];
  gint i;

  file_list->path = NULL;
  file_list->show_folders = TRUE;
  file_list->sort_mode = GTK_FILE_LIST_SORT_TYPE;
  file_list->filter = NULL;

  GTK_ICON_LIST(file_list)->text_space = 150;

  GTK_ICON_LIST(file_list)->compare_func = (GCompareFunc)sort_list;

  pixmap_data[GTK_FILE_LIST_FILE] = file_xpm;
  pixmap_data[GTK_FILE_LIST_FOLDER] = folder_xpm;
  pixmap_data[GTK_FILE_LIST_EXEC] = exec_xpm;
  pixmap_data[GTK_FILE_LIST_CPP] = cpp_xpm;
  pixmap_data[GTK_FILE_LIST_C] = c_xpm;
  pixmap_data[GTK_FILE_LIST_H] = h_xpm;
  pixmap_data[GTK_FILE_LIST_F] = f_xpm;
  pixmap_data[GTK_FILE_LIST_JAVA] = java_xpm;
  pixmap_data[GTK_FILE_LIST_DOC] = doc_xpm;
  pixmap_data[GTK_FILE_LIST_SOUND] = sound_xpm;
  pixmap_data[GTK_FILE_LIST_PS] = ps_xpm;
  pixmap_data[GTK_FILE_LIST_PDF] = pdf_xpm;
  pixmap_data[GTK_FILE_LIST_TEXT] = text_xpm;
  pixmap_data[GTK_FILE_LIST_IMG] = image_xpm;
  pixmap_data[GTK_FILE_LIST_ARCH] = arch_xpm;
  pixmap_data[GTK_FILE_LIST_PKG] = package_xpm;
  pixmap_data[GTK_FILE_LIST_DEB] = deb_xpm;
  pixmap_data[GTK_FILE_LIST_RPM] = rpm_xpm;
  pixmap_data[GTK_FILE_LIST_HTML] = html_xpm;
  pixmap_data[GTK_FILE_LIST_MOVIE] = movie_xpm;
  pixmap_data[GTK_FILE_LIST_CAT] = cat_xpm;
  pixmap_data[GTK_FILE_LIST_CORE] = core_xpm;

  for(i = 0; i <= GTK_FILE_LIST_CORE; i++)
    file_list->pixmaps = g_list_append(file_list->pixmaps, pixmap_data[i]);

  file_list->ntypes = 0;

  types = default_types;
  while(types->extension){
    file_list->types = g_list_append(file_list->types, types);
    types++;
    file_list->ntypes++;
  }
}

static gint
sort_list(gpointer a, gpointer b)
{
  GtkFileList *file_list;
  GtkIconListItem *itema;
  GtkIconListItem *itemb;
  GtkFileListItem *filea;
  GtkFileListItem *fileb;
  gint compare_value;

  itema = (GtkIconListItem *)a;
  itemb = (GtkIconListItem *)b;
  filea = (GtkFileListItem *)itema->link;
  fileb = (GtkFileListItem *)itemb->link;
  file_list = GTK_FILE_LIST(itema->entry->parent);
  if(!file_list) return 0;

  switch(file_list->sort_mode){
    case GTK_FILE_LIST_SORT_TYPE:
       compare_value = filea->type - fileb->type;
       if(compare_value == 0)
              compare_value = strcmp(itema->label, itemb->label);
       break;
    case GTK_FILE_LIST_SORT_NAME:
    default:
       compare_value = strcmp(itema->label, itemb->label);
       break;
  }

  return compare_value;
}

static void
gtk_file_list_realize(GtkWidget *widget)
{
  GtkFileList *file_list;

  GTK_WIDGET_CLASS(parent_class)->realize (widget);

  file_list = GTK_FILE_LIST(widget);
  gtk_file_list_open_dir(file_list, file_list->path);
}

void
gtk_file_list_open_dir(GtkFileList *file_list, gchar *dir_path)
{
  DIR *dir;
  struct dirent *dirent;
  struct stat fileinfo, linkinfo;
  GtkIconList *icon_list;
  GtkIconListItem *item;
  GtkFileListItem *file_item;
  GtkFileListType *file_type;
  GtkWidget *widget;
  gchar *full_name, *file;
  gchar *real_path;
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  gchar **icon;
  gint width, height;
  GdkGC *gc;
  GList *types, *files, *list;
  gint type;
  gboolean show_file;

  widget = GTK_WIDGET(file_list);
  icon_list = GTK_ICON_LIST(widget);

  if(dir_path)
     real_path = g_strdup(dir_path);
  else
     real_path = g_strdup("/");

  if(file_list->path)
      g_free(file_list->path);

  file_list->path = real_path;

  if((dir = opendir(file_list->path)) == NULL){
    g_warning("Can not open folder: %s",file_list->path);
    return;
  }

  gtk_icon_list_freeze(icon_list);

  files = icon_list->icons;
  while(files){
    item = (GtkIconListItem *)files->data;
    file_item = (GtkFileListItem *)item->link;
    g_free(file_item->file_name);
    g_free(item->link);
    item->link = NULL;
    files = files->next;
  }

  gtk_icon_list_clear(icon_list);

  files = NULL;
  while((dirent=readdir(dir))!=NULL){
        full_name = g_strconcat(real_path,dirent->d_name,NULL);
        file = g_strdup(dirent->d_name);
        stat(full_name, &fileinfo);

        if(S_ISDIR(fileinfo.st_mode))
         {
          type = GTK_FILE_LIST_FOLDER;
         }
        else
         {
          type = GTK_FILE_LIST_FILE;
          if(fileinfo.st_mode & 0111)
                 type = GTK_FILE_LIST_EXEC;
         }

        show_file = TRUE;
        if(strcmp(file,".") == 0 || strcmp(file,"..") == 0)
             show_file = FALSE;
        if(!file_list->show_folders && type == GTK_FILE_LIST_FOLDER)
             show_file = FALSE;
        if(file_list->filter && fnmatch(file_list->filter, file, (1 << 4)) != 0)
             show_file = FALSE;

        if(show_file){
            file_item = (GtkFileListItem *)g_new(GtkFileListItem, 1);
            file_item->file_name = file;
            file_item->type = type;

            file_item->is_link = FALSE;
            if(lstat(full_name, &linkinfo) == -1)
                   g_warning("Can not resolve link: %s",full_name);
            if(S_ISLNK(linkinfo.st_mode))
                file_item->is_link = TRUE;
            files = g_list_append(files, file_item);
        }

        g_free(full_name);
  }

  list = files;
  while(list){
       file_item = (GtkFileListItem *)list->data;
       type = file_item->type;

       types = file_list->types;
       while(types){
         file_type = (GtkFileListType *)types->data;
         if(fnmatch((gchar *)file_type->extension, file_item->file_name, (1 << 4)) == 0){
           type = file_type->type;
           break;
         }
         types = types->next;
       }

       icon = (gchar **)g_list_nth_data(file_list->pixmaps, type);
       file_item->type = type;

       item = gtk_icon_list_add_from_data(icon_list, icon,
                                          file_item->file_name, file_item);

       if(file_item->is_link){
               pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                      gdk_colormap_get_system(),
                                                      &mask, NULL,
                                                      symlink_xpm);
               gdk_window_get_size(GTK_PIXMAP(item->pixmap)->pixmap,
                                   &width, &height);
               gc = gdk_gc_new(pixmap);
               gdk_draw_pixmap(GTK_PIXMAP(item->pixmap)->pixmap,
                               gc,
                               pixmap,
                               0, 0,
                               width - 7, height - 7,
                               7, 7);
               gdk_gc_unref(gc);
               gc = gdk_gc_new(mask);
               gdk_draw_pixmap(GTK_PIXMAP(item->pixmap)->mask,
                               gc,
                               mask,
                               0, 0,
                               width - 7, height - 7,
                               7, 7);
               gdk_gc_unref(gc);

               gdk_pixmap_unref(pixmap);
       }

       list = list->next;
  }

  closedir(dir);

  gtk_icon_list_thaw(icon_list);

  g_list_free(files);
}

void
gtk_file_list_set_filter(GtkFileList *filelist, gchar *filter)
{
  filelist->filter = g_strdup(filter);
  gtk_file_list_open_dir(filelist, filelist->path);
}

gchar *
gtk_file_list_get_path(GtkFileList *file_list)
{
  return(file_list->path);
}

gchar *
gtk_file_list_get_filename(GtkFileList *file_list)
{
  GList *list;
  gpointer selection=NULL;
  gchar *file;

  file=NULL;
  list = GTK_ICON_LIST(file_list)->selection;

  if(list) selection=list->data;

  if(selection){
       selection = ((GtkIconListItem *)selection)->link;
       file = ((GtkFileListItem *)selection)->file_name;
  }
  return file;
}

