/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_timer.c,v 1.4 1999/07/21 06:08:29 slouken Exp $";
#endif

#include <stdio.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include "SDL_timer.h"

/* The first ticks value of the application */
static struct timeval start;

void SDL_StartTicks(void)
{
	/* Set first ticks value */
	gettimeofday(&start, NULL);
}

Uint32 SDL_GetTicks (void)
{
	struct timeval now;
	Uint32 ticks;

	gettimeofday(&now, NULL);
	ticks=(now.tv_sec-start.tv_sec)*1000+(now.tv_usec-start.tv_usec)/1000;
	return(ticks);
}

void SDL_Delay (Uint32 ms)
{
	int was_error;
#ifndef linux	/* Non-Linux implementations need to calculate time left */
	Uint32 then, now, elapsed;
#endif
	struct timeval tv;

	/* Set the timeout interval - Linux only needs to do this once */
#ifdef linux
	tv.tv_sec = ms/1000;
	tv.tv_usec = (ms%1000)*1000;
#else
	then = SDL_GetTicks();
#endif
	do {
		errno = 0;
#ifndef linux
		/* Calculate the time interval left (in case of interrupt) */
		now = SDL_GetTicks();
		elapsed = (now-then);
		then = now;
		if ( elapsed >= ms ) {
			break;
		}
		ms -= elapsed;
		tv.tv_sec = ms/1000;
		tv.tv_usec = (ms%1000)*1000;
#endif
		was_error = select(0, NULL, NULL, NULL, &tv);
	} while ( was_error && (errno == EINTR) );
}

/* Data to handle a single periodic alarm */
static SDL_TimerCallback alarm_callback = NULL;
static Uint32 alarm_interval;

static void HandleAlarm(int sig)
{
	Uint32 ms;

	if ( alarm_callback ) {
		ms = (*alarm_callback)(alarm_interval);
		if ( ms != alarm_interval ) {
			SDL_SetTimer(ms, alarm_callback);
		}
	}
}

int SDL_SetTimer (Uint32 ms, SDL_TimerCallback callback)
{
	struct itimerval timer;

	alarm_callback = NULL;
	alarm_interval = ms;
	if ( ms ) {
		/* Set a new alarm */
		alarm_callback = callback;
	}
	timer.it_value.tv_sec = (ms/1000);
	timer.it_value.tv_usec = (ms%1000)*1000;
	timer.it_interval.tv_sec = (ms/1000);
	timer.it_interval.tv_usec = (ms%1000)*1000;
	setitimer(ITIMER_REAL, &timer, NULL);
	return(0);
}

int SDL_TimerInit(void)
{
	struct sigaction action;

	/* Set the alarm handler (Linux specific) */
	memset(&action, 0, sizeof(action));
	action.sa_handler = HandleAlarm;
	action.sa_flags = SA_RESTART;
	sigemptyset(&action.sa_mask);
	sigaction(SIGALRM, &action, NULL);
	return(0);
}

void SDL_TimerQuit(void)
{
	SDL_SetTimer(0, NULL);
}

