/*
	SDL - Simple DirectMedia Layer
	Copyright (C) 1997, 1998, 1999  Sam Lantinga

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Library General Public
	License as published by the Free Software Foundation; either
	version 2 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Library General Public License for more details.

	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	Sam Lantinga
	slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_fbvideo.c,v 1.4 1999/12/11 06:10:56 hercules Exp $";
#endif

/* Framebuffer console based SDL video driver implementation.
*/

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <asm/page.h>		/* For definition of PAGE_SIZE */

#include "SDL.h"
#include "SDL_error.h"
#include "SDL_video.h"
#include "SDL_mouse.h"
#include "SDL_sysvideo.h"
#include "SDL_pixels_c.h"
#include "SDL_events_c.h"
#include "SDL_fbvideo.h"
#include "SDL_fbmouse_c.h"
#include "SDL_fbevents_c.h"

/* Initialization/Query functions */
static int FB_VideoInit(_THIS, SDL_PixelFormat *vformat);
static SDL_Rect **FB_ListModes(_THIS, SDL_PixelFormat *format, Uint32 flags);
static SDL_Surface *FB_SetVideoMode(_THIS, SDL_Surface *current, int width, int height, int bpp, Uint32 flags);
static int FB_SetColors(_THIS, int firstcolor, int ncolors);
static void FB_VideoQuit(_THIS);

/* Hardware surface functions */
static int FB_AllocHWSurface(_THIS, SDL_Surface *surface);
static int FB_LockHWSurface(_THIS, SDL_Surface *surface);
static void FB_UnlockHWSurface(_THIS, SDL_Surface *surface);
static void FB_FreeHWSurface(_THIS, SDL_Surface *surface);

/* FB driver bootstrap functions */

static int FB_Available(void)
{
	int console;

	console = open("/dev/fb0", O_RDWR, 0);
	if ( console >= 0 ) {
		close(console);
	}
	return(console >= 0);
}

static void FB_DeleteDevice(SDL_VideoDevice *device)
{
	free(device->hidden);
	free(device);
}

static SDL_VideoDevice *FB_CreateDevice(int devindex)
{
	SDL_VideoDevice *device;

	/* Initialize all variables that we clean on shutdown */
	device = (SDL_VideoDevice *)malloc(sizeof(SDL_VideoDevice));
	if ( device ) {
		memset(device, 0, (sizeof *device));
		device->hidden = (struct SDL_PrivateVideoData *)
				malloc((sizeof *device->hidden));
	}
	if ( (device == NULL) || (device->hidden == NULL) ) {
		SDL_OutOfMemory();
		if ( device ) {
			free(device);
		}
		return(0);
	}
	memset(device->hidden, 0, (sizeof *device->hidden));

	/* Set the function pointers */
	device->VideoInit = FB_VideoInit;
	device->ListModes = FB_ListModes;
	device->SetVideoMode = FB_SetVideoMode;
	device->SetColors = FB_SetColors;
	device->UpdateRects = NULL;
	device->VideoQuit = FB_VideoQuit;
	device->AllocHWSurface = FB_AllocHWSurface;
	device->CheckHWBlit = NULL;
	device->FillHWRect = NULL;
	device->SetHWColorKey = NULL;
	device->SetHWAlpha = NULL;
	device->LockHWSurface = FB_LockHWSurface;
	device->UnlockHWSurface = FB_UnlockHWSurface;
	device->FlipHWSurface = NULL;
	device->FreeHWSurface = FB_FreeHWSurface;
	device->SetIcon = NULL;
	device->SetCaption = NULL;
	device->GetWMInfo = NULL;
	device->FreeWMCursor = FB_FreeWMCursor;
	device->CreateWMCursor = FB_CreateWMCursor;
	device->ShowWMCursor = FB_ShowWMCursor;
	device->WarpWMCursor = FB_WarpWMCursor;
	device->InitOSKeymap = FB_InitOSKeymap;
	device->PumpEvents = FB_PumpEvents;

	device->free = FB_DeleteDevice;

	return device;
}

VideoBootStrap FBCON_bootstrap = {
	"fbcon", FB_Available, FB_CreateDevice
};

int FB_VideoInit(_THIS, SDL_PixelFormat *vformat)
{
	struct fb_fix_screeninfo finfo;
	struct fb_var_screeninfo vinfo;
	int i, index;
	int keyboard;

	/* Initialize the library */
	console_fd = open("/dev/fb0", O_RDWR, 0);
	if ( console_fd < 0 ) {
		SDL_SetError("Unable to open framebuffer console");
		return(-1);
	}

	/* Get the type of video hardware */
	if ( ioctl(console_fd, FBIOGET_FSCREENINFO, &finfo) < 0 ) {
		SDL_SetError("Couldn't get cosole hardware info\n");
		FB_VideoQuit(this);
		return(-1);
	}
	switch (finfo.visual) {
		case FB_VISUAL_TRUECOLOR:
		case FB_VISUAL_PSEUDOCOLOR:
		case FB_VISUAL_STATIC_PSEUDOCOLOR:
			break;
		default:
			SDL_SetError("Unsupported console hardware\n");
			FB_VideoQuit(this);
			return(-1);
	}

	/* Determine the screen depth (use default 8-bit depth) */
	if ( ioctl(console_fd, FBIOGET_VSCREENINFO, &vinfo) < 0 ) {
		SDL_SetError("Couldn't get cosole pixel format\n");
		FB_VideoQuit(this);
		return(-1);
	}
	vformat->BitsPerPixel = vinfo.bits_per_pixel;
	for ( i=0; i<vinfo.red.length; ++i ) {
		vformat->Rmask <<= 1;
		vformat->Rmask |= (0x00000001<<vinfo.red.offset);
	}
	for ( i=0; i<vinfo.green.length; ++i ) {
		vformat->Gmask <<= 1;
		vformat->Gmask |= (0x00000001<<vinfo.green.offset);
	}
	for ( i=0; i<vinfo.blue.length; ++i ) {
		vformat->Bmask <<= 1;
		vformat->Bmask |= (0x00000001<<vinfo.blue.offset);
	}
	video_mode.x = 0;
	video_mode.y = 0;
	video_mode.w = vinfo.xres;
	video_mode.h = vinfo.yres;
	index = ((vinfo.bits_per_pixel+7)/8)-1;
	SDL_modelist[index] = (SDL_Rect **)malloc(2*sizeof(SDL_Rect *));
	if ( SDL_modelist[index] == NULL ) {
		SDL_OutOfMemory();
		FB_VideoQuit(this);
		return(-1);
	}
	SDL_modelist[index][0] = &video_mode;
	SDL_modelist[index][1] = NULL;

	/* Reset the virtual panning */
	vinfo.xoffset = 0;
	vinfo.yoffset = 0;
	ioctl(console_fd, FBIOPUT_VSCREENINFO, &vinfo);

	/* Save hardware palette, if needed */
	if ( finfo.visual == FB_VISUAL_PSEUDOCOLOR ) {
		struct fb_cmap cmap;

		cmap.start = 0;
		cmap.len = 1<<vinfo.bits_per_pixel;
		cmap.red = &console_pal[0*256];
		cmap.green = &console_pal[1*256];
		cmap.blue = &console_pal[2*256];
		cmap.transp = NULL;
		ioctl(console_fd, FBIOGETCMAP, &cmap);
	}

	/* Fill in our hardware acceleration capabilities */
	this->info.hw_available = 1;
	this->info.video_mem = finfo.smem_len/1024;

	/* Enable mouse and keyboard support */
	if ( FB_OpenKeyboard(this) < 0 ) {
		SDL_SetError("Unable to open keyboard");
		FB_VideoQuit(this);
		return(-1);
	}
	if ( FB_OpenMouse(this) < 0 ) {
		SDL_SetError("Unable to open mouse");
		FB_VideoQuit(this);
		return(-1);
	}

	/* We're done! */
	return(0);
}

SDL_Rect **FB_ListModes(_THIS, SDL_PixelFormat *format, Uint32 flags)
{
	return(SDL_modelist[((format->BitsPerPixel+7)/8)-1]);
}

/* Various screen update functions available */
static void FB_DirectUpdate(_THIS, int numrects, SDL_Rect *rects);

SDL_Surface *FB_SetVideoMode(_THIS, SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	struct fb_fix_screeninfo finfo;
	caddr_t address;
	long map_offset;

	/* Get the fixed information about the console hardware */
	if ( ioctl(console_fd, FBIOGET_FSCREENINFO, &finfo) < 0 ) {
		SDL_SetError("Couldn't get cosole hardware info\n");
		return(NULL);
	}

	/* Unmap the current buffer, if any */
	if ( current->pixels ) {
		munmap(current->pixels, finfo.smem_len);
	}

	/* Memory map the device, compensating for buggy PPC mmap() */
	map_offset = (((long)finfo.smem_start) -
		     (((long)finfo.smem_start)&~(PAGE_SIZE-1)));
	address = mmap(NULL, finfo.smem_len+map_offset, PROT_READ|PROT_WRITE, MAP_SHARED, console_fd, 0);
	if ( address == (caddr_t)-1 ) {
		SDL_SetError("Unable to memory map the video hardware");
		return(NULL);
	}

	/* Set up the new mode framebuffer */
	current->flags = (SDL_FULLSCREEN|SDL_HWSURFACE);
	current->w = width;
	current->h = height;
	current->pitch = finfo.line_length;
	current->pixels = address+map_offset;

	/* Set the blit function */
	this->UpdateRects = FB_DirectUpdate;

	/* We're done */
	return(current);
}

/* We don't actually allow hardware surfaces other than the main one */
static int FB_AllocHWSurface(_THIS, SDL_Surface *surface)
{
	return(-1);
}
static void FB_FreeHWSurface(_THIS, SDL_Surface *surface)
{
	return;
}
static int FB_LockHWSurface(_THIS, SDL_Surface *surface)
{
	return(0);
}
static void FB_UnlockHWSurface(_THIS, SDL_Surface *surface)
{
	return;
}

static void FB_DirectUpdate(_THIS, int numrects, SDL_Rect *rects)
{
	/* The application is already updating the visible video memory */
	return;
}

int FB_SetColors(_THIS, int firstcolor, int ncolors)
{
	SDL_Color *colors;
	int i, which;
	__u16 r[256];
	__u16 g[256];
	__u16 b[256];
	struct fb_cmap cmap;

	/* Set up the colormap */
	colors = this->screen->format->palette->colors;
	for ( i=ncolors-1, which=firstcolor+ncolors-1; i>=0; --i, --which ) {
		r[i] = ((__u16)colors[which].r)<<8;
		g[i] = ((__u16)colors[which].g)<<8;
		b[i] = ((__u16)colors[which].b)<<8;
	}
	cmap.start = firstcolor;
	cmap.len = ncolors;
	cmap.red = r;
	cmap.green = g;
	cmap.blue = b;
	cmap.transp = NULL;

	if ( ioctl(console_fd, FBIOPUTCMAP, &cmap) < 0 ) {
		ncolors = this->screen->format->palette->ncolors;
		cmap.start = 0;
		cmap.len = ncolors;
		if ( ioctl(console_fd, FBIOGETCMAP, &cmap) == 0 ) {
			for ( i=ncolors-1; i>=0; --i ) {
				colors[i].r = (r[i]>>8);
				colors[i].g = (g[i]>>8);
				colors[i].b = (b[i]>>8);
			}
		}
		return(0);
	}
	return(1);
}

/* Note:  If we are terminated, this could be called in the middle of
   another SDL video routine -- notably UpdateRects.
*/
void FB_VideoQuit(_THIS)
{
	int i;

	if ( this->screen ) {
		/* Clear screen, and unmap video memory */
		if ( this->screen->pixels ) {
			struct fb_fix_screeninfo finfo;

			if ( ioctl(console_fd, FBIOGET_FSCREENINFO, &finfo) == 0 ) {
				munmap(this->screen->pixels, finfo.smem_len);
			}
			this->screen->pixels = NULL;
		}

		/* Restore the original palette */
		if ( this->screen->format->palette ) {
			struct fb_cmap cmap;

			cmap.start = 0;
			cmap.len = this->screen->format->palette->ncolors;
			cmap.red = &console_pal[0*256];
			cmap.green = &console_pal[1*256];
			cmap.blue = &console_pal[2*256];
			cmap.transp = NULL;
			ioctl(console_fd, FBIOPUTCMAP, &cmap);
		}
	}

	/* Clean up defined video modes */
	for ( i=0; i<4; ++i ) {
		if ( SDL_modelist[i] != NULL ) {
			free(SDL_modelist[i]);
			SDL_modelist[i] = NULL;
		}
	}

	/* Close console and input file descriptors */
	if ( console_fd > 0 ) {
		close(console_fd);
		console_fd = -1;
	}
	FB_CloseMouse(this);
	FB_CloseKeyboard(this);
}
