/********************************************************************************
*                                                                               *
*                S p l i t t e r   W i n d o w   O b j e c t                    *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXSplitter.cpp,v 1.8 1999/11/13 06:15:15 jeroen Exp $                    *
********************************************************************************/
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXDrawable.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXSplitter.h"


/*
  Notes:
  - Reversed split option starts parting out from right [bottom].
  - Minimal partition of a split is 0 pixels.
  - Minimum width/height determined by number of visible children & barsize.
  - For convenience, width/height of <=1 replaced by minimum width/height,
    but only if both width and height of child is <=1 at the same time.
*/

// Splitter styles
#define SPLITTER_MASK     (SPLITTER_REVERSED|SPLITTER_VERTICAL|SPLITTER_TRACKING)


/*******************************************************************************/

// Map
FXDEFMAP(FXSplitter) FXSplitterMap[]={
  FXMAPFUNC(SEL_MOTION,0,FXSplitter::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXSplitter::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXSplitter::onLeftBtnRelease),
  FXMAPFUNC(SEL_FOCUS_NEXT,0,FXSplitter::onFocusNext),
  FXMAPFUNC(SEL_FOCUS_PREV,0,FXSplitter::onFocusPrev),
  FXMAPFUNC(SEL_FOCUS_UP,0,FXSplitter::onFocusUp),
  FXMAPFUNC(SEL_FOCUS_DOWN,0,FXSplitter::onFocusDown),
  FXMAPFUNC(SEL_FOCUS_LEFT,0,FXSplitter::onFocusLeft),
  FXMAPFUNC(SEL_FOCUS_RIGHT,0,FXSplitter::onFocusRight),
  };


// Object implementation
FXIMPLEMENT(FXSplitter,FXComposite,FXSplitterMap,ARRAYNUMBER(FXSplitterMap))

  
// Make a splitter
FXSplitter::FXSplitter(){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  window=NULL;
  split=0;
  offset=0;
  barsize=4;
  }


// Make a splitter; it has no interior padding, and no borders
FXSplitter::FXSplitter(FXComposite* p,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXComposite(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  defaultCursor=(options&SPLITTER_VERTICAL) ? getApp()->vsplitCursor : getApp()->hsplitCursor;
  dragCursor=defaultCursor;
  window=NULL;
  split=0;
  offset=0;
  barsize=4;
  }


// Get default width
FXint FXSplitter::getDefaultWidth(){
  FXWindow* child;
  FXint numc,w;
  if(options&SPLITTER_VERTICAL){
    w=1;
    }
  else{
    numc=w=0;
    for(child=getFirst(); child; child=child->getNext()){
      if(child->shown()){ w+=1; numc++; }
      }
    if(numc>1) w+=(numc-1)*barsize;
    }
  return w;
  }
  

// Get default height
FXint FXSplitter::getDefaultHeight(){
  FXWindow* child;
  FXint numc,h;
  if(options&SPLITTER_VERTICAL){
    numc=h=0;
    for(child=getFirst(); child; child=child->getNext()){
      if(child->shown()){ h+=1; numc++; }
      }
    if(numc>1) h+=(numc-1)*barsize;
    }
  else{
    h=1;
    }
  return h;
  }


// Recompute layout
void FXSplitter::layout(){
  FXint pos,w,h;
  FXWindow* child;
  if(options&SPLITTER_VERTICAL){          // Vertical
    if(options&SPLITTER_REVERSED){
      pos=height;
      child=getLast();
      while(child){
        if(child->shown()){
          w=child->getWidth();
          h=child->getHeight();
          if(w<=1 && h<=1) h=child->getDefaultHeight();
          if(child==getFirst()){ h=pos; if(h<0) h=0; }
          child->position(0,pos-h,width,h);
          pos=pos-h-barsize;
          }
        child=child->getPrev();
        }
      }
    else{
      pos=0;
      child=getFirst();
      while(child){
        if(child->shown()){
          w=child->getWidth();
          h=child->getHeight();
          if(w<=1 && h<=1) h=child->getDefaultHeight();
          if(child==getLast()){ h=height-pos; if(h<0) h=0; }
          child->position(0,pos,width,h);
          pos=pos+h+barsize;
          }
        child=child->getNext();
        }
      }
    }
  else{                                   // Horizontal
    if(options&SPLITTER_REVERSED){
      pos=width;
      child=getLast();
      while(child){
        if(child->shown()){
          w=child->getWidth();
          h=child->getHeight();
          if(w<=1 && h<=1) w=child->getDefaultWidth();
          if(child==getFirst()){ w=pos; if(w<0) w=0; }
          child->position(pos-w,0,w,height);
          pos=pos-w-barsize;
          }
        child=child->getPrev();
        }
      }
    else{
      pos=0;
      child=getFirst();
      while(child){
        if(child->shown()){
          w=child->getWidth();
          h=child->getHeight();
          if(w<=1 && h<=1) w=child->getDefaultWidth();
          if(child==getLast()){ w=width-pos; if(w<0) w=0; }
          child->position(pos,0,w,height);
          pos=pos+w+barsize;
          }
        child=child->getNext();
        }
      }
    }
  flags&=~FLAG_DIRTY;
  }


// Adjust horizontal layout
void FXSplitter::adjustHLayout(){
  FXWindow *child;
  FXint w,h,pos;
  if(options&SPLITTER_REVERSED){
    pos=window->getX()+window->getWidth();
    window->position(split,0,pos-split,height);
//     window->setX(split);
//     window->setWidth(pos-split);
    pos=split-barsize;
    for(child=window->getPrev(); child; child=child->getPrev()){
      if(child->shown()){
        w=child->getWidth();
        h=child->getHeight();
        if(w<=1 && h<=1) w=child->getDefaultWidth();
        if(child==getFirst()){ w=pos; if(w<0) w=0; }
        child->position(pos-w,0,w,height);
//         child->setX(pos-w);
//         child->setWidth(w);
        pos=pos-w-barsize;
        }
      }
    }
  else{
    pos=window->getX();
    window->position(pos,0,split-pos,height);
//     window->setX(pos);
//     window->setWidth(split-pos);
    pos=split+barsize;
    for(child=window->getNext(); child; child=child->getNext()){
      if(child->shown()){
        w=child->getWidth();
        h=child->getHeight();
        if(w<=1 && h<=1) w=child->getDefaultWidth();
        if(child==getLast()){ w=width-pos; if(w<0) w=0; }
        child->position(pos,0,w,height);
//         child->setX(pos);
//         child->setWidth(w);
        pos=pos+w+barsize;
        }
      }
    }
  }


// Adjust vertical layout
void FXSplitter::adjustVLayout(){
  FXWindow *child;
  FXint w,h,pos;
  if(options&SPLITTER_REVERSED){
    pos=window->getY()+window->getHeight();
    window->position(0,split,width,pos-split);
//     window->setY(split);
//     window->setHeight(pos-split);
    pos=split-barsize;
    for(child=window->getPrev(); child; child=child->getPrev()){
      if(child->shown()){
        w=child->getWidth();
        h=child->getHeight();
        if(w<=1 && h<=1) h=child->getDefaultHeight();
        if(child==getFirst()){ h=pos; if(h<0) h=0; }
        child->position(0,pos-h,width,h);
//         child->setY(pos-h);
//         child->setHeight(h);
        pos=pos-h-barsize;
        }
      }
    }
  else{
    pos=window->getY();
    window->position(0,pos,width,split-pos);
//     window->setY(pos);
//     window->setHeight(split-pos);
    pos=split+barsize;
    for(child=window->getNext(); child; child=child->getNext()){
      if(child->shown()){
        w=child->getWidth();
        h=child->getHeight();
        if(w<=1 && h<=1) h=child->getDefaultHeight();
        if(child==getLast()){ h=height-pos; if(h<0) h=0; }
        child->position(0,pos,width,h);
//         child->setY(pos);
//         child->setHeight(h);
        pos=pos+h+barsize;
        }
      }
    }
  }


// Find child just before split
FXWindow* FXSplitter::findHSplit(FXint pos){
  FXWindow* child;
  if(options&SPLITTER_REVERSED){
    child=getFirst();
    while(child){
      if(child->shown()){
        if(child->getX()-barsize<=pos && pos<child->getX()) return child;
        }
      child=child->getNext();
      }
    }
  else{
    child=getFirst();
    while(child){
      if(child->shown()){
        if(child->getX()+child->getWidth()<=pos && pos<child->getX()+child->getWidth()+barsize) return child;
        }
      child=child->getNext();
      }
    }
  return NULL;
  }


// Find child just before split
FXWindow* FXSplitter::findVSplit(FXint pos){
  FXWindow* child;
  if(options&SPLITTER_REVERSED){
    child=getFirst();
    while(child){
      if(child->shown()){
        if(child->getY()-barsize<=pos && pos<child->getY()) return child;
        }
      child=child->getNext();
      }
    }
  else{
    child=getFirst();
    while(child){
      if(child->shown()){
        if(child->getY()+child->getHeight()<=pos && pos<child->getY()+child->getHeight()+barsize) return child;
        }
      child=child->getNext();
      }
    }
  return NULL;
  }


// Move the horizontal split intelligently
void FXSplitter::moveHSplit(FXint pos){
  FXint smin,smax;
  FXASSERT(window);
  if(options&SPLITTER_REVERSED){
    smin=0;
    smax=window->getX()+window->getWidth();
    }
  else{
    smin=window->getX();
    smax=width-barsize;
    }
  split=pos;
  if(split<smin) split=smin;
  if(split>smax) split=smax;
  }


// Move the vertical split intelligently
void FXSplitter::moveVSplit(FXint pos){
  FXint smin,smax;
  FXASSERT(window);
  if(options&SPLITTER_REVERSED){
    smin=0;
    smax=window->getY()+window->getHeight();
    }
  else{
    smin=window->getY();
    smax=width-barsize;
    }
  split=pos;
  if(split<smin) split=smin;
  if(split>smax) split=smax;
  }


// Button being pressed
long FXSplitter::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* ev=(FXEvent*)ptr;
  if(isEnabled()){
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(options&SPLITTER_VERTICAL){
      window=findVSplit(ev->win_y);
      if(window){
        if(options&SPLITTER_REVERSED)
          split=window->getY();
        else
          split=window->getY()+window->getHeight();
        offset=ev->win_y-split;
        if(!(options&SPLITTER_TRACKING)){
          drawVSplit(split);
          }
        flags|=FLAG_PRESSED;
        flags&=~FLAG_UPDATE;
        }
      }
    else{
      window=findHSplit(ev->win_x);
      if(window){
        if(options&SPLITTER_REVERSED)
          split=window->getX();
        else
          split=window->getX()+window->getWidth();
        offset=ev->win_x-split;
        if(!(options&SPLITTER_TRACKING)){
          drawHSplit(split);
          }
        flags|=FLAG_PRESSED;
        flags&=~FLAG_UPDATE;
        }
      }
    return 1;
    }
  return 0;
  }
  

// Button being released
long FXSplitter::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXuint flgs=flags;
  if(isEnabled()){
    ungrab();
    flags|=FLAG_UPDATE;
    flags&=~FLAG_CHANGED;
    flags&=~FLAG_PRESSED;
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(flgs&FLAG_PRESSED){
      if(!(options&SPLITTER_TRACKING)){
        if(options&SPLITTER_VERTICAL){
          drawVSplit(split);
          adjustVLayout();
          }
        else{
          drawHSplit(split);
          adjustHLayout();
          }
        if(flgs&FLAG_CHANGED){
          if(target) target->handle(this,MKUINT(message,SEL_CHANGED),NULL);
          }
        }
      if(flgs&FLAG_CHANGED){
        if(target) target->handle(this,MKUINT(message,SEL_COMMAND),NULL);
        }
      }
    return 1;
    }
  return 0;
  }


// Button being released
long FXSplitter::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* ev=(FXEvent*)ptr;
  FXint oldsplit;
  if(flags&FLAG_PRESSED){
    oldsplit=split;
    if(options&SPLITTER_VERTICAL){
      moveVSplit(ev->win_y-offset);
      if(split!=oldsplit){
        if(!(options&SPLITTER_TRACKING)){
          drawVSplit(oldsplit);
          drawVSplit(split);
          }
        else{
          adjustVLayout();
          if(target) target->handle(this,MKUINT(message,SEL_CHANGED),NULL);
          }
        flags|=FLAG_CHANGED;
        }
      }
    else{
      moveHSplit(ev->win_x-offset);
      if(split!=oldsplit){
        if(!(options&SPLITTER_TRACKING)){
          drawHSplit(oldsplit);
          drawHSplit(split);
          }
        else{
          adjustHLayout();
          if(target) target->handle(this,MKUINT(message,SEL_CHANGED),NULL);
          }
        flags|=FLAG_CHANGED;
        }
      }
    return 1;
    }
  return 0;
  }


// Focus moved to next
long FXSplitter::onFocusNext(FXObject* sender,FXSelector sel,void* ptr){
  return (options&SPLITTER_VERTICAL) ? onFocusDown(sender,sel,ptr) : onFocusRight(sender,sel,ptr);
  }


// Focus moved to previous
long FXSplitter::onFocusPrev(FXObject* sender,FXSelector sel,void* ptr){
  return (options&SPLITTER_VERTICAL) ? onFocusUp(sender,sel,ptr) : onFocusLeft(sender,sel,ptr);
  }


// Focus moved up
long FXSplitter::onFocusUp(FXObject*,FXSelector sel,void* ptr){
  FXWindow *child;
  if(options&SPLITTER_VERTICAL){
    if(getFocus())
      child=getFocus()->getPrev();
    else
      child=getLast();
    while(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)) return 1;
      child=child->getPrev();
      }
    }
  return 0;
  }


// Focus moved down
long FXSplitter::onFocusDown(FXObject*,FXSelector sel,void* ptr){
  FXWindow *child;
  if(options&SPLITTER_VERTICAL){
    if(getFocus())
      child=getFocus()->getNext();
    else
      child=getFirst();
    while(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)) return 1;
      child=child->getNext();
      }
    }
  return 0;
  }


// Focus moved to left
long FXSplitter::onFocusLeft(FXObject*,FXSelector sel,void* ptr){
  FXWindow *child;
  if(!(options&SPLITTER_VERTICAL)){
    if(getFocus())
      child=getFocus()->getPrev();
    else
      child=getLast();
    while(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)) return 1;
      child=child->getPrev();
      }
    }
  return 0;
  }


// Focus moved to right
long FXSplitter::onFocusRight(FXObject*,FXSelector sel,void* ptr){
  FXWindow *child;
  if(!(options&SPLITTER_VERTICAL)){
    if(getFocus())
      child=getFocus()->getNext();
    else
      child=getFirst();
    while(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)) return 1;
      child=child->getNext();
      }
    }
  return 0;
  }


// Draw the horizontal split
void FXSplitter::drawHSplit(FXint pos){
  FXDCWindow dc(this);
  dc.clipChildren(FALSE);
  dc.setFunction(BLT_NOT_DST);     // Does this always show up?
  dc.fillRectangle(pos,0,barsize,height);
  }


// Draw the vertical split
void FXSplitter::drawVSplit(FXint pos){
  FXDCWindow dc(this);
  dc.clipChildren(FALSE);
  dc.setFunction(BLT_NOT_DST);     // Does this always show up?
  dc.fillRectangle(0,pos,width,barsize);
  }


// Return splitter style
FXuint FXSplitter::getSplitterStyle() const {
  return (options&SPLITTER_MASK);
  }


// Set horizontal or vertical
void FXSplitter::setSplitterStyle(FXuint style){
  FXuint opts=(options&~SPLITTER_MASK) | (style&SPLITTER_MASK);
  if(options!=opts){

    // Split direction changed; need re-layout of everything
    if((opts&SPLITTER_VERTICAL)!=(options&SPLITTER_VERTICAL)){
      for(FXWindow *child=getFirst(); child; child=child->getNext()){
        if(child->shown()){
          child->setWidth(0);
          child->setHeight(0);
          }
        }
      setDefaultCursor((opts&SPLITTER_VERTICAL) ? getApp()->vsplitCursor : getApp()->hsplitCursor);
      setDragCursor(getDefaultCursor());
      recalc();
      }

    // Split mode reversal; re-layout first and last only
    if((opts&SPLITTER_REVERSED)!=(options&SPLITTER_REVERSED)){
      if(getFirst()){
        getFirst()->setWidth(0);
        getFirst()->setHeight(0);
        getLast()->setWidth(0);
        getLast()->setHeight(0);
        }
      recalc();
      }
    options=opts;
    }
  }


// Change bar size
void FXSplitter::setBarSize(FXint bs){
  if(bs!=barsize){
    barsize=bs;
    recalc();
    }
  }


// Save object to stream
void FXSplitter::save(FXStream& store) const {
  FXComposite::save(store);
  store << barsize;
  }


// Load object from stream
void FXSplitter::load(FXStream& store){
  FXComposite::load(store);
  store >> barsize;
  }  


// Zap it
FXSplitter::~FXSplitter(){
  window=(FXWindow*)-1;
  }
