%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id$
%%

-module(pman_buf_utils).

%%-compile(export_all).
-export([textformat/1, split/4]).


%% ---------------------------------------------------------------
%% Funtion which format the trace message
%% ---------------------------------------------------------------

textformat(died) ->
    "\n\nProcess died\n";
textformat({died, Pid}) ->
    io_lib:format("~w Process died.~n",[Pid]);
textformat({shell_died, Old, New}) ->
    io_lib:format("~w Shell Process died. Restarted as ~w~n~n",[Old,New]);


textformat(to_buffer) ->
    "\nAppending trace log to Buffer\n\n";
textformat(to_file) ->
    "\nAppending trace log to File\n\n";
textformat(cut_buffer) ->
    "\nCUT BUFFER\n\n";
textformat({trace, From, 'receive', Msg}) ->
    io_lib:format("~w: rec   ~s~n", [From,
				       tuple_space(Msg)]);
textformat({trace, From, send, Msg, To}) ->
    io_lib:format("~w:  !    To: ~w Msg: ~s~n", [From,
						   To,
						   tuple_space(Msg)]);
textformat({trace, From, call, Func}) ->
    io_lib:format("~w: call  ~s~n",[From, ffunc(Func)]);
textformat({trace, From, spawn, Data}) ->
    io_lib:format("~w: spawn ~p~n", [From, Data]);
textformat({trace, From, link, Data}) ->
    io_lib:format("~w: link  ~p~n", [From,  Data]);
textformat({trace, From, unlink, Data}) ->
    io_lib:format("~w: U-lnk ~p~n", [From,  Data]);

textformat({trace, From, Op, Data}) ->
    io_lib:format("~w: ~w   ~p~n", [From, Op, Data]);

textformat({print, Format, Args}) ->
    io_lib:format(Format, Args);
textformat(Other) ->
    io_lib:format("~p~n",[Other]).
    




ffunc({M,F, Argl}) ->
    io_lib:format("~w:~w(~s)", [M, F, fargs(Argl)]);
ffunc(X) -> tuple_space(X).
fargs([]) -> [];
fargs([A]) -> tuple_space(A);  %% last arg
fargs([A|Args]) -> [tuple_space(A),", "|fargs(Args)].


tuple_space(X) when tuple(X) -> print(size(X), X, "}");
tuple_space(X)               -> io_lib:format("~p",[X]).

print(0  , X, Buff) -> ["{"|Buff];
print(1  , X, Buff) -> 
    Str =  tuple_space(element(1, X)),
    ["{",Str|Buff];
print(Num, X, Buff) ->
    Str =  tuple_space(element(Num, X)),
    print(Num-1, X, [", ",Str|Buff]).



%% ----------------------------------------------------------------
%% splits the list at element Size, returns Size, and the 2 lists
%% If the list is not long enough, it returns {size(List),[],List}


split([],_,Length,Buff) ->
    {Length,[],lists:reverse(Buff)};
split(Rest,0,Length,Buff) ->
    {Length,Rest,lists:reverse(Buff)};
split([L|List],Size,Length,Buff) ->
    split(List,Size-1,Length+1,[L|Buff]).



