// ---------------------------------------------------------------------------
// - cterm.cxx                                                               -
// - standard system library : c terminal function implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cio.hxx"
#include "cterm.hxx"
#include "cstring.hxx"
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <term.h>

namespace aleph {

  // check if the stream id is a terminal

  bool c_istty (const int sid) {
    return (isatty (sid) == 1) ? true : false;
  }

  // get the terminal attributes

  void* c_gtattr (const int sid) {
    struct termios* tattr = new struct termios;

    // check for terminal first
    if (c_istty (sid) == false) return 0;
    // get the terminal attributes
    if (tcgetattr (sid, tattr) != 0) return 0;
    return tattr;
  }

  // free the terminal attributes structure

  void c_ftattr (void* ptr) {
    struct termios* tattr = (struct termios*) ptr;
    delete tattr;
    TERMINAL* term = set_curterm (NULL);
    del_curterm (term);
  }

  // set the terminal attributes

  void c_stattr (const int sid, void* tattr) {
    // check for structure and tty
    if (tattr == 0) return;
    if (c_istty (sid) == false) return;
 
    // set terminal attributes
    tcsetattr (sid, TCSANOW, (struct termios*) tattr);
  }

  // reset terminal mode - put in non canonical mode

  bool c_stcanon (const int sid) {
    // check for terminal first
    if (c_istty (sid) == false) return false;

    // reset canonical mode and echo
    struct termios tattr;

    tcgetattr (STDIN_FILENO, &tattr);
    tattr.c_lflag    &= ~(ICANON|ECHO);
    tattr.c_cc[VMIN]  = 1;
    tattr.c_cc[VTIME] = 0;
    if (tcsetattr (STDIN_FILENO, TCSAFLUSH, &tattr) != 0) return false;
    return true;
  }

  // this function checks if a string is bounded in the tinfo array
  static bool check_tinfo (char** tinfo, long size, char* name) {
    for (long i = 0; i < size; i++) 
      if (c_strcmp (tinfo[i],name) == true) return true;
    return false;
  }

  // these functions fix some inconsistencies in the terminfo/termcap world
  // this is only coming from various experiments
  static char** fix_tinfo_input (char** tinfo) {
    if (check_tinfo (tinfo,ITERM_MAX,XTERM_ARROW_UP) == false) 
      tinfo[ITERM_STD_UP] = c_strdup (XTERM_ARROW_UP);
    if (check_tinfo (tinfo,ITERM_MAX,XTERM_ARROW_DOWN) == false) 
      tinfo[ITERM_STD_DOWN] = c_strdup (XTERM_ARROW_DOWN);
    if (check_tinfo (tinfo,ITERM_MAX,XTERM_ARROW_RIGHT) == false) 
      tinfo[ITERM_STD_RIGHT] = c_strdup (XTERM_ARROW_RIGHT);
    if (check_tinfo (tinfo,ITERM_MAX,XTERM_ARROW_LEFT) == false) 
      tinfo[ITERM_STD_LEFT ] = c_strdup (XTERM_ARROW_LEFT);
    if (check_tinfo (tinfo,ITERM_MAX,XTERM_DELETE_KEY) == false) 
      tinfo[ITERM_STD_DELETE] = c_strdup (XTERM_DELETE_KEY);
    if (check_tinfo (tinfo,ITERM_MAX,XTERM_INSERT_KEY) == false) 
      tinfo[ITERM_STD_INSERT] = c_strdup (XTERM_INSERT_KEY);
    // and hopefully, it is working
    return tinfo;
  }

  static char** fix_tinfo_output (char** tinfo) {
    // check for cub1 - if not set we use backspace
    if (tinfo[OTERM_MOVE_LEFT] == 0)
      tinfo[OTERM_MOVE_LEFT] = c_strdup ("\b");
    // check for cuf1 - if not set we use the standard right arrow
    if (tinfo[OTERM_MOVE_RIGHT] == 0)
      tinfo[OTERM_MOVE_RIGHT] = c_strdup (XTERM_ARROW_RIGHT);

  // hopefully, we are set
    return tinfo;
  }

  // return an array of terminal capabilities

  char** c_rtinfo (bool imode) {
    int status = 0;
    // read terminfo database
    if (setupterm (getenv ("TERM"),STDOUT_FILENO,&status) != OK) return 0;

    // create an array of capabilities and initialize it
    long len = imode ? ITERM_MAX : OTERM_MAX;
    char** result = (char**) malloc (len * sizeof (char*));
    for (int i = 0; i < len; i++) result[i] = 0;

    // query input capabilities
    if (imode == true) {
      result[ITERM_BACKSPACE]   = c_strdup (tigetstr ("kbs"));
      result[ITERM_DELETE]      = c_strdup (tigetstr ("kdch1"));
      result[ITERM_ARROW_UP]    = c_strdup (tigetstr ("kcuu1"));
      result[ITERM_ARROW_DOWN]  = c_strdup (tigetstr ("kcud1"));
      result[ITERM_ARROW_LEFT]  = c_strdup (tigetstr ("kcub1"));
      result[ITERM_ARROW_RIGHT] = c_strdup (tigetstr ("kcuf1"));
      result[ITERM_INSERT_KEY]  = c_strdup (tigetstr ("kich1"));
      result[ITERM_STD_UP]      = 0;
      result[ITERM_STD_DOWN]    = 0;
      result[ITERM_STD_RIGHT]   = 0;
      result[ITERM_STD_LEFT]    = 0;
      result[ITERM_STD_DELETE]  = 0;
      result[ITERM_STD_INSERT]  = 0;
    } else {
      result[OTERM_DELETE_CHAR] = c_strdup (tigetstr ("dch1"));
      result[OTERM_MOVE_LEFT]   = c_strdup (tigetstr ("cub1"));
      result[OTERM_MOVE_RIGHT]  = c_strdup (tigetstr ("cuf1"));
      result[OTERM_INSERT_CHAR] = c_strdup (tigetstr ("ich1"));
      result[OTERM_IMODE_START] = c_strdup (tigetstr ("smir"));
      result[OTERM_IMODE_END]   = c_strdup (tigetstr ("rmir"));
    }
    // here is our array
    return imode ? fix_tinfo_input (result) : fix_tinfo_output (result);
  }

  // send a character to the standard output

  void c_tparm (const int sid, char** tinfo, const long index) {
    if ((tinfo == 0) || (index >= OTERM_MAX)) return;
    char* data = tinfo[index];
    if (data == 0) return;
    char* buffer = tparm (data);
    long  length = c_strlen (buffer);
    c_write (sid,buffer,length);
  }
}
