// ---------------------------------------------------------------------------
// - NameTable.cpp                                                           -
// - standard object library - name table class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "NameTable.hpp"
#include "Exception.hpp"

namespace aleph {
  
  // the name table bucket
  struct s_bucket {
    // the object name
    String d_name;
    // the object 
    Object* p_object;
    // next record in the list
    s_bucket* p_next;
    // simple constructor
    s_bucket (void) {
      p_object = nilp;
      p_next   = nilp;
    }
    // simple destructor
    ~s_bucket (void) {
      Object::dref (p_object);
      delete p_next;
    }
  };
  
  // find a bucket by name given its root bucket
  static inline s_bucket* getbucket (s_bucket* bucket, const String& name) {
    // simple check as fast as we can
    if (bucket == nilp) return nilp;
    // loop until we have a match
    while (bucket != nilp) {
      if (bucket->d_name == name) return bucket;
      bucket = bucket->p_next;
    }
    // no bucket found
    return nilp;
  }
  
  // extract a bucket by name given its root bucket . This procedure remove the
  // bucket if it is found and maintain the link list.
  static inline s_bucket* rmbucket (s_bucket** root, const String& name) {
    s_bucket* bucket = *root;
    // simple check as fast as we can
    if (bucket == nilp) return nilp;
    // first case for the root bucket
    if (bucket->d_name == name) {
      *root = bucket->p_next;
      bucket->p_next = nilp;
      return bucket;
    }
    // loop until we have a match
    while (bucket->p_next != nilp) {
      if (bucket->p_next->d_name == name) {
	s_bucket* result = bucket->p_next;
	bucket->p_next = result->p_next;
	result->p_next = nilp;
	return result;
      }
      bucket = bucket->p_next;
    } 
    // no node found
    return nilp;
  }
  
  // Create a new name table
  
  NameTable::NameTable (void) {
    p_table = nilp;
  }
  
  // delete this name table but not the objects, norr the parent
  
  NameTable::~NameTable (void) {
    delete p_table;
  }

  // return the class name

  String NameTable::repr (void) const {
    return "NameTable";
  }

  // set or create an object in this table
  
  void NameTable::add (const String& name, Object* object) {
    // protect the object
    Object::iref (object);
    // look for existing symbol
    s_bucket* bucket = getbucket (p_table,name);
    if (bucket != nilp) {
      Object::dref (bucket->p_object);
      bucket->p_object = object;
      return;
    }
    // the bucket does not exist, create it 
    bucket           = new s_bucket;
    bucket->d_name   = name;
    bucket->p_object = object;
    bucket->p_next   = p_table;
    p_table          = bucket;
  }
  
  // get an object by name. If the name is not found, nilp is returned
  
  Object* NameTable::get (const String& name) const {
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table,name);
    if (bucket != nilp) return bucket->p_object;;
    return nilp;
  }

  // get an object by name. If the name is not found an exception is raised

  Object* NameTable::lookup (const String& name) const {
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table,name);
    if (bucket != nilp) return bucket->p_object;;
    throw Exception ("name-error", "name not found", name);
  }
  
  // return true if a name exists in this table

  bool NameTable::exists (const String& name) const {
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table,name);
    if (bucket != nilp) return true;
    return false;
  }
  
  // remove an object by name. 
  
  void NameTable::remove (const String& name) {
    // extract the bucket
    s_bucket* bucket = rmbucket (&p_table,name);
    delete bucket;
  }
  
  // return a vector of names defined in this name table
  
  Strvec NameTable::names (void) const {
    Strvec result;
    s_bucket* bucket = p_table;
    // loop through the list
    while (bucket != nilp) {
      result.add (bucket->d_name);
      bucket = bucket->p_next;
    }
    return result;
  }
}
