// ---------------------------------------------------------------------------
// - Qualified.hpp                                                           -
// - aleph engine - qualified name class definition                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_QUALIFIED_HPP
#define  ALEPH_QUALIFIED_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

#ifndef  ALEPH_STRVEC_HPP
#include "Strvec.hpp"
#endif

namespace aleph {

  /// The qualified class is similiar to the lexical one, except that it 
  /// defines an object with a path. Each component of the qualified name is
  /// a lexical name. During construction, the qualified name is checked for
  /// adherence to this rule.
  /// @author amaury darsch

  class Qualified : public Literal {
  private:
    /// the qualified name
    String d_name;
    /// the qualified path
    Strvec d_path;
    /// the line number
    long   d_lnum;

  public:
    /// create an empty qualified
    Qualified (void);

    /// create a new qualified with a name and a line number
    /// @param name the qualified name
    /// @param line the line number
    Qualified (const String& name, const long lnum);

    /// copy constructor for this qualified
    /// @param that the qualified to copy
    Qualified (const Qualified& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this qualified
    inline String toLiteral (void) const {
      return d_name;
    }

    /// @return a string representation of this qualified
    inline String toString (void) const {
      return d_name;
    }

    /// @return the length of the qualified name path
    inline long length (void) const {
      return d_path.length ();
    }

    /// @return a partial lexical name by index
    inline String getname (const long index) const {
      return d_path.get (index);
    }

    /// @return the qualified line number
    inline long getlnum (void) const {
      return d_lnum;
    }

    /// set this qualified name as a const symbol
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* cdef (Interp* interp, Nameset* nset, Object* object);

    /// set this qualified name with an object
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate this qualified name in the current nameset
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    Object* eval (Interp* interp, Nameset* nset);
  
  private:
    // make the assignment operator private
    Qualified& operator = (const Qualified&);
  };
}

#endif
