// ---------------------------------------------------------------------------
// - Interp.hpp                                                              -
// - aleph engine - interpreter class definition                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_INTERP_HPP
#define  ALEPH_INTERP_HPP

#ifndef  ALEPH_STRVEC_HPP
#include "Strvec.hpp"
#endif

#ifndef  ALEPH_TERMINAL_HPP
#include "Terminal.hpp"
#endif

#ifndef  ALEPH_GLOBALSET_HPP
#include "Globalset.hpp"
#endif

#ifndef  ALEPH_STACK_HPP
#include "Stack.hpp"
#endif

namespace aleph {

  /// The Interp class is the main aleph interpreter engine. The interpreter
  /// operates on three streams, the standard input, the standard output and
  /// the standard error stream. By default, the interpreter install a 
  /// terminal object, unless the three streams are specified at construction.
  /// The interpreter holds a series of registers. For each basic types,
  /// like integer, real, character and boolean, the interpreter has 256 
  /// local thread registers. The operations associated with the registers
  /// are type dependant. For integer and real, these are mostly arithmetic 
  /// operations. For boolean, these are logical operations. The interpreter 
  /// has a top level (or global) nameset. The global nameset is shared 
  /// between all interpreters and is referenced as '...'. On top of this, 
  /// the interpreter provides a 'compile' method which takes a form and 
  /// generates a new one. The sole purpose of the compile method is to
  /// perform static checking and generate a new form ready for execution.
  /// @author amaury darsch

  class Interp : public Object {
  public:
    /// maximum number of register
    static const int MAX_REGS = 256;

  private:
    /// the default terminal
    Terminal* p_term;
    /// the standard input stream
    Input* p_is;
    /// the standard output stream
    Output* p_os;
    /// the standard error stream
    Output* p_es;
    /// the assert flag
    bool d_assert;
    /// the local integer registers
    t_long* d_li;
    /// the local real registers
    double* d_lr;
    /// the local character registers
    char* d_lc;
    /// the local boolean registers
    bool* d_lb;
    /// the interpreter global nameset
    Globalset* p_gset;
    /// the execution stack
    Stack* p_stk;
    /// the vector arguments
    Vector* p_argv;
    /// the vector of shared libraries
    Vector* p_shlib;

  public:
    /// create a default interpreter
    Interp (void);

    /// create a new interpreter
    /// @param is the standard input stream
    /// @param os the standard output stream
    /// @param es the standard error stream
    Interp (Input* is, Output* os, Output* es);

    /// destroy this interpreter
    ~Interp (void);

    /// @return the class name
    String repr (void) const;

    /// @return the interpreter input stream
    inline Input* getis (void) const {
      return p_is;
    }

    /// @return the interpreter output stream
    inline Output* getos (void) const {
      return p_os;
    }

    /// @return the interpreter error stream
    inline Output* getes (void) const {
      return p_es;
    }

    /// set the interpreter arguments
    /// @param args the arguments to set
    void setargs (const Strvec& args);

    /// set the assert flag
    /// @param flag the flag to set
    void setassert (const bool flag);

    // create a new reserved name in the global nameset
    /// @param name the reserved name
    /// @param object the object to bind
    void mkrsv (const String& name, Object* object);

    /// create a new nameset in reference to a parent one
    /// @param name the nameset name to create
    /// @param parent the parent nameset
    Nameset* mknset (const String& name, Nameset* parent);

    /// open a new dynamic library by name
    /// @param name the library name
    /// @param args the arguments for initialization
    Object* library (const String& name, Cons* args);

    /// @return the assert flag
    inline bool getassert (void) const {
      return d_assert;
    }

    /// @return the interpreter stack
    inline Stack* getstk (void) const {
      return p_stk;
    }

    /// @return the interpreter root nameset
    inline Nameset* getnset (void) const {
      return p_gset;
    }

    /// run the read-eval loop on the standard streams
    /// @return false if something bad happen
    bool loop (void);

    /// run the read-eval loop with a file
    /// @param fname the file name to read
    /// @return false if something bad happen
    bool loop (const String& fname);

    /// evaluate an interpreter method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this interpreter with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the copy constructor private
    Interp (const Interp&);
    // make the assignment operator private
    Interp& operator = (const Interp&);
  };
}

#endif
