/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <time.h>

#include "TFFileRC.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"
#include "flexarray.h"
#include "terraform.h"

#define BUFSIZE 128
#define SBUFSIZE 8


// defined in TFOptionsTable.h
extern void hash_printer (gpointer key, gpointer data, gpointer user_data);
extern void hash_key_printer (gpointer key, gpointer data, gpointer user_data);

/*
 *  constructor: give everyting a meaningful default value
 */
TFFileRC::TFFileRC (const char *fName, TFOptionsTable *optTbl)
	      : FileIO (fName)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ TFFileRC\n");

	p_optionsTable = optTbl;
	d_lines = d_items = 0;
}


/*
 *  destructor: clean up
 */
TFFileRC::~TFFileRC ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- TFFileRC\n");
}


/*
 *  parseRCFile: open the RC file and parse it. This method exits if an 
 * 	error is encountered. 
 */
TFOptionsTable *TFFileRC::parseRCFile ()
{
	char		buf[BUFSIZE];
	int		rc=0, l;

	if (!p_optionsTable)
		p_optionsTable = new TFOptionsTable ();
	this->open ("r");

	sprintf (buf, "Parsing RC file %s\n", this->getName());
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	if (!this->getFilePtr())
		{
		this->close ();
		return NULL;
		}

	while (!feof (this->getFilePtr()) && !rc && d_lines < TF_MAX_RC_LINES)
		{
		fgets (buf, BUFSIZE, this->getFilePtr());
		if (feof (this->getFilePtr()))
			break;

		l = strlen (buf);
		buf[l-1] = '\0';		// erase newline 
		d_lines++;
		if (buf[0] == '#' || buf[0] == ';' || !strlen(buf))
			rc = 0;
		else
			rc = parseOption (buf);
		}

	this->close ();
	if (rc)
		{
		fprintf (stderr, _("Syntax error in %s on line %d ...\n"),
			this->getName(), d_lines);
		fprintf (stderr, _("Line reads: [%s]\n"), buf);
		fprintf (stderr, _("Exiting ...\n"));
		exit (1);
		}

	//if (GlobalTrace::isSet (GlobalTrace::TRACE_DEBUG))
	//	p_optionsTable->foreach (hash_printer, stdout);

	return (p_optionsTable);
}



/*
 *  parseOption: parse options which from input stream/buffer
 */
int TFFileRC::parseOption (char *buf)
{
	char 	*equal = NULL,
		left[BUFSIZE],
		right[BUFSIZE];

	equal=strchr (buf, '=');
	if (!equal)
		{
		return 1; 
		}
	else
		{
		strncpy (left, buf, (int)((equal)-buf));
		left[equal-buf] = '\0';
		strncpy (right, equal+1, strlen(buf)-((equal-1)-buf));
		d_items++;
		//printf ("inserting [%s], [%s]\n", left, right);
		if (p_optionsTable->setKeyVal (strdup(left), strdup(right)))
			{
			fprintf (stderr, "Unknown token: [%s]\n", left);
			return 1;
			}
		else
			return 0;
		}
}



/*
 * writeRCFile: write the current setting to the user's RC file
 */
void TFFileRC::writeRCFile (TFOptionsTable *optTbl)
{
	char	*home,
		buf[80];
	FileIO	*fio;
	FILE	*fp;
	time_t	now;

	SanityCheck::bailout ((!optTbl), "optTbl==NULL", "TFFileRC::writeRCFile");

	home = getenv ("HOME");
	SanityCheck::warning((!home), "can't determine $HOME ... RC file save cancelled", "TFFileRC::writeRCFile");
	if (!home)
		return;

	sprintf (buf, "%s/%s", home, TF_LOCAL_RC_FILE);
	fio = new FileIO (buf);
	if (!fio->open ("w"))
		{
		delete fio;
		return;
		}

	if (!(fp = fio->getFilePtr()))
		{
		delete fio;	
		return;
		}

	now = time (NULL);
	fprintf (fp, "#\n# Terraform resource file created %s#\n\n", 
		 ctime(static_cast<const time_t*>(&now))); 

	FlexArray       *flArr = optTbl->collectOptions ();

	for (int i=0; i<flArr->getSize(); i++)
 		{
 		char *k = static_cast<char*>(flArr->El(i));
 		char *v = static_cast<char*>(optTbl->lookup(k));
 		hash_printer (k, v, fp);
 		}

	fio->close();
	delete fio;
}


