/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Uses:      GTK+ 1.2                                                       *
 * Modul:     file.c                                                         *
 *            GTK load/save files for paul                                   *
 * Author:    Andreas Tille                                                  *
 * Date:      14.06.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdlib.h>
#include <unistd.h>

#include "paul.h"
#include "callback.h"
#include "names.h"

static void FileLoadOk(GtkWidget *widget, GtkFileSelection *fs)
/*
 * --- Parameter: ---
 * GtkWidget        *widget : OK-button inside GtkFileSelection widget (can be ignored)
 * GtkFileSelection *fs     : GtkFileSelection widget
 *                            connected data: PAUL *p
 */
{
  PAUL      *p;
  PICTURE   *bild;
  char      *filename;

  g_return_if_fail ( GTK_IS_FILE_SELECTION(fs) );
  g_return_if_fail ( (p = gtk_object_get_user_data(GTK_OBJECT(fs)) ) );
  g_return_if_fail ( IS_PAUL(p) );

  if ( p->spiclist ) {
    g_warning(_("Unable to load new files while current operation is not finished."));
    gtk_widget_destroy(GTK_WIDGET(fs));
    return;
  }
  
  filename = ChdirToLoadDir(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));

  bild = ReadPic(filename, p->opt);
  gtk_widget_destroy(GTK_WIDGET(fs));
  if ( !bild ) {
    FileLoad(p);
    return;
  }
   
  AddImageToFilelist(p, bild);
}

static void FileSaveOk(GtkWidget *widget, GtkFileSelection *fs)
/*
 * --- Parameter: ---
 * GtkWidget        *widget : OK-button inside GtkFileSelection widget (can be ignored)
 * GtkFileSelection *fs     : fileselection widget, PAUL *p set as user_data
 */
{
  PAUL    *p;
  PICTURE *bild;

  g_return_if_fail ( GTK_IS_FILE_SELECTION(fs) ) ;
  g_return_if_fail ( (p = gtk_object_get_user_data(GTK_OBJECT(fs)) ) ) ;
  g_return_if_fail ( IS_PAUL(p) ) ;
  g_return_if_fail ( (bild = BILD(p->activ)) );

  if ( FilenameExtractFromSelection(fs, bild) ) {
    gtk_clist_set_text(GTK_CLIST(p->filelist), ACT_ROW(p), 0, bild->file);
    gdk_window_set_title(p->show->window, bild->file);
  }
  FileSave(p);
}

void FileLoad(PAUL *p)
/*
 * --- Parameter: ---
 * PAUL      *p     : PAUL structure
 */
{
  char    *filename = NULL;
  PICTURE *bild;
   
  g_return_if_fail ( IS_PAUL(p) ) ;
  
  bild = BILD(g_list_last(p->piclist));
  if ( bild ) g_return_if_fail ( (filename = ImgFileName(bild)) );

  FileSelectionLoadSave(p, GTK_SIGNAL_FUNC(FileLoadOk), filename, Load_image, NULL);
  FREE(filename);
}

static PICTURE *FileSaveCheckValid(PAUL *p)
/* Check if saving could be performed
 */
{
  g_return_val_if_fail ( IS_PAUL(p), NULL );
  
  if ( !p->piclist ) {
    g_warning(_("Sorry, there are no images to save."));
    return NULL;
  }

  return BILD(p->activ);
}
		   
void FileSave(PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *widget: menu entry for "Save" file (can be ignored)
 * PAUL      *p     : PAUL structure
 */
{
  register PICTURE *bild; 
  GList            *pl;

  g_return_if_fail ( (bild = FileSaveCheckValid(p)) );
  
  if ( !bild->flag ) {
    g_warning(_("No changes in %s need to be saved."), bild->file);
    return;
  }
  pl = g_list_append(NULL, bild);
  SavePictures(pl, SAFEFLAG);
  G_LIST_FREE(pl);
  bild->flag = 0;   /* to recognize further changes */
}
		   
void FilesSaveAll(PAUL *p)
/* save all files in list, which has changes in flag set
 * --- Parameter: ---
 * GtkWidget *widget: menu entry for "Save" file (can be ignored)
 * PAUL      *p     : PAUL structure
 */
{
  PICTURE *bild; 
  GList   *pl, *sl = NULL;

  g_return_if_fail ( FileSaveCheckValid(p) );

  for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) 
    if ( bild->flag ) sl = g_list_append(sl, bild);
  
  SavePictures(sl, SAFEFLAG);

  for ( bild = BILD(pl = sl); pl; bild = BILD(pl = pl->next) ) 
    bild->flag = 0;   /* to recognize further changes */

  G_LIST_FREE(sl);
}
		   
void FileSaveAs(PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *widget: menu entry for "Save AS" file (can be ignored)
 * PAUL      *p     : PAUL structure
 */
{
  g_return_if_fail ( IS_PAUL(p) );

  BILD(p->activ)->flag |= SAFEFLAG;
  FileSelectionLoadSave(p, GTK_SIGNAL_FUNC(FileSaveOk), BILD(p->activ)->file, Save_img_as, NULL);
}

static void FirstPictureLoaded(GtkWidget *widget, GtkFileSelection *fs)
/*
 * --- Parameter: ---
 * GtkWidget        *widget : OK-button inside GtkFileSelection widget (can be ignored)
 * GtkFileSelection *fs     : GtkFileSelection widget
 *                            connected data: PAUL *p
 */
{
  PAUL      *p;
  PICTURE   *bild;
  char      *filename;

  g_return_if_fail ( widget );
  g_return_if_fail ( GTK_IS_WIDGET(widget) );
  g_return_if_fail ( fs );
  g_return_if_fail ( GTK_IS_WIDGET(fs) );
  g_return_if_fail ( (p = gtk_object_get_user_data(GTK_OBJECT(fs))) );
  g_return_if_fail ( IS_PAUL(p) ) ;
  g_return_if_fail ( p->piclist == NULL ) ;
  
  filename = ChdirToLoadDir(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
  g_return_if_fail ( filename ) ;
      
  bild = ReadPic(filename, p->opt);
  gtk_widget_destroy(GTK_WIDGET(fs));
  if ( !bild ) {
    LoadFirstPicture(p);
    return;
  }
   
  p->piclist = g_list_append(NULL, bild);
  CreateShowPixmaps(p->activ = p->piclist);
  GtkInterface(p);
}

void LoadFirstPicture(PAUL *p)
{
  GtkWidget *window = NULL;

  g_return_if_fail ( IS_PAUL(p) );
  window = gtk_file_selection_new(Load_image);

  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_MOUSE);
  gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);

  gtk_object_set_user_data(GTK_OBJECT(window), p);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(window)->ok_button), "clicked", 
                     GTK_SIGNAL_FUNC(FirstPictureLoaded), window);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(window)->cancel_button), "clicked", 
                     GTK_SIGNAL_FUNC(DoPaulExit), p);
  
  gtk_widget_show_all(window);
}

static void SureDeleteFile(GtkWidget *button, char *file)
/*
 * --- Parameter: ---
 * GtkWidget *button: OK-button to delete file (window connected)
 * char      *file  : file to delete
 */
{
  if ( unlink(file) ) g_warning(_("Unable to delete %s"), file);
}

static void DeleteFromList(GtkWidget *button, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *button: Delete from list button
 * PAUL      *p     : PAUL structure
 */
{
  PICTURE *bild;
   
  g_return_if_fail ( IS_PAUL(p) );
  
  bild = BILD(p->activ);
  RemoveActivImageFromFilelist(p);
  FreeBild(bild);
}

static void DeleteFromDisk(GtkWidget *button, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *button: Delete from disk button
 * PAUL      *p     : PAUL structure
 */
{
  char    *file, *buf;
  PICTURE *bild;

  g_return_if_fail ( IS_PAUL(p) );
  g_return_if_fail ( IS_PICTURE(bild = BILD(p->activ)) );
  
  buf = NULL;
  if ( bild->dir ) {
    char *cwd;
      
    if ( strcmp((cwd = g_get_current_dir()), bild->dir) ) 
       buf = g_strdup_printf("%s/", bild->dir);
    FREE(cwd);
  }
   
  file = g_strdup_printf("%s%s%s%s", buf ? buf : "", bild->file, bild->ext ? "." : "", 
                                                                 bild->ext ? bild->ext : "");
  FREE(buf);

  if ( !CheckFile(file) ) return;
  
  buf = g_strdup_printf(Delete_f_d, file);
  AreYouSure(file, GTK_SIGNAL_FUNC(SureDeleteFile), buf);
  FREE(buf);
  DeleteFromList(button, p);
}

void FileDelete(PAUL *p)
/*
 * --- Parameter: ---
 * PAUL      *p     : PAUL structure
 */
{
  static    GtkWidget *window = NULL;
  GtkWidget           *main_vbox, /* what should this mean ?? :)        */
                      *button;    /* OK / Cancel                        */
  char                *buf;

  g_return_if_fail ( IS_PAUL(p) );

  if ( p->spiclist ) {
    g_warning(_("Unable to delete files while current operation is not finished."));
    return;
  }

  if ( !(p->activ) ) {
    g_warning(_("There are no images loaded which could be deleted"));
    return ;
  }

  if ( window ) {
     gtk_widget_destroy(window);
     return;
  }
  
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_MOUSE);
  gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);
  gtk_window_set_title(GTK_WINDOW(window), _("Delete image from list"));
  main_vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 10);
  gtk_container_add(GTK_CONTAINER(window), main_vbox);
     
  buf = g_strdup_printf(_("Delete %s from %s's image list?"), BILD(p->activ)->file, exename);
  button = gtk_button_new_with_label(buf);
  FREE(buf);
  gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(DeleteFromList), p);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                            GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(main_vbox), button, FALSE, TRUE, 5);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(button);
      
  buf = g_strdup_printf(Delete_f_d, BILD(p->activ)->file);
  button = gtk_button_new_with_label(buf);
  FREE(buf);
  gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(DeleteFromDisk), p);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                            GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(main_vbox), button, TRUE, TRUE, 5);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);

  CreateCancelButton(window, main_vbox);
    
  gtk_widget_show_all(window);
}
		   
