/* GtkDatabox - An extension to the gtk+ library
 * Copyright (C) 1998-1999 Roland Bock
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <gtk/gtk.h>
#include <gtkdatabox.h>
#include <math.h>
/* Some versions of math.h have a PI problem... */
#ifndef PI
#define PI 3.14159265358979323846
#endif

#define POINTS 2000

/*----------------------------------------------------------------
 *  databox basics
 *----------------------------------------------------------------*/

static void
create_basics(void)
{
  static GtkWidget *window = NULL;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *close_button;
  GtkWidget *box;
  GtkWidget *label;
  GtkWidget *separator;
  gint index;
  gfloat *X;
  gfloat *Y;
  GdkColor color;
  gint i;
  guint size;
  GtkDataboxDataType type;
  
  if (!window)
  {
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize (window, 300, 300);  

    gtk_signal_connect (GTK_OBJECT (window), "destroy",
			  GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			  &window);

    gtk_window_set_title (GTK_WINDOW (window), "Databox Basics");
    gtk_container_border_width (GTK_CONTAINER (window), 0);

    box1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), box1);

    label=gtk_label_new("Draw a selection with the left button pressed,\nThan click into the selection.\nUse the right mouse button to zoom out.\nShift+mouse button zooms to default.");
    gtk_box_pack_start (GTK_BOX (box1), label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);
  
    box=gtk_databox_new();
    gtk_signal_connect (GTK_OBJECT (box), "destroy",
			  GTK_SIGNAL_FUNC(gtk_databox_data_destroy_all),
			  NULL);
    
    X=g_new0(gfloat, POINTS);
    Y=g_new0(gfloat, POINTS);
    
    for (i = 0; i < POINTS; i++ ) {
      X[i]=i;
      Y[i]=100.*sin(i*2*PI/POINTS);
    }
    color.red=0;
    color.green=0;
    color.blue=0;
    
    index=gtk_databox_data_add_x_y(GTK_DATABOX(box), POINTS, 
    			X, Y, color, 
    			GTK_DATABOX_LINES, 
    			7);
    gtk_databox_get_data_type(GTK_DATABOX(box), index, &type, &size);
    printf("type: %d, size: %d\n", type, size);
    gtk_databox_set_data_type(GTK_DATABOX(box), index, GTK_DATABOX_POINTS, 1);
    gtk_databox_get_data_type(GTK_DATABOX(box), index, &type, &size);
    printf("type: %d, size: %d\n", type, size);

    Y=g_new0(gfloat, POINTS);
    
    for (i = 0; i < POINTS; i++ ) {
      Y[i]=100.*cos(i*2*PI/POINTS);
    }
    color.red=65535;
    color.green=0;
    color.blue=0;
    
    gtk_databox_data_add_y(GTK_DATABOX(box), POINTS, 
    			Y, index, color, 
    			GTK_DATABOX_LINES, 
    			5);
    			
    gtk_databox_rescale(GTK_DATABOX(box));
    gtk_box_pack_start (GTK_BOX (box1), box, TRUE, TRUE, 0);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);

    box2 = gtk_vbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_end (GTK_BOX (box1), box2, FALSE, TRUE, 0);
    close_button = gtk_button_new_with_label ("close");
    gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked",
			 GTK_SIGNAL_FUNC(gtk_widget_destroy),
			 GTK_OBJECT (window));
    gtk_box_pack_start (GTK_BOX (box2), close_button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (close_button);
  }
  
  if (!GTK_WIDGET_VISIBLE (window)) {
    gtk_widget_show_all (window);
  } else {
    gtk_widget_destroy (window);
  }
}

/*----------------------------------------------------------------
 *  databox colors and misc
 *----------------------------------------------------------------*/

static void menu_color_change_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget);
static void menu_options_show_coord_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget);
static void menu_options_show_rulers_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget);
static void menu_options_show_scrollbars_cb	(gpointer callback_data, guint callback_action, GtkWidget *widget);
static void menu_options_enable_zoom_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget);

static GtkItemFactoryEntry
menu_main_entries[]=
{
  {"/_Color", 				NULL, NULL, 		0, 	"<Branch>"},
  {"/Color/Change Color #0",		NULL, menu_color_change_cb, 	0, 	NULL},
  {"/Color/Change Color #1",		NULL, menu_color_change_cb,	1, 	NULL},
  {"/Color/Change Color #2",		NULL, menu_color_change_cb,	2, 	NULL},
  {"/Color/Change Color #3",		NULL, menu_color_change_cb,	3, 	NULL},
  {"/Color/Change Color #4",		NULL, menu_color_change_cb,	4, 	NULL},
  {"/_Options", 			NULL, NULL, 		0, 	"<Branch>"},
  {"/Options/Show coordinates",		NULL, menu_options_show_coord_cb,	1, 	"<RadioItem>"},
  {"/Options/Hide coordinates",		NULL, menu_options_show_coord_cb,	0, 	"/Options/Show coordinates"},
  {"/Options/Show rulers",		NULL, menu_options_show_rulers_cb,	1, 	"<RadioItem>"},
  {"/Options/Hide rulers",		NULL, menu_options_show_rulers_cb,	0, 	"/Options/Show rulers"},
  {"/Options/Show scrollbars",		NULL, menu_options_show_scrollbars_cb,	1, 	"<RadioItem>"},
  {"/Options/Hide scrollbars",		NULL, menu_options_show_scrollbars_cb,	0, 	"/Options/Show scrollbars"},
  {"/Options/Enable zoom",		NULL, menu_options_enable_zoom_cb,	1, 	"<RadioItem>"},
  {"/Options/Disable zoom",		NULL, menu_options_enable_zoom_cb,	0, 	"/Options/Enable zoom"},
  {"/_Help", 				NULL, NULL, 				0, 	"<LastBranch>"},
  {"/Help/Do not hesitate to ask\nrbock@eudoxos.de", 	NULL, NULL, 		0, 	NULL},
};

static gint
menu_main_num = sizeof (menu_main_entries) / sizeof (menu_main_entries[0]);

typedef struct {
  GtkWidget *selector;
  GtkWidget *box;
  gint index;
} col_sel;

static void 
get_color_cb(GtkWidget *widget, col_sel *sel)
{
  gdouble color[8];
  GdkColor gdkcolor;
  
  g_return_if_fail(GTK_IS_COLOR_SELECTION_DIALOG(sel->selector));
  g_return_if_fail(GTK_IS_DATABOX(sel->box));

  gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(sel->selector)->colorsel), color);
  gdkcolor.red=(gushort)(color[0]*65535);
  gdkcolor.green=(gushort)(color[1]*65535);
  gdkcolor.blue=(gushort)(color[2]*65535);
  
  printf("return value of gtk_databox_set_color: %d\n", gtk_databox_set_color(GTK_DATABOX(sel->box), sel->index, gdkcolor));
  gtk_databox_redraw(GTK_DATABOX(sel->box));
}

static void
menu_color_change_cb(gpointer callback_data, guint callback_action, GtkWidget *widget)
{
  GtkWidget *selector;
  gchar title[20];
  col_sel *sel=g_new0(col_sel, 1);
  GdkColor gdkcolor;
  gdouble color[8];
  
  sprintf(title, "Choose color #%d", callback_action);
  selector=gtk_color_selection_dialog_new(title);
  gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(selector)->help_button);
  sel->selector=selector;
  sel->box=GTK_WIDGET(callback_data);
  sel->index=callback_action;
  
  gtk_databox_get_color(GTK_DATABOX(sel->box), sel->index, &gdkcolor);
  color[0]=gdkcolor.red/65535.;
  color[1]=gdkcolor.green/65535.;
  color[2]=gdkcolor.blue/65535.;
  color[3]=1;		/* Opacity? Dunno how to use it. */
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(sel->selector)->colorsel), color);
  
  gtk_signal_connect_object(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(selector)->cancel_button), "clicked",
		 GTK_SIGNAL_FUNC(gtk_widget_destroy),
		 GTK_OBJECT(selector));
  gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(selector)->ok_button), "clicked",
		 GTK_SIGNAL_FUNC(get_color_cb),
		 (gpointer) sel);
  gtk_widget_show(selector);

  return;
}

static void menu_options_show_coord_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget)
{
  if (callback_action)
    gtk_databox_show_cross(GTK_DATABOX(callback_data));
  else
    gtk_databox_hide_cross(GTK_DATABOX(callback_data));
  gtk_databox_redraw(GTK_DATABOX(callback_data));
}

static void menu_options_show_rulers_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget)
{
  if (callback_action)
    gtk_databox_show_rulers(GTK_DATABOX(callback_data));
  else
    gtk_databox_hide_rulers(GTK_DATABOX(callback_data));
  gtk_databox_redraw(GTK_DATABOX(callback_data));
}

static void menu_options_show_scrollbars_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget)
{
  if (callback_action)
    gtk_databox_show_scrollbars(GTK_DATABOX(callback_data));
  else
    gtk_databox_hide_scrollbars(GTK_DATABOX(callback_data));
  gtk_databox_redraw(GTK_DATABOX(callback_data));
}

static void menu_options_enable_zoom_cb		(gpointer callback_data, guint callback_action, GtkWidget *widget)
{
  if (callback_action)
    gtk_databox_enable_zoom(GTK_DATABOX(callback_data));
  else
    gtk_databox_disable_zoom(GTK_DATABOX(callback_data));
  gtk_databox_redraw(GTK_DATABOX(callback_data));
}

static void
create_colors(void)
{
  static GtkWidget *window = NULL;
  GtkWidget *box1=NULL;
  GtkWidget *box2=NULL;
  GtkWidget *close_button=NULL;
  GtkWidget *box=NULL;
  GtkWidget *separator;
  GtkWidget *label;
  GtkWidget *menu_bar;
  GtkItemFactory *item_factory;
  gfloat *X=NULL;
  gfloat *Y=NULL;
  gint i, j;
  GdkColor color;
  gint index=0;
  gchar title[]="<Main>";
  
  if (!window)
  {
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize (window, 300, 300);  

    gtk_signal_connect (GTK_OBJECT (window), "destroy",
			  GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			  &window);

    gtk_window_set_title (GTK_WINDOW (window), "Colors & Misc");
    gtk_container_border_width (GTK_CONTAINER (window), 0);

    box1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), box1);

    box=gtk_databox_new ();
    item_factory=gtk_item_factory_new(GTK_TYPE_MENU_BAR, title, NULL);
    gtk_item_factory_create_items(item_factory, menu_main_num, menu_main_entries, box);
    menu_bar=gtk_item_factory_get_widget(item_factory, title);
    gtk_box_pack_start(GTK_BOX (box1), menu_bar, FALSE, TRUE, 0);

    label=gtk_label_new("You may change the colors of the shown datasets,\nthe drawing style and behaviour of\ncoordinate axes and/or scrollbars.");
    gtk_box_pack_start (GTK_BOX (box1), label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);
  
    gtk_signal_connect (GTK_OBJECT (box), "destroy",
			  GTK_SIGNAL_FUNC(gtk_databox_data_destroy_all),
			  NULL);
    
    
    for (i=0; i<5; i++) {
      if (!i) X=g_new0(gfloat, POINTS);
      Y=g_new0(gfloat, POINTS);
      for (j=0; j<POINTS; j++ ) {
        X[j]=j;
        Y[j]=100.*sin((i+1)*2*j*PI/POINTS);
      }
      color.red=65535*(i%2);
      color.green=(65535/2)*(i%3);
      color.blue=(65535/3)*(i%4);
      if (!i) index=gtk_databox_data_add_x_y(GTK_DATABOX(box), POINTS, 
    			X, Y, color, 
    			GTK_DATABOX_POINTS, 
    			0);
      else printf("index: %d\n", gtk_databox_data_add_y(GTK_DATABOX(box), POINTS, 
                        Y, index, color, 
                        GTK_DATABOX_POINTS, 
                        0));
    }
    
    gtk_databox_rescale(GTK_DATABOX(box));
    gtk_box_pack_start (GTK_BOX (box1), box, TRUE, TRUE, 0);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);

    box2 = gtk_vbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_end (GTK_BOX (box1), box2, FALSE, TRUE, 0);
    close_button = gtk_button_new_with_label ("close");
    gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked",
			 GTK_SIGNAL_FUNC(gtk_widget_destroy),
			 GTK_OBJECT (window));
    gtk_box_pack_start (GTK_BOX (box2), close_button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (close_button);
  }
  
  if (!GTK_WIDGET_VISIBLE (window))
    gtk_widget_show_all (window);
  else {
    gtk_widget_destroy (window);
  }
}

/*----------------------------------------------------------------
 *  databox lissajous
 *----------------------------------------------------------------*/

static gfloat *lissajousX=NULL;
static gfloat *lissajousY=NULL;

static gint     lissajous_idle=0;
static gfloat   lissajous_frequency=3.*PI/2.;
static GtkWidget *lissajous_label=NULL;
static guint    lissajous_counter=0;

static gboolean
lissajous_idle_func(GtkDatabox *box)
{
  gfloat freq;
  gfloat off;
  gchar label[10];
  gint i;

  if (!GTK_IS_DATABOX(box)) return FALSE;

  lissajous_frequency+=0.001;
  off=lissajous_counter*4*PI/POINTS;
  
  freq=14+10*sin(lissajous_frequency);
  for (i = 0; i < POINTS; i++ ) {
     lissajousX[i]=100.*sin(i*4*PI/POINTS+off);
     lissajousY[i]=100.*cos(i*freq*PI/POINTS+off);
  }
  
  
  gtk_databox_redraw(GTK_DATABOX(box));
  
  sprintf(label,"%d", lissajous_counter++);
  gtk_entry_set_text(GTK_ENTRY(lissajous_label), label);
  
  return TRUE;  
} 

static void
lissajous_destroy(GtkWidget *widget, GtkWidget **window)
{
  if (lissajous_idle) {
    gtk_idle_remove(lissajous_idle);
    lissajous_idle=0;
  }
  *window=NULL;  
}

static void
create_lissajous(void)
{
  static GtkWidget *window = NULL;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *close_button;
  GtkWidget *box;
  GtkWidget *label;
  GtkWidget *separator;
  GdkColor color;
  gint i;
  
  lissajous_frequency=0;
  if (!window)
  {
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize (window, 300, 300);  

    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                          GTK_SIGNAL_FUNC(lissajous_destroy),
                          &window);

    gtk_window_set_title (GTK_WINDOW (window), "Databox Lissajous");
    gtk_container_border_width (GTK_CONTAINER (window), 0);

    box1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), box1);

    label=gtk_label_new("This example resembles an oszilloscope\nreceiving two signals, one is a sine (horizontal), \nthe other is a cosine with variable frequency(vertical).\nThe counter is synchron with the updates.");
    gtk_box_pack_start (GTK_BOX (box1), label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);
    lissajous_label=gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(lissajous_label), "0");
    gtk_box_pack_start (GTK_BOX (box1), lissajous_label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);

    lissajous_idle=0;
    lissajous_frequency=3.*PI/2.;
    lissajous_counter=0;

    box=gtk_databox_new();
    gtk_signal_connect (GTK_OBJECT (box), "destroy",
			  GTK_SIGNAL_FUNC(gtk_databox_data_destroy_all),
			  NULL);
    
    lissajousX=g_new0(gfloat, POINTS);
    lissajousY=g_new0(gfloat, POINTS);
    
    for (i = 0; i < POINTS; i++ ) {
      lissajousX[i]=100.*sin(i*4*PI/POINTS);
      lissajousY[i]=100.*cos(i*4*PI/POINTS);
    }
    color.red=0;
    color.green=0;
    color.blue=0;
    
    gtk_databox_data_add_x_y(GTK_DATABOX(box), POINTS, 
    			lissajousX, lissajousY, color, 
    			GTK_DATABOX_POINTS, 
    			0);

    gtk_databox_rescale(GTK_DATABOX(box));

    gtk_box_pack_start (GTK_BOX (box1), box, TRUE, TRUE, 0);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);

    box2 = gtk_vbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_end (GTK_BOX (box1), box2, FALSE, TRUE, 0);
    close_button = gtk_button_new_with_label ("close");

    gtk_signal_connect_object_after (GTK_OBJECT (close_button), "clicked",
			 GTK_SIGNAL_FUNC(gtk_widget_destroy),
			 GTK_OBJECT (window));

    gtk_box_pack_start (GTK_BOX (box2), close_button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (close_button);
    lissajous_idle=gtk_idle_add((GtkFunction) lissajous_idle_func, box);
  }
  
  if (!GTK_WIDGET_VISIBLE (window))
    gtk_widget_show_all (window);
  else {
    gtk_widget_destroy (window);
  }
}

/*----------------------------------------------------------------
 *  databox bars
 *----------------------------------------------------------------*/

static gfloat *barsY1=NULL;
static gfloat *barsY2=NULL;

static gint     bars_idle=0;
static GtkWidget *bars_label=NULL;
static guint    bars_counter=0;

static gboolean
bars_idle_func(GtkDatabox *box)
{
  gfloat off;
  gchar label[10];
  gint i;

  if (!GTK_IS_DATABOX(box)) return FALSE;
   
  off=bars_counter/500.;
  
  for (i = 0; i < 10; i++ ) {
     barsY1[i]=100.*sin(i*PI/10+off);
     barsY2[i]=-100.*cos(i*PI/10+off/3);
  }
  
  
  gtk_databox_redraw(GTK_DATABOX(box));
  
  sprintf(label,"%d", bars_counter++);
  gtk_entry_set_text(GTK_ENTRY(bars_label), label);
  
  return TRUE;  
} 

static void
bars_destroy(GtkWidget *widget, GtkWidget **window)
{
  if (bars_idle) {
    gtk_idle_remove(bars_idle);
    bars_idle=0;
  }
  *window=NULL;  
}

static void
create_bars(void)
{
  static GtkWidget *window = NULL;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *close_button;
  GtkWidget *box;
  GtkWidget *label;
  GtkWidget *separator;
  gint index;
  GdkColor color;
  gfloat *X=NULL;
  gint i;
  
  if (!window)
  {
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize (window, 300, 300);  

    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                          GTK_SIGNAL_FUNC(bars_destroy),
                          &window);

    gtk_window_set_title (GTK_WINDOW (window), "Databox Bars");
    gtk_container_border_width (GTK_CONTAINER (window), 0);

    box1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), box1);

    label=gtk_label_new("This example shows an animated example of\nthe possible use of bars.\n\nThe counter is synchron with the updates.");
    gtk_box_pack_start (GTK_BOX (box1), label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);
    bars_label=gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(bars_label), "0");
    gtk_box_pack_start (GTK_BOX (box1), bars_label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);

    bars_idle=0;
    bars_counter=0;

    box=gtk_databox_new();
    gtk_signal_connect (GTK_OBJECT (box), "destroy",
			  GTK_SIGNAL_FUNC(gtk_databox_data_destroy_all),
			  NULL);
    
    X=g_new0(gfloat, 10);
    barsY1=g_new0(gfloat, 10);
    
    for (i = 0; i < 10; i++ ) {
      X[i]=i;
      barsY1[i]=100.*sin(i*PI/10);
    }
    color.red=0;
    color.green=0;
    color.blue=65535;
    
    index=gtk_databox_data_add_x_y(GTK_DATABOX(box), 10, 
    			X, barsY1, color, 
    			GTK_DATABOX_BARS, 
    			5);

    X=g_new0(gfloat, 10);
    barsY2=g_new0(gfloat, 10);
    
    for (i = 0; i < 10; i++ ) {
      X[i]=i+0.5;
      barsY2[i]=-100.*cos(i*PI/10);
    }
    color.red=65535;
    color.green=0;
    color.blue=0;
    
   gtk_databox_data_add_x_y(GTK_DATABOX(box), 10, 
    			X, barsY2, color, 
    			GTK_DATABOX_BARS, 
    			3);

    gtk_databox_rescale(GTK_DATABOX(box));

    gtk_box_pack_start (GTK_BOX (box1), box, TRUE, TRUE, 0);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);

    box2 = gtk_vbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_end (GTK_BOX (box1), box2, FALSE, TRUE, 0);
    close_button = gtk_button_new_with_label ("close");

    gtk_signal_connect_object_after (GTK_OBJECT (close_button), "clicked",
			 GTK_SIGNAL_FUNC(gtk_widget_destroy),
			 GTK_OBJECT (window));

    gtk_box_pack_start (GTK_BOX (box2), close_button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (close_button);
    bars_idle=gtk_idle_add((GtkFunction) bars_idle_func, box);
  }
  
  if (!GTK_WIDGET_VISIBLE (window))
    gtk_widget_show_all (window);
  else {
    gtk_widget_destroy (window);
  }
}

/*----------------------------------------------------------------
 *  databox movement
 *----------------------------------------------------------------*/

static gfloat *movementX=NULL;
static gfloat *movementY=NULL;

static gint     movement_idle=0;
static gfloat   movement_length=100;
static GtkWidget *movement_label=NULL;
static guint    movement_counter=0;

static gboolean
movement_idle_func(GtkDatabox *box)
{
  gchar label[10];
  gint i;

  if (!GTK_IS_DATABOX(box)) return FALSE;

  for (i = 0; i < movement_length; i++ ) {
  movementX[i]=100.*sin((movement_counter+i)*2*PI/POINTS)+2*sin(i*2*PI/20);
  movementY[i]=100.*cos((movement_counter+i)*10*PI/POINTS)+2*cos(i*2*PI/20);
  }
  
  gtk_databox_redraw(GTK_DATABOX(box));
  
  sprintf(label,"%d", movement_counter++);
  gtk_entry_set_text(GTK_ENTRY(movement_label), label);
  
  return TRUE;  
} 

static void
movement_destroy(GtkWidget *widget, GtkWidget **window)
{
  if (movement_idle) {
    gtk_idle_remove(movement_idle);
    movement_idle=0;
  }
  *window=NULL;  
}

static void
create_movement(void)
{
  static GtkWidget *window = NULL;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *close_button;
  GtkWidget *box=NULL;
  GtkWidget *label;
  GtkWidget *separator;
  GdkColor color;
  gfloat *X;
  gfloat *Y;
  gint i;
  
  if (!window)
  {
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize (window, 300, 300);  

    gtk_signal_connect (GTK_OBJECT (window), "destroy",
                          GTK_SIGNAL_FUNC(movement_destroy),
                          &window);

    gtk_window_set_title (GTK_WINDOW (window), "Databox Movement");
    gtk_container_border_width (GTK_CONTAINER (window), 0);

    box1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), box1);

    label=gtk_label_new("This example shows one constant signal and\na variable signal, that resembles a movement.\nMaybe we should write a game?\nThe counter is synchron with the updates.");
    gtk_box_pack_start (GTK_BOX (box1), label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);
    movement_label=gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(movement_label), "0");
    gtk_box_pack_start (GTK_BOX (box1), movement_label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);

    movement_idle=0;
    movement_counter=0;

    box=gtk_databox_new();
    gtk_signal_connect (GTK_OBJECT (box), "destroy",
			  GTK_SIGNAL_FUNC(gtk_databox_data_destroy_all),
			  NULL);
    
    X=g_new0(gfloat, POINTS);
    Y=g_new0(gfloat, POINTS);
    
    for (i = 0; i < POINTS; i++ ) {
      X[i]=100.*sin(i*2*PI/POINTS);
      Y[i]=100.*cos(i*10*PI/POINTS);
    }
    
    color.red=0;
    color.green=0;
    color.blue=0;
    
    gtk_databox_data_add_x_y(GTK_DATABOX(box), POINTS, 
    			X, Y, color, 
    			GTK_DATABOX_POINTS, 
    			0);

    movementX=X=g_new0(gfloat, movement_length);
    movementY=Y=g_new0(gfloat, movement_length);
    
    for (i=0; i<movement_length; i++ ) {
      X[i]=100.*sin(i*2*PI/POINTS);
      Y[i]=100.*cos(i*10*PI/POINTS);
    }
    
    color.red=65535;
    color.green=0;
    color.blue=0;
    
    gtk_databox_data_add_x_y(GTK_DATABOX(box), movement_length, 
    			X, Y, color, 
    			GTK_DATABOX_POINTS, 
    			4);

    gtk_databox_rescale(GTK_DATABOX(box));
    gtk_box_pack_start (GTK_BOX (box1), box, TRUE, TRUE, 0);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);

    box2 = gtk_vbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_end (GTK_BOX (box1), box2, FALSE, TRUE, 0);
    close_button = gtk_button_new_with_label ("close");

    gtk_signal_connect_object_after (GTK_OBJECT (close_button), "clicked",
			 GTK_SIGNAL_FUNC(gtk_widget_destroy),
			 GTK_OBJECT (window));

    gtk_box_pack_start (GTK_BOX (box2), close_button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (close_button);
    movement_idle=gtk_idle_add((GtkFunction) movement_idle_func, box);
  }
  
  if (!GTK_WIDGET_VISIBLE (window)) {
    gtk_widget_show_all (window);
    gtk_databox_hide_scrollbars(GTK_DATABOX(box));
    gtk_databox_hide_rulers(GTK_DATABOX(box));
    gtk_databox_hide_cross(GTK_DATABOX(box));
    gtk_databox_disable_zoom(GTK_DATABOX(box));
  } else {
    gtk_widget_destroy (window);
  }
}

/*----------------------------------------------------------------
 *  databox events
 *----------------------------------------------------------------*/

static gint
events_zoomed(GtkDatabox *box, GtkDataboxValue *top_left, GtkDataboxValue *bottom_right, GtkWidget *text)
{
  gchar *message;
  GdkColor fore;
  GdkColor back;
  
  fore.red=65535;
  fore.green=0;
  fore.blue=0;
  
  back.red=back.green=back.blue=65535;
  
  message=g_strdup_printf("--signal: zoomed\n  values: top_left (X,Y)=(%g, %g), bottom_right (X,Y)=(%g, %g)\n", top_left->x, top_left->y, bottom_right->x, bottom_right->y);
  gtk_text_insert(GTK_TEXT(text), text->style->font,
  		&fore, &back,
  		message, strlen(message));
  
  g_free(message);
  
  return 0;
}

static gint
events_marked(GtkDatabox *box, GtkDataboxCoord *marked, GtkWidget *text)
{
  gchar *message;
  GdkColor fore;
  GdkColor back;
  GtkDataboxValue value;
  
  gtk_databox_data_get_value(box, *marked, &value);
  
  fore.red=0;
  fore.green=35535;
  fore.blue=0;
  
  back.red=back.green=back.blue=65535;
  
  message=g_strdup_printf("--signal: marked\n  values: marked (X,Y)=(%g, %g)\n", value.x, value.y);
  gtk_text_insert(GTK_TEXT(text), text->style->font,
  		&fore, &back,
  		message, strlen(message));
  
  g_free(message);
  
  return 0;
}

static gint
events_selection_started(GtkDatabox *box, GtkDataboxCoord *marked, GtkWidget *text)
{
  gchar *message;
  GdkColor fore;
  GdkColor back;
  GtkDataboxValue value;
  
  gtk_databox_data_get_value(box, *marked, &value);
  
  fore.red=65535;
  fore.green=35535;
  fore.blue=0;
  
  back.red=back.green=back.blue=65535;
  
  message=g_strdup_printf("--signal: selection started\n  values: marked (X,Y)=(%g, %g)\n", value.x, value.y);
  gtk_text_insert(GTK_TEXT(text), text->style->font,
  		&fore, &back,
  		message, strlen(message));
  
  g_free(message);
  
  return 0;
}

static gint
events_selection_changed(GtkDatabox *box, GtkDataboxCoord *marked, GtkDataboxCoord *select, GtkWidget *text)
{
  gchar *message;
  GdkColor fore;
  GdkColor back;
  GtkDataboxValue value1;
  GtkDataboxValue value2;
  
  gtk_databox_data_get_value(box, *marked, &value1);
  gtk_databox_data_get_value(box, *select, &value2);
  
  fore.red=0;
  fore.green=0;
  fore.blue=65535;
  
  back.red=back.green=back.blue=65535;
  
  message=g_strdup_printf("--signal: selection changed\n  values: corner1 (X,Y)=(%g, %g) corner2 (X,Y)=(%g, %g)\n", value1.x, value1.y, value2.x, value2.y);
  gtk_text_insert(GTK_TEXT(text), text->style->font,
  		&fore, &back,
  		message, strlen(message));
  
  g_free(message);
  
  return 0;
}

static gint
events_selection_stopped(GtkDatabox *box, GtkDataboxCoord *marked, GtkDataboxCoord *select, GtkWidget *text)
{
  gchar *message;
  GdkColor fore;
  GdkColor back;
  GtkDataboxValue value1;
  GtkDataboxValue value2;
  
  gtk_databox_data_get_value(box, *marked, &value1);
  gtk_databox_data_get_value(box, *select, &value2);
  
  fore.red=0;
  fore.green=35535;
  fore.blue=65535;
  
  back.red=back.green=back.blue=65535;
  
  message=g_strdup_printf("--signal: selection stopped\n  values: corner1 (X,Y)=(%g, %g) corner2 (X,Y)=(%g, %g)\n", value1.x, value1.y, value2.x, value2.y);
  gtk_text_insert(GTK_TEXT(text), text->style->font,
  		&fore, &back,
  		message, strlen(message));
  
  g_free(message);
  
  return 0;
}

static gint
events_selection_canceled(GtkDatabox *box, GtkWidget *text)
{
  gchar *message;
  GdkColor fore;
  GdkColor back;
  
  fore.red=35535;
  fore.green=35535;
  fore.blue=65535;
  
  back.red=back.green=back.blue=65535;
  
  message=g_strdup_printf("--signal: selection canceled\n  values: none\n");
  gtk_text_insert(GTK_TEXT(text), text->style->font,
  		&fore, &back,
  		message, strlen(message));
  
  g_free(message);
  
  return 0;
}

static void
create_events(void)
{
  static GtkWidget *window = NULL;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *close_button;
  GtkWidget *box=NULL;
  GtkWidget *hbox=NULL;
  GtkWidget *label;
  GtkWidget *separator;
  GtkWidget *text;
  GtkWidget *scrollbar;
  GdkColor color;
  gfloat *X;
  gfloat *Y;
  GtkAdjustment *vadj;
  gint i;
  
  if (!window)
  {
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize (window, 600, 600);  
    gtk_signal_connect (GTK_OBJECT (window), "destroy",
			  GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			  &window);


    gtk_window_set_title (GTK_WINDOW (window), "Databox Events");
    gtk_container_border_width (GTK_CONTAINER (window), 0);

    box1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), box1);

    label=gtk_label_new("This example shows the use of the\nevents emitted by the GtkDatabox.\nThe signal name and its data is shown in the text field.");
    gtk_box_pack_start (GTK_BOX (box1), label, FALSE, FALSE, 0);
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);
    
    hbox=gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start (GTK_BOX (box1), hbox, FALSE, FALSE, 0);

    vadj=(GtkAdjustment *) gtk_adjustment_new(0, 0, 10, 1, 5, 10);
    text=gtk_text_new(NULL, vadj);
    gtk_text_set_editable(GTK_TEXT(text), FALSE);
    gtk_box_pack_start (GTK_BOX (hbox), text, TRUE, TRUE, 0);
    
    scrollbar=gtk_vscrollbar_new(vadj);
    gtk_box_pack_start (GTK_BOX (hbox), scrollbar, FALSE, FALSE, 0);
    
    
    separator=gtk_hseparator_new();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);

    box=gtk_databox_new();
    gtk_signal_connect (GTK_OBJECT (box), "destroy",
			  GTK_SIGNAL_FUNC(gtk_databox_data_destroy_all),
			  NULL);
    gtk_signal_connect (GTK_OBJECT (box), "gtk_databox_zoomed",
			  GTK_SIGNAL_FUNC(events_zoomed),
			  text);
    gtk_signal_connect (GTK_OBJECT (box), "gtk_databox_marked",
			  GTK_SIGNAL_FUNC(events_marked),
			  text);
    gtk_signal_connect (GTK_OBJECT (box), "gtk_databox_selection_started",
			  GTK_SIGNAL_FUNC(events_selection_started),
			  text);
    
    gtk_signal_connect (GTK_OBJECT (box), "gtk_databox_selection_changed",
			  GTK_SIGNAL_FUNC(events_selection_changed),
			  text);
    gtk_signal_connect (GTK_OBJECT (box), "gtk_databox_selection_stopped",
			  GTK_SIGNAL_FUNC(events_selection_stopped),
			  text);
    gtk_signal_connect (GTK_OBJECT (box), "gtk_databox_selection_canceled",
			  GTK_SIGNAL_FUNC(events_selection_canceled),
			  text);
    
    X=g_new0(gfloat, POINTS);
    Y=g_new0(gfloat, POINTS);
    
    for (i = 0; i < POINTS; i++ ) {
      X[i]=100.*sin(i*2*PI/POINTS);
      Y[i]=100.*cos(i*10*PI/POINTS);
    }
    
    color.red=0;
    color.green=0;
    color.blue=0;
    
    gtk_databox_data_add_x_y(GTK_DATABOX(box), POINTS, 
    			X, Y, color, 
    			GTK_DATABOX_POINTS, 
    			0);

    gtk_databox_rescale(GTK_DATABOX(box));
    gtk_box_pack_start (GTK_BOX (box1), box, TRUE, TRUE, 0);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);

    box2 = gtk_vbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_end (GTK_BOX (box1), box2, FALSE, TRUE, 0);
    close_button = gtk_button_new_with_label ("close");

    gtk_signal_connect_object_after (GTK_OBJECT (close_button), "clicked",
			 GTK_SIGNAL_FUNC(gtk_widget_destroy),
			 GTK_OBJECT (window));

    gtk_box_pack_start (GTK_BOX (box2), close_button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (close_button);
  }
  
  if (!GTK_WIDGET_VISIBLE (window)) {
    gtk_widget_show_all (window);
  } else {
    gtk_widget_destroy (window);
  }
}

static void
do_exit (GtkWidget *widget, GtkWidget *window)
{
  gtk_widget_destroy (window);
  gtk_main_quit ();
}

static void
create_main_window (void)
{
  struct {
    char *label;
    void (*func)();
  } buttons[] =
    {
      { "databox basics", create_basics },
      { "databox colors and misc", create_colors },
      { "bars example", create_bars },
      { "lissajous example", create_lissajous },
      { "movement example", create_movement },
      { "events example", create_events },
    };
  int nbuttons = sizeof (buttons) / sizeof (buttons[0]);
  GtkWidget *window;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *button;
  GtkWidget *separator;
  int i;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy (GTK_WINDOW (window), FALSE, FALSE, FALSE);
  gtk_widget_set_name (window, "Gtk Databox");
  gtk_widget_set_usize (window, 200, 200);
  gtk_widget_set_uposition (window, 20, 20);

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC(gtk_main_quit),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (window), "delete-event",
		      GTK_SIGNAL_FUNC (gtk_false),
		      NULL);

  box1 = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), box1);

  gtk_container_border_width (GTK_CONTAINER (box1), 10);

  for (i = 0; i < nbuttons; i++)
    {
      button = gtk_button_new_with_label (buttons[i].label);
      if (buttons[i].func)
        gtk_signal_connect (GTK_OBJECT (button), 
			    "clicked", 
			    GTK_SIGNAL_FUNC(buttons[i].func),
			    NULL);
      else
        gtk_widget_set_sensitive (button, FALSE);
      gtk_box_pack_start (GTK_BOX (box1), button, TRUE, TRUE, 0);
    }

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);

  box2 = gtk_vbox_new (FALSE, 10);
  gtk_container_border_width (GTK_CONTAINER (box2), 10);
  gtk_box_pack_start (GTK_BOX (box1), box2, FALSE, TRUE, 0);

  button = gtk_button_new_with_label ("close");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (do_exit),
		      window);
  gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);

  gtk_widget_show_all (window);
}

gint 
main(gint argc, char *argv[])
{
  gtk_init(&argc, &argv);
  
  create_main_window();
  gtk_main();

  return 0;
}
