/* This program will print out exercise lines to the display, and
   then check for correct typing.
*/

#include <stdio.h>
#include <ctype.h>
#include <signal.h>

#if defined(ultrix)
#include <cursesX.h>
#else
#include <curses.h>
#endif
#if defined(VMS)
#define cbreak() crmode()
#define nocbreak() nocrmode()
#endif

#include <errno.h>
#if defined(sony_news)
extern int errno;
#endif

typedef unsigned char BoolType;
#if !defined(FALSE)
#define FALSE 0
#define TRUE 1
#endif

#if !defined(NULL)
#define NULL 0
#endif

#define ASCII_ESC 27
#define ASCII_GS  29
#define ASCII_BS  8
#define ASCII_RUBOUT 127

#if defined(VMS)
#define ESCAPER ASCII_GS
#else
#define ESCAPER ASCII_ESC
#endif

#define STR_SIZE 1024
typedef char StrType[STR_SIZE + 1];

#define MAX_LINES 1500
char *cached_lines[MAX_LINES];

int linesInLesson;
int exerciseStart;
int newStart;

FILE *lesson_file;
StrType current_line;

wait_user()
/* Get any response from user before proceeding */
{
move(23, 0);
standout();
addstr("Press any key to continue...");
standend();
refresh();
getch();
}

void get_end_line()
/* Read end of line, should be smarter... */
{
refresh();
getch();
}

int cleanup()
/* cleanup routine, esp. SIGINT */
{
endwin();
printf("\n");
exit(1);
}

char *get_lesson()
/* Ask user for desired lesson */
{
static StrType response;

move(0,0);
addstr("  Several lessons are available:\n");
addstr("        Quick QWERTY course (q1 - q5)\n");
addstr("        Long QWERTY course  (r1 - r14)\n");
addstr("        QWERTY touch typing (t1 - t16)\n");
addstr("        Yet Another QWERTY  (v1 - v20)\n");
addstr("        QWERTY Review       (u1 - u13)\n");
addstr("        Dvorak touch typing (d1 - d14)\n");
addstr("        Typing drills       (m1 - m11)\n");
addstr("        Speed drills        (s1 - s4)\n");
addstr("        Calculator keypad   (n1 - n3)\n");
addstr("\n");
addstr(" Press return to exit or else type the desired lesson name: ");
refresh();
echo();
nocbreak();
getstr(response);
noecho();
cbreak();
if (strlen(response) == 0)
  cleanup();
return response;
}

BoolType get_lesson_line()
/* Read a line of input from user, in canonical mode */
{
if (feof(lesson_file) || ferror(lesson_file))
  return FALSE;
if (fgets(current_line, STR_SIZE, lesson_file) == NULL)
  current_line[0] = '\0';
else {
  if (current_line[strlen(current_line) - 1] == '\n')
    current_line[strlen(current_line) - 1] = '\0';
  }
return TRUE;
}

BoolType find_lesson(name)
/* locate given lesson, leave lesson_file open so next line begins */
char *name;
{
StrType fullName;
StrType target;
StrType response;
BoolType done = FALSE;

/* gf: Added LESSONDIR */
#ifndef LESSONDIR
#define LESSONDIR "."
#endif
sprintf(fullName, "%s/%c.typ", LESSONDIR, name[0]);
sprintf(target, "*%c%s*", toupper(name[0]), &name[1]);
lesson_file = fopen(fullName, "r");
if (lesson_file == NULL) {
  standout();
  sprintf(response, "open() error on %s:  errno=%d\n", fullName, errno);
  addstr(response);
  standend();
  wait_user();
  return FALSE;
  }
while (!done) {
  if (!get_lesson_line())
    break;
  if (strncmp(current_line, target, 5) == 0) {
    done = TRUE;
    break;
    }
  }
if (!done) {
  sprintf(response, "Lesson %s is not available.\n", &name[1]);
  standout();
  addstr(response);
  standend();
  wait_user();
  return FALSE;
  }
linesInLesson = 0;
done = FALSE;
while (!done) {
  if (!get_lesson_line()) {
    done = TRUE;
    break;
    }
  if (current_line[0] == '*') {
    done = TRUE;
    break;
    }
  if (linesInLesson == MAX_LINES) {
    }
  if (cached_lines[linesInLesson] != NULL) {
    free(cached_lines[linesInLesson]);
    cached_lines[linesInLesson] = NULL;
    }
  cached_lines[linesInLesson] = (char *)malloc(strlen(current_line) + 1);
  if (cached_lines[linesInLesson] == NULL) {
    endwin();
    fprintf(stderr, "malloc() failure\n");
    exit(1);
    }
  strcpy(cached_lines[linesInLesson], current_line);
  linesInLesson ++;
  }
return TRUE;
}

void get_exercise(startLine, kind, newStart)
/* determine length and kind of exercise */
int startLine;
char *kind;
int *newStart;
{
char *thisLine;
int lineLength;

*newStart = startLine;
*kind = 'I'; /* default... */
/* Determine end of exercise */
while (TRUE) {
  if (*newStart >= linesInLesson)
    break; /* done */

  thisLine = cached_lines[*newStart];
  lineLength = strlen(thisLine);
  if (thisLine[lineLength - 2] == '\\') { /* expected terminator */
    *kind = thisLine[lineLength - 1];
    thisLine[lineLength - 2] = '\0'; /* remove */
    (*newStart) ++;
    break;
    }
  if (thisLine[lineLength - 1] == '\\') /* compatibility with BASIC */
    thisLine[lineLength - 1] = '\0'; /* also remove */
  (*newStart) ++;
  }
}

void displaySpeed(totalChars, elapsed, errs)
/* show how fast you were */
int totalChars;
int elapsed;
int errs;
{
  double testTime = (double)elapsed / (double)60.0; /* time in minutes */
  double words = (double)totalChars / 5.0;
  double speed = words / testTime;
  double adjustedSpeed;
  StrType message;

  words -= errs;
  adjustedSpeed = words / testTime;

  move(19, 19);
  standout();
  sprintf(message, "raw speed      = %f wpm", speed);
  addstr(message);
  move(20, 19);
  sprintf(message, "adjusted speed = %f wpm", adjustedSpeed);
  addstr(message);
  move(21, 19);
  sprintf(message, "            with %f%s errors",
    (double)100.0 * errs / totalChars, "%");
  addstr(message);
  standend();
  wait_user();
}

void do_drill()
/* \D, a drill */
{
int tries;
int errors;
int totalChars;
int i, j;
char *thisLine;
int thisLength;
char ch;
long startTime, endTime;

tries = 0;
do { /* a try */
  totalChars = 0;
  errors = 0;
  tries ++;

  /* display drill pattern */
  for (i = exerciseStart; i < newStart; i ++) {
    move(3 + (i - exerciseStart) * 2, 0);
    addstr(cached_lines[i]);
    totalChars += strlen(cached_lines[i]) + 1; /* +1 for NULL */
    }
  refresh();

  /* run the drill */
  for (i = exerciseStart; i < newStart; i ++) {
    thisLine = cached_lines[i];
    thisLength = strlen(thisLine);

    move(4 + (i - exerciseStart) * 2, 0);
    for (j = 0; j < thisLength; j ++) {
      refresh();
      ch = getch();
      if (i == exerciseStart && j == 0)
        startTime = time(NULL); /* timer doesn't run till first stroke */
      if (ch == ESCAPER) {
        clear();
        goto ALL_DONE;
        }
      if (ch != thisLine[j]) {
        addch('X');
#if 0
        beep();
#else
        printf("\007");
        fflush(stdout);
#endif
        errors ++;
        }
      else
        addch(ch);
      }
    get_end_line();
    }
  endTime = time(NULL);

  displaySpeed(totalChars, endTime - startTime, errors);
  clear();
  if (errors > 0 && tries < 3) {
    move(0, 0);
    addstr("Try again...");
    }
  tries ++;
  } while (errors > 0 && tries < 4);
ALL_DONE:;
}

void do_para()
/* \P, practice paragraph */
{
int i, j;
char *thisLine;
int thisLength;
char ch;
int totalChars = 0;
int errors = 0;
int startTime, endTime;

/* print out practice text */
for (i = exerciseStart; i < newStart; i ++) {
  move(3 + (i - exerciseStart), 0);
  addstr(cached_lines[i]);
  totalChars += strlen(cached_lines[i]);
  }

move(3, 0);
refresh();
for (i = exerciseStart; i < newStart; i ++) {
  move(3 + (i - exerciseStart), 0);
  thisLine = cached_lines[i];
  thisLength = strlen(thisLine);
  for (j = 0; j < thisLength; j ++) {
    refresh();
    ch = getch();
    if (i == exerciseStart && j == 0)
      startTime = time(NULL); /* timer doesn't run till first stroke */
    if (ch == ESCAPER) {
      clear();
      goto ALL_DONE;
      }
    /* Allow backspace */
    if (j != 0 && (ch == ASCII_BS || ch == ASCII_RUBOUT)) {
      j --;
      move(3 + (i - exerciseStart), j);
      addch(thisLine[j]);
      move(3 + (i - exerciseStart), j);
      j --;
      continue; /* but notice that errors accumulate */
      }
    if (ch != thisLine[j]) {
      move(3 + (i - exerciseStart), j);
      standout();
      addch(thisLine[j]);
      standend();
      errors ++;
      }
    else
      move(3 + (i - exerciseStart), j + 1);
    }
    get_end_line();
  }
  endTime = time(NULL);

  displaySpeed(totalChars, endTime - startTime, errors);
ALL_DONE:;
}

void do_type()
/* \T, type out a lesson */
{
int i;

clear();
for (i = exerciseStart; i < newStart; i ++) {
  move(1 + (i - exerciseStart), 0);
  addstr(cached_lines[i]);
  }
wait_user();
clear();
}

void do_instruction()
/* \I, instruction line */
{
clear();
move(0, 0);
addstr(cached_lines[exerciseStart]);
}

void do_clear()
/* \B, clear screen */
{
clear();
move(1, 29);
addstr(cached_lines[exerciseStart]);
wait_user();
}

void give_lesson()
/* dance through lesson file */
{
char kind;

exerciseStart = 0;
while (TRUE) {
  get_exercise(exerciseStart, &kind, &newStart);
  switch (kind) {
    case 'B':
      do_clear();
      break;

    case 'P':
      do_para();
      break;

    case 'T':
      do_type();
      break;

    case 'D':
      do_drill();
      break;

    case 'I':
      do_instruction();
      break;

    default:
      endwin();
      fprintf(stderr, "Unknown exercise near line %d of this lesson\n",
          newStart - 1);
      exit(1);
      break;
    }
  exerciseStart = newStart;
  if (exerciseStart >= linesInLesson)
    break;
  }
}

int main(argc, argv)
int argc;
char *argv[];
{
char *lessonName;
int i;

initscr();
clear();
refresh();
cbreak();
noecho();
signal(SIGINT, cleanup);

for (i = 0; i < MAX_LINES; i ++)
  cached_lines[i] = NULL;

/* Start and format screen.  Open and read exercise file.
   Inquire for lesson to run */
while (TRUE) {
  clear();
  lessonName = get_lesson();
  if (!find_lesson(lessonName))
    continue;
  give_lesson();
  }
}
