## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys, os, string, time, types
try:                                                                #bundle#
    from cPickle import Pickler, Unpickler, UnpicklingError         #bundle#
except ImportError:                                                 #bundle#
    from pickle import Pickler, Unpickler, UnpicklingError          #bundle#

# PySol imports
from mfxutil import Struct, EnvError                                #bundle#


# /***********************************************************************
# // constants
# ************************************************************************/

PACKAGE = "PySol"                                                   #bundle#
VERSION = "3.00"

# Suits values are 0-3. This maps to colors 0-1.
SUITS = ("Club", "Spade", "Heart", "Diamond")
COLORS = ("black", "red")

# Card ranks are 0-12.  We also define symbolic names for the
# picture cards.
RANKS = ("Ace", "2", "3", "4", "5", "6", "7", "8", "9", "10", "Jack", "Queen", "King")
ACE = 0
JACK = 10
QUEEN = 11
KING = 12

# Special values for Stack.cap:
ANY_SUIT = -1
ANY_COLOR = -1
ANY_RANK = -1
NO_SUIT = 4         # no card can ever match this suit
NO_COLOR = 2        # no card can ever match this color
NO_RANK = 13        # no card can ever match this rank

#
NO_REDEAL = 0
UNLIMITED_REDEALS = -1
VARIABLE_REDEALS = -2


# /***********************************************************************
# // simple benchmarking
# ************************************************************************/

class Timer:
    def __init__(self, msg = ""):
        self.msg = msg
        self.clock = time.time
        if os.name == "nt":
            self.clock = time.clock
        self.start = self.clock()
    def reset(self):
        self.start = self.clock()
    def get(self):
        return self.clock() - self.start
    def __repr__(self):
        return "%-20s %6.3f seconds" % (self.msg, self.clock() - self.start)


# /***********************************************************************
# // DataLoader
# ************************************************************************/

class DataLoader:
    def __init__(self, argv0, filenames, path=[]):
        self.dir = None
        if type(filenames) == types.StringType:
            filenames = (filenames,)
        assert type(filenames) in (types.TupleType, types.ListType)
        #$ init path
        path = path[:]
        head, tail = os.path.split(argv0)
        if not head:
            head = os.curdir
        path.append(head)
        path.append(os.path.join(head, "data"))
        path.append(os.path.join(head, os.pardir, "data"))          #bundle#
        #$ you can add your extra directories here
        if os.name == "posix":
            pass
        if os.name == "nt":
            pass
        if os.name == "mac":
            pass
        #$ add standard Unix directories to search path
        if 1 and os.name == "posix":
            for v in (VERSION, ""):
                for prefix in ("@prefix@", "/usr/local", "/usr"):
                    try:
                        if os.path.isdir(prefix):
                            path.append(os.path.join(prefix,"share/pysol",v))
                            path.append(os.path.join(prefix,"lib/pysol",v))
                            path.append(os.path.join(prefix,"share/games/pysol",v))
                            path.append(os.path.join(prefix,"lib/games/pysol",v))
                            path.append(os.path.join(prefix,"games/share/pysol",v))
                            path.append(os.path.join(prefix,"games/lib/pysol",v))
                    except EnvError:
                        pass
        #$ check path for valid directories
        self.path = []
        for p in path:
            if not p: continue
            try:
                np = os.path.normpath(p)
                if np and (not np in self.path) and os.path.isdir(np):
                    self.path.append(np)
            except EnvError:
                pass
        #$ now try to find all filenames along path
        for p in self.path:
            n = 0
            for filename in filenames:
                try:
                    f = os.path.join(p, filename)
                    if os.path.isfile(f):
                        n = n + 1
                    else:
                        break
                except EnvError:
                    pass
            if n == len(filenames):
                self.dir = p
                break
        else:
            raise os.error, str(argv0) + ": DataLoader could not find " + str(filenames)
        ##print path, self.path, self.dir


    def __findFile(self, func, filename, subdirs=None):
        if subdirs is None:
            subdirs = ("",)
        elif type(subdirs) == types.StringType:
            subdirs = (subdirs,)
        for dir in subdirs:
            f = os.path.join(self.dir, dir, filename)
            f = os.path.normpath(f)
            if func(f):
                return f
        raise os.error, "DataLoader could not find " + filename + " in " + self.dir + " " + str(subdirs)

    def findFile(self, filename, subdirs=None):
        return self.__findFile(os.path.isfile, filename, subdirs)

    def findDir(self, filename, subdirs=None):
        return self.__findFile(os.path.isdir, filename, subdirs)


# /***********************************************************************
# // Cardset
# ************************************************************************/

class Cardset(Struct):
    def __init__(self):
        Struct.__init__(self,
            ident = "",
            name = "",
            version = 1,
            type = 0,
            ncards = 52,
            dir = "",
            ext = ".gif",
            CARDW = 0,
            CARDH = 0,
            CARDD = 0,
            CARD_UP_YOFFSET = 0,
            CARD_DOWN_YOFFSET = 0,      # not used
            SHADOW_XOFFSET = 0,
            SHADOW_YOFFSET = 0,
            backnames = (),
            back_index = 0,
            # initialized by CardsetManager
            index = -1,
        )


class CardsetManager:
    def __init__(self):
        self.__cs = []

    def len(self):
        return len(self.__cs)

    def register(self, cardset):
        assert cardset.index == -1
        cardset.index = len(self.__cs)
        self.__cs.append(cardset)

    def get(self, index):
        return self.__cs[index]

    def getByName(self, name):
        for cs in self.__cs:
            if cs.name == name:
                return cs
        return None


# /***********************************************************************
# // Images
# ************************************************************************/

class ImagesCardback:
    def __init__(self, index, name, image, menu_image=None):
        if menu_image is None: menu_image = image
        self.index = index
        self.name = name
        self.image = image
        self.menu_image = menu_image


class Images:
    def __init__(self, dataloader, cs):
        self.d = dataloader
        self.cs = cs
        # copy from cardset
        self.CARDW, self.CARDH, self.CARDD = cs.CARDW, cs.CARDH, cs.CARDD
        self.CARD_XOFFSET = 12
        self.CARD_YOFFSET = max(10, cs.CARD_UP_YOFFSET)
        self.SHADOW_XOFFSET, self.SHADOW_YOFFSET = cs.SHADOW_XOFFSET, cs.SHADOW_YOFFSET
        self.__back_index = cs.back_index
        # other
        self.__shade_index = 0
        self.__card = []
        self.__back = []
        self.__bottom = []
        self.__letter = []
        self.__shadow = []
        self.__shade = []

    def loadImage(self, file):
        raise SubclassResponsibility

    def __loadCard(self, filename, check_w=1, check_h=1):
        f = os.path.join(self.cs.dir, filename)
        img = self.loadImage(file=f)
        w, h = img.width(), img.height()
        if self.CARDW < 0:
            self.CARDW, self.CARDH = w, h
        else:
            if ((check_w and w != self.CARDW) or
                (check_h and h != self.CARDH)):
                raise Exception, "Invalid size %dx%d of image %s" % (w, h, f)
        return img

    def load(self, progress=None, fast=0):
        ##fast = 1                                                    #bundle#
        ext = self.cs.ext[1:]
        # load cards
        i = 0
        for name in self.cs.backnames:
            im1 = self.__loadCard(name)
            r = max(self.CARDW / 40.0, self.CARDH / 60.0, 1)
            r = int(round(r))
            if r < 2: r = 2
            im2 = im1.subsample(r)
            self.__back.append(ImagesCardback(i, name, im1, im2))
            i = i + 1
        if progress: progress.update(step=1)
        for rank in range(13):
            for suit in "cshd":
                name = "%02d%s.%s" % (rank + 1, suit, ext)
                self.__card.append(self.__loadCard(name))
                if progress: progress.update(step=1)
        for rank in range(4):
            name = "l%02d.%s" % (rank + 1, ext)
            self.__letter.append(self.__loadCard(name))
            if progress: progress.update(step=1)
        for i in range(7):
            name = "bottom%02d.%s" % (i + 1, ext)
            self.__bottom.append(self.__loadCard(name))
            if progress: progress.update(step=1)
        # support shadow for max. 13 cards (memory & performance reasons)
        for i in range(14):
            name = "shadow%02d.%s" % (i, ext)
            if fast:
                self.__shadow.append(None)
            else:
                self.__shadow.append(self.__loadCard(name, check_w=0, check_h=0))
            if progress: progress.update(step=1)
        if fast:
            self.__shade.append(None)
            if progress: progress.update(step=1)
        else:
            self.__shade.append(self.__loadCard("shade." + ext))
            if progress: progress.update(step=1)

    def getFace(self, suit, rank):
        assert 0 <= suit <= 3
        assert ACE <= rank <= KING
        index = rank * 4 + suit
        ## print suit, rank, index
        assert 0 <= index <= 51
        return self.__card[index]

    def getBack(self, deck):
        return self.__back[self.__back_index].image

    def getTalonBottom(self):
        return self.__bottom[0]

    def getReserveBottom(self):
        return self.__bottom[0]

    def getSuitBottom(self, suit=-1):
        assert type(suit) == types.IntType
        assert -1 <= suit <= 3
        if suit == -1: return self.__bottom[1]
        return self.__bottom[3 + suit]

    def getBraidBottom(self):
        return self.__bottom[2]

    def getLetter(self, rank):
        assert 0 <= rank <= 3
        return self.__letter[rank]

    def getShadow(self, ncards):
        assert ncards >= 0
        if ncards >= len(self.__shadow):
            ##ncards = len(self.__shadow) - 1
            return None
        return self.__shadow[ncards]

    def getShade(self):
        return self.__shade[self.__shade_index]

    #
    # card back
    #

    def getCardbacks(self):
        return self.__back

    def getCardbackIndex(self):
        return self.__back_index

    def setCardback(self, index):
        i = self.__back_index
        if index is None:
            self.__back_index = (self.__back_index + 1) % len(self.__back)
        else:
            self.__back_index = index
        return i != self.__back_index


# /***********************************************************************
# // memory util/debugging
# ************************************************************************/

cyclops = None

