## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import math, os, re, string, sys, types
import Tkinter, Canvas
import tkColorChooser, tkFileDialog

# PySol imports
from mfxutil import destruct, Struct, kwdefault                     #bundle#
from util import PACKAGE, VERSION                                   #bundle#
from gamedb import GI                                               #bundle#
from stats import PysolStatsWriter                                  #bundle#
from actions import PysolMenubarActions                             #bundle#

# toolkit imports
from tkconst import EVENT_HANDLED, EVENT_PROPAGATE                  #bundle#
from tkutil import bind                                             #bundle#
from selectgame import SelectGameDialog                             #bundle#
from tkstats import StatsDialog, StatsLogDialog                     #bundle#


# /***********************************************************************
# //
# ************************************************************************/

class MfxMenubar(Tkinter.Menu):
    addPath = None

    def __init__(self, master, **kw):
        assert kw.get("name")
        self.n = kw["tearoff"] = int(kw.get("tearoff", 0))
        apply(Tkinter.Menu.__init__, (self, master, ), kw)

    def labeltoname(self, label):
        underline = -1
        m = re.search(r'^(.*)\&([^\&].*)$', label)
        if m:
            l1, l2 = m.group(1), m.group(2)
            l1 = re.sub(r'\&\&', r'&', l1)
            l2 = re.sub(r'\&\&', r'&', l2)
            label = l1 + l2
            underline = len(l1)
        name = string.lower(re.sub(r'[^0-9a-zA-Z]', '', label))
        return name, label, underline

    def add(self, itemType, cnf={}):
        label = cnf.get("label")
        if label:
            name, label, underline = self.labeltoname(label)
            cnf["label"] = label
            cnf["underline"] = cnf.get("underline", underline)
            path = str(self._w) + "." + name
            if self.addPath:
                self.addPath(path, self, self.n, cnf.get("menu"))
        Tkinter.Menu.add(self, itemType, cnf)
        self.n = self.n + 1


class MfxMenu(MfxMenubar):
    def __init__(self, master, label, underline=None, **kw):
        name, label, label_underline = self.labeltoname(label)
        kwdefault(kw, name=name)
        apply(MfxMenubar.__init__, (self, master,), kw)
        if underline is None:
            underline = label_underline
        master.add_cascade(menu=self, label=label, underline=underline)


# /***********************************************************************
# // - create menubar
# // - update menubar
# // - menu actions
# ************************************************************************/

class PysolMenubar(PysolMenubarActions):
    def __init__(self, app, top):
        PysolMenubarActions.__init__(self, app, top)
        # init columnbreak
        sh = self.top.winfo_screenheight()
        self.__cb_max = 25
        if sh >= 600: self.__cb_max = 30
        if sh >= 768: self.__cb_max = 35
        # create menus
        self.__menubar = None
        self.__menupath = {}
        self.__keybindings = {}
        self._createMenubar()
        # set the menubar
        self.updateBackgroundImages()
        self.top.config(menu=self.__menubar)


    # create a GTK-like path
    def _addPath(self, path, menu, index, submenu):
        if not self.__menupath.has_key(path):
            ##print path, menu, index, submenu
            self.__menupath[path] = (menu, index, submenu)

    #
    # create the menubar
    #

    def _createMenubar(self):
        MfxMenubar.addPath = self._addPath
        self.__menubar = MfxMenubar(self.top, name="menubar")

        # init keybindings
        bind(self.top, "<KeyPress>", self._keyPressHandler)

        m = "Ctrl-"
        if os.name == "mac": m = "Cmd-"

        menu = self._createMenu("&File")
        menu.add_command(label="&New game", command=self.mNewGame, accelerator="N")
        self._addSelectGameMenu(menu)
        menu.add_command(label="Select &random game", command=self.mSelectRandomGame, accelerator=m+"R")
        menu.add_command(label="Select game by nu&mber...", command=self.mSelectGameById, accelerator=m+"M")
        menu.add_separator()
        menu.add_command(label="&Open...", command=self.mOpen, accelerator=m+"O")
        menu.add_command(label="&Save", command=self.mSave, accelerator=m+"S")
        menu.add_command(label="Save &as...", command=self.mSaveAs)
        menu.add_separator()
        menu.add_command(label="&Quit", command=self.mQuit, accelerator=m+"Q")

        menu = self._createMenu("&Edit")
        menu.add_command(label="&Undo", command=self.mUndo, accelerator="Z")
        menu.add_command(label="&Redo", command=self.mRedo, accelerator="R")
        menu.add_command(label="Redo &all", command=self.mRedoAll)
        menu.add_separator()
        menu.add_command(label="Restart &game", command=self.mRestart, accelerator=m+"G")

        menu = self._createMenu("&Game")
        menu.add_command(label="&Deal cards", command=self.mDeal, accelerator="D")
        menu.add_command(label="&Auto drop", command=self.mDrop, accelerator="A")
        menu.add_separator()
        menu.add_command(label="S&tatus...", command=self.mStatus, accelerator="T")

        menu = self._createMenu("&Assist")
        menu.add_command(label="&Hint", command=self.mHint, accelerator="H")
        menu.add_command(label="Highlight p&iles", command=self.mHighlightPiles, accelerator="I")
        menu.add_separator()
        menu.add_command(label="&Demo", command=self.mDemo, accelerator=m+"D")
        menu.add_command(label="Demo (&all games)", command=self.mMixedDemo)

        menu = self._createMenu("&Options")
##        menu.add_command(label="&Player name...", command=self.mOptPlayerName)
        menu.add_checkbutton(label="&Confirm quit", variable=self.tkopt.confirm, command=self.mOptConfirm)
        submenu = MfxMenu(menu, label="&Automatic play")
        submenu.add_checkbutton(label="Auto &face up", variable=self.tkopt.autofaceup, command=self.mOptAutoFaceUp)
        submenu.add_checkbutton(label="&Auto drop", variable=self.tkopt.autodrop, command=self.mOptAutoDrop)
        submenu.add_checkbutton(label="Auto &deal", variable=self.tkopt.autodeal, command=self.mOptAutoDeal)
        submenu.add_separator()
        submenu.add_checkbutton(label="&Quick play", variable=self.tkopt.quickplay, command=self.mOptQuickPlay)
        submenu = MfxMenu(menu, label="Assist &level")
        submenu.add_checkbutton(label="Enable &undo", variable=self.tkopt.undo, command=self.mOptEnableUndo)
        submenu.add_checkbutton(label="Enable &hint", variable=self.tkopt.hint, command=self.mOptEnableHint)
        submenu.add_checkbutton(label="Enable highlight &piles", variable=self.tkopt.highlight_piles, command=self.mOptEnableHighlightPiles)
        submenu.add_checkbutton(label="Enable highlight &cards", variable=self.tkopt.highlight_cards, command=self.mOptEnableHighlightCards)
        submenu.add_checkbutton(label="Enable highlight same &rank", variable=self.tkopt.highlight_samerank, command=self.mOptEnableHighlightSameRank)
        menu.add_separator()
##        menu.add_checkbutton(label="Sound", variable=self.tkopt.sound, command=self.mOptSound)
        if 1 or len(self.app.cm.len()) > 1:
            submenu = MfxMenu(menu, label="Cards&et")
            cb = (25, self.__cb_max) [ self.app.cm.len() > 4 * 25 ]
            for i in range(self.app.cm.len()):
                columnbreak = i > 0 and (i % cb) == 0
                submenu.add_radiobutton(label=self.app.cm.get(i).name, variable=self.tkopt.cardset, value=i,
                                        command=self.mOptCardset, columnbreak=columnbreak)
        # this submenu will get set by updateBackgroundImages()
        submenu = MfxMenu(menu, label="Card &background")
        tiles = self.app.tiles
        if 1 or len(tiles) > 1:
            submenu = MfxMenu(menu, label="Table ti&le")
            state = Tkinter.NORMAL
            if len(tiles) == 1:
                state = Tkinter.DISABLED
            cb = (25, self.__cb_max) [ len(tiles) > 4 * 25 ]
            for i in range(len(tiles)):
                columnbreak = i > 0 and (i % cb) == 0
                underline = (i == 0) - 1
                submenu.add_radiobutton(label=tiles[i][0], variable=self.tkopt.tabletile, value=i,
                                        command=self.mOptTableTile, columnbreak=columnbreak, state=state,
                                        underline=underline)
        menu.add_command(label="Table colo&r...", command=self.mOptTableColor)
        submenu = MfxMenu(menu, label="A&nimations")
        submenu.add_radiobutton(label="&None", variable=self.tkopt.animations, value=0, command=self.mOptAnimations)
        submenu.add_radiobutton(label="&Fast", variable=self.tkopt.animations, value=1, command=self.mOptAnimations)
        submenu.add_radiobutton(label="&Timer based", variable=self.tkopt.animations, value=2, command=self.mOptAnimations)
        submenu.add_radiobutton(label="&Slow", variable=self.tkopt.animations, value=3, command=self.mOptAnimations)
        submenu.add_radiobutton(label="&Very slow", variable=self.tkopt.animations, value=4, command=self.mOptAnimations)
        menu.add_checkbutton(label="Card shado&w", variable=self.tkopt.shadow, command=self.mOptShadow)
        menu.add_checkbutton(label="Shade &legal moves", variable=self.tkopt.shade, command=self.mOptShade)
        menu.add_separator()
        menu.add_command(label="&Hint speed...", command=self.mOptHintSpeed)
        menu.add_command(label="&Demo speed...", command=self.mOptDemoSpeed)
        menu.add_separator()
        submenu = MfxMenu(menu, label="&Toolbar")
        submenu.add_radiobutton(label="&Hide", variable=self.tkopt.toolbar, value=0, command=self.mOptToolbar)
        submenu.add_radiobutton(label="&Top", variable=self.tkopt.toolbar, value=1, command=self.mOptToolbar)
        submenu.add_radiobutton(label="&Bottom", variable=self.tkopt.toolbar, value=2, command=self.mOptToolbar)
        menu.add_checkbutton(label="Stat&usbar", variable=self.tkopt.statusbar, command=self.mOptStatusbar)
        menu.add_separator()
        menu.add_command(label="&Save options", command=self.mOptSave)

        menu = self._createMenu("&Help")
        menu.add_command(label="&Contents", command=self.mHelp, accelerator="F1")
        menu.add_command(label="&How to play", command=self.mHelpHowToPlay)
        menu.add_command(label="&Rules for this game", command=self.mHelpRules)
        menu.add_command(label="&License terms", command=self.mHelpLicense)
        menu.add_separator()
        menu.add_command(label="&About PySol...", command=self.mHelpAbout)

        if os.name == "mac":
            menu = self._createMenu("Apple")
            menu.add_command(label="About PySol...", command=self.mHelpAbout)
            menu.add_command(label="PySol help", command=self.mHelp, accelerator="F1")

        MfxMenubar.addPath = None

        ### FIXME: all key bindings should be *added* to keyPressHandler
        ctrl = "Control-"
        if os.name == "mac": ctrl = "Command-"
        self._bindKey("",   "n", self.mNewGame)
        self._bindKey("",   "g", self.mSelectGameDialog)
        self._bindKey(ctrl, "r", self.mSelectRandomGame)
        self._bindKey(ctrl, "m", self.mSelectGameById)
        self._bindKey(ctrl, "n", self.mNewGameWithNextId)
        self._bindKey(ctrl, "o", self.mOpen)
        self._bindKey(ctrl, "s", self.mSave)
        self._bindKey(ctrl, "q", self.mQuit)
        self._bindKey("",   "z", self.mUndo)
        self._bindKey("",   "s", self.mUndo)            # undocumented
        self._bindKey("",   "BackSpace", self.mUndo)    # undocumented
        self._bindKey("",   "r", self.mRedo)
        self._bindKey(ctrl, "g", self.mRestart)
        self._bindKey("",   "d", self.mDeal)
        self._bindKey("",   "space", self.mDeal)        # undocumented
        self._bindKey("",   "a", self.mDrop)
        self._bindKey(ctrl, "a", self.mDrop1)
        self._bindKey("",   "t", self.mStatus)
        self._bindKey(ctrl, "t", self.mPlayerStats)     # undocumented
        self._bindKey("",   "h", self.mHint)
        self._bindKey(ctrl, "h", self.mHint1)           # undocumented
        ##self._bindKey("",   "Shift_L", self.mHighlightPiles)
        ##self._bindKey("",   "Shift_R", self.mHighlightPiles)
        self._bindKey("",   "i", self.mHighlightPiles)
        self._bindKey(ctrl, "d", self.mDemo)
        self._bindKey(ctrl, "b", self.mOptChangeCardback) # undocumented
        self._bindKey(ctrl, "i", self.mOptChangeTableTile) # undocumented
        self._bindKey(ctrl, "p", self.mOptPlayerName)   # undocumented
        self._bindKey("",   "F1", self.mHelp)
        self._bindKey(ctrl, "F1", self.mHelpRules)      # undocumented

        if 1 and self.app.debug:
            self._bindKey(ctrl, "Prior", self.mSelectPrevGameByName)
            self._bindKey(ctrl, "Next", self.mSelectNextGameByName)
            self._bindKey(ctrl, "Up", self.mSelectPrevGameById)
            self._bindKey(ctrl, "Down", self.mSelectNextGameById)

    #
    # key binding utility
    #

    def _bindKey(self, modifier, key, func):
        if 0 and not modifier and len(key) == 1:
            self.__keybindings[string.lower(key)] = func
            self.__keybindings[string.upper(key)] = func
            return
        sequence = "<" + modifier + "KeyPress-" + key + ">"
        try:
            bind(self.top, sequence, func)
            if len(key) == 1 and key != string.upper(key):
                key = string.upper(key)
                sequence = "<" + modifier + "KeyPress-" + key + ">"
                bind(self.top, sequence, func)
        except:
            raise                                                   #bundle#
            pass


    def _keyPressHandler(self, *event):
        r = EVENT_PROPAGATE
        if event and event[0] and self.game:
            keypress = event[0]
            ##print keypress.__dict__
            if self.game.demo:
                # stop the demo by setting self.game.demo.keypress
                if keypress.char:    # ignore Ctrl/Shift/etc.
                    self.game.demo.keypress = keypress.char
                    r = EVENT_HANDLED
            func = self.__keybindings.get(keypress.char)
            if func and (keypress.state & ~2) == 0:
                func(event)
                r = EVENT_HANDLED
        return r


    # utility
    def _createMenu(self, label, tearoff=0):
        # It took me hours to find out how to place the help-menu
        # on the right side of the menubar. Finally I discovered
        # that there seems to be some magic with the widget name...
        if os.name == "posix":
            tearoff = 1
        menu = MfxMenu(self.__menubar, label, tearoff=tearoff, underline=-1)
        return menu


    #
    # Select Game menu entry
    #

    # new implementation - use a tree widget
    def _addSelectGameMenu(self, menu):
        if 0:
            menu.add_command(label="Select &game...", command=self.mSelectGameDialog, accelerator="G")
            return
        games = map(self.app.getGameInfo, self.app.getGamesIdSortedByName())
        games = tuple(games)
        submenu = MfxMenu(menu, label="Select &game")
        submenu.add_command(label="All games...", command=self.mSelectGameDialog, accelerator="G")
        data = ("Popular games", lambda gi, l=GI.POPULAR_GAMES: gi.id in l)
        self._addSelectGameSubMenu(submenu, games, (data, ),
                                   self.mSelectGamePopular, self.tkopt.gameid_popular)
        submenu.add_separator()
        self._addSelectGameSubMenu(submenu, games, GI.SELECT_GAME_BY_TYPE,
                                   self.mSelectGame, self.tkopt.gameid)


    def _addSelectGameSubMenu(self, menu, games, select_data, command, variable):
        ##print select_data
        for label, select_func in select_data:
            g = filter(select_func, games)
            if not g:
                continue
            submenu = MfxMenu(menu, label=label)
            cb = (25, self.__cb_max) [ len(g) > 4 * 25 ]
            for i in range(len(g)):
                gi = g[i]
                columnbreak = i > 0 and (i % cb) == 0
                submenu.add_radiobutton(command=command, variable=variable,
                                        columnbreak=columnbreak,
                                        value=gi.id, label=gi.name)

    def mSelectGameDialog(self, *event):
        if self._cancelDrag(): return
        d = SelectGameDialog(self.top, title="Select Game", gameid=self.game.id)
        if d.status == 0 and d.num == 0 and d.gameid != self.game.id:
            self.tkopt.gameid.set(d.gameid)
            self.tkopt.gameid_popular.set(d.gameid)
            if 0:
                self.mSelectGame()
            else:
                # don't ask areYouSure()
                self.game._quitGame(d.gameid)
        return EVENT_HANDLED


    #
    # menubar overrides
    #

    def updateBackgroundImages(self):
        # delete all entries
        submenu = self.__menupath[".menubar.options.cardbackground"][2]
        submenu.delete(0, "last")
        # insert new cardbacks
        mbacks = self.app.images.getCardbacks()
        cb = int(math.ceil(math.sqrt(len(mbacks))))
        for i in range(len(mbacks)):
            columnbreak = i > 0 and (i % cb) == 0
            submenu.add_radiobutton(label=mbacks[i].name, image=mbacks[i].menu_image, variable=self.tkopt.cardback, value=i,
                                    command=self.mOptCardback, columnbreak=columnbreak, indicatoron=0, hidemargin=0)


    #
    # menu updates
    #

    def setMenuState(self, state, path):
        #print state, path
        s = (Tkinter.DISABLED, Tkinter.NORMAL)[state]
        path = ".menubar." + path
        menu, index, submenu = self.__menupath[path]
        menu.entryconfig(index, state=s)

    def setToolbarState(self, state, path):
        #print state, path
        s = (Tkinter.DISABLED, Tkinter.NORMAL)[state]
        w = getattr(self.app.toolbar, path + "_button")
        w["state"] = s


    #
    # menu actions
    #

    FILETYPES = (("PySol files", "*.pso"), ("All files", "*"))

    def mOpen(self, *event):
        if self._cancelDrag(): return
        filename = self.game.filename
        if filename:
            idir, ifile = os.path.split(os.path.normpath(filename))
        else:
            idir, ifile = "", ""
        if not idir:
            idir = self.app.dn.savegames
        filename = tkFileDialog.askopenfilename(
                filetypes=self.FILETYPES,
                initialdir=idir, initialfile=ifile)
        if filename:
            filename = os.path.normpath(filename)
            ##filename = os.path.normcase(filename)
            if os.path.isfile(filename):
                self.game.loadGame(filename)

    def mSaveAs(self, *event):
        if self._cancelDrag(): return
        if not self.menustate.save_as:
            return
        filename = self.game.filename
        if not filename:
            filename = self.app.getGameSaveName(self.game.id)
            if os.name == "posix":
                filename = filename + "-" + self.game.getFullId(format=0)
            else:
                filename = filename + "-01"
            filename = filename + ".pso"
        idir, ifile = os.path.split(os.path.normpath(filename))
        if not idir:
            idir = self.app.dn.savegames
        filename = tkFileDialog.asksaveasfilename(
                filetypes=self.FILETYPES,
                initialdir=idir, initialfile=ifile)
        if filename:
            filename = os.path.normpath(filename)
            ##filename = os.path.normcase(filename)
            self.game.saveGame(filename)
            self.updateMenus()

    def _mDoStats(self, player, header, title):
        d = StatsDialog(self.top, title, self.app, player, header)
        if d.status == 0:
            self._mHandleStats(player, header, title, d.num)

    def mPlayerLog(self, *args):
        if self._cancelDrag(): return
        player = self.app.opt.player
        title="Log for " + player
        d = StatsLogDialog(self.top, title, self.app, player, header="")

    def _mOptCardback(self, index):
        if self._cancelDrag(): return
        if self.app.images.setCardback(index):
            index = self.app.images.getCardbackIndex()
            for card in self.game.cards:
                image = self.app.images.getBack(card.deck)
                card.updateCardBackground(image=image)
            self.app.canvas.update_idletasks()
            self.tkopt.cardback.set(index)
            self.app.opt.cardback_name = self.app.images.getCardbacks()[index].name

    def mOptCardback(self, *event):
        self._mOptCardback(self.tkopt.cardback.get())

    def mOptChangeCardback(self, *event):
        self._mOptCardback(None)

    def _mOptTableTile(self, i):
        if self.app.tabletile_index != i:
            if self.game.canvas.setTile(self.app.tiles[i][1]):
                self.game.canvas.setTextColor(self.app.tiles[i][3])
                if self.app.debug: self.game.updateStatus(gameid=self.app.tiles[i][2])
                self.tkopt.tabletile.set(i)
                self.app.tabletile_index = i
                self.app.opt.tabletile_name = self.app.tiles[i][2]

    def mOptTableTile(self, *event):
        if self._cancelDrag(): return
        self._mOptTableTile(self.tkopt.tabletile.get())

    def mOptChangeTableTile(self, *event):
        if self._cancelDrag(): return
        i = (self.tkopt.tabletile.get() + 1) % len(self.app.tiles)
        self._mOptTableTile(i)

    def mOptTableColor(self, *event):
        if self._cancelDrag(): return
        c = tkColorChooser.askcolor(initialcolor=self.app.opt.tablecolor,
                                    title="Select table color")
        if c and c[1]:
            self._mOptTableTile(0)
            self.app.opt.tablecolor = c[1]
            self.game.canvas.config(bg=self.app.opt.tablecolor)
            self.top.config(bg=self.app.opt.tablecolor)

    def mOptToolbar(self, *event):
        if self._cancelDrag(): return
        self.setToolbarSide(self.tkopt.toolbar.get())

    def mOptToolbarRelief(self, *event):
        if self._cancelDrag(): return
        self.setToolbarRelief(self.tkopt.toolbar_relief.get())

    def mOptStatusbar(self, *event):
        if self._cancelDrag(): return
        if not self.app.statusbar: return
        side = self.tkopt.statusbar.get()
        self.app.opt.statusbar = side
        if self.app.statusbar.show(side):
            self.top.update_idletasks()


    #
    # toolbar support
    #

    def setToolbarSide(self, side):
        if self._cancelDrag(): return
        self.app.opt.toolbar = side
        self.tkopt.toolbar.set(side)                    # update radiobutton
        if self.app.toolbar.show(side):
            self.top.update_idletasks()

    def setToolbarRelief(self, relief):
        if self._cancelDrag(): return
        self.app.opt.toolbar_relief = relief
        self.tkopt.toolbar_relief.set(relief)           # update radiobutton
        ### FIXME
        self.top.update_idletasks()

