## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import os, re, sys, string, time

# PySol imports
from mfxutil import EnvError, SubclassResponsibility                #bundle#
from mfxutil import Struct, destruct                                #bundle#
from random import constructRandom                                  #bundle#
from stats import PysolStatsWriter                                  #bundle#

# toolkit imports
from pysoltk import EVENT_HANDLED, EVENT_PROPAGATE                  #bundle#
from pysoltk import MfxDialog, MfxSimpleSlider, MfxSimpleEntry      #bundle#
from pysoltk import MfxExceptionDialog                              #bundle#
from pysoltk import MfxCheckMenuItem, MfxRadioMenuItem              #bundle#
from pysoltk import getFont                                         #bundle#
from help import helpAbout, helpHTML                                #bundle#


# /***********************************************************************
# // menubar
# ************************************************************************/

class PysolMenubarActions:
    def __init__(self, app, top):
        self.app = app
        self.top = top
        self.game = None
        # this is set by updateMenuState()
        self.menustate = Struct(
            save = 0,
            save_as = 0,
            undo = 0,
            redo = 0,
            restart = 0,
            deal = 0,
            hint = 0,
            autofaceup = 0,
            autodrop = 0,
            autodeal = 0,
            quickplay = 0,
            demo = 0,
            highlight_piles = 0,
            rules = 0,
        )
        # structure to convert menu-options to Toolkit variables
        self.tkopt = Struct(
            gameid = MfxRadioMenuItem(self),
            gameid_popular = MfxRadioMenuItem(self),
            confirm = MfxCheckMenuItem(self),
            autofaceup = MfxCheckMenuItem(self),
            autodrop = MfxCheckMenuItem(self),
            autodeal = MfxCheckMenuItem(self),
            quickplay = MfxCheckMenuItem(self),
            undo = MfxCheckMenuItem(self),
            hint = MfxCheckMenuItem(self),
            highlight_piles = MfxCheckMenuItem(self),
            highlight_cards = MfxCheckMenuItem(self),
            highlight_samerank = MfxCheckMenuItem(self),
            sound = MfxCheckMenuItem(self),
            cardset = MfxRadioMenuItem(self),
            cardback = MfxRadioMenuItem(self),
            tabletile = MfxRadioMenuItem(self),
            animations = MfxRadioMenuItem(self),
            shadow = MfxCheckMenuItem(self),
            shade = MfxCheckMenuItem(self),
            toolbar = MfxRadioMenuItem(self),
            toolbar_relief = MfxRadioMenuItem(self),
            statusbar = MfxCheckMenuItem(self),
        )

    def connectGame(self, game):
        self.game = game
        if game is None:
            return
        assert self.app is game.app
        # set state of the menu items
        self.tkopt.gameid.set(self.game.id)
        self.tkopt.gameid_popular.set(self.game.id)
        self.tkopt.confirm.set(self.app.opt.confirm)
        self.tkopt.autofaceup.set(self.app.opt.autofaceup)
        self.tkopt.autodrop.set(self.app.opt.autodrop)
        self.tkopt.autodeal.set(self.app.opt.autodeal)
        self.tkopt.quickplay.set(self.app.opt.quickplay)
        self.tkopt.undo.set(self.app.opt.undo)
        self.tkopt.hint.set(self.app.opt.hint)
        self.tkopt.highlight_piles.set(self.app.opt.highlight_piles)
        self.tkopt.highlight_cards.set(self.app.opt.highlight_cards)
        self.tkopt.highlight_samerank.set(self.app.opt.highlight_samerank)
        self.tkopt.sound.set(self.app.opt.sound)
        self.tkopt.cardset.set(self.app.cardset.index)
        self.tkopt.cardback.set(self.app.images.getCardbackIndex())
        self.tkopt.tabletile.set(self.app.tabletile_index),
        self.tkopt.animations.set(self.app.opt.animations)
        self.tkopt.shadow.set(self.app.opt.shadow)
        self.tkopt.shade.set(self.app.opt.shade)
        self.tkopt.toolbar.set(self.app.opt.toolbar)
        self.tkopt.toolbar_relief.set(self.app.opt.toolbar_relief)
        self.tkopt.statusbar.set(self.app.opt.statusbar)

    # will get called after a new cardset has been loaded
    def updateBackgroundImages(self):
        pass


    #
    # delegation to Game
    #

    def _finishDrag(self):
        return self.game is None or self.game._finishDrag()

    def _cancelDrag(self):
        return self.game is None or self.game._cancelDrag()

    def changed(self):
        assert self.game is not None
        return self.game.changed()


    #
    # menu updates
    #

    def setMenuState(self, state, path):
        raise SubclassResponsibility

    def setToolbarState(self, state, path):
        raise SubclassResponsibility

    # update self.menustate for menu items and toolbar
    def _updateMenuState(self):
        game = self.game
        assert game is not None
        opt = self.app.opt
        ms = self.menustate
        for k in ms.__dict__.keys():
            ms.__dict__[k] = 0
        # 0 = DISABLED, 1 = ENABLED
        ms.save_as = game.canSaveGame()
        if game.filename and ms.save_as:
            ms.save = 1
        if opt.undo:
            if game.moves.index > 0:
                ms.undo = 1
            if game.moves.index < len(game.moves.history):
                ms.redo = 1
        if game.moves.index > 0:
            ms.restart = 1
        if game.canDealCards():
            ms.deal = 1
        if game.getHintClass() is not None:
            if opt.hint:
                ms.hint = 1
            ###if not game.demo:       # if not already running
            ms.demo = 1
        autostacks = game.getAutoStacks()
        if autostacks[0]:
            ms.autofaceup = 1
        if autostacks[1] and game.s.foundations:
            ms.autodrop = 1
        if game.s.waste:
            ms.autodeal = 1
        if autostacks[2]:
            ms.quickplay = 1
        ms.highlight_piles = 0
        if opt.highlight_piles and game.getHighlightPilesStacks():
            ms.highlight_piles = 1
        if game.app.getGameRulesFilename(game.id):  # note: this may return ""
            ms.rules = 1

    # update menu items and toolbar
    def updateMenus(self):
        if self.game is None:
            return
        self._updateMenuState()
        ms = self.menustate
        # File menu
        self.setMenuState(ms.save, "file.save")
        self.setMenuState(ms.save_as, "file.saveas")
        # Edit menu
        self.setMenuState(ms.undo, "edit.undo")
        self.setMenuState(ms.redo, "edit.redo")
        self.setMenuState(ms.redo, "edit.redoall")
        self.setMenuState(ms.restart, "edit.restartgame")
        # Game menu
        self.setMenuState(ms.deal, "game.dealcards")
        self.setMenuState(ms.autodrop, "game.autodrop")
        # Assist menu
        self.setMenuState(ms.hint, "assist.hint")
        self.setMenuState(ms.highlight_piles, "assist.highlightpiles")
        self.setMenuState(ms.demo, "assist.demo")
        self.setMenuState(ms.demo, "assist.demoallgames")
        # Options menu
        self.setMenuState(ms.autofaceup, "options.automaticplay.autofaceup")
        self.setMenuState(ms.autodrop, "options.automaticplay.autodrop")
        self.setMenuState(ms.autodeal, "options.automaticplay.autodeal")
        self.setMenuState(ms.quickplay, "options.automaticplay.quickplay")
        # Help menu
        self.setMenuState(ms.rules, "help.rulesforthisgame")
        # Toolbar
        self.setToolbarState(ms.restart, "restart")
        self.setToolbarState(ms.save_as, "save")
        self.setToolbarState(ms.undo, "undo")
        self.setToolbarState(ms.redo, "redo")
        self.setToolbarState(ms.autodrop, "autodrop")
        self.setToolbarState(ms.rules, "rules")


    #
    # menu actions
    #

    def mNewGame(self, *args):
        if self._cancelDrag(): return
        if self.changed():
            if not self.game.areYouSure("New game"): return
        if self.app.nextgame.cardset_index == self.app.cardset.index:
            self.game.newGame()
        else:
            self.game._quitGame(self.game.id)

    def _mSelectGame(self, id):
        if self._cancelDrag(): return
        if self.game.id == id:
            return
        if self.changed():
            if not self.game.areYouSure("Select game"):
                # restore radiobutton settings
                self.tkopt.gameid.set(self.game.id)
                self.tkopt.gameid_popular.set(self.game.id)
                return
        self.tkopt.gameid.set(id)
        self.tkopt.gameid_popular.set(id)
        self.game._quitGame(id)

    def mSelectGame(self, *args):
        self._mSelectGame(self.tkopt.gameid.get())

    def mSelectGamePopular(self, *args):
        self._mSelectGame(self.tkopt.gameid_popular.get())

    def _mNewGameBySeed(self, seed):
        try:
            id, random = constructRandom(seed)
            if id is None:
                id = self.game.id
            if random is None:
                return
        except (ValueError, TypeError), ex:
            d = MfxDialog(self.top, title="Invalid game number",
                          text="Invalid game number\n" + str(seed),
                          bitmap="error")
            return
        if id == self.game.id and self.app.nextgame.cardset_index == self.app.cardset.index:
            self.game.newGame(random=random)
        else:
            self.game._quitGame(id, random=random)

    def mNewGameWithNextId(self, *args):
        if self.changed():
            if not self.game.areYouSure("Select next game number"): return
        r = self.game.random
        seed = r.increaseSeed(r.initial_seed)
        seed = r.str(seed)
        self._mNewGameBySeed(seed)

    def mSelectGameById(self, *args):
        if self._cancelDrag(): return
        id, f = None, self.game.getFullId(format=1)
        d = MfxSimpleEntry(self.top, "Select new game number", "\n\nEnter new game number", f,
                           strings=("Ok", "Next game", "Cancel"), default=0,
                           e_width=25)
        if d.status != 0: return
        if d.num == 2: return
        if d.num == 1:
            self.mNewGameWithNextId()
            return
        if self.changed():
            if not self.game.areYouSure("Select new game number"): return
        self._mNewGameBySeed(d.value)

    def mSelectRandomGame(self, *args):
        if self._cancelDrag(): return
        if self.changed():
            if not self.game.areYouSure("Select random game"): return
        while 1:
            id = self.app.getRandomGameId()
            if 1 and id == self.game.id:        # force change of game
                continue
            break
        self.game._quitGame(id)

    def _mSelectNextGameFromList(self, gl, step):
        if self._cancelDrag(): return
        id = self.game.id
        if len(gl) < 2 or not id in gl:
            return
        if self.changed():
            if not self.game.areYouSure("Select next game"): return
        index = (gl.index(id) + step) % len(gl)
        self.game._quitGame(gl[index])

    def mSelectNextGameById(self, *args):
        self._mSelectNextGameFromList(self.app.getGamesIdSortedById(), 1)

    def mSelectPrevGameById(self, *args):
        self._mSelectNextGameFromList(self.app.getGamesIdSortedById(), -1)

    def mSelectNextGameByName(self, *args):
        self._mSelectNextGameFromList(self.app.getGamesIdSortedByName(), 1)

    def mSelectPrevGameByName(self, *args):
        self._mSelectNextGameFromList(self.app.getGamesIdSortedByName(), -1)

    def mSave(self, *args):
        if self._cancelDrag(): return
        if self.menustate.save_as:
            if self.game.filename:
                self.game.saveGame(self.game.filename)
            else:
                self.mSaveAs()

    def mQuit(self, *args):
        if self._cancelDrag(): return
        if self.changed():
            if not self.game.areYouSure("Quit PySol"): return
        self.game._quitGame()

    def mUndo(self, *args):
        if self._cancelDrag(): return
        if self.app.opt.undo:
            self.game.undo()

    def mRedo(self, *args):
        if self._cancelDrag(): return
        if self.app.opt.undo:
            self.game.redo()
            self.game.checkForWin()

    def mRedoAll(self, *args):
        if self._cancelDrag(): return
        if self.app.opt.undo:
            while self.game.moves.index < len(self.game.moves.history):
                self.game.redo()
                if self.game.checkForWin():
                    break

    def mRestart(self, *args):
        if self._cancelDrag(): return
        if self.game.moves.index == 0:
            return
        if self.changed():
            if not self.game.areYouSure("Restart game","Restart this game ?"): return
        self.game.restartGame()

    def mDeal(self, *args):
        if self._cancelDrag(): return
        self.game.dealCards()

    def mDrop(self, *args):
        if self._cancelDrag(): return
        self.game.autoPlay(autofaceup=-1, autodrop=1)

    def mDrop1(self, *args):
        if self._cancelDrag(): return
        self.game.autoPlay(autofaceup=1, autodrop=1)

    def mStatus(self, *args):
        game, stats = self.game, self.game.stats
        if self._cancelDrag(): return
        w = ""
        if game.s.talon:
            w = w + "\nRedeals: " + str(game.s.talon.round - 1)
            w = w + "\nCards in Talon: " + str(len(game.s.talon.cards))
        if game.s.waste:
            w = w + "\nCards in Waste: " + str(len(game.s.waste.cards))
        if game.s.foundations:
            n = 0
            for s in game.s.foundations:
                n = n + len(s.cards)
            w = w + "\nCards in Foundations: " + str(n)
        d = MfxDialog(self.top,title="Game status",
                   text=game.getTitleName() + "\n" +
                        game.getFullId(format=2) + "\n" +
                        "Playing time: " + game.getTime() + "\n\n" +
                        "Moves: " + str(game.moves.index) + "\n" +
                        "Undo moves: " + str(stats.undo_moves) + "\n" +
                        "Demo moves: " + str(stats.demo_moves) + "\n\n" +
###                        "Total player moves: " + str(stats.player_moves) + "\n" +
###                        "Total moves in this game: " + str(stats.total_moves) + "\n" +
                        "Hints: " + str(stats.hints) + "\n" +
                        "Highlight piles: " + str(stats.highlight_piles) + "\n" +
                        "Highlight cards: " + str(stats.highlight_cards) + "\n" +
                        "Highlight same rank: " + str(stats.highlight_samerank) + "\n" +
                        w,
                   strings=("Ok", "Stats...", "Log...", "Demo..."),
                   padx=30, separatorwidth=2)
        if d.status == 0:
            if d.num == 1:
                self.mPlayerStats()
            if d.num == 2:
                self.mPlayerLog()
            elif d.num == 3:
                self.mDemoStats()

    def _mHandleStats(self, player, header, title, num):
        n = self.game.getTitleName()
        # print to file
        if num == 3:
            file = None
            filename, text = "stats.txt", "Your statistics"
            if player is None:
                filename, text = "stats_demo.txt", "Demo statistics"
            filename = os.path.join(self.app.dn.config, filename)
            filename = os.path.normpath(filename)
            try:
                file = open(filename, "a")
                a = PysolStatsWriter(self.app)
                writer = a.FileWriter(file)
                a.writeFullStats(writer, player, header, title)
                destruct(a)
            except EnvError, ex:
                if file: file.close()
                d = MfxExceptionDialog(self.top, ex,
                              text="Error while writing to file")
            else:
                if file: file.close()
                d = MfxDialog(self.top, title="PySol Info", bitmap="info",
                              text=text + " were appended to\n" + filename)
            return
        # reset
        if player is not None:
            if num == 1:
                if self.game.areYouSure("Reset player statistics",
                                        "Reset your statistics for\n" + n + " ?",
                                        confirm=1, default=1):
                    self.app.stats.resetStats(player, self.game.id)
            elif num == 2:
                if self.game.areYouSure("Reset player statistics",
                                        "Reset ALL statistics for\nplayer " + player + " ?",
                                        confirm=1, default=1):
                    self.app.stats.resetStats(player, 0)
            self.game.updateStatus(stats=self.app.stats.getStats(player, self.game.id))
        else:
            if num == 1:
                if self.game.areYouSure("Reset demo statistics",
                                        "Reset demo statistics for\n" + n + " ?",
                                         confirm=1, default=1):
                    self.app.stats.resetStats(player, self.game.id)
            elif num == 2:
                if self.game.areYouSure("Reset demo statistics",
                                        "Reset ALL demo statistics ?",
                                        confirm=1, default=1):
                    self.app.stats.resetStats(player, 0)

    # default implementation uses a simple dialog with a fixed size font
    def _mDoStats(self, player, header, title):
        a = PysolStatsWriter(self.app)
        writer = a.StringWriter()
        a.writeStats(writer, player, header)
        text = writer.text
        destruct(a)
        font = buttonfont = getFont("fixed")
        d = MfxDialog(self.top, title=title,
                      text=text, justify="left",
                      font=font, buttonfont=buttonfont,
                      strings=("Ok", "Reset game", "Reset all", "Print to file"),
                      padx=40, separatorwidth=2)
        if d.status == 0:
            self._mHandleStats(player, header, title, d.num)

    def mPlayerStats(self, *args):
        if self._cancelDrag(): return
        player = self.app.opt.player
        self._mDoStats(player, player, title="Statistics for " + player)

    def mDemoStats(self, *args):
        if self._cancelDrag(): return
        self._mDoStats(None, "Demo games", title="PySol demo statistics")

    def mPlayerLog(self, *args):
        pass

    def mHint(self, *args):
        if self._cancelDrag(): return
        if self.app.opt.hint:
            if self.game.showHint(0, self.app.opt.hint_sleep):
                self.game.stats.hints = self.game.stats.hints + 1

    def mHint1(self, *args):
        if self._cancelDrag(): return
        if self.app.opt.hint:
            if self.game.showHint(1, self.app.opt.hint_sleep):
                self.game.stats.hints = self.game.stats.hints + 1

    def mHighlightPiles(self, *args):
        if self._cancelDrag(): return
        if self.app.opt.highlight_piles:
            if self.game.highlightPiles(self.app.opt.highlight_piles_sleep):
                self.game.stats.highlight_piles = self.game.stats.highlight_piles + 1

    def mDemo(self, *args):
        if self._cancelDrag(): return
        if self.game.getHintClass() is not None:
            self._mDemo(mixed=0)

    def mMixedDemo(self, *args):
        if self._cancelDrag(): return
        self._mDemo(mixed=1)

    def _mDemo(self, mixed):
        if self._cancelDrag(): return
        if self.changed():
            # only ask if there have been no demo moves or hints yet
            if self.game.stats.demo_moves == 0 and self.game.stats.hints == 0:
                if not self.game.areYouSure("Play demo"): return
        ##self.app.demo_counter = 0
        self.game.startDemo(mixed=mixed)

    def mOptPlayerName(self, *args):
        if self._cancelDrag(): return
        d = MfxSimpleEntry(self.top, "Set player name", "\nPlease enter your name",
                           self.app.opt.player, e_width=20)
        if d.status == 0:
            n = string.strip(d.value)
            if 0 < len(n) <= 30:
                self.app.opt.player = n
                self.game.updateStatus(player=self.app.opt.player)
                self.game.updateStatus(stats=self.app.stats.getStats(self.app.opt.player, self.game.id))

    def mOptConfirm(self, *args):
        if self._cancelDrag(): return
        self.app.opt.confirm = self.tkopt.confirm.get()

    def mOptAutoFaceUp(self, *args):
        if self._cancelDrag(): return
        self.app.opt.autofaceup = self.tkopt.autofaceup.get()
        if self.app.opt.autofaceup:
            self.game.autoPlay()

    def mOptAutoDrop(self, *args):
        if self._cancelDrag(): return
        self.app.opt.autodrop = self.tkopt.autodrop.get()
        if self.app.opt.autodrop:
            self.game.autoPlay()

    def mOptAutoDeal(self, *args):
        if self._cancelDrag(): return
        self.app.opt.autodeal = self.tkopt.autodeal.get()
        if self.app.opt.autodeal:
            self.game.autoPlay()

    def mOptQuickPlay(self, *args):
        if self._cancelDrag(): return
        self.app.opt.quickplay = self.tkopt.quickplay.get()

    def mOptEnableUndo(self, *args):
        if self._cancelDrag(): return
        self.app.opt.undo = self.tkopt.undo.get()
        self.game.updateMenus()

    def mOptEnableHint(self, *args):
        if self._cancelDrag(): return
        self.app.opt.hint = self.tkopt.hint.get()
        self.game.updateMenus()

    def mOptEnableHighlightPiles(self, *args):
        if self._cancelDrag(): return
        self.app.opt.highlight_piles = self.tkopt.highlight_piles.get()
        self.game.updateMenus()

    def mOptEnableHighlightCards(self, *args):
        if self._cancelDrag(): return
        self.app.opt.highlight_cards = self.tkopt.highlight_cards.get()
        self.game.updateMenus()

    def mOptEnableHighlightSameRank(self, *args):
        if self._cancelDrag(): return
        self.app.opt.highlight_samerank = self.tkopt.highlight_samerank.get()
        self.game.updateMenus()

    def mOptSound(self, *args):
        if self._cancelDrag(): return
        self.app.opt.sound = self.tkopt.sound.get()

    def mOptCardset(self, *args):
        if self._cancelDrag(): return
        self.app.nextgame.cardset_index = self.tkopt.cardset.get()

    def mOptAnimations(self, *args):
        if self._cancelDrag(): return
        self.app.opt.animations = self.tkopt.animations.get()

    def mOptShadow(self, *args):
        if self._cancelDrag(): return
        self.app.opt.shadow = self.tkopt.shadow.get()

    def mOptShade(self, *args):
        if self._cancelDrag(): return
        self.app.opt.shade = self.tkopt.shade.get()

    def mOptIrregularPiles(self, *args):
        if self._cancelDrag(): return
        self.app.opt.irregular_piles = self.tkopt.irregular_piles.get()

    def mOptDemoSpeed(self, *args):
        if self._cancelDrag(): return
        d = MfxSimpleSlider(self.top, "Set demo speed", "Set demo delay in seconds",
                            self.app.opt.demo_sleep, 0.2, 9.9, 0.1)
        if d.status == 0:
            self.app.opt.demo_sleep = d.value

    def mOptHintSpeed(self, *args):
        if self._cancelDrag(): return
        d = MfxSimpleSlider(self.top, "Set hint speed", "Set hint delay in seconds",
                            self.app.opt.hint_sleep, 0.2, 9.9, 0.1)
        if d.status == 0:
            self.app.opt.hint_sleep = d.value

    def mOptSave(self, *args):
        if self._cancelDrag(): return
        try:
            self.app.saveOptions()
        except Exception, ex:
            d = MfxExceptionDialog(self.top, ex,
                                   text="Error while saving options")
        else:
            # tell the player where their config files reside
            d = MfxDialog(self.top, title="PySol Info",
                          text="Options were saved to\n\n" + self.app.fn.opt,
                          bitmap="info")

    def mHelp(self, *args):
        if self._cancelDrag(): return
        helpHTML(self.app, "index.html", "html")

    def mHelpHowToPlay(self, *args):
        if self._cancelDrag(): return
        helpHTML(self.app, "usage.html", "html")

    def mHelpRules(self, *args):
        if self._cancelDrag(): return
        if not self.menustate.rules:
            return
        dir = os.path.join("html", "rules")
        ## FIXME: plugins
        helpHTML(self.app, self.app.getGameRulesFilename(self.game.id), dir)

    def mHelpLicense(self, *args):
        if self._cancelDrag(): return
        helpHTML(self.app, "gpl.html", "html")

    def mHelpAbout(self, *args):
        if self._cancelDrag(): return
        helpAbout(self.app)


# /***********************************************************************
# // toolbar
# ************************************************************************/

class PysolToolbarActions:
    def __init__(self):
        self.game = None
        self.menubar = None

    #
    # public methods
    #

    def connectGame(self, game, menubar):
        self.game = game
        self.menubar = menubar


    #
    # button event handlers - delegate to menubar
    #

    def _busy(self):
        raise SubclassResponsibility

    def mNewGame(self, *args):
        if not self._busy():
            self.menubar.mNewGame()
        return 1

    def mOpen(self, *args):
        if not self._busy():
            self.menubar.mOpen()
        return 1

    def mRestart(self, *args):
        if not self._busy():
            self.menubar.mRestart()
        return 1

    def mSave(self, *args):
        if not self._busy():
            self.menubar.mSaveAs()
        return 1

    def mUndo(self, *args):
        if not self._busy():
            self.menubar.mUndo()
        return 1

    def mRedo(self, *args):
        if not self._busy():
            self.menubar.mRedo()
        return 1

    def mDrop(self, *args):
        if not self._busy():
            self.menubar.mDrop()
        return 1

    def mStatus(self, *args):
        if not self._busy():
            self.menubar.mStatus()
        return 1

    def mPlayerStats(self, *args):
        if not self._busy():
            self.menubar.mPlayerStats()
        return 1

    def mHelpRules(self, *args):
        if not self._busy():
            self.menubar.mHelpRules()
        return 1

    def mQuit(self, *args):
        if not self._busy():
            self.menubar.mQuit()
        return 1

    def mOptPlayerName(self, *args):
        if not self._busy():
            self.menubar.mOptPlayerName()
        return 1

