/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 by the Free Software Foundation.               *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define FIND_DEFENSE  0
#define READLAD1      1
#define SAVESTONE2    2
#define SAVESTONE3    3

#define ATTACK        4
#define READLAD2      5
#define BASICNET3     6

#define UNUSED(x)  x=x

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "liberty.h"
#include "hash.h"
#include <assert.h>


/*
 * The functions in reading.c are used to read wether groups 
 * can be captured or not.  See the file docs/READING for debugging
 * suggestions including the use of gdb and sgf files.
 *
 * NULL POINTERS: Many functions in this file can use pointers
 * to return the locations of recommended plays. These can be
 * set NULL in which case these values are not returned.
 */



static void chain(int i, int j, char ma[MAX_BOARD][MAX_BOARD],
		  char ml[MAX_BOARD][MAX_BOARD], int *adj, int adji[MAXCHAIN],
		  int adjj[MAXCHAIN], int adjsize[MAXCHAIN], 
		  int adjlib[MAXCHAIN], int mark);
static int relative_break_chain(int, int, int *, int *, int, int);
static int special_rescue(int si, int sj, int ai, int aj, int *ti, int *tj);

/* ================================================================ */


static int
get_read_result(/*int color, */int routine, int si, int sj, 
		Read_result **read_result)
{
  Hashnode      * hashnode;
  int             retval;

#if HASHING
#if CHECK_HASHING
  Hashposition    pos;
  unsigned long   hash;

  UNUSED(color);

  /* Check the hash table to see if we have had this position before. */
  board_to_position(p, ko_i, ko_j, /*color,*/ &pos);
  hash = board_hash(p, ko_i, ko_j/*, color*/);
  assert(hash == hashitem.hashval);
  assert(hashposition_compare(&pos, &(hashitem.hashpos)) == 0);

  /* Find this position in the table.  If it wasn't found, enter it. */
  hashnode = hashtable_search(movehash, &pos, hash);
  if (hashnode != NULL)
    RTRACE("We found position %d in the hash table...", hash);
  else
    hashnode = hashtable_enter_position(movehash, &pos, hash);
#else
  /* Find this position in the table.  If it wasn't found, enter it. */
  hashnode = hashtable_search(movehash, &(hashitem.hashpos), hashitem.hashval);
  if (hashnode != NULL)
    RTRACE("We found position %d in the hash table...", hashitem.hashval);
  else
    hashnode = hashtable_enter_position(movehash,
					&(hashitem.hashpos), hashitem.hashval);
#endif

#else  /* HASHING */
  hashnode = NULL;
#endif

  retval = 0;
  if (hashnode == NULL) {
    /* No hash node, thus we can't enter a result into it.  :-( */
    *read_result = NULL;

  } else {

    /* We found it!  Now see if we can find a previous result. */
    *read_result = hashnode_search(hashnode, routine, si, sj);

   if  (*read_result != NULL) 
     retval = 1;
   else {
      RTRACE("...but no previous result for routine %d and (%m)...",
	     routine, si, sj);

      /* Only store the result if stackp <= depth. Above that, there
	 is no branching, so we won't gain anything. */
      if (stackp > depth) 
	*read_result = NULL;
      else {
	*read_result = hashnode_new_result(movehash, hashnode, 
					   routine, si, sj);
	if (*read_result == NULL)
	  RTRACE("...and unfortunately there was no room for one.\n");
	else
	  RTRACE("...so we allocate a new one.\n");
      } 
    }
  }

  return retval;
}


/* ================================================================ */


/* If si, sj points to a string with exactly one liberty readlad1 
 * determines whether it can be saved by extending or capturing
 * a boundary chain having one liberty. The function returns 1 if the string
 * can be saved, otherwise 0.
 *
 * The pair readlad1-readlad2 call each other recursively to
 * read situations such as ladders. They read all ladders to the end.
 * If the reading ply (stackp) is deeper than the deep-reading cutoff
 * parameter depth, whose default value DEPTH is defined in liberty, then a
 * string is assumed alive if it can get 3 liberties. When stackp < depth, a
 * string is considered alive if it can get 4 liberties.  */

int 
readlad1(int si, int sj, int *i, int *j) 
{
  int color;
  int di, dj;
  int liberties;
  int             found_read_result;
  Read_result   * read_result;

  DEBUG(DEBUG_LADDER, "readlad1(%m)\n", si, sj);
  assert(p[si][sj]!=EMPTY);
  if (approxlib(si,sj,p[si][sj],2)!=1) {
    verbose=4;
    dump_stack();
    abort;
  }

  RTRACE("try to escape atari on %m.\n",  si, sj);

  color=p[si][sj];

  if (hashflags & HASH_READLAD1) {
  
    found_read_result = get_read_result(/*color, */READLAD1, si, sj, &read_result);
    if (found_read_result
	&& rr_get_depthleft(*read_result) >= depth - stackp)
    {
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_depthleft(*read_result, 
				   READLAD1, si, sj, depth - stackp);
    }
  } else
    read_result = NULL;

  /* If we can capture a bounding chain and live, then do so. */
  if (break_chain(si, sj, &di, &dj, NULL, NULL)) {
    if (i) *i=di;
    if (j) *j=dj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, di, dj);
    }
    return 1;
  }

  /* (di, dj) will be the liberty of the string. */
  approxlib(si, sj, color, 2);
  ASSERT(lib==1, si, sj);
  di=libi[0];
  dj=libj[0];

  /* Try to extend along our own liberty. */
  RTRACE("extending to %m.\n", di, dj);
  if (trymove(di, dj, color, "readlad1-A", si, sj)) {
    liberties=approxlib(di, dj, color, 4);

    /* If we are still in atari, or have no liberties at all, we lose. */
    if (liberties < 2) {
      RTRACE("still atari : it dies!\n");
      popgo();
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 0, 0, 0);
      }
      return 0;
    }

    /* If we are still within the deep reading cutoff, then read further
       even if the group has 3 liberties. */
    if ((stackp<depth) && (liberties==3)) {
      
      /* If the group can be killed in spite of the 3 liberties, return 0. */
      if (basicnet3(si, sj, NULL, NULL)) {
	RTRACE("it dies!\n");
	popgo();
	if (read_result) {
	  rr_set_result_ri_rj(*read_result, 0, 0, 0);
	}
	return 0;

      } else {
	RTRACE("it lives at %m!\n", di, dj);
	if (i) *i=di;
	if (j) *j=dj;
	popgo();
	if (read_result) {
	  rr_set_result_ri_rj(*read_result, 1, di, dj);
	}
	return 1;
      }
    }

    /* If we are here and liberties is >2 (must be 3), then stackp must
       be bigger than the deep reading cutoff, and then we consider 3
       liberties to live. 
       */
    if (liberties>2) {
      RTRACE("it lives at %m!\n", di, dj);
      if (i) *i=di;
      if (j) *j=dj;
      popgo();
      if (read_result) {
	  rr_set_result_ri_rj(*read_result, 1, di, dj);
      }
      return 1;
    }

    /* If we are here, then liberties must be 2 and we don't care if
       stackp is greater than depth. We continue the ladder anyway. */
    ASSERT(liberties==2, si, sj);
    
    if (readlad2(si, sj, NULL, NULL)) {
      RTRACE("DEAD!\n");
      popgo();
      if (read_result) {
	  rr_set_result_ri_rj(*read_result, 0, 0, 0);
      }
      return 0;

    } else {
      RTRACE("ALIVE at %m!\n", di, dj);
      popgo();
      if (i) *i=di;
      if (j) *j=dj;
      if (read_result) {
	  rr_set_result_ri_rj(*read_result, 1, di, dj);
      }
      return 1;
    }
  }

  /* If we get here, then trymove() failed => the ladder works. */
  RTRACE( "Cannot extend : die in ladder\n");

  if (read_result) {
    rr_set_result_ri_rj(*read_result, 0, 0, 0);
  }
  return 0;
}


/* If si, sj points to a group with exactly two liberties
 * readlad2 determines whether it can be captured in ladder or net.
 * If yes, *i, *j is the killing move. i & j may be null if caller 
 * is only interested in whether it can be captured.
 *  
 * Returns 2 if the killing move involves backfilling.
 *
 * See the comment before readlad1 about ladders and reading depth.
 */

int 
readlad2(int si, int sj, int *i, int *j) 
{
  int color, other;
  int ai, aj, bi, bj;
  int di, dj, gi, gj, hi, hj;
  int liberties, r;
  int acount=0, bcount=0;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN], adjsize[MAXCHAIN], adjlib[MAXCHAIN];
  int savei=-1, savej=-1;
  int             found_read_result;
  Read_result   * read_result;

  assert(p[si][sj]!=EMPTY);
  assert(approxlib(si,sj,p[si][sj],3)==2);
  DEBUG(DEBUG_LADDER, "readlad2(%m)\n", si, sj);

  RTRACE("checking attack on %m with 2 liberties\n", si, sj);

  color=p[si][sj];
  other=OTHER_COLOR(color);

  if (hashflags & HASH_READLAD2) {
  
    found_read_result = get_read_result(/*color, */READLAD2, si, sj, &read_result);
    if (found_read_result
	&& rr_get_depthleft(*read_result) >= depth - stackp)
    {
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_depthleft(*read_result, 
				   READLAD2, si, sj, depth - stackp);
    }
  } else
    read_result = NULL;

  /* The attack may fail if a boundary string is in atari and cannot 
   * be defended.  First we must try defending such a string. 
   *
   * We start by just trying to extend the boundary string at
   * (adji[r], adji[r]) to (hi, hj). If that doesn't work, then we 
   * try to find a boundary string to (adji[r], adjj[r]) which is in 
   * atari, and we try capturing at (gi, gj).
   */

  chainlinks(si, sj, &adj, adji, adjj, adjsize, adjlib);
  for (r=0; r<adj; r++) {
    if (adjlib[r]!=1) 
      continue;

    if (stackp > depth) {
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 0, 0, 0);
      }
      return 0;
    }

    if (relative_break_chain(adji[r], adjj[r], &gi, &gj, si, sj)) {
      if (i) *i=gi;
      if (j) *j=gj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, gi, gj);
      }
      return 1;
    }

    approxlib(adji[r], adjj[r], p[adji[r]][adjj[r]], 2);
    hi=libi[0];
    hj=libj[0];
    if (trymove(hi, hj, other, "readlad2-A", si, sj)) {
      if (attack(si, sj, NULL, NULL) 
	  && !find_defense(si, sj, NULL, NULL)) {
	if (i) *i=hi;
	if (j) *j=hj;
	if (read_result) {
	  rr_set_result_ri_rj(*read_result, 1, hi, hj);
	}
	popgo();
	return 1;
      }
      popgo();
    }
  }

  /* Get the two liberties of (si, sj) into (ai, aj) and (bi, bj). */ 
  liberties=approxlib(si, sj, color, 3);
  ASSERT(liberties==2, si, sj);
  ai=libi[0];
  aj=libj[0];
  bi=libi[1];
  bj=libj[1];

  /* if (bi, bj) looks more promising we wish to switch the two liberties.
     We check whether (bi, bj) is adjacent to more open liberties than (ai,aj).
  */

  if ((ai>0) && p[ai-1][aj]==EMPTY)
    acount++;
  if ((ai<board_size-1) && p[ai+1][aj]==EMPTY)
    acount++;
  if ((aj>0) && p[ai][aj-1]==EMPTY)
    acount++;
  if ((aj<board_size-1) && p[ai][aj+1]==EMPTY)
    acount++;
  if ((bi>0) && p[bi-1][bj]==EMPTY)
    bcount++;
  if ((bi<board_size-1) && p[bi+1][bj]==EMPTY)
    bcount++;
  if ((bj>0) && p[bi][bj-1]==EMPTY)
    bcount++;
  if ((bj<board_size-1) && p[bi][bj+1]==EMPTY)
    bcount++;

  if (bcount>acount) {
    ai=libi[1];
    aj=libj[1];
    bi=libi[0];
    bj=libj[0];
  }

  RTRACE("considering atari at %m\n", ai, aj);

  /* we only want to consider the move at (ai,aj) if:
     stackp < backfill_depth                   -or-
     stackp<depth and it is an isolated stone  -or-
     it is not in immediate atari
  */

  if ( (stackp < backfill_depth) 
       || ((stackp < depth) 
	   && ((ai==0) || (p[ai-1][aj] != other))
	   && ((ai==board_size-1) || (p[ai+1][aj] != other))
	   && ((aj==0) || (p[ai][aj-1] != other))
	   && ((aj==board_size-1) || (p[ai][aj+1] != other)))
       || approxlib(ai, aj, other, 2) >1 )
    if (trymove(ai, aj, other, "readlad2-B", si, sj))
      {
	if (!readlad1(si, sj, &di, &dj)) {
	  if (i) *i=ai;
	  if (j) *j=aj;
	  if (read_result) {
	    rr_set_result_ri_rj(*read_result, 1, ai, aj);
	  }
	  RTRACE("%m captures !!\n", ai, aj);
	  popgo();
	  return 1;
	}
	popgo();
      }


  if ((stackp < backfill_depth) && approxlib(ai, aj, other, 2)==1) {
    int ui=-1, uj=-1;

    ui=libi[0];
    uj=libj[0];
    if (trymove(ui, uj, other, "readlad2-C", si, sj))  {
      if (attack(si, sj, NULL, NULL) && 
	  !find_defense(si, sj, NULL, NULL)) {
	savei=ui;
	savej=uj;
      }
      popgo();
    }
  } else
    RTRACE("%m is not a legal move\n", ai, aj);
  
  RTRACE("first atari didn't work - try %m\n", bi, bj);
  
  /* we only want to consider the move at (bi,bj) if
     either it is not in immediate atari, or else
     stackp<depth and it is an isolated stone. */

  if (((stackp < depth) 
       && ((bi==0) || (p[bi-1][bj]!=other))
       && ((bi==board_size-1) || (p[bi+1][bj]!=other))
       && ((bj==0) || (p[bi][bj-1]!=other))
       && ((bj==board_size-1) || (p[bi][bj+1]!=other)))
      || approxlib(bi, bj, other, 2) >1) 
    if (trymove(bi, bj, other, "readlad2-D", si, sj))
      {
	if (!readlad1(si, sj, &di, &dj)) {
	  if (i) *i=bi;
	  if (j) *j=bj;
	  if (read_result) {
	    rr_set_result_ri_rj(*read_result, 1, bi, bj);
	  }
	  RTRACE("%m captures !!\n", bi, bj);
	  popgo();
	  return 1;
	}
	popgo();
      }

  if ((stackp < backfill_depth) && (approxlib(bi, bj, other, 2)==1)) {
    int ui=-1, uj=-1;
      
    ui=libi[0];
    uj=libj[0];
    if (trymove(ui, uj, other, "readlad2-E", si, sj)) {
      if (attack(si, sj, NULL, NULL) 
	  && !find_defense(si, sj, NULL, NULL)) {
	savei=ui;
	savej=uj;
      }
      popgo();
    }
  } else
    RTRACE("second atari at %m illegal\n", bi, bj);

  /* The simple ataris didn't work. Try something more fancy. */
  {
    int  xi, xj;

    if (find_cap2(si, sj, &xi, &xj)) {
      if (i) *i=xi;
      if (j) *j=xj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, xi, xj);
      }
      return 1;
    }
  }

  if (savei==-1) {
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 0, 0, 0);
    }
    RTRACE("ALIVE!!\n");
    return 0;
  }

  RTRACE("%m rescued by backfilling at %m\n", si, sj, savei, savej);
  if (i) *i=savei;
  if (j) *j=savej;
  if (read_result) {
    rr_set_result_ri_rj(*read_result, 2, savei, savej);
  }
  return 2;
}



/* If si, sj points to a group with two liberties, savestone2 determines
 * whether the group can be saved by extending, or by capturing part of
 * its surrounding chain. A group is considered safe if either part of
 * the surrounding chain may be captured, or if it can get 3
 * liberties. It is presumed that the opponent could kill if tenuki.
 * If both extensions work, it prefers the one which maximizes 
 * liberties.
 *
 * i and j return the move to save the stones. Can be NULL if caller
 * is not interested in the details.
*/

int 
savestone2(int si, int sj, int *i, int *j)
{
  int color, ai, aj, bi, bj, ci, cj, firstlib, secondlib;
  int savei=-1, savej=-1;
  int bc=0;
  int             found_read_result;
  Read_result   * read_result;

  RTRACE("trying to rescue %m\n",  si, sj);
  color=p[si][sj];

  assert(p[si][sj]!=EMPTY);
  assert(approxlib(si,sj,p[si][sj],3)==2);

  if (hashflags & HASH_SAVESTONE2) {
  
    found_read_result = get_read_result(/*color, */SAVESTONE2,
					si, sj, &read_result);
    if (found_read_result
	&& rr_get_depthleft(*read_result) >= depth - stackp)
    {
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_depthleft(*read_result, 
				   SAVESTONE2, si, sj, depth - stackp);
    }
  } else
    read_result = NULL;

  /* It is best to defend by capturing part of the surrounding chain
   * if this is possible. But simply moving the break_chain tests to
   * the front results in thrashing. As a compromise, we try to
   * capture part of the surrounding chain first if stackp==0, 
   * otherwise, we try this last.
   */

  bc=break_chain(si, sj, &ci, &cj, NULL, NULL);
  if (bc == 1) {
    if (i) *i=ci;
    if (j) *j=cj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, ci, cj);
    }
    return 1;
  }

  if (bc == 2) {
    savei=ci;
    savej=cj;
  }
  approxlib(si, sj, color, 3);
  ASSERT(lib==2, si, sj);
  ai=libi[0];
  aj=libj[0];
  bi=libi[1];
  bj=libj[1];

  /* before trying (ai,aj) check if (bi,bj) gives immediate success */

  if (approxlib(bi, bj, color,5) > 3) {
    if (i) *i=bi;
    if (j) *j=bj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, bi, bj);
    }
    return 1;
  }

  RTRACE("trying extension to %m\n", ai, aj);
  if (trymove(ai, aj, color, "savestone2-A", si, sj)) {
    firstlib=approxlib(si, sj, color, 5);
    if ((firstlib==2) && (readlad2(si, sj, NULL, NULL))) firstlib=0;
    if ((stackp<depth) && (firstlib==3) && (basicnet3(si, sj, NULL, NULL))) firstlib=0;
    popgo();
    if (firstlib>1) {
      RTRACE("%m rescues\n", ai, aj);
      if (i) *i=ai;
      if (j) *j=aj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, ai, aj);
      }
      return 1;
    }
    else 
      RTRACE("%m does not rescue. Is other move better?\n", ai, aj);
  } else {
    RTRACE("%m isn't a legal move!\n", ai, aj);
    firstlib=0;
  }
  if (firstlib<2 || i) {
    RTRACE("trying extension to %m\n", bi, bj);
    if (trymove(bi, bj, color, "savestone2-B", si, sj)) {
      secondlib=approxlib(si, sj, color, 5);
      if ((firstlib>=secondlib)&&(firstlib>2))
	{
	  RTRACE("%m is not superior to first move\n",  bi, bj);
	  if (i) *i=ai;
	  if (j) *j=aj;
	  if (read_result) {
	    rr_set_result_ri_rj(*read_result, 1, ai, aj);
	  }
	  popgo();
	  return 1;
	}
      if ((secondlib==2)&&(readlad2(si, sj, NULL, NULL))) secondlib=0;
      if ((stackp<depth) && (secondlib==3) && (basicnet3(si, sj, NULL, NULL))) secondlib=0;
      if ((secondlib>=firstlib) && (secondlib>1)) {
	if (firstlib>1)
	  RTRACE("%m also rescues, we'll use it\n", bi, bj);
	else
	  RTRACE("%m rescues, we'll use it\n",  bi, bj);
	if (i) *i=bi;
	if (j) *j=bj;
	if (read_result) {
	  rr_set_result_ri_rj(*read_result, 1, bi, bj);
	}
	popgo();
	return 1;
      }
      if (firstlib>1) {
	RTRACE("%m doesn't work, use first move\n", bi, bj);
	if (i) *i=ai;
	if (j) *j=aj;
	if (read_result) {
	  rr_set_result_ri_rj(*read_result, 1, ai, aj);
	}
	popgo();
	return 1;
      }
      popgo();
    }
  } 

  if (stackp < depth) {
    if (break_chain2(si, sj, &ci, &cj)) {
      if (i) *i=ci;
      if (j) *j=cj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, ci, cj);
      }
      return 1;
    }
  }
  else {
    if (double_atari_chain2(si, sj, &ci, &cj)) {
      if (i) *i=ci;
      if (j) *j=cj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, ci, cj);
      }
      return 1;
    }
  }
  
  {
    int xi, xj;

    if ((stackp < depth ) && (special_rescue(si, sj, ai, aj, &xi, &xj))) {
      if (i) *i=xi;
      if (j) *j=xj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, xi, xj);
      }
      return 1;
    }
  
    if ((stackp < depth ) && (special_rescue(si, sj, bi, bj, &xi, &xj))) {
      if (i) *i=xi;
      if (j) *j=xj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, xi, xj);
      }
      return 1;
    }
  }

  if (savei != -1) {
    if (i) *i=savei;
    if (j) *j=savej;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, savei, savej);
    }
    return 1;
  }


  RTRACE("failed to find rescuing move.\n");
  if (read_result) {
    rr_set_result_ri_rj(*read_result, 0, 0, 0);
  }

  return 0;
}


/*
 * special_rescue(si, sj, ai, aj, *ti, *tj) is called with (si, sj) a
 * string having a liberty at (ai, aj). The saving move is returned
 * in (*ti, *tj).
 *
 * This checks whether there is a rescuing move in the following shape:
 *
 *   .
 *  O.*
 *   O
 *
 * This will occasionally save a string where no other move will.
 */

int
special_rescue(int si, int sj, int ai, int aj, int *ti, int *tj)
{
  int n=0;
  int other=OTHER_COLOR(p[si][sj]);
  int ui=-1, uj=-1, vi=-1, vj=-1;

  /* Not applicable at the edge. */
  if ((ai==0) 
      || (ai==board_size-1) 
      || (aj==0)
      || (aj==board_size-1))
    return 0;

  /* See which neighbours to the liberty (ai, aj) are of the same color 
     as (si, sj).  Count these.  There must be no stones of other color
     and the stones of the own color must add up to 2. After this, 
     (ui, uj) and (vi, vj) will be the two liberties which we
     can try later to see if the special saving move works. */
  if (p[ai-1][aj]==other)
    return 0;
  else if (p[ai-1][aj]==p[si][sj])
    n++;
  else {
    if (ui == -1) {
      ui=ai-1;
      uj=aj;
    }
    else {
      vi=ai-1;
      vj=aj;
    }
  }

  if (p[ai+1][aj]==other)
    return 0;
  else if (p[ai+1][aj]==p[si][sj])
    n++;
  else {
    if (ui == -1) {
      ui=ai+1;
      uj=aj;
    }
    else {
      vi=ai+1;
      vj=aj;
    }
  }

  if (p[ai][aj-1]==other)
    return 0;
  else if (p[ai][aj-1]==p[si][sj])
    n++;
  else {
    if (ui == -1) {
      ui=ai;
      uj=aj-1;
    }
    else {
      vi=ai;
      vj=aj-1;
    }
  }

  if (p[ai][aj+1]==other)
    return 0;
  else if (p[ai][aj+1]==p[si][sj])
    n++;
  else {
    if (ui == -1) {
      ui=ai;
      uj=aj+1;
    }
    else {
      vi=ai;
      vj=aj+1;
    }
  }

  /* If wrong number of friendly neighbours, the pattern does not fit. */
  if (n != 2)
    return 0;

  /* We now have two candidates for the saving move. Try each of them. */
  if (trymove(ui, uj, p[si][sj], "special_rescue2-A", si, sj)) {
    if (!attack(si, sj, NULL, NULL)) {
      if (ti) *ti=ui;
      if (tj) *tj=uj;
      popgo();
      return 1;
    }
    popgo();
  }
  if (trymove(vi, vj, p[si][sj], "special_rescue2-B", si, sj)) {
    if (!attack(si, sj, NULL, NULL)) {
      if (ti) *ti=vi;
      if (tj) *tj=vj;
      popgo();
      return 1;
    }
    popgo();
  }

  /* Sadly, we couldn't find a way to save the string. Return 0. */
  return 0;
}


/* savestone3(si, sj, *i, *j) attempts to find a move rescuing the 
 * string at (si, sj) with 3 liberties.  If such a move can be found,
 * it returns true, and if the pointers i, j are not NULL, 
 * then it returns the saving move in (*i, *j).
 */

int 
savestone3(int si, int sj, int *i, int *j)
{
  int color, ai, aj, bi, bj, ci, cj;
  int savei=-1, savej=-1;
  int bc=0;
  int             found_read_result;
  Read_result   * read_result;

  RTRACE("trying to rescue %m\n",  si, sj);
  color=p[si][sj];

  assert(p[si][sj]!=EMPTY);
  assert(approxlib(si,sj,p[si][sj],3)==3);

  /* If we can capture a surrounding string, this is the best way to
     defend. */

  if (hashflags & HASH_SAVESTONE3) {
    found_read_result = get_read_result(/*color, */SAVESTONE3, 
					si, sj, &read_result);
    if (found_read_result
	&& rr_get_depthleft(*read_result) >= depth - stackp)
    {
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_depthleft(*read_result, 
				   SAVESTONE3, si, sj, depth - stackp);
    }
  } else
    read_result = NULL;

  bc=break_chain(si, sj, &ai, &aj, &ci, &cj);
  if (bc == 1) {
    if (i) *i=ai;
    if (j) *j=aj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, ai, aj);
    }
    return 1;
  } 

  /* We prefer not to sacrifice unnecessarily. If the attacking
   * move works but can itself be captured, we cache its location
   * in (savei, savej). Then if we cannot find a better way, we 
   * use it at the end.
   */       

  if (bc == 2) {
    savei=ai;
    savej=aj;
  }

  /* Get the three liberties into (ai, aj), (bi, bj) and (ci, cj). */
  approxlib(si, sj, color, 4);
  ASSERT(lib==3, si, sj);
  ai=libi[0];
  aj=libj[0];
  bi=libi[1];
  bj=libj[1];
  ci=libi[2];
  cj=libj[2];

  /* before trying seriously, check if there is a simple solution. */
  if (approxlib(ai, aj, color, 5) > 3) {
    if (i) *i=ai;
    if (j) *j=aj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, ai, aj);
    }
    return 1;
  }
  if (approxlib(bi, bj, color, 5) > 3) {
    if (i) *i=bi;
    if (j) *j=bj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, bi, bj);
    }
    return 1;
  }
  if (approxlib(ci, cj, color, 5) > 3) {
    if (i) *i=ci;
    if (j) *j=cj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, ci, cj);
    }
    return 1;
  }

  /* Nope, no quick solution available. Try reading further instead. */

  RTRACE("trying extension to %m\n", ai, aj);
  if (trymove(ai, aj, color, "savestone3-A", si, sj)) {
    if (!attack(si, sj, NULL, NULL)) {
      if (i) *i=ai;
      if (j) *j=aj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, ai, aj);
      }
      popgo();
      return 1;
    }
    popgo();
  } 

  RTRACE("trying extension to %m\n", bi, bj);
  if (trymove(bi, bj, color, "savestone3-B", si, sj)) {
    if (!attack(si, sj, NULL, NULL)) {
      if (i) *i=bi;
      if (j) *j=bj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, bi, bj);
      }
      popgo();
      return 1;
    }
    popgo();
  } 

  RTRACE("trying extension to %m\n", ci, cj);
  if (trymove(ci, cj, color, "savestone3-C", si, sj)) {
    if (!attack(si, sj, NULL, NULL)) {
      if (i) *i=ci;
      if (j) *j=cj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, ci, cj);
      }
      popgo();
      return 1;
    }
    popgo();
  } 

  if ((stackp<backfill_depth) && break_chain2(si, sj, &ai, &aj)) {
    if (i) *i=ai;
    if (j) *j=aj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 1, ai, aj);
    }
    return 1;
  }

  if (savei != -1) {
    if (i) *i=savei;
    if (j) *j=savej;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 0, savei, savej);
    }
  }

  RTRACE("failed to find rescuing move.\n");    
  return 0;
}



/* find_lunch(m, n, *wi, *wj, *ai, *aj) looks for a worm adjoining the
 * string at (m,n) which can be easily captured but also defended.
 *
 * Returns the location of the string in (*wi, *wj), and the location
 * of the attacking move in (*ai, *aj).
 */
	
int
find_lunch(int m, int n, int *wi, int *wj, int *ai, int *aj)
{
  int i, j, vi, vj, ki, kj;

  ASSERT(p[m][n] != 0, m, n);
  ASSERT(stackp==0, m, n);

  vi = -1;
  vj = -1;
  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++)
      if (p[i][j]==OTHER_COLOR(p[m][n]))
	if (((i>0) && worm[i-1][j].origini==worm[m][n].origini 
	     && worm[i-1][j].originj ==worm[m][n].originj)
	    || ((i<board_size-1) && worm[i+1][j].origini==worm[m][n].origini 
		&& worm[i+1][j].originj==worm[m][n].originj)
	    || ((j>0) && worm[i][j-1].origini==worm[m][n].origini
		&& worm[i][j-1].originj==worm[m][n].originj)
	    || ((j<board_size-1) && worm[i][j+1].origini==worm[m][n].origini
		&& worm[i][j+1].originj==worm[m][n].originj))
	  /*
 * If several adjacent lunches are found, we pick the juiciest.
 * First maximize cutstone, then minimize liberties. We can only
 * do if the worm data is available, i.e. if stackp==0.
 */
	  if (((stackp==0) && (worm[i][j].attacki != -1) 
	       && (worm[i][j].defendi != -1) && (!worm[i][j].ko))
	      || ((stackp>0) && attack(i, j, &ki, &kj) 
		  && find_defense(i, j, NULL, NULL))) {
	    if ((vi) == -1 || 
		(stackp==0 && (worm[i][j].cutstone > worm[vi][vj].cutstone ||
			       (worm[i][j].cutstone == worm[vi][vj].cutstone &&
				worm[i][j].liberties < worm[vi][vj].liberties)))) {
	      vi = worm[i][j].origini;
	      vj = worm[i][j].originj;
	      if (stackp>0) {
		if (ai) *ai = ki;
		if (aj) *aj = kj;
	      }
	      else 
		{
		  if (ai) *ai=worm[i][j].attacki;
		  if (aj) *aj=worm[i][j].attackj;
		}
	    }
	  }
  if (vi != -1) {
    if (wi) *wi=vi;
    if (wj) *wj=vj;
    return (1);
  }
  else
    return (0);
}


/* find_defense(m, n *i, *j) attempts to find a move that will save
 * The string at (m,n). It returns true if such a move is found, with
 * (*i, *j) the location of the saving move. It is not checked that
 * tenuki defends, so this may give an erroneous answer if !attack(m,n).
 */

int 
find_defense(int m, int n, int *i, int *j)
{
  int color = p[m][n];
  int di, dj;
  int mylib=approxlib(m, n, p[m][n], 5);
  int             found_read_result;
  Read_result   * read_result;

  RTRACE("Can we rescue %m?\n", m, n);
	  
  if (hashflags & HASH_FIND_DEFENSE) {
    found_read_result = get_read_result(/*color, */FIND_DEFENSE,
					m, n, &read_result);
    if (found_read_result
	&& rr_get_depthleft(*read_result) >= depth - stackp)
    {
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_depthleft(*read_result, 
				   FIND_DEFENSE, m, n, depth - stackp);
    }
  } else
    read_result = NULL;


  if (mylib==1) {
    if (readlad1(m, n, &di, &dj)) {
      if (legal(di, dj, color)) {
	RTRACE("saving move for %m found at %m!\n", m, n, di, dj);
	if (i) *i=di;
	if (j) *j=dj;
	if (read_result) {
	  rr_set_result_ri_rj(*read_result, 1, di, dj);
	}
	return (1);
      }
    }
  }
  else if (mylib==2) {
    if (savestone2(m, n, &di, &dj)) {
      RTRACE("saving move for %m found at %m!\n", m, n, di, dj);
      if (i) *i=di;
      if (j) *j=dj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, di, dj);
      }
      return (1);
    }
  }
  else if (mylib==3) {
    if (stackp >= DEPTH) 
      return 1;
    if (savestone3(m, n, &di, &dj)) {
      RTRACE("saving move for %m found at %m!\n", m, n, di, dj);
      if (i) *i=di;
      if (j) *j=dj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, di, dj);
      }
      return (1);
    }
  }

  if (read_result) {
    rr_set_result_ri_rj(*read_result, 0, 0, 0);
  }
  return 0;
}

/* 
 * (si, sj) points to a string. 
 * 
 * break_chain(si, sj, *i, *j, *k, *l) returns 1 if:
 *   Part of some surrounding string is in atari, and if capturing
 *   this string results in a live string at (si, sj). 
 *
 * Returns 2 if: 
 *   The capturing string can be taken (as in a snapback). 
 *
 * (i,j), if not NULL, are left pointing to the appropriate defensive move.
 * (k,l), if not NULL, are left pointing to the boundary string which
 *   is in atari.
 */

int
break_chain(int si, int sj, int *i, int *j, int *k, int *l)
{
  int color;
  int r;
  int ai, aj, ci, cj;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN], adjsize[MAXCHAIN], adjlib[MAXCHAIN];
  int savei = -1, savej = -1, savek=-1, savel=-1;
  
  RTRACE("in break_chain at %m\n", si, sj);
  color=p[si][sj];
  chainlinks(si, sj, &adj, adji, adjj, adjsize, adjlib);
  if (verbose) {
    RTRACE("chain surrounding %m (liberties): ", si, sj);
    for (r=0; r<adj; r++) {
      ai=adji[r];
      aj=adjj[r];
      RTRACE("%o%m(%d) ", ai, aj, adjlib[r]);  /* %o cancels indent */
    }
    RTRACE("%o\n");
  }
  for (r=0; r<adj; r++) {
    ai=adji[r];
    aj=adjj[r];
    if (approxlib(ai, aj, p[ai][aj], 2)==1) {
      ci=libi[0];
      cj=libj[0];
      if (stackp<depth) {
	if (trymove(ci, cj, color, "break_chain", si, sj)) {
	  if (!attack(si, sj, NULL, NULL)) {
	    RTRACE("%m found to defend %m by attacking %m\n", ci, cj, si, sj, ai, aj);
	    if (i && attack(ci, cj, NULL, NULL)) {
	      savei=ci;
	      savej=cj;
	      savek=ai;
	      savel=aj;
	      popgo();
	    }
	    else {
	      if (i) *i=ci;
	      if (j) *j=cj;
	      if (k) *k=ai;
	      if (l) *l=aj;
	      popgo();
	      return (1);
	    }
	  }
	  else popgo();
	}
      }	else {
	RTRACE("%m found to defend %m by attacking %m\n", ci, cj, si, sj, ai, aj);
	if (i) *i=ci;
	if (j) *j=cj;
	if (k) *k=ai;
	if (l) *l=aj;
	return (1);
      }
    }
  }
  if (savei != -1) {
    if (i) *i=savei;
    if (j) *j=savej;
    if (k) *k=savek;
    if (l) *l=savel;
    return 2;
  }
  return(0);
}

/*
 * si, sj points to a group. break_chain2(si, sj, *i, *j)
 * returns 1 if there is a string in the surrounding chain having
 * exactly two liberties whose attack leads to the rescue of
 * (si, sj). Then *i, *j points to the location of the attacking move.
 * 
 * Returns 2 if the attacking stone can be captured, 1 if it cannot.
 */

int 
break_chain2(int si, int sj, int *i, int *j)
{
  int color, other;
  int r;
  int u=0, v;
  int ai, aj;
  int saveti=-1, savetj=-1;
  int adj;
  int adji[MAXCHAIN], adjj[MAXCHAIN], adjsize[MAXCHAIN], adjlib[MAXCHAIN];
  int ti[MAXCHAIN], tj[MAXCHAIN];
  int mw[MAX_BOARD][MAX_BOARD];

  memset(mw, 0, sizeof(mw));

  RTRACE("in break_chain2 at %m\n", si, sj);
  color=p[si][sj];
  other=OTHER_COLOR(color);
  chainlinks(si, sj, &adj, adji, adjj, adjsize, adjlib);
  RTRACE("chain around %m (liberties): ", si, sj);
  for (r=0;r<adj;r++) {
    ai=adji[r];
    aj=adjj[r];
    if (adjlib[r] == 2) {
      approxlib(ai, aj, p[ai][aj], 3);
      if (!mw[libi[0]][libj[0]]) {
	mw[libi[0]][libj[0]]=1;
	ti[u]=libi[0];
	tj[u]=libj[0];
	u++;
      }
      if (!mw[libi[1]][libj[1]]) {
	mw[libi[1]][libj[1]]=1;
	ti[u]=libi[1];
	tj[u]=libj[1];
	u++;
      }
    }
    RTRACE("%o%m(%d) ", ai, aj, adjlib[r]);
  }
  RTRACE("%o\n");

/* we do not wish to consider the move if it can be 
 * immediately recaptured, unless stackp<backfill_depth.
 */

  for (v=0; v<u; v++)
    if (stackp<backfill_depth 
	|| approxlib(ti[v], tj[v], color, 2)>1)
      if (trymove(ti[v], tj[v], color, "break_chain2", si, sj)) {
	if (!attack(si, sj, i, j)) {
	  if (i && attack(ti[v], tj[v], NULL, NULL)) {
	    saveti=ti[v];
	    savetj=tj[v];
	    popgo();
	  }
	  else {
	    if (i) *i=ti[v];
	    if (j) *j=tj[v];
	    popgo();
	    return (1);
	  }
	}
	else popgo();
      }
  if (saveti != -1) {
    if (i) *i=saveti;
    if (j) *j=savetj;
    return (2);
  }
  return(0);
}

/*
 * If si, sj points to a group, double_atari_chain2(si, sj, *i, *j)
 * returns 1 if there is move which makes a double atari on 
 * a string in the surrounding chain, and if this move rescues
 * the string. This is a faster, more specific function than
 * break_chain2, and substitutes for it while reading ladders
 * when stackp>depth.
 * 
 * If not NULL, *i, *j points to the location of the attacking move.
 * 
 */

int 
double_atari_chain2(int si, int sj, int *i, int *j)
{
  int r;
  int color=p[si][sj];
  int ai, aj;
  int adj;
  int adji[MAXCHAIN], adjj[MAXCHAIN], adjsize[MAXCHAIN], adjlib[MAXCHAIN];
  int mw[MAX_BOARD][MAX_BOARD];
  int w=0;
  int v;
  int ti[MAXCHAIN], tj[MAXCHAIN]; /* list of double atari moves */

  memset(mw, 0, sizeof(mw));

  color=p[si][sj];
  chainlinks(si, sj, &adj, adji, adjj, adjsize, adjlib);
  RTRACE("chain around %m (liberties): ", si, sj);
  for (r=0;r<adj;r++) {
    ai=adji[r];
    aj=adjj[r];
    if (adjlib[r] == 2) {
      approxlib(ai, aj, p[ai][aj], 3);
      mw[libi[0]][libj[0]]++;
      mw[libi[1]][libj[1]]++;      
      if (mw[libi[0]][libj[0]]>1) {
	/* found a double atari */
	ti[w]=libi[0];
	tj[w]=libj[0];
	w++;
      }
      if (mw[libi[1]][libj[1]]>1) {
	/* found a double atari */
	ti[w]=libi[1];
	tj[w]=libj[1];
	w++;
      }
    }
  }
  for (v=0; v<w; v++)
    if (trymove(ti[v], tj[v], color, "double_atari_chain2", si, sj)) {
      if (!attack(si, sj, NULL, NULL)) {
	popgo();
	if (i) *i=ti[v];
	if (j) *j=tj[v];
	return 1;
      }
      else popgo();
    }
  return 0;
}

/* 
 * relative_break_chain(si, sj, *i, *j, ti, tj) is a variant of
 * break_chain. The strings (si, sj) and (ti, tj) are of
 * opposite color, and (ti, tj) is under attack. This function
 * looks for a boundary string to (si, sj) which is in atari,
 * and asks whether capturing it will result in the capture
 * of (ti, tj).
 */

int
relative_break_chain(int si, int sj, int *i, int *j, int ti, int tj)
{
  int color;
  int r;
  int ai, aj, ci, cj;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN], adjsize[MAXCHAIN], adjlib[MAXCHAIN];
  
  if (stackp>depth)
    return (0);

  RTRACE("in relative_break_chain at %m (%m)\n", si, sj, ti, tj);
  color=p[si][sj];
  chainlinks(si, sj, &adj, adji, adjj, adjsize, adjlib);
  if (verbose) {
    RTRACE("chain surrounding %m (liberties): ", si, sj);
    for (r=0; r<adj; r++) {
      ai=adji[r];
      aj=adjj[r];
      RTRACE("%o%m(%d) ", ai, aj, adjlib[r]);  /* %o cancels indent */
    }
    RTRACE("%o\n");
  }
  for (r=0; r<adj; r++) {
    ai=adji[r];
    aj=adjj[r];
    if (approxlib(ai, aj, p[ai][aj], 2)==1) {
      ci=libi[0];
      cj=libj[0];
      if (trymove(ci, cj, color, "break_chain", si, sj)) {
	if (attack(ti, tj, NULL, NULL) && !find_defense(ti, tj, NULL, NULL)) {
	  RTRACE("%m found to attack %m by defending %m\n", ci, cj, ti, tj, si, sj);
	  if (i) *i=ci;
	  if (j) *j=cj;
	  popgo();
	  return (1);
	}
	else popgo();
      }
    }
  }
  return(0);
}



/* snapback(si, sj, i, j, color) considers a move by COLOR at (I, J)
 * and returns true if the move is a snapback.
 *
 * Algorithm: It removes dead pieces of the other color, then returns 1 
 * if the stone at (si, sj) has <2 liberties. The purpose of this test 
 * is to avoid snapbacks. 
 *
 * (i, j) and (si, sj) may be either same or different. Also returns 1
 * if the move at (i, j) is illegal, with the message
 * "ko violation" which is the only way I think this could happen.
 *
 * It is not a snapback if the capturing stone can
 * be recaptured on its own, eg
 *
 *   XXOOOOO
 *   X*XXXXO
 *   -------
 *
 * O capturing at * is in atari, but this is not a snapback.
 *
 * FIXME : the capture
 *
 *   XXXOOOOOOOO
 *   XO*XXXXXXXO
 *   -----------
 *
 * is rejected as a snapback, yet O captures more than it gives up
 *
 */


int 
snapback(int si, int sj, int i, int j, int color)
{

  /* approxlib writes the size of the group to 'size'. If approxlib returns
   * 1, we know it has done an exhaustive search, and therefore size is correct
   */

  if (trymove(i, j, color, "snapback", si, sj)) {
    if (approxlib(si, sj, p[si][sj], 3)<2  &&  (size > 1) ) {
      RTRACE("SNAPBACK at %m!\n", i, j);
      popgo();
      return 1;
    } else {
      popgo();
      return 0;
    }
  } 
  RTRACE("KO VIOLATION at %m!\n", i, j);
  return 1;
}

/* chainlinks returns (in adji, adjj arrays) the chains surrounding
 * the group at i, j. Adapted from count. Marks only one stone on
 * each link. If stackp<depth, these are sorted by size (largest
 * first.
 */

static char chainlinks_ma[MAX_BOARD][MAX_BOARD];
static char chainlinks_ml[MAX_BOARD][MAX_BOARD];
static char chainlinks_mark;

void
init_chainlinks(void)
{
  /* set all pieces as unmarked */
  memset(chainlinks_ma, 0, sizeof(chainlinks_ma));
  memset(chainlinks_ml, 0, sizeof(chainlinks_ml));
  chainlinks_mark = 1;
}

void 
chainlinks(int m, int n, int *adj, int adji[MAXCHAIN], int adjj[MAXCHAIN],
	   int adjsize[MAXCHAIN], int adjlib[MAXCHAIN])
{
  if (chainlinks_mark == 0) { /* We have wrapped around, reinitialize. */
    init_chainlinks();
  }
  
  (*adj)=0;
  
  chain(m, n, chainlinks_ma, chainlinks_ml, adj, adji, adjj, adjsize, adjlib,
	chainlinks_mark);
  chainlinks_mark++;
}


/* worker fn for chainlinks.
 * ma and ml store state about where we have been.
 * ma stores which elements of the surrounded chain have been visited.
 * ml (via count()) stores which surrounding stones have been visited.
 *
 * Algorithm :
 *   Mark ma[i][j] as done.
 *   explore the stones surrounding i,j :
 *    - for any which are the same color and not yet visited,
 *      save them in an internal stack for later visit.
 *    - any which are the opposite colour, make a note if not visited,
 *      and use count() to mark them all as visited.
 */

static int seedi[MAXCHAIN];
static int seedj[MAXCHAIN];

static void 
chain(int i, int j, char ma[MAX_BOARD][MAX_BOARD],
      char ml[MAX_BOARD][MAX_BOARD], int *adj, int adji[MAXCHAIN],
      int adjj[MAXCHAIN], int adjsize[MAXCHAIN], int adjlib[MAXCHAIN],
      int mark)
{
  int k;
  int seedp;
  int color=p[i][j];
  int other=OTHER_COLOR(color);

  seedi[0] = i;
  seedj[0] = j;
  seedp = 1;

  while (seedp>0) {
    seedp--;
    i = seedi[seedp];
    j = seedj[seedp];

    /* We may already have visited here. */
    if (ma[i][j] == mark)
      continue;
    ma[i][j] = mark;

    /* check North neighbor */
    if (i != 0) {
      if ((p[i-1][j] == other) && ma[i-1][j] != mark && ml[i-1][j] != mark) {
	adjlib[(*adj)] = count(i-1, j, other, ml, 99999, mark);
	for(k=0; k<lib; k++)
	  ml[libi[k]][libj[k]] = mark-1;
	adji[(*adj)] = i-1;
	adjj[(*adj)] = j;
	adjsize[(*adj)] = size;
	++(*adj);
	ma[i-1][j] = mark;
      } 
      else {
	if ((p[i-1][j] == color) && ma[i-1][j] != mark) {
	  seedi[seedp] = i-1;
	  seedj[seedp] = j;
	  seedp++;
	}
      }
    }
    /* check South neighbor */
    if (i != board_size-1) {
      if ((p[i+1][j] == other) && ma[i+1][j] != mark && ml[i+1][j] != mark) {
	adjlib[(*adj)] = count(i+1, j, other, ml, 99999, mark);
	for(k = 0; k < lib; k++)
	  ml[libi[k]][libj[k]] = mark-1;
	adji[(*adj)] = i+1;
	adjj[(*adj)] = j;
	adjsize[(*adj)] = size;
	++(*adj);
	ma[i+1][j] = mark;
      }
      else {
	if ((p[i+1][j] == color) && ma[i+1][j] != mark) {
	  seedi[seedp] = i+1;
	  seedj[seedp] = j;
	  seedp++;
	}
      }
    }
    /* check West neighbor */
    if (j != 0) {
      if ((p[i][j-1] == other) && ma[i][j-1] != mark && ml[i][j-1] !=mark) {
	adjlib[(*adj)] = count(i, j-1, other, ml, 99999, mark);
	for(k = 0; k < lib; k++)
	    ml[libi[k]][libj[k]] = mark-1;
	adji[(*adj)] = i;
	adjj[(*adj)] = j-1;
	adjsize[(*adj)] = size;
	++(*adj);
	ma[i][j-1] = mark;
      }
      else {
	if ((p[i][j-1] == color) && ma[i][j-1] != mark) {
	  seedi[seedp] = i;
	  seedj[seedp] = j-1;
	  seedp++;
	}
      }
    }
    /* check East neighbor */
    if (j != board_size-1) {
      if ((p[i][j+1] == other) && ma[i][j+1] != mark && ml[i][j+1] != mark) {
	adjlib[(*adj)] = count(i, j+1, other, ml, 99999, mark);
	for(k = 0; k < lib; k++)
	  ml[libi[k]][libj[k]] = mark-1;
	adji[(*adj)] = i;
	adjj[(*adj)] = j+1;
	adjsize[(*adj)] = size;
	++(*adj);
	ma[i][j+1] = mark;
	}
      else {
	if ((p[i][j+1] == color) && ma[i][j+1] != mark) {
	  seedi[seedp] = i;
	  seedj[seedp] = j+1;
	  seedp++;
	}
      }
    }
  }
} 



/* basicnet3(ti, tj, *i, *j) is used when (ti, tj) points to a group with
 * three liberties. It returns true if it finds a way to kill the group.
 *
 * If non-NULL (*i, *j) will be set to the move which makes the
 * attack succeed.
 */

int 
basicnet3(int ti, int tj, int *i, int *j)
{
  int color=p[ti][tj];
  int other=OTHER_COLOR(color);
  int ai, aj, bi, bj, ci, cj;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN], adjsize[MAXCHAIN], adjlib[MAXCHAIN];
  int r, gi, gj;
  int mx[MAX_BOARD][MAX_BOARD];    /* mark moves already tried */
  int             found_read_result;
  Read_result   * read_result;
  
  assert(p[ti][tj]!=EMPTY);

  if (hashflags & HASH_BASICNET3) {
    found_read_result = get_read_result(/*color, */BASICNET3,
					ti, tj, &read_result);
    if (found_read_result
	&& rr_get_depthleft(*read_result) >= depth - stackp)
    {
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_depthleft(*read_result, 
				   BASICNET3, ti, tj, depth - stackp);
    }
  } else
    read_result = NULL;

  if (stackp>depth) {
    if (read_result) {
      rr_set_result_ri_rj(*read_result, 0, 0, 0);
    }
    return (0);
  }

  memset(mx, 0, sizeof(mx));

/* The attack may fail if a boundary string is in atari and cannot be defended.
 * First we must try defending such a string.
 * 
 * FIXME: we need a similar treatment for boundary strings with 2 liberties.
 */

  chainlinks(ti, tj, &adj, adji, adjj, adjsize, adjlib);
  for (r=0; r<adj; r++)
    if (adjlib[r]==1) {
      if (stackp > depth) {
	if (read_result) {
	  rr_set_result_ri_rj(*read_result, 0, 0, 0);
	}
	return (0);
      }
      approxlib(adji[r], adjj[r], p[adji[r]][adjj[r]], 2);
      gi=libi[0];
      gj=libj[0];
      /* If moving out results in more than one liberty,
         we resume the attack. */
      if (!mx[gi][gj] && (approxlib(gi, gj, other, 2) > 1)) {
	if (trymove(gi, gj, other, "basicnet3-A", ti, tj)) {
	  mx[gi][gj]=1;
	  if (attack(ti, tj, NULL, NULL) && !find_defense(ti, tj, NULL, NULL)) {
	    if (i) *i=gi;
	    if (j) *j=gj;
	    if (read_result) {
	      rr_set_result_ri_rj(*read_result, 1, gi, gj);
	    }
	    popgo();
	    return (1);
	  }
	  popgo();
	}
      }
      { /* try to rescue the boundary piece by capturing */
	int xi, xj;
	int result;
	
	result = relative_break_chain(adji[r], adjj[r], &xi, &xj, ti, tj);
	if (result) {
	  if (read_result) {
	    rr_set_result_ri_rj(*read_result, 1, xi, xj);
	  }
	  return 1;
	}
      }
    }

  RTRACE("checking attack on %m with 3 liberties\n", ti, tj);
  approxlib(ti, tj, color, 4);
  ASSERT (lib==3, ti, tj);
  ai=libi[0];
  aj=libj[0];
  bi=libi[1];
  bj=libj[1];
  ci=libi[2];
  cj=libj[2];
  if (!mx[ai][aj] && trymove(ai, aj, other, "basicnet3-B", ti, tj)) {
    mx[ai][aj]=1;
    RTRACE("try attacking at %m ...\n", ai, aj);
    if ((approxlib(ti, tj, color, 3)==2) && (!savestone2(ti, tj, NULL, NULL)) && (readlad2(ti, tj, NULL, NULL))) {
      if (i) *i=ai;
      if (j) *j=aj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, ai, aj);
      }
      popgo();
      return (1);
    }
    popgo();
  }
  if (!mx[bi][bj] && trymove(bi, bj, other, "basicnet3-C", ti, tj)) {
    mx[bi][bj]=1;
    RTRACE("try attacking at %m ...\n", bi, bj);
    if ((approxlib(ti, tj, color, 3)==2) && (!savestone2(ti, tj, NULL, NULL)) && (readlad2(ti, tj, NULL, NULL))) {
      if (i) *i=bi;
      if (j) *j=bj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, bi, bj);
      }
      popgo();
      return (1);
    }
    popgo();
  }
  if (!mx[ci][cj] && trymove(ci, cj, other, "basicnet3-D", ti, tj)) {
    RTRACE("try attacking at %m ...\n", ci, cj);
    if ((approxlib(ti, tj, color, 3)==2) && (!savestone2(ti, tj, NULL, NULL)) && (readlad2(ti, tj, NULL, NULL))) {
      if (i) *i=ci;
      if (j) *j=cj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, ci, cj);
      }
      popgo();
      return (1);
    }
    popgo();
  }

  if (read_result) {
    rr_set_result_ri_rj(*read_result, 0, 0, 0);
  }
  return (0);
}



/* attack(m, n, *i, *j) determines if the string at (m, n) can be 
 * attacked, and if so, (*i, *j) returns the attacking move.
 *
 * Return 1 if the attack succeeds, otherwise 0.
 */

int
attack(int m, int n, int *i, int *j)
{
  int color = p[m][n];
  int other = OTHER_COLOR(color);
  int xi, xj;
  int libs;
  int result;
  int             found_read_result;
  Read_result   * read_result;

  ASSERT(color != 0, m, n);

  if (color == 0)      /* if assertions are turned off, silently fails */
    return (0);

  if (hashflags & HASH_ATTACK) {
    found_read_result = get_read_result(/*color, */ATTACK, m, n, &read_result);
    if (found_read_result
	&& rr_get_depthleft(*read_result) >= depth - stackp)
    {
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_depthleft(*read_result, 
				   ATTACK, m, n, depth - stackp);
    }
  } else
    read_result = NULL;

  /* Treat the attack differently depending on how many liberties the 
     string at (m, n) has. */
  libs=approxlib(m, n, p[m][n], 5);
  xi=libi[0];
  xj=libj[0];

  if (libs == 1) {

    /* (m,n) has one liberty. We can capture it, but it's a
     * waste of time if it is a snapback.
     * It is bad to rely on this, but snapback returns 1 if
     * the move at i,j is illegal, so net result will
     * be return of 0 if is illegal, which is what we want.
     */
    result = !snapback(xi, xj, xi, xj, other );
    if (i) *i=xi;
    if (j) *j=xj;
    if (read_result) {
      rr_set_result_ri_rj(*read_result, result, xi, xj);
    }
    return result;
  } 
  else if (libs == 2) {

    /* readlad2() can return 0, 1 or 2. */
    result = readlad2(m, n, &xi, &xj);
    if (result) {
      if (i) *i=xi;
      if (j) *j=xj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, xi, xj);
      }
      return 1;
    }

  } else if (libs == 3) {

    result =  basicnet3(m, n, &xi, &xj);
    if (result) {
      if (i) *i=xi;
      if (j) *j=xj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, xi, xj);
      }
      return 1;
    }

    /* reading with 4 liberties is temporarily turned off */
  } else if (0 && (stackp==0) &&(libs == 4)) {

    /* reading with 4 liberties is temporarily turned off */

    result =  basicnet4(m, n, &xi, &xj);
    if (result) {
      if (i) *i=xi;
      if (j) *j=xj;
      if (read_result) {
	rr_set_result_ri_rj(*read_result, 1, xi, xj);
      }
      return 1;
    }
  }

  /* More than 4 liberties.  The attack fails. 
   * (This is an oversimplification, but it's the one we use here.)
   */
  if (read_result) {
    rr_set_result_ri_rj(*read_result, 0, 0, 0);
  }
  return 0;
}
    

/* If (m,n) points to a string with 2 liberties, find_cap2(m,n,&i,&j)
 * looks for a configuration of the following type:
 *
 *  O.
 *  .*
 *
 * where O is an element of the string in question. It tries the
 * move at * and returns true this move captures the string, leaving
 * (i,j) pointing to *. 
*/

int
find_cap2(int m, int n, int *i, int *j)
{
  int ai, aj, bi, bj;
  int ti=-1, tj=-1;

  /* Get the liberties into (ai, aj) and (bi, bj). */
  if (approxlib(m, n, p[m][n], 4) != 2)
    return 0;
  ai=libi[0];
  aj=libj[0];
  bi=libi[1];
  bj=libj[1];

  /* Check if the two liberties are located like the figure above. */
  if (((ai==bi+1) || (ai==bi-1)) 
      && ((aj==bj+1) || (aj==bj-1))) 
  {
    /* Which of the two corner points should we use? One of them is 
       always occupied by the string at (m, n), the other one is either
       free or occupied by something else. */
    if (p[bi][aj]==EMPTY) {
      ti=bi;
      tj=aj;
    }
    if (p[ai][bj]==EMPTY) {
      ti=ai;
      tj=bj;
    }

    /* If we didn't find a free intersection, we couldn't make the move. */
    if (ti==-1)
      return 0;

    /* Ok, we found the spot. Now see if the move works. */
    RTRACE("trying to capture %m with capping move at %m\n", m, n, ti, tj);
    if (trymove(ti, tj, OTHER_COLOR(p[m][n]),"find_cap2", m, n)) {
      if (savestone2(m, n, NULL, NULL)) {
	RTRACE("cap failed!\n", m, n, ti, tj);
	popgo();
	return 0;
      }
      RTRACE("cap succeeded!\n", m, n, ti, tj);
      popgo();
      if (i) *i=ti;
      if (j) *j=tj;
      return 1;
    }
  }

  return 0;
}    



static int safe_move_cache[MAX_BOARD][MAX_BOARD][2];
static int safe_move_cache_when[MAX_BOARD][MAX_BOARD][2];

void
clear_safe_move_cache(void)
{
  int i,j;
  for (i=0;i<MAX_BOARD;i++)
    for (j=0;j<MAX_BOARD;j++) {
      safe_move_cache_when[i][j][0] = -1;
      safe_move_cache_when[i][j][1] = -1;
    }
}

/* safe_move(i, j, color) checks whether a move at (i, j) is illegal
 * or can immediately be captured. If stackp==0 the result is cached.
 */

int 
safe_move(int i, int j, int color)
{
  int safe=0;

  if (stackp == 0 && safe_move_cache_when[i][j][color==BLACK] == movenum)
    return safe_move_cache[i][j][color==BLACK];

  if (trymove(i, j, color, "safe_move", -1, -1)) {
    if (!attack(i, j, NULL, NULL))
      safe = 1;
    popgo();
  }
  
  if (stackp == 0) {
    safe_move_cache_when[i][j][color==BLACK] = movenum;
    safe_move_cache[i][j][color==BLACK] = safe;
  }
  return safe;
}

/* basicnet4 tries to capture a string with 4 liberties. Not used in GNU Go 2.4. */

int 
basicnet4(int i, int j, int *ti, int *tj)
{
  int color=p[i][j];
  int other=OTHER_COLOR(color);
  int ai, aj, bi, bj, ci, cj, di, dj;
  
  assert(p[i][j]!=EMPTY);

  if (stackp>depth)
    return (0);

/* The attack may fail if a boundary string is in atari and cannot be defended.
 * First we must try defending such a string.
 * 
 * FIXME: we need a similar treatment for boundary strings with 2 liberties.
 */

  RTRACE("checking attack on %m with 3 liberties\n", i, j);
  approxlib(i, j, color, 5);
  ASSERT (lib==4, i, j);
  ai=libi[0];
  aj=libj[0];
  bi=libi[1];
  bj=libj[1];
  ci=libi[2];
  cj=libj[2];
  di=libi[3];
  dj=libj[3];

  if (trymove(ai, aj, other, "basicnet4-A", i, j)) {
    RTRACE("try attacking at %m ...\n", ai, aj);
    if ((approxlib(i, j, color, 3)==3) && attack(i, j, NULL, NULL) && !find_defense(i, j, NULL, NULL)) {
      if (ti) *ti=ai;
      if (tj) *tj=aj;
      popgo();
      return (1);
    }
    popgo();
  }
  if (trymove(bi, bj, other, "basicnet4-B", i, j)) {
    RTRACE("try attacking at %m ...\n", bi, bj);
    if ((approxlib(i, j, color, 3)==3) && attack(i, j, NULL, NULL) && !find_defense(i, j, NULL, NULL)) {
      if (ti) *ti=bi;
      if (tj) *tj=bj;
      popgo();
      return (1);
    }
    popgo();
  }
  if (trymove(ci, cj, other, "basicnet4-C", i, j)) {
    RTRACE("try attacking at %m ...\n", ci, cj);
    if ((approxlib(i, j, color, 3)==3) && attack(i, j, NULL, NULL) && !find_defense(i, j, NULL, NULL)) {
      if (ti) *ti=ci;
      if (tj) *tj=cj;
      popgo();
      return (1);
    }
    popgo();
  }
  if (trymove(di, dj, other, "basicnet4-D", i, j)) {
    RTRACE("try attacking at %m ...\n", di, dj);
    if ((approxlib(i, j, color, 3)==3) && attack(i, j, NULL, NULL) && !find_defense(i, j, NULL, NULL)) {
      if (ti) *ti=di;
      if (tj) *tj=dj;
      popgo();
      return (1);
    }
    popgo();
  }
  return (0);
}




/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
