//  MenuAction.cpp version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
#include "MainWidget.h"
#include "Yudit.h"

#include <Translate.h>
#include <UFreeType.h>
#include <UPrint.h>
#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <stdiostream.h>

#include <UTextDialog.h>
#include <UCommon.h>

#include <string.h>
#include <strings.h>
#include <math.h>

#define XK_MISCELLANY
#define XK_LATIN1
#include <X11/keysymdef.h>

void
MainWidget::menuAction (MenuId id, int menuItem)
{
	UComponent*	comp;
	UTextMenu*	item;
	const char*	str;
	AString		fileNameIn;
	AString		errorMessage;
	UColor		color (top->display);
	const char*	gotErrors;
	int		wasEditable;
	char*		filePathName=0;
	MainWidget*	mainWidget;
	

	if (language)
	{
		TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
	}

	if ((int) id > (int)UMENUMAX || (int)id < 0) return;
	comp = popUp[id]->itemAt(menuItem);
	if (comp ==0 || !comp->isA (UComponent::TEXT_MENU)) return;
	item = (UTextMenu*) comp;
	str = item->getName();
	if (str==0) return;
	switch (id)
	{
	case UFILE:
		if (strcmp (str, "New Window") == 0)
		{
			mainWidget = new MainWidget (config);
			mainWidget->show ();
		}
		if (strcmp (str, "Close") == 0)
		{
			quit ();
			return;
		}
		if (strcmp (str, "Exit All") == 0)
		{
			if(ask ("Do you really want to exit?"))
			{
				exit (0);
				return;
			}
		}
		if (strcmp (str, "Save") ==0)
		{
			if (!editArea->textFrame->uIsEditable())
			{
				warn (Translate ("Buffer is not editable.\nTry 'Save As...'"));
				return;
			}
			if (fileName.isNull())
			{
				warn (Translate ("Buffer has no name.\nTry 'SaveAs...'"));
				return;
			}
			if (checkEncoding ()== 0) return;
			saveFile ((const char*) fileName);
			if (strncmp ((const char*) fileName, "file:",
				strlen("file:"))==0
				&& strcmp (&fileName[strlen("file:")],
					config.getConfigPath()) == 0)
			{
				gotErrors=config.testConfig ("yuditrc");
				if (gotErrors==0) return;
				errorMessage = (const char*) Translate (
					"The yuditrc config file you\n"
					"have just updated has errors.\n"
					"Do not quit until you fix them.\n\n");
				errorMessage+=gotErrors;
				warn ((const char*) errorMessage);
				return;
			}
			encoding = (const char*) config.getChecked (Config::ENCODING);
			return;
		}
		if (!strcmp (str, "Save As..."))
		{
			if (checkEncoding ()== 0) return;
			saveAs ();
			setTitle (fileName.isNull() ? "Yudit" :
				(const char*)fileName);
			break;
		}
		if (!strcmp (str, "Open..."))
		{
			if (editArea->textFrame->uIsModified())
			{
				if (!ask ("Buffer has been modified.\nDiscard changes?"))
				{
					return;
				}
			}
			if (checkEncoding ()== 0) return;
			load ();
			setTitle (fileName.isNull() ? "Yudit" :
				(const char*)fileName);
			break;
		}
		if (!strcmp (str, "Append..."))
		{
			if (!editArea->textFrame->uIsEditable())
			{
				warn ("Buffer is not editable.\nTry 'Load'");
				return;
			}
			append ();
			setTitle (fileName.isNull() ? "Yudit" :
				(const char*)fileName);
			break;
		}
		if (strcmp (str, "Re-Load") ==0)
		{
			if (fileName.isNull())
			{
				warn ("Buffer has no name.\nTry 'Save As...' first.");
				return;
			}
			if (editArea->textFrame->uIsModified())
			{
				if (!ask ("Buffer has been modified.\nDiscard changes?"))
				{
					return;
				}
			}
			if (checkEncoding ()== 0) return;
			fileNameIn= (const char*) fileName;
			wasEditable = editArea->textFrame->uIsEditable ();
			editArea->textFrame->uSetEditable (UWidget::EDITABLE);
			loadFile ((const char*) fileNameIn);
			encoding = (const char*) config.getChecked (Config::ENCODING);
			if (wasEditable==0)
			{
				editArea->textFrame->uSetEditable (UWidget::VIEW);
			}
			return;
		}
		if (strcmp (str, "Config") == 0)
		{
			mainWidget = new MainWidget (config, config.getConfigPath(), "UTF8", UWidget::EDITABLE);
			mainWidget->show ();
			return;
		}
		if (!strcmp (str, "Print..."))
		{
			print ();
			break;
		}
		break;
	case UENCODING:
		if (findDialog) findDialog->hide ();
		if (editArea->textFrame->uSetConverter(str) == UString::ERROR)
		{
			warn (Translate ("Can not set desired encoding."));
			return;
		}
		checkItem (UENCODING, str);
		config.setChecked (Config::ENCODING, str);
		break;

	case UINPUT:
		if (findDialog) findDialog->hide ();
		if (strcasecmp (str, "Straight") ==0
			|| strcasecmp (str, "None")==0)
		{
			editArea->textFrame->uSetInput (0);
		}
		else if (editArea->textFrame->uSetInput(str) == UWidget::ERROR)
		{
			warn (Translate ("Can not set desired input map."));
			return;
		}
		if (strcmp ((const char*) config.getChecked (Config::INPUT), str))
		{
			lastInput = (const char*) config.getChecked (Config::INPUT);
		}
		checkItem (UINPUT, str);
		config.setChecked (Config::INPUT, str);
		break;

	case UXINPUT:
		if (findDialog) findDialog->hide ();
		if (setXInput (str))
		{
			warn (Translate ("Can not set desired xinput."));
			return;
		}
		checkItem (UXINPUT, str);
		config.setChecked (Config::XINPUT, str);
		break;

	case UFONTNAME:
		if (findDialog) findDialog->hide ();
		if (editArea->textFrame->uSetFont (str)== UWidget::ERROR)
		{
			warn (Translate ("Can not set desired font."));
			break;
		}
		editArea->textFrame->uSetFontSize (atoi((const char*)config.getChecked(Config::FONTSIZE)));
		editArea->textFrame->uSetFontSlant ((const char*)config.getChecked(Config::FONTSLANT));
		editArea->textFrame->uSetFontWeight ((const char*)config.getChecked(Config::FONTWEIGHT));
		editArea->textFrame->uSetFontSpacing ((const char*)config.getChecked(Config::FONTSPACING));
		editArea->textFrame->uSetFontAvgWidth ((const char*)config.getChecked(Config::FONT_AVG_WIDTH));
		editArea->textFrame->uSetFontAddStyle ((const char*)config.getChecked(Config::FONT_ADD_STYLE));
		editArea->textFrame->uRefresh();
		checkItem (UFONTNAME, str);
		config.setChecked (Config::FONTNAME, str);

		if (editFont && editFont != GetDefaultFont() ) delete editFont;

		editFont = UNewFont (top->display, top->screen, 
		(const char*) config.getChecked (Config::FONTNAME));
		if (editFont==0)
		{
			editFont = GetDefaultFont();
			return;
		}
		editFont->setPixel (atoi((const char*)config.getChecked(Config::FONTSIZE)));
		editFont->setWeight ((const char*)config.getChecked(Config::FONTWEIGHT));
		editFont->setSlant ((const char*)config.getChecked(Config::FONTSLANT));
		editFont->setSpacing ((const char*)config.getChecked(Config::FONTSPACING));
		editFont->setAvgWidth ((const char*)config.getChecked(Config::FONT_AVG_WIDTH));
		editFont->setAddStyle ((const char*)config.getChecked(Config::FONT_ADD_STYLE));

		break;
	case UFONT_ADD_STYLE:
		if (findDialog) findDialog->hide ();
		editArea->textFrame->uSetFontAddStyle (str);
		if (editFont!=GetDefaultFont())
			editFont->setAddStyle (str);
		checkItem (UFONT_ADD_STYLE, str);
		config.setChecked (Config::FONT_ADD_STYLE, str);
		editArea->textFrame->uRefresh();
		break;
	case UFONT_AVG_WIDTH:
		if (findDialog) findDialog->hide ();
		editArea->textFrame->uSetFontAvgWidth (str);
		if (editFont!=GetDefaultFont())
			editFont->setAvgWidth (str);
		checkItem (UFONT_AVG_WIDTH, str);
		config.setChecked (Config::FONT_AVG_WIDTH, str);
		editArea->textFrame->uRefresh();
		break;
	case UFONTSIZE:
		if (findDialog) findDialog->hide ();
		editArea->textFrame->uSetFontSize (atoi(str));
		if (editFont!=GetDefaultFont())
			editFont->setPixel (atoi(str));
		checkItem (UFONTSIZE, str);
		config.setChecked (Config::FONTSIZE, str);
		editArea->textFrame->uRefresh();
		break;
	case UFONTWEIGHT:
		if (findDialog) findDialog->hide ();
		editArea->textFrame->uSetFontWeight (str);
		if (editFont!=GetDefaultFont())
			editFont->setWeight (str);
		checkItem (UFONTWEIGHT, str);
		config.setChecked (Config::FONTWEIGHT, str);
		editArea->textFrame->uRefresh();
		break;
	case UFONTSLANT:
		if (findDialog) findDialog->hide ();
		editArea->textFrame->uSetFontSlant (str);
		if (editFont!=GetDefaultFont())
			editFont->setSlant (str);
		checkItem (UFONTSLANT, str);
		config.setChecked (Config::FONTSLANT, str);
		editArea->textFrame->uRefresh();
		break;
	case UFONTSPACING:
		editArea->textFrame->uSetFontSpacing (str);
		if (editFont!=GetDefaultFont())
			editFont->setSpacing (str);
		checkItem (UFONTSPACING, str);
		config.setChecked (Config::FONTSPACING, str);
		editArea->textFrame->uRefresh();
		break;
	case UTABSIZE:
		editArea->textFrame->uSetTabSize (atoi(str));
		checkItem (UTABSIZE, str);
		config.setChecked (Config::TABSIZE, str);
		editArea->textFrame->uRefresh();
		break;
	case UWINDOW_BACKGROUND:
		editArea->textFrame->setBackground (str);
		checkItem (UWINDOW_BACKGROUND, str);
		config.setChecked (Config::WINDOW_BACKGROUND, str);
		editArea->textFrame->uRefresh();
		break;
	case UWINDOW_FOREGROUND:
		editArea->textFrame->setForeground (str);
		checkItem (UWINDOW_FOREGROUND, str);
		config.setChecked (Config::WINDOW_FOREGROUND, str);
		editArea->textFrame->uRefresh();
		break;
	case UCURSOR_COLOR:
		color = str;
		editArea->textFrame->uSetCursorColor (color.getPixel());
		checkItem (UCURSOR_COLOR, str);
		config.setChecked (Config::CURSOR_COLOR, str);
		editArea->textFrame->uRefresh();
		break;
	case ULANGUAGE:
		if (findDialog) findDialog->hide ();
		setLanguage (str);
		checkItem (ULANGUAGE, str);
		config.setChecked (Config::LANGUAGE, str);
		break;
	case USEARCH:
		if (strcasecmp (str, "Find/Replace")==0)
		{
			doFind();
			break;
		}
		if (strcasecmp (str, "Go To")==0)
		{
			doGoto();
			break;
		}
		break;
	case UHELP:
		if (strcmp (str, "About...")==0)
		{
			info (Translate (UWidgetVersion()));
			return;
		}
		if (strcmp (str, "Test Page")==0)
		{
			str = "TestPage";
		}
		fileNameIn = (const char*) str;
		fileNameIn += ".help";
		filePathName = UFindFile (UGetMapPath (), 
			Translate ((const char*) fileNameIn));
		if (filePathName==0)
		{
			warn ("Can not find help file.");
			return;
		}
		mainWidget = new MainWidget (config, filePathName,
			"UTF8", UWidget::VIEW);
		mainWidget->show ();
		delete filePathName;
		return;
		
		
	default: break;
	}
}
void
MainWidget::info (const char* message)
{
	UTextDialog*	dialog;
	if (language)
	{
		TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
	}
	dialog = new UTextDialog (this, UTextDialog::BCANCEL);
	dialog->setButtonFont (buttonFont);
	dialog->setButtonText (UTextDialog::BCANCEL, Translate ("DISMISS"));

	dialog->setTextFont (textFont);
	// WARN
	// We may end up re-translating here....
	//
	dialog->setText (Translate (message));
	dialog->setTitle (Translate ("Yudit: information"));
	dialog->getValue();
	delete dialog;
}

void
MainWidget::warn (const char* message)
{
	UTextDialog*	dialog;
	if (language)
	{
		TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
	}
	dialog = new UTextDialog (this, UTextDialog::BCANCEL);
	dialog->setButtonFont (buttonFont);
	dialog->setButtonText (UTextDialog::BCANCEL, Translate ("DISMISS"));

	dialog->setTextFont (textFont);
	// WARN
	// We may end up re-translating here....
	//
	dialog->setText (Translate (message));
	dialog->setTitle (Translate ("Yudit: warning"));
	dialog->getValue();
	delete dialog;
}

int
MainWidget::ask (const char* message)
{
	UTextDialog*		dialog;
	UTextDialog::UValue	value;
	dialog = new UTextDialog (this, UTextDialog::BOK);
	dialog->setButtonFont (buttonFont);
	dialog->setButtonText (UTextDialog::BCANCEL, Translate ("NO"));
	dialog->setButtonText (UTextDialog::BOK, Translate ("YES"));

	dialog->setTextFont (textFont);
	// WARN
	// We may end up re-translating here....
	//
	dialog->setText (Translate (message));
	dialog->setTitle (Translate ("Yudit: question"));
	value = dialog->getValue();
	delete dialog;
	if (value==UDialog::OK) return 1;
	return 0;
}

void
MainWidget::setFileDialog (UFileDialog::UFileDialogType dialogType)
{
	const XInputMap*        xInputMap;
	if (fileDialog==0)
	{
		fileDialog = new UFileDialog (this, dialogType);
	}
	fileDialog->setDialogType (dialogType);
	xInputMap = config.getXInputMap ((const char*)config.getChecked(Config::XINPUT));
	if (xInputMap!=0)
	{
		fileDialog->setXInput ((const char*)config.getChecked(Config::XINPUT),
			(const char*) xInputMap->list[XInputMap::TYPE],
			(const char*) xInputMap->list[XInputMap::STYLE],
			(const char*) xInputMap->list[XInputMap::TEXT_TYPE],
			(const char*) xInputMap->list[XInputMap::ENCODING],
			(const char*) xInputMap->list[XInputMap::FONTMAP],
			atof ((const char*)xInputMap->list[XInputMap::VERSION]));
	}
	if (language)
	{
		TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
	}
	fileDialog->setButtonFont (buttonFont);
	fileDialog->setTextFont (textFont);
	fileDialog->setText (UFileDialog::OK, Translate ("OK"));
	fileDialog->setText (UFileDialog::CD, Translate ("CD To"));
	fileDialog->setText (UFileDialog::CANCEL, Translate ("CANCEL"));
	fileDialog->setText (UFileDialog::FILE_LABEL, Translate ("File:"));
	fileDialog->setText (UFileDialog::FILES_LABEL, Translate ("Files:"));
	fileDialog->setText (UFileDialog::DIRS_LABEL, Translate ("Directories:"));
	fileDialog->setText (UFileDialog::TOGGLE_TEXT, Translate ("Hidden"));
	fileDialog->setInput ((const char*)config.getChecked(Config::INPUT));
	fileDialog->setExists (Translate ("File exists.\nDo you want to overwrite it?"));
	fileDialog->setExistsYes (Translate ("YES"));
	fileDialog->setExistsNo (Translate ("NO"));
	fileDialog->setEncoding ((const char*)config.getChecked(Config::ENCODING));
}
void
MainWidget::saveAs ()
{
	const char*	gotFileName;
	AString		fileNameIn;

	UDialog::UValue		value;
	setFileDialog (UFileDialog::SAVE);
	fileDialog->setTitle (Translate ("Yudit: Save"));
	value = fileDialog->getValue ();
	if (value!=UDialog::OK) return;
	gotFileName = fileDialog->getSelected();
	if (gotFileName==0||gotFileName[0]==0)
	{
		warn ("No file specified.");
		return;
	}
	fileNameIn="file:";
	fileNameIn+=gotFileName;
	editArea->textFrame->uSetCursor (UCursor::WAIT, 1);
	saveFile ((const char*) fileNameIn);
	editArea->textFrame->uSetCursor (UCursor::INSERT, 1);
	fileName = (const char*) fileNameIn;
	encoding = (const char*) config.getChecked (Config::ENCODING);
	editArea->textFrame->uSetEditable (UWidget::EDITABLE);
	return;
	
}
void
MainWidget::load ()
{
	const char*	gotFileName;
	AString		fileNameIn;

	UDialog::UValue		value;
	setFileDialog (UFileDialog::LOAD);
	fileDialog->setTitle (Translate ("Yudit: Open"));
	value = fileDialog->getValue ();
	if (value!=UDialog::OK) return;

	gotFileName = fileDialog->getSelected();
	if (gotFileName==0||gotFileName[0]==0)
	{
		warn ("No file specified.");
		return;
	}
	fileNameIn="file:";
	fileNameIn+=gotFileName;
	editArea->textFrame->uSetCursor (UCursor::WAIT, 1);
	editArea->textFrame->uSetEditable (UWidget::EDITABLE);
	loadFile ((const char*)fileNameIn);
	editArea->textFrame->uSetCursor (UCursor::INSERT, 1);
	encoding = (const char*) config.getChecked (Config::ENCODING);
}
void
MainWidget::append ()
{
	const char*	gotFileName;
	AString		fileNameIn;

	UDialog::UValue		value;
	setFileDialog (UFileDialog::LOAD);
	fileDialog->setTitle (Translate ("Yudit: Append"));
	value = fileDialog->getValue ();
	if (value!=UDialog::OK) return;

	gotFileName = fileDialog->getSelected();
	if (gotFileName==0||gotFileName[0]==0)
	{
		warn ("No file specified.");
		return;
	}
	fileNameIn="file:";
	fileNameIn+=gotFileName;
	editArea->textFrame->uSetCursor (UCursor::WAIT, 1);
	editArea->textFrame->uSetEditable (UWidget::EDITABLE);
	appendFile ((const char*)fileNameIn);
	editArea->textFrame->uSetCursor (UCursor::INSERT, 1);
}

void
MainWidget::print ()
{
	UDialog::UValue		value;
	AStringList		printTranslated;
	int			i, j;
	const char*		labelStr;
	const char*		format=0;
	char			message[1024];

	FILE*			file=0;
#ifdef HAVE_STDIOSTREAM
	stdiostream*		oPStream=0;
#else
	ostdiostream*   	oPStream=0;
#endif
	ostream*		oStream=0;
	ofstream*		oFStream=0;

	UPrint*			uPrint;
	UString			titleString;
	const UCS2* const*	text;
	int			textSize;
	const int*		textSizes;
	UCS2* 			header = 0;
	int 			headerLen = 0;
	const char*		fontFile=0;
	const char*		printerCommand=0;

	text = editArea->textFrame->uGetTextBuffer();
	textSize = editArea->textFrame->uGetTextSize ();
	textSizes = editArea->textFrame->uGetTextSizes ();
	if (text == 0 || textSize==0 || (textSize==1 && text[0][0]==0))
	{
		warn ("Nothing to print.\n");
		return;
	}

	if (printDialog==0)
	{
		printDialog = new UChoiceDialog (this, PRINT_COMP_MAX);
		// get print defualts
		for (i=0; i<PRINT_COMP_MAX; i++)
		{
			if (setupPrinter(i)) return;
		}
	}
	if (language)
	{
		TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
	}
	printDialog->setButtonFont (buttonFont);
	printDialog->setTextFont (textFont);
	printDialog->setText (UChoiceDialog::OK, Translate ("PRINT"));
	printDialog->setText (UChoiceDialog::CANCEL, Translate ("CANCEL"));
	printDialog->setText (UChoiceDialog::LABEL, fileName.isNull() ? 
		Translate ("Untitled") : (const char*) fileName);

	// set up menu
	for (i=0; i<PRINT_COMP_MAX; i++)
	{
		switch (i)
		{
		case PRINTER:
			labelStr = Translate ("Printer:");
			break;
		case FONT_NAME:
			labelStr = Translate ("Font:");
			break;
		case FONT_SIZE:
			labelStr = Translate ("Font Size:");
			break;
		case FONT_WEIGHT:
			labelStr = Translate ("Font Weight:");
			break;
		case FONT_SLANT:
			labelStr = Translate ("Font Slant:");
			break;
		case TAB_SIZE:
		default:
			labelStr = Translate ("Tab Size:");
			break;
		}
		printTranslated.clear ();
		for (j=0; j<printList[i].getSize(); j++)
		{
			printTranslated.add (Translate (printList[i].at(j)));
		}
		printDialog->setChoice (i, labelStr, (const char*) printDefault[i], 
			printList[i], printTranslated);
	}

	printDialog->setTitle (Translate ("Yudit: Print"));
	value = printDialog->getValue ();

	// get defaults
	for (i=0; i<PRINT_COMP_MAX; i++)
	{
		printDefault[i] = printDialog->getSelected (i); 
	}
	if (value!=UDialog::OK) return;

	printerCommand = config.getPrinterCommand (
		(const char*) printDefault[PRINTER]);
	if (printerCommand==0)
	{
		format = Translate ("No printer command\nconfigured for printer '%s'.\n");
		sprintf (message, format,  
			(const char*) printDefault[PRINTER]);
		warn (message);
		return;
	}
	UFreeType::UWeight weight = 
		(strcmp ((const char*) printDefault[FONT_WEIGHT], "Bold")==0) 
		  ? UFreeType::BOLD: UFreeType::MEDIUM;
	UFreeType::USlant slant = 
		(strcmp ((const char*) printDefault[FONT_SLANT], "Italic")==0) 
		  ? UFreeType::ITALIC: UFreeType::ROMAN;
	UFreeType* freeType = 
		UGetFreeTypePrint ((const char*) printDefault[FONT_NAME]);
	if (freeType==0)
	{
		format = Translate ("Can not get True Type font '%s'.\n");
		sprintf (message, format,  
			(const char*) printDefault[FONT_NAME]);
		warn (message);
		return;
	}
	fontFile = freeType->getFileName (weight, slant);
	if (fontFile==0)
	{
		format = Translate ("Can not get True Type font file '%s'.\n");
		sprintf (message, format,  
			(const char*) printDefault[FONT_NAME]);
		warn (message);
		return;
	}
	// This is a command.
	if (printerCommand[0] == '|')
	{
		format = Translate ("Can not open printer command: '%s'\n"); 
		file = popen (&printerCommand[1], "w");
		if (file==0)
		{
			sprintf (message, format,  &printerCommand[1]);
			warn (message);
			return;
		}
#ifdef HAVE_STDIOSTREAM
 		oPStream = new stdiostream (file);
#else
 		oPStream = new ostdiostream (file);
#endif
		if (oPStream==0)
		{
			sprintf (message, format,  &printerCommand[1]);
			warn (message);
			pclose (file);
			return;
		}
		oStream = new iostream (oPStream->rdbuf());
	}
	else
	{
		oFStream =  new ofstream (printerCommand);
                if (oFStream==0 ||  oFStream->rdbuf() == 0
                        || !oFStream->rdbuf()->is_open())
		{
			format = Translate ("Can not open print target file: '%s'\n");
			sprintf (message, format,  printerCommand);
			if (oFStream != 0) delete oFStream;

			warn (message);
			return;
			
		}
		oStream = oFStream;
	}

	uPrint = new UPrint (fontFile, atof (printDefault[FONT_SIZE]), 
			atoi ((const char*) printDefault[TAB_SIZE]));
	titleString.setConverter ("UTF8");
	if (!fileName.isNull())
	{
		titleString.putString ((const unsigned char*)((const char*) fileName));
		header = titleString.getUString(&headerLen);
	}

	// Here SGC
	UPrint::UStatus status = uPrint->print (oStream, text,
			textSizes, textSize, header, headerLen);

	if (header) delete header;

	if (status!=UPrint::OK)
	{
		warn ("Print failed.\nYou might not have printing properly\nconfigured in your config file.");
		delete uPrint;
		if (printerCommand[0] == '|')
		{
			delete oStream; delete oPStream; pclose (file);
		}
		else
		{
			oFStream->close(); delete oFStream;
		}
		return;
	}
	format = Translate ("Printed %d page(s)");
	sprintf (message, format, uPrint->getPageCount());
	if (uPrint->getWrappedCount() > 0)
	{
		format = Translate ("\nWrapped %d line(s)");
		sprintf (&message[strlen(message)], 
			format, uPrint->getPageCount());
	}
	delete uPrint;
	if (printerCommand[0] == '|')
	{
		delete oStream; delete oPStream; pclose (file);
	}
	else
	{
		oFStream->close(); delete oFStream;
	}
	info (message);
	return;
}

int
MainWidget::setupPrinter (int pos)
{
	int		i;
	const char*	fName;

	printList[pos].clear();
	printDefault[pos] = 0;

	switch (pos)
	{
	case PRINTER:
		if (config.getPrinter()==0)
		{
			warn (Translate ("No printer configured."));
			return 1;
		}
		printList[PRINTER] = config.getPrinterList();
		printDefault[PRINTER] = config.getPrinter ();
		break;
	case FONT_NAME:
		for (i=0; i<config.getCheckList(Config::FONTNAME).getSize()>0;i++)
		{
			fName = config.getCheckList(Config::FONTNAME).at(i);
			if (UGetFreeTypePrint (fName)==0) continue;
			printList[FONT_NAME].add (fName);
			if ( strcmp ((const char*) config.getChecked (Config::FONTNAME), fName)==0)
			{
				printDefault[FONT_NAME]=fName;
			}
		}
		if (printList[FONT_NAME].getSize() ==0)
		{
			warn (Translate ("No True Type fonts configured."));
			return 1;
		}
		if (printDefault[FONT_NAME].isNull())
		{
			printDefault[FONT_NAME] = printList[FONT_NAME].at (0);
		}
		break;
	case FONT_SIZE:
		printList[FONT_SIZE] = config.getCheckList(Config::FONTSIZE);
		printDefault[FONT_SIZE] = config.getChecked(Config::FONTSIZE);
		break;
	case FONT_WEIGHT:
		printList[FONT_WEIGHT].add ("Medium");
		printList[FONT_WEIGHT].add ("Bold");
		printDefault[FONT_WEIGHT] = "Medium";
		break;
	case FONT_SLANT:
		printList[FONT_SLANT].add ("Roman");
		printList[FONT_SLANT].add ("Italic");
		printDefault[FONT_SLANT] = "Roman";
		break;
	case TAB_SIZE:
		printList[TAB_SIZE] = config.getCheckList(Config::TABSIZE);
		printDefault[TAB_SIZE] = config.getChecked (Config::TABSIZE);
	}
	return 0;
}

void
MainWidget::doGoto()
{
	UDialog::UValue		value;
	const char*		str;

	if (promptDialog==0)
	{
		promptDialog = new UPromptDialog (this);
		promptDialog->setTitle (Translate ("Yudit: Go To"));
	}
	promptDialog->setButtonFont (buttonFont);
	promptDialog->setTextFont (textFont);
	promptDialog->setText (UPromptDialog::OK, Translate ("OK"));
	promptDialog->setText (UPromptDialog::CANCEL, Translate ("CANCEL"));
	promptDialog->setText (UPromptDialog::LABEL, Translate ("Line Number:"));
	promptDialog->setInput ((const char*)config.getChecked(Config::INPUT));
	promptDialog->setEncoding ((const char*)config.getChecked(Config::ENCODING));
	value = promptDialog->getValue();
	if (value!=UDialog::OK) return;
	str = promptDialog->getSelected ();
	editArea->textFrame->uGoTo(atoi(str));
}

void
MainWidget::doFind()
{
	UBestSize		bs;
	const XInputMap*        xInputMap;

	if (findDialog==0)
	{
		findDialog = new UFindDialog (this);
		findDialog->setTitle (Translate ("Yudit: Find/Replace"));
	}

	xInputMap = config.getXInputMap ((const char*)config.getChecked(Config::XINPUT));
	if (xInputMap!=0)
	{
		findDialog->setXInput ((const char*)config.getChecked(Config::XINPUT),
			(const char*) xInputMap->list[XInputMap::TYPE],
			(const char*) xInputMap->list[XInputMap::STYLE],
			(const char*) xInputMap->list[XInputMap::TEXT_TYPE],
			(const char*) xInputMap->list[XInputMap::ENCODING],
			(const char*) xInputMap->list[XInputMap::FONTMAP],
			atof ((const char*)xInputMap->list[XInputMap::VERSION]));
	}
	findDialog->setButtonFont (buttonFont);
	findDialog->setTextFont (textFont);
	findDialog->setEntryFont (editFont);

	findDialog->setText (UFindDialog::FIND, Translate ("FIND"));
	findDialog->setText (UFindDialog::REPLACE, Translate ("FIND/REPLACE"));
	findDialog->setText (UFindDialog::CANCEL, Translate ("DISMISS"));
	findDialog->setText (UFindDialog::FIND_LABEL, Translate ("Find Text:"));
	findDialog->setText (UFindDialog::REPLACE_LABEL, Translate ("Replacement:"));
	findDialog->setInput ((const char*)config.getChecked(Config::INPUT));
	findDialog->setEncoding ((const char*)config.getChecked(Config::ENCODING));
	findDialog->setFindArea (editArea->textFrame);
	bs = findDialog->getBestSize();
	findDialog->resize (bs.width, bs.height);
	XResizeWindow (top->display, findDialog->window, bs.width, bs.height);
	findDialog->show ();
}

int
MainWidget::checkEncoding ()
{
	char            buffer[255];
	char            format[255];
	const char*     enc=(const char*) config.getChecked(Config::ENCODING);

	if (fileName.isNull() || enc==0 || encoding.isNull())
	{
		return 1;
	}
	if (strcmp ((const char*) encoding, enc)==0)
	{
		return 1;
	}
	strcpy (format, Translate (
		"The last file transaction on this buffer used encoding '%s'.\n"
		"Now your encoding is '%s'.\n\nShall we proceed?"));
	sprintf (buffer, format, (const char*) encoding, enc);
	if (language)
	{
		TranslateLocale ((const char*) language->list[Language::SHORT_NAME]);
	}
	if (!ask (buffer))
	{
		return 0;
	}
	return 1;
}


int
MainWidget::saveFile (const char* filenameIn)
{
	unsigned char*		text;
	ofstream*		file;
	int			textLen;
	int			error;

	if (strlen(filenameIn) > strlen ("file:") 
		&& strncmp (filenameIn, "file:", strlen ("file:")) == 0)
	{
		file = new ofstream (filenameIn + strlen ("file:"));
	}
	else
	{
		file = new ofstream (filenameIn);
	}
	if (file->rdbuf() == 0 || !file->rdbuf()->is_open())
	{
		editArea->textFrame->uSetCursor (UCursor::INSERT, 1);
		warn (Translate ("Write failed."));
		file->close();
		delete file;
		return -1;
	}
	error=0;
	text = editArea->textFrame->uGetBuffer(&textLen);
	if (text !=0)
	{
		if (UWriteBuffer(file, text, textLen) ==-1) error=1;
		delete text;
	}
	file->close();
	delete file;
	if (error)
	{
		editArea->textFrame->uSetCursor (UCursor::INSERT, 1);
		warn (Translate ("Write failed."));
		return -1;
	}
	editArea->textFrame->uClearModified ();
	return 1;
}

int
MainWidget::loadFile (const char* filenameIn)
{
	int	retVle;
	editArea->textFrame->uClear();
	retVle =  appendFile (filenameIn);
	editArea->textFrame->uClearModified ();
	if (retVle>=0)
	{
		fileName = (const char*) filenameIn;
		encoding = (const char*) config.getChecked (Config::ENCODING);
	}
	else
	{
		fileName = (const char*) 0;
	}
	return retVle;
}


int
MainWidget::appendFile (const char* filenameIn)
{
	int	ret_vle;
	ret_vle = appendFileBase (filenameIn);
	editArea->textFrame->uRefresh();
	if (ret_vle>=0)
	{
		if (fileName.isNull ())
		{
			fileName = (const char*) filenameIn;
			encoding = (const char*) config.getChecked (Config::ENCODING);
		}
	}
	return ret_vle;
}

int
MainWidget::appendFileBase (const char* filenameIn)
{
	ifstream*	file;
	unsigned char* 	buffer;
	int		bufferLen;

	if (strlen(filenameIn) > strlen ("file:") 
		&& strncmp (filenameIn, "file:", strlen ("file:")) == 0)
	{
		file = new ifstream (filenameIn + strlen ("file:"));
	}
	else
	{
		file = new ifstream ((const char*) filenameIn);
	}

	if (file->rdbuf() == 0 || !file->rdbuf()->is_open())
	{
		cerr << "Yudit: can not open '" << 
			filenameIn << "' to read.\n";
		editArea->textFrame->uSetCursor (UCursor::INSERT, 1);
		warn ("Can not read file.");
		file->close ();
		delete file;
		return -1;
	}
	buffer = UReadBuffer (file, &bufferLen);
	editArea->textFrame->uAppendText (buffer, bufferLen, 0, 0);
	delete buffer;
	file->close ();
	delete file;
	return 1;
}

//
// Get some hot-keys.
//
int
MainWidget::filterKeyEvent (XEvent* xevent)
{
	KeySym          key;
	int             count;
	char            asci[16];
	AString		last;

	if (xevent->type != KeyPress) return 0;

	if (xevent->xkey.state & ControlMask)
	{
		count = XLookupString( &xevent->xkey, asci, 16, &key, 0);
		switch (key)
		{
		// toggle lastInput in...
		case XK_i:
		case XK_I:
			last = (const char*) lastInput;
			lastInput = config.getChecked(Config::INPUT);
			if (strcmp (last, (const char*) lastInput)==0)
			{
				return 1;
			}
			if (findDialog) findDialog->hide ();
			if (strcasecmp ((const char*)last, "Straight") ==0
				|| strcasecmp ((const char*)last, "None")==0)
			{
				editArea->textFrame->uSetInput (0);
			}
			else 
			{
				editArea->textFrame->uSetInput(last);
			}
			checkItem (UINPUT, last);
			config.setChecked (Config::INPUT, last);
			return 1;
		default:
			break;
		}
	}
	return 0;
}
