//  UTextLabel.cpp version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
#include "UTextLabel.h"
#include <UFontFreeType.h>
#include <UString.h>

// #define WASTE_RESOURCES

UTextLine::UTextLine (const UCS2* text_, int from, int to)
{
	int	i;
	text = new UCS2 [to-from+1];
	for (i=from; i<to; i++)
	{
		text[i-from] = text_[i];
	}
	text[to-from] = (UCS2) 0;
	textSize = to-from;
	drawType = NORMAL;
	length = 0;
	coordx = 0;
} 

UTextLine::~UTextLine()
{
	delete text;
}

UTextLabel::UTextLabel (UFrame* parent_, const char* text_,
	UTextLine::UAlign align_)
	: UComponent(parent_->top, parent_->top->display) 
{
	XGCValues	gcv;

	font =0;
	parent = parent_;
	top = parent->top;

	// has to create a window from nothing.
	rectangle.width = 4;
	rectangle.height = 4;
	rectangle.x = 0;
	rectangle.y = 0;

	window = XCreateSimpleWindow (top->display,
		parent->window, rectangle.x, rectangle.y,
		rectangle.width, rectangle.height,
		0, parent->getBackground().getPixel(), 
		parent->getBackground().getPixel());

	gcv.background = parent->getBackground().getPixel();
	gcv.foreground = parent->getForeground().getPixel();

	fgGC = XCreateGC (top->display, window, 
		GCForeground | GCBackground, &gcv);

	UColor rback (top->display);
	UColor rfore (top->display);

	rback.assign (0, 0, 0x8b8b);
	rfore.assign (0xffff, 0xffff, 0xffff);

	gcv.foreground = rfore.getPixel();
	gcv.background = rback.getPixel();

	rvGC = XCreateGC (top->display, window, 
		GCForeground | GCBackground, &gcv);

	gcv.foreground = rback.getPixel();
	gcv.background = rfore.getPixel();

	rvfGC = XCreateGC (top->display, window, 
		GCForeground | GCBackground, &gcv);
	
	setBackground (parent->getBackground());
	placement.margin = 0;

	XMapRaised (top->display, window);
	// If you need more you select
	select (ExposureMask);

	bestSize.width = rectangle.width;
	bestSize.height = rectangle.height;

	// You have to do it if you dont inherit frame
	UTOP->addComponent (this);
	parent->addChild (this);

	lineCount = 0;
	line = 0;
	alignWest = 0;
	alignEast = 0;
	alignment = align_;
	setFont (parent->getFont());
	setText (text_);
	calculateBestSize();
	baseCoord = 0;
	vertOffset = 0;
}

UTextLabel::~UTextLabel()
{
	setText ((const UCS2*) 0);
	// SGC - GC should go away with window. 
	// It dumps with free type fonts for some reason.
	//XFreeGC (top->display, fgGC);
	//XFreeGC (top->display, rvGC);
	//XFreeGC (top->display, rvfGC);
	XDestroyWindow (top->display, window);
	UTOP->deleteComponent (this);
	parent->deleteChild (this);
#ifdef WASTE_RESOURCES
	if (font) delete font;
#endif
	
}

void
UTextLabel::setText (const char* text)
{
	if (text==0)
	{
		setText ((const UCS2*) 0);
		return;
	}
	UString		str((const unsigned char*) text, "UTF8");
	int len;
	UCS2* ustr = str.getUString (&len);
	setText (ustr);
	delete ustr;
}

void
UTextLabel::setText (const unsigned char* text)
{
	if (text==0)
	{
		setText ((const UCS2*) 0);
		return;
	}
	UString	str (text, "UTF8");
	int len;
	UCS2* ustr = str.getUString (&len);
	setText (ustr);
}

void
UTextLabel::setText (const UCS2* text_)
{
	int	i;
	int	len;
	int	cl;
	int	from;
	int 	to;

	vertOffset = 0;
	for (i=0; i<lineCount; i++)
	{
		delete line[i];
	}
	if (lineCount) delete line;
	lineCount = 0;
	line = 0;
	if (text_==0)
	{
		calculateBestSize();
		return;
	}
	len = UCS2NullLen(text_);
	// Break it down to lines
	cl = 0;
	for (i=0; i<len; i++)
	{
		if (text_[i] == '\n') cl++;
	}
	if (text_[i] != '\n') cl++;
	line = new UTextLine* [cl];
	CHECKNULL (line);
	from=0; to=0; cl=0;
	for (i=0; i<len; i++)
	{
		if (text_[i] == '\n')
		{
			line[cl] = new UTextLine (text_, from, to);
			CHECKNULL (line[cl]);
			from = to+1;
			cl++;
		}
		to++;
	}
	if (text_[i] != '\n')
	{
		line[cl] = new UTextLine (text_, from, to);
		CHECKNULL (line[cl]);
		cl++;
	}
	lineCount = cl;
	calculateBestSize();
	if (isShown (this))
	{
		XClearArea (top->display, window, 0, 0, 
			rectangle.width, rectangle.height, True);
	}
	return;
}

void
UTextLabel::resize (int width, int height)
{
	XRectangle	clip;

	if (rectangle.width == width 
		&& rectangle.height ==height) return;

	XClearArea (top->display, window, alignWest, 0, 
		rectangle.width-alignEast-alignWest, rectangle.height, 
		False);

	rectangle.width = (width>2)? width : 2;
	rectangle.height = (height>2)? height : 2;
	
	XResizeWindow (top->display, window, 
		rectangle.width, rectangle.height);

	clip.x = alignWest; clip.y = 0;
	clip.width = rectangle.width - alignEast - alignWest;
	clip.height = rectangle.height;

#if 0
	XSetClipRectangles (top->display, fgGC, 0, 0, &clip, 1, Unsorted);
	XSetClipRectangles (top->display,  rvGC, 0, 0, &clip, 1, Unsorted);
#endif

	redraw (0, 0, width, height);
}

void
UTextLabel::move (int x, int y)
{
	if (rectangle.x == x && rectangle.y ==y) return;
	rectangle.x = x;
	rectangle.y = y;
	XMoveWindow (top->display, window, rectangle.x, rectangle.y);
}

void
UTextLabel::setAlignment (UTextLine::UAlign align)
{
	alignment = align;
	XClearArea (top->display, window, 0, 0, 
		rectangle.width, rectangle.height, True);
	calculateBestSize ();
}

void
UTextLabel::setMargin (int m)
{
	placement.margin = m;
	calculateBestSize ();
}

void
UTextLabel::calculateBestSize()
{
	UBestSize 	bs;
	int		widest;
	int		i;
	int		len;

	bs.height = (font==0) ? 2 : lineCount * font->fontHeight 
		+ 2*placement.margin;
	if (bs.height < 2)  bs.height = 2;

	widest = 0;
	for (i=0; i<lineCount; i++)
	{
		len = drawLine (i, UTextLine::MEASURE);
		line[i]->length = len;
		switch (alignment)
		{
		case UTextLine::LEFT: 
			line[i]->coordx = 0;
			break;
		case UTextLine::RIGHT: 
			line[i]->coordx = -len;
			break;
		case UTextLine::CENTER:
			line[i]->coordx = -len/2;
			break;
		}
		if (len>widest) widest=len;
	}
	
	switch (alignment)
	{
	case UTextLine::LEFT: 
		baseCoord = 0;
		break;
	case UTextLine::RIGHT: 
		baseCoord = widest;
		break;
	case UTextLine::CENTER:
		baseCoord = widest/2;
		break;
	}
	bs.width = widest + alignWest + alignEast + 2*placement.margin;
	bestSize.width = bs.width;
	bestSize.height = bs.height;
}

void
UTextLabel::setFont (UFont* font_)
{
#ifdef WASTE_RESOURCES
	if (font!=0)
	{
		delete font;
	}
	if (font_!=0)
	{
		if (font_->isA (UFont::UX11))
		{
			font = new UFontX11 (*((UFontX11*)font_));
		}
		else if (font_->isA (UFont::UFREE))
		{
			font = new UFontFreeType (*((UFontFreeType*)font_));
		}
		else
		{
			font = 0;
		}
	}
#else
	font = font_;
#endif
	if (font !=0 && font->isNew())  font->getFont();
	XClearArea (top->display, window, 0, 0, 
		rectangle.width, rectangle.height, True);
	calculateBestSize();
	return;
}

void
UTextLabel::setInsets (int west_, int east_)
{
	bestSize.width += west_ - alignWest + east_ - alignEast;
	alignWest = west_;
	alignEast = east_;
	XClearArea (top->display, window, 0, 0, 
		rectangle.width, rectangle.height, True);
}

void
UTextLabel::eventDown (UEvent* event)
{
	XRectangle 	rect;

	switch (event->type)
	{
		case UEvent::TIMER:
			break;
		case UEvent::X:
		switch (event->xevent.type)
		{
		case Expose:
			rect.x = event->xevent.xexpose.x;
			rect.y = event->xevent.xexpose.y;
			rect.width = event->xevent.xexpose.width;
			rect.height = event->xevent.xexpose.height;
			redraw (rect.x, rect.y, rect.width, rect.height);
			break;

		case GraphicsExpose:
cerr << "Graphicsexpose\n";
			rect.x = event->xevent.xgraphicsexpose.x;
			rect.y = event->xevent.xgraphicsexpose.y;
			rect.width = event->xevent.xgraphicsexpose.width;
			rect.height = event->xevent.xgraphicsexpose.height;
			redraw (rect.x, rect.y, rect.width, rect.height);
			break;
		default:
			parent->eventUp (event);
		}
	default:
		break;
	}
}

void 
UTextLabel::setBackground (const UColor& color)
{
	background = color;
	XSetWindowBackground (background.getDisplay(), window,
		background.getPixel());

	XSetBackground (top->display, fgGC, background.getPixel());

	XClearArea (top->display, window, 0, 0, rectangle.width,
		rectangle.height, True);
}

void 
UTextLabel::setBackground (const char* color)
{
	background = color;
	setBackground (background);
}

void 
UTextLabel::setForeground (const UColor& color)
{
	foreground = color;
	XSetForeground (top->display, fgGC, foreground.getPixel());
}

void 
UTextLabel::setForeground (const char* color)
{
	foreground = color;
	setForeground (foreground);
}
//
// I just redraw the whole text
//
void
UTextLabel::redraw (int x, int y, int width, int height)
{
	int		i;
	int		from;
	int		to;
	if (!isShown(this)) return;
	if (font==0) return;
	if (font->fontHeight < 1) return;

	from = (y+vertOffset)/font->fontHeight;
	to = (y+height+vertOffset)/font->fontHeight+1;
	if (from>=lineCount) return;
	if (to>lineCount) to=lineCount;
	for (i=from; i<to; i++)
	{
		drawLine (i, line[i]->drawType);
	}
	
}

int
UTextLabel::drawLine (int lineNo, UTextLine::UDrawType dt)
{
	int		len;
	int		from;
	int		newTo;
	int		to;
	UFontCache*	fontCache;
	int		offset;
	int		start;

	if (font==0) return 0;
	if (lineNo > lineCount) return 0;

	len = 0;
	offset = 0;
	switch (alignment)
	{
	case UTextLine::LEFT: 
		offset = alignWest + baseCoord;
		break;
	case UTextLine::RIGHT: 
		offset = rectangle.width - bestSize.width + baseCoord + alignWest + 2 * placement.margin;
		break;
	case UTextLine::CENTER:
		offset = (rectangle.width-bestSize.width)/2 + baseCoord + alignWest + placement.margin;
		break;
	}
	for (from=0, to=line[lineNo]->textSize;
		from< to; from=newTo)
	{
		fontCache = font->getFont (line[lineNo]->text,
			from, &newTo);
		if (fontCache==0)  newTo = from + 1;
		len += drawSegment (fontCache, dt, 
			len+line[lineNo]->coordx + offset, lineNo, from, newTo);
	}
	if (dt == UTextLine::REVERSE || dt == UTextLine::NORMAL_FILL)
	{
		start = line[lineNo]->coordx + offset;
		if (dt == UTextLine::REVERSE)
		{
			if (start > alignWest)
			XFillRectangle (top->display, window, rvfGC,
				alignWest, lineNo * font->fontHeight - vertOffset, 
				start-alignWest, font->fontHeight);
			if (len+start < rectangle.width - alignEast)
			XFillRectangle (top->display, window, rvfGC,
				len+start, lineNo * font->fontHeight - vertOffset, 
				rectangle.width-alignEast-len-start, 
				font->fontHeight);
		}
		else
		{
			if (start > alignWest)
			XClearArea (top->display, window, 
				alignWest, lineNo * font->fontHeight - vertOffset, 
				start-alignWest, font->fontHeight, False);
			if (len+start < rectangle.width - alignEast)
			XClearArea (top->display, window,
				len+start, lineNo * font->fontHeight - vertOffset, 
				rectangle.width - alignEast-len-start, 
				font->fontHeight, False);
		}
	}
	return len;
}
//
// Simple drawing routine. Much simpler than the big text screen's 
int
UTextLabel::drawSegment (UFontCache* fnt, UTextLine::UDrawType dt,
	int coordx, int lineNo, int from, int to)
{
	GC			gc;
	int			coordy;
	XChar2b*		char2BBuf;
	int			len;
	const UFontMapStruct*   fontMapStruct;
	UCS2			converted;
	UCS2*			text;
	int			i;
	int			count;
	int			lastLength;

	gc = (dt==UTextLine::NORMAL || dt==UTextLine::NORMAL_FILL) 
			? fgGC : rvGC;
	coordy = lineNo * font->fontHeight - vertOffset;
	if (fnt==0)
	{
		// Draw something...
		if (dt==UTextLine::NORMAL_FILL)
		{
			XClearArea (top->display, window, 
					coordx, coordy, font->fontWidth,
					font->fontHeight, False);
		}
		if (dt==UTextLine::REVERSE)
		{
			XFillRectangle (top->display, window, rvfGC, 
				coordx, coordy, font->fontWidth,
				font->fontHeight);
		}
		XDrawRectangle (top->display, window, gc,
			coordx+1, coordy+1, 
			font->fontWidth-2, font->fontHeight-2);
		return font->fontWidth;
	}

	char2BBuf = new XChar2b[to-from+1];
	CHECKNULL (char2BBuf);
	len = 0;
	text = line [lineNo]->text;
	fontMapStruct = font->getFontMapStruct (text[from]);

	count = 0;
	for (i=from; i<to; i++)
	{
		converted = (fontMapStruct==0) ?  text[i]
			: fontMapStruct->encoder->encode (text[i]);
		char2BBuf[count].byte1 = converted >> 8;
		char2BBuf[count].byte2 = converted & 0xff;
		char2BBuf[count+1].byte2 = char2BBuf[count+1].byte1 = 0;
		lastLength = fnt->getWidth (char2BBuf[count]);
		lastLength = fnt->getWidth (char2BBuf[count]);
		// if this happens we have to do something
		if (lastLength ==0)
		{
			if (i>from)
			{
				lastLength = drawSegment (fnt, dt, coordx, 
					lineNo, from, i);
			}
			lastLength += drawSegment (0, dt, lastLength+coordx, 
				lineNo,  i, to);
			return lastLength;
		}
		len += lastLength;
		count++;
	}
	// Draw the converted string
	switch (dt)
	{
	case  UTextLine::NORMAL:
		fnt->drawString (top->display, window, gc,
		   coordx, coordy, font->fontAscent, char2BBuf, to-from, 0);
		break;
	case  UTextLine::NORMAL_FILL:
		fnt->drawImageString (top->display, window, gc,
		   coordx, coordy, font->fontAscent, char2BBuf, to-from, 0);
		break;
	case  UTextLine::REVERSE:
		fnt->drawImageString (top->display, window, gc,
		   coordx, coordy, font->fontAscent, char2BBuf, to-from, 0);
		break;
	default:
		break;
	}
	if (dt==UTextLine::NORMAL_FILL|| UTextLine::REVERSE)
	{
		if (font->fontAscent - fnt->getAscent() > 0)
		{
			if (dt==UTextLine::NORMAL_FILL)
			{
				XClearArea (top->display, window,
					coordx, coordy, len,
					font->fontAscent - fnt->getAscent(),
					False);
			}
			if (dt==UTextLine::REVERSE)
			{
				XFillRectangle (top->display, window, rvfGC,
					coordx, coordy, len,
					font->fontAscent - fnt->getAscent());
			}
		}
		if (font->fontDescent - fnt->getDescent() > 0)
		{
			if (dt==UTextLine::NORMAL_FILL)
			{
				XClearArea (top->display, window,
					coordx, coordy+font->fontAscent+fnt->getDescent(), len,
					font->fontDescent - fnt->getDescent(),
					False);
			}
			if (dt==UTextLine::REVERSE)
			{
				XFillRectangle (top->display, window, rvfGC,
					coordx, coordy+font->fontAscent+fnt->getDescent(), len,
					font->fontDescent - fnt->getDescent());
			}
		}
	}

	delete char2BBuf;
	return len;
}

int
UTextLabel::isA (UComponent::UType type_)
{
	if (type_ == UComponent::TEXT_LABEL) return 1;
	return UComponent::isA (type_);
}

void 
UTextLabel::selectLine (int line_)
{
	if (line_<0 || line_ >=lineCount) return;
	line[line_]->drawType = UTextLine::REVERSE;
	drawLine (line_, line[line_]->drawType);
}
void 
UTextLabel::unselectLine (int line_)
{
	if (line_<0 || line_ >=lineCount) return;
	line[line_]->drawType = UTextLine::NORMAL_FILL;
	drawLine (line_, line[line_]->drawType);
	line[line_]->drawType = UTextLine::NORMAL;
}

const UCS2*
UTextLabel::getText (int line_)
{
	if (line_<0 || line_ >=lineCount) return 0;
	return line[line_]->text;
}
