/************************************************************************/
/*									*/
/*  Read a plain text file into a BufferDocument			*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>
#   include	<charnames.h>

#   include	<debugon.h>

#   include	"docRtf.h"

/************************************************************************/
/*  An empty tag table.							*/
/************************************************************************/
RtfControlWord	docRtfEmptyTable[]=
    {
	{ 0, }
    };

/************************************************************************/
/*  Reset the attributes in an RtfReadingContext to the defaults.	*/
/************************************************************************/
void docRtfInitReadingContext(	RtfReadingContext *	rrc	)
    {
    int		i;

    rrc->rrcLevel= DOClevOUT;
    rrc->rrcSplitLevel= DOClevOUT;
				/****************************************/
				/*  The depth in the document		*/
				/*  == Type of BufferItem.		*/
				/****************************************/
    rrc->rrcComplainUnknown= 1;
				/****************************************/
				/*  Complain about unknown control.	*/
				/****************************************/
    rrc->rrcCharacterAhead= EOF;

    rrc->rrcCurrentLine= 1;
    rrc->rrcBd= (BufferDocument *)0;
    rrc->rrcBi= (BufferItem *)0;
				/****************************************/
				/*  The current position in the input.	*/
				/****************************************/
    docInitTextAttribute( &(rrc->rrcCurrentTextAttribute) );
    docInitBorderProperties( &(rrc->rrcTextBorderProperties) );
    rrc->rrcInDeletedText= 0;

    rrc->rrcCurrentTextAttribute.taFontSizeHalfPoints= 24;
    rrc->rrcTextShadingPattern= DOCspSOLID;
    rrc->rrcTextShadingLevel= 0;
    rrc->rrcTextForeground= -1;
    rrc->rrcTextBackground= -1;
				/****************************************/
				/* Attributes of the current position.	*/
				/****************************************/
    docInitParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitRowProperties( &(rrc->rrcRowProperties) );
    docInitCellProperties( &(rrc->rrcCellProperties) );
    docInitBorderProperties( &(rrc->rrcBorderProperties) );
    /*docInitShapeProperties( &(rrc->rrcShapeProperties) );*/
    docInitShape( &(rrc->rrcShape) );
    docInitTabStop( &(rrc->rrcTabStop) );
    appInitDocumentGeometry( &(rrc->rrcDocumentGeometry) );
    rrc->rrcColor.rgb8Red= rrc->rrcColor.rgb8Green= rrc->rrcColor.rgb8Blue= 0;
    rrc->rrcGotComponent= 0;
				/****************************************/
				/*  Paragraph properties.		*/
				/****************************************/
    rrc->rrcDefaultFont= -1;
				/****************************************/
				/*  Document properties.		*/
				/****************************************/
    docInitFont( &(rrc->rrcCurrentFont) );
    rrc->rrcStyleNumber= 0;
    rrc->rrcStyleBasedOn= 0;
    rrc->rrcStyleNext= 0;
				/****************************************/
				/*  For reading the font table.		*/
				/****************************************/
    rrc->rrcInsertedObject= (InsertedObject *)0;
    rrc->rrcFieldNumber= -1;
				/****************************************/
				/*  For reading 'objects' and pictures.	*/
				/*  For reading 'fields'.		*/
				/****************************************/
    rrc->rrcJustAfterPntext= 0;
				/****************************************/
				/*  For coping with the way word saves	*/
				/*  {\pntext ... }			*/
				/****************************************/

    for ( i= 0; i < 256; i++ )
	{ rrc->rrcInputMapping[i]= i;	}
    }

void docRtfCleanReadingContext(	RtfReadingContext *	rrc	)
    {
    docCleanParagraphProperties( &(rrc->rrcParagraphProperties) );
    docCleanRowProperties( &(rrc->rrcRowProperties) );
    /*docCleanCellProperties( &(rrc->rrcRowProperties) );*/
    /*docCleanShapeProperties( &(rrc->rrcShapeProperties) );*/
    docCleanShape( &(rrc->rrcShape) );
    docCleanFont( &(rrc->rrcCurrentFont) );
    }

int docRtfCopyReadingContext(	RtfReadingContext *		to,
				const RtfReadingContext *	from )
    {
    ParagraphProperties		ppCopy;
    RowProperties		rpCopy;

    docInitParagraphProperties( &ppCopy );
    if  ( docCopyParagraphProperties( &ppCopy,
					    &(from->rrcParagraphProperties) ) )
	{ LDEB(1); return -1;	}

    docInitRowProperties( &rpCopy );
    if  ( docCopyRowProperties( &rpCopy, &(from->rrcRowProperties) ) )
	{ LDEB(1); return -1;	}

    *to= *from;

    to->rrcParagraphProperties= ppCopy;
    to->rrcRowProperties= rpCopy;

    return 0;
    }


void docRtfCopyReadingContextBack(	RtfReadingContext *	to,
					RtfReadingContext *	from )
    {
    to->rrcBi= from->rrcBi;
    to->rrcLevel= from->rrcLevel;
    to->rrcSplitLevel= from->rrcSplitLevel;
    to->rrcCurrentLine= from->rrcCurrentLine;
    to->rrcFieldNumber= from->rrcFieldNumber;
    to->rrcJustAfterPntext= from->rrcJustAfterPntext;

    to->rrcCurrentFont= from->rrcCurrentFont;
    docInitFont( &(from->rrcCurrentFont) );
    }

/************************************************************************/
/*									*/
/*  Read a control word from the input stream.				*/
/*									*/
/************************************************************************/
static int docRtfReadControlWord(	SimpleInputStream *	sis,
					RtfReadingContext *	rrc,
					int *			pC,
					char *			controlWord,
					int *			pGotArg,
					int *			pArg )
    {
    int		c;
    int		len= 0;
    int		arg;
    int		sign= 1;

    c= sioInGetCharacter( sis );
    while( c == '\n' || c == '\r' )
	{
	if  ( c == '\n' )
	    { rrc->rrcCurrentLine++;	}
	c= sioInGetCharacter( sis );
	}

    if  ( c != '\\' )
	{ CDEB(c); return -1;	}

    c= sioInGetCharacter( sis );
    if  ( ! isalpha( c ) )
	{
	switch( c )
	    {
	    case '\n':
		rrc->rrcCurrentLine++;
		/*FALLTHROUGH*/
	    case '\r':
		strcpy( controlWord, "par" );
		*pGotArg= 0;
		return 0;
	    case '\t':
		strcpy( controlWord, "tab" );
		*pGotArg= 0;
		return 0;
	    default:
		*pC= c; return 1;
	    }
	}
    controlWord[len++]= c;

    for (;;)
	{
	c= sioInGetCharacter( sis );
	if  ( ! isalpha( c ) )
	    { controlWord[len]= '\0'; break;	}
	if  ( len >= TEDszRTFCONTROL )
	    { LLDEB(len,TEDszRTFCONTROL); return -1;	}
	controlWord[len++]= c;
	}

    /*HACK*/
    if  ( ! strcmp( controlWord, "endash" )	||
          ! strcmp( controlWord, "emdash" )	)
	{ *pC= ISO1_hyphen; return 2;	}

    if  ( ! strcmp( controlWord, "rquote" )	)
	{ *pC= ISO1_quoteright; return 2;	}

    if  ( ! strcmp( controlWord, "lquote" )	)
	{ *pC= ISO1_quoteleft; return 2;	}

    if  ( ! strcmp( controlWord, "ldblquote" )	)
	{ *pC= ISO1_quotedbl; return 2;	}

    if  ( ! strcmp( controlWord, "rdblquote" )	)
	{ *pC= ISO1_quotedbl; return 2;	}

    if  ( ! strcmp( controlWord, "bullet" )	)
	{ *pC= ISO1_periodcentered; return 2;	}

    if  ( c == '-' )
	{ sign= -1; c= sioInGetCharacter( sis );	}

    *pGotArg= isdigit( c );
    if  ( *pGotArg )
	{
	arg= c- '0'; c= sioInGetCharacter( sis );
	while( isdigit( c ) )
	    { arg= 10* arg+ c- '0'; c= sioInGetCharacter( sis ); }

	*pArg= sign* arg;
	}

    if  ( c != ' ' )
	{ sioInUngetLastRead( sis );	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find a Control word.						*/
/*									*/
/************************************************************************/
const RtfControlWord * docRtfFindWord(	const char *		controlWord,
					const RtfControlWord *	contolWords,
					int			listOnly )
    {
    if  ( ! contolWords )
	{ return (RtfControlWord *)0;	}

    while( contolWords->rcwWord )
	{
	if  ( ! strcmp( contolWords->rcwWord, controlWord ) )
	    { return contolWords;	}

	contolWords++;
	}

    if  ( ! listOnly )
	{
	contolWords= docRtfPropertyWords;

	while( contolWords->rcwWord )
	    {
	    if  ( ! strcmp( contolWords->rcwWord, controlWord ) )
		{ return contolWords;	}

	    contolWords++;
	    }
	}

    return (RtfControlWord *)0;
    }

int docRtfApplyControlWord(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			gotArg,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( ! gotArg )
	{ arg= -1;	}

    if  ( (*rcw->rcwApply) ( sis, rcw, arg, rrc ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Just ignore a control word, a text particule.			*/
/*									*/
/************************************************************************/

int docRtfIgnoreWord(	SimpleInputStream *	sis,
			const RtfControlWord *	rcw,
			int			arg,
			RtfReadingContext *	rrc	)
    { return 0;	}

int docRtfIgnoreParticule(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    { /* SDEB(text); */ return 0;	}


/************************************************************************/
/*									*/
/*  Find out what to do.						*/
/*									*/
/*  Return	RTFfiCTRLGROUP:	Found a control group.			*/
/*  Return	RTFfiWORD:	Found a control word.			*/
/*  Return	RTFfiCHAR:	Found a character.			*/
/*  Return	RTFfiCLOSE:	Found an unescaped brace.		*/
/*  Return	-1:		Error.					*/
/*									*/
/************************************************************************/

int docRtfFindControl(		SimpleInputStream *	sis,
				RtfReadingContext *	rrc,
				int *			pC,
				char *			controlWord,
				int *			pGotArg,
				int *			pArg		)
    {
    int				c;
    int				res;

    if  ( rrc->rrcCharacterAhead != EOF )
	{
	*pC= rrc->rrcCharacterAhead;
	rrc->rrcCharacterAhead= EOF;
	return RTFfiCHAR;
	}

    for (;;)
	{
	c= sioInGetCharacter( sis );

	switch( c )
	    {
	    case EOF:
		XDEB(c); *pC= '}'; return RTFfiCLOSE;
	    case '\\':
		sioInUngetLastRead( sis );
		res= docRtfReadControlWord( sis, rrc, &c,
						controlWord, pGotArg, pArg );

		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		if  ( res > 0 )
		    {
		    switch( c )
			{
			case '\\': case '{': case '}':
			    goto defaultCase;
			case '~':
			    c= ISO1_nobreakspace;
			    goto defaultCase;
			case '\'':
			    if  ( res == 1 )
				{
				char	b[3];

				b[0]= sioInGetCharacter( sis );
				b[1]= sioInGetCharacter( sis );
				b[2]= '\0';

				sscanf( b, "%x", &c );
				}
			    goto defaultCase;
			default:
			    goto defaultCase;
			}
		    }
		return RTFfiWORD;
	    case '{':
		c= sioInGetCharacter( sis );
		while( c == '\n' || c == '\r' )
		    {
		    if  ( c == '\n' )
			{ rrc->rrcCurrentLine++;	}
		    c= sioInGetCharacter( sis );
		    }
		if  ( c == '\\' )
		    {
		    sioInUngetLastRead( sis );
		    res= docRtfReadControlWord( sis, rrc, &c,
						controlWord, pGotArg, pArg );
		    if  ( res < 0 )
			{ LDEB(res); return -1;	}
		    if  ( res > 0 )
			{
			if  ( c == '*' )
			    { return RTFfiSTARGROUP;	}

			if  ( c=='\'' )
			    {
			    char	b[3];

			    b[0]= sioInGetCharacter( sis );
			    b[1]= sioInGetCharacter( sis );
			    b[2]= '\0';

			    sscanf( b, "%x", &c );
			    }

			rrc->rrcCharacterAhead= c;
			return RTFfiTEXTGROUP;
			}

		    return RTFfiCTRLGROUP;
		    }

		sioInUngetLastRead( sis );
		return RTFfiTEXTGROUP;
	    case '\n':
		rrc->rrcCurrentLine++;
		/*FALLTHROUGH*/
	    case '\r':
		continue;
	    case '}':
		*pC= c; return RTFfiCLOSE;
	    case '\t':
		*pC= c; return RTFfiTAB;
	    default: defaultCase:
		*pC= c; return RTFfiCHAR;
	    }
	}
    }

/************************************************************************/
/*									*/
/*  Adjust the level in the hierarchy for docRtfReadGroup().		*/
/*									*/
/*  1)  Empty paragraphs get one empty particule to have a height.	*/
/*									*/
/************************************************************************/
int docRtfAdjustLevel(	RtfReadingContext *	rrc,
			int			toLevel,
			int			textLevel	)
    {
    BufferItem *	bi= rrc->rrcBi;

    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    /*  1  */
    if  ( toLevel < rrc->rrcLevel	&&
	  bi->biLevel == DOClevPARA	&&
	  bi->biParaParticuleCount == 0	)
	{
	if  ( docInflateTextString( bi, 0 ) )
	    { LDEB(bi->biParaStrlen); return -1;	}
	bi->biParaString[bi->biParaStrlen]= '\0';

	if  ( ! docInsertTextParticule( bi, -1, 0, 0,
				DOCkindTEXT, rrc->rrcCurrentTextAttribute ) )
	    { LDEB(bi->biParaParticuleCount); return -1;	}
	}

    while( toLevel < rrc->rrcLevel )
	{ bi= bi->biParent; rrc->rrcLevel--; }
    rrc->rrcSplitLevel= rrc->rrcLevel;

    while( rrc->rrcLevel < toLevel && rrc->rrcLevel < textLevel )
	{
	BufferItem *	fresh;

	fresh= docInsertItem( bi, -1, rrc->rrcLevel+ 1 );
	if  ( ! fresh )
	    {
	    SLDEB(docLevelStr(rrc->rrcLevel),fresh);
	    SSDEB(docLevelStr(toLevel),docLevelStr(textLevel));
	    return -1;
	    }

	bi= fresh;
	
	rrc->rrcLevel++;

	if  ( rrc->rrcLevel == DOClevPARA )
	    {
	    BufferItem *	cellBi= bi->biParent;
	    BufferItem *	rowBi= cellBi->biParent;

	    if  ( docCopyParagraphProperties( &(bi->biParaProperties),
					 &(rrc->rrcParagraphProperties) ) )
		{ LDEB(1); return -1;	}

	    if  ( rowBi && bi->biParaInTable )
		{
		if  ( cellBi->biGroupChildCount == 1	&&
		      rowBi->biGroupChildCount == 1	)
		    {
		    if  ( ! rowBi->biRowHasTableParagraphs )
			{
			if  ( docCopyRowProperties( &(rowBi->biRowProperties),
						 &(rrc->rrcRowProperties) ) )
			    { LDEB(1); return -1;	}

			rowBi->biRowHasTableParagraphs= 1;
			}
		    }
		else{
		    if  ( ! rowBi->biRowHasTableParagraphs )
			{
			BufferItem *	insBi;
			if  ( cellBi->biGroupChildCount > 1 )
			    {
			    if  ( docSplitGroupItem( &insBi, cellBi,
						    bi->biNumberInParent ) )
				{ LDEB(1); return -1;	}
			    }

			if  ( rowBi->biGroupChildCount > 1 )
			    {
			    if  ( docSplitGroupItem( &insBi, rowBi,
						cellBi->biNumberInParent ) )
				{ LDEB(1); return -1;	}
			    }

			if  ( docCopyRowProperties( &(rowBi->biRowProperties),
						 &(rrc->rrcRowProperties) ) )
			    { LDEB(1); return -1;	}

			rowBi->biRowHasTableParagraphs= 1;
			}
		    }
		}
	    else{
		if  ( rowBi && rowBi->biRowHasTableParagraphs )
		    { LDEB(rowBi->biRowHasTableParagraphs); }
		}
	    }

	if  ( rrc->rrcLevel == DOClevROW )
	    {
	    if  ( docCopyRowProperties( &(bi->biRowProperties),
					 &(rrc->rrcRowProperties) ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( rrc->rrcLevel == DOClevCELL )
	    {
	    if  ( docCopyCellProperties( &(bi->biCellProperties),
						&(rrc->rrcCellProperties) ) )
		{ LDEB(1); return -1;	}
	    }
	}

    rrc->rrcSplitLevel= rrc->rrcLevel;
    rrc->rrcBi= bi;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume free text upto a control word or a group.			*/
/*									*/
/************************************************************************/
static int docRtfReadText(	SimpleInputStream *	sis,
				int			c,
				int *			pC,

				char *			controlWord,
				int *			pGotArg,
				int *			pArg,

				RtfReadingContext *	rrc,
				RtfAddTextParticule	addParticule	)
    {
    int				res;

    static unsigned char *	collectedText;
    static int			collectedSize;

    unsigned char *		fresh;
    int				size;
    int				len;

    len= 0;
    if  ( len >= collectedSize )
	{
	size= 100;

	fresh= (unsigned char *)realloc( collectedText, size+ 2 );
	if  ( ! fresh )
	    { LXDEB(size,fresh); return -1;	}

	collectedText= fresh;
	collectedSize= size;
	}

    collectedText[len++]= c;
    collectedText[len  ]= '\0';

    for (;;)
	{
	c= sioInGetCharacter( sis );

	switch( c )
	    {
	    case EOF:
		LDEB(c); return -1;
	    case '}':
		if  ( len > 0 )
		    {
		    if  ( (*addParticule)( rrc, collectedText, len ) )
			{ LDEB(len); return -1;	}
		    }
		return RTFfiCLOSE;
	    case '{':
		sioInUngetLastRead( sis );
		res= docRtfFindControl( sis, rrc, &c,
						controlWord, pGotArg, pArg );
		if  ( res < 0 )
		    { LDEB(res); return res;	}
		if  ( (*addParticule)( rrc, collectedText, len ) )
		    { LDEB(len); return -1;	}
		*pC= c; return res;
	    case '\n':
		rrc->rrcCurrentLine++;
		/*FALLTHROUGH*/
	    case '\r' :
		continue;
	    case '\\':
		sioInUngetLastRead( sis );
		res= docRtfFindControl( sis, rrc, &c,
						controlWord, pGotArg, pArg );
		if  ( res < 0 )
		    { LDEB(res); return res;	}

		if  ( res != RTFfiCHAR )
		    {
		    if  ( (*addParticule)( rrc, collectedText, len ) )
			{ LDEB(len); return -1;	}

		    return res;
		    }
		/*FALLTHROUGH*/
	    default:
		if  ( len >= collectedSize )
		    {
		    size= ( 3* collectedSize+ 2 )/ 2;

		    if  ( size < len )
			{ size= len+ 2;	}

		    fresh= (unsigned char *)realloc( collectedText, size+ 2 );
		    if  ( ! fresh )
			{ LXDEB(size,fresh); return -1;	}

		    collectedText= fresh;
		    collectedSize= size;
		    }
		collectedText[len++]= c;
		collectedText[len  ]= '\0';
		break;
	    case '\t':
		if  ( (*addParticule)( rrc, collectedText, len ) )
		    { LDEB(len); return -1;	}
		strcpy( controlWord, "tab" ); *pGotArg= 0; *pArg= -1;
		res= RTFfiWORD;
		return res;
	    }
	}
    }

/************************************************************************/
/*									*/
/*  Skip an unknown or superfluous group.				*/
/*									*/
/************************************************************************/
int docRtfSkipGroup(	SimpleInputStream *	sis,
			const RtfControlWord *	rcw,
			int			textLevel,
			RtfReadingContext *	rrc		)
    {
    int		complainUnknown= rrc->rrcComplainUnknown;

    rrc->rrcComplainUnknown= 0;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
		(RtfControlWord *)0, 0, 0, rrc,
		docRtfEmptyTable, docRtfEmptyTable, docRtfIgnoreParticule ) )
	{ SDEB(rcw->rcwWord); return -1;	}

    rrc->rrcComplainUnknown= complainUnknown;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Read an RTF control group. typically the document.			*/
/*									*/
/************************************************************************/
int docRtfReadGroup(	SimpleInputStream *	sis,
			int			textLevel,

			const RtfControlWord *	applyFirst,
			int			gotArg,
			int			arg,

			RtfReadingContext *	externRrc,

			const RtfControlWord *	contolWords,
			const RtfControlWord *	groupWords,
			RtfAddTextParticule	addParticule	)
    {
    int				rval;
    RtfReadingContext		internRrc;

    docRtfInitReadingContext( &internRrc );
    docRtfCopyReadingContext( &internRrc, externRrc );

    externRrc->rrcCharacterAhead= EOF;

    if  ( applyFirst )
	{
	rval= docRtfApplyControlWord( sis, applyFirst,
						    gotArg, arg, &internRrc );
	if  ( rval )
	    { SDEB(applyFirst->rcwWord);	}
	}

    rval= docRtfConsumeGroup( sis, textLevel, &internRrc,
				    contolWords, groupWords, addParticule );

    if  ( rval )
	{ LDEB(rval);	}

    docRtfCopyReadingContextBack( externRrc, &internRrc );
    docRtfCleanReadingContext( &internRrc );

    return rval;
    }

int docRtfConsumeGroup(	SimpleInputStream *	sis,
			int			textLevel,

			RtfReadingContext *	rrc,

			const RtfControlWord *	contolWords,
			const RtfControlWord *	groupWords,
			RtfAddTextParticule	addParticule	)
    {
    int				res;
    const RtfControlWord *	rcw;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				arg= -1;
    int				c;

    int				resAhead= -2;
    int				gotArgAhead;
    int				argAhead;

    res= docRtfFindControl( sis, rrc, &c, controlWord, &gotArg, &arg );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}

    for (;;)
	{
	int	complainUnknown;

	switch( res )
	    {
	    case RTFfiCLOSE:
		return 0;
	    case RTFfiCHAR:
		if  ( textLevel != DOClevANY				&&
		      docRtfAdjustLevel( rrc, textLevel, textLevel )	)
		    { LLDEB(textLevel,rrc->rrcLevel); return -1; }

		res= docRtfReadText( sis, c, &c, controlWord,
				    &gotArg, &arg, rrc, addParticule );
		if  ( res < 0 )
		    { SLDEB(controlWord,res); return -1;	}
		break;
	    case RTFfiTAB:
		strcpy( controlWord, "tab" );
		/*FALLTHROUGH*/
	    case RTFfiWORD:
		rcw= docRtfFindWord( controlWord, contolWords, 0 );
		if  ( ! rcw )
		    {
		    if  ( rrc->rrcComplainUnknown )
			{ LSDEB(rrc->rrcCurrentLine,controlWord);	}
		    }
		else{
		    if  ( rcw->rcwLevel != DOClevANY			&&
			  docRtfAdjustLevel( rrc,
					    rcw->rcwLevel, textLevel )	)
			{ SDEB(controlWord); return -1; }

		    if  ( rcw->rcwPrepare )
			{
			if  ( (*rcw->rcwPrepare) ( sis, rcw, arg, rrc ) )
			    { SLDEB(rcw->rcwWord,arg); return -1;	}
			}

		    if  ( rcw->rcwDetailWords )
			{
			for (;;)
			    {
			    const RtfControlWord *	rcwAhead;

			    resAhead= docRtfFindControl( sis, rrc, &c,
					controlWord, &gotArgAhead, &argAhead );

			    if  ( resAhead != RTFfiWORD )
				{ break;	}

			    rcwAhead= docRtfFindWord( controlWord,
						    rcw->rcwDetailWords, 1 );

			    if  ( ! rcwAhead )
				{ break;	}

			    resAhead= docRtfApplyControlWord( sis,
				    rcwAhead, gotArgAhead, argAhead, rrc  );

			    if  ( resAhead < 0 )
				{ LSDEB(resAhead,controlWord); return -1; }
			    }
			}

		    res= docRtfApplyControlWord( sis, rcw,
							gotArg, arg, rrc );

		    if  ( res < 0 )
			{ LSDEB(res,controlWord); return -1;	}
		    }

		if  ( resAhead == -2 )
		    {
		    res= docRtfFindControl( sis, rrc, &c,
						controlWord, &gotArg, &arg );
		    }
		else{
		    res= resAhead; arg= argAhead; gotArg= gotArgAhead;
		    resAhead= -2;
		    }

		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		break;
	    case RTFfiCTRLGROUP:
		rcw= docRtfFindWord( controlWord, groupWords, 1 );
		if  ( ! rcw )
		    {
		    rcw= docRtfFindWord( controlWord, contolWords, 0 );
		    if  ( rcw )
			{
			if  ( rcw->rcwLevel != DOClevANY		&&
			      docRtfAdjustLevel( rrc,
					    rcw->rcwLevel, textLevel )	)
			    { SDEB(controlWord); return -1; }

			if  ( docRtfReadGroup( sis, textLevel,
						rcw, gotArg, arg, rrc,
						contolWords, groupWords,
						addParticule ) )
			    { SDEB(rcw->rcwWord); return -1;	}
			}
		    else{
			complainUnknown= rrc->rrcComplainUnknown;

			if  ( rrc->rrcComplainUnknown )
			    { SDEB(controlWord);	}

			rrc->rrcComplainUnknown= 0;

			if  ( docRtfReadGroup( sis,
				    rrc->rrcLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfIgnoreParticule ) )
			    { SDEB(rcw->rcwWord); return -1;	}

			rrc->rrcComplainUnknown= complainUnknown;
			}
		    }
		else{
		  groupFound:
		    if  ( rcw->rcwLevel != DOClevANY			&&
			  docRtfAdjustLevel( rrc,
					    rcw->rcwLevel, textLevel )	)
			{ SDEB(controlWord); return -1; }

		    res= docRtfApplyControlWord( sis, rcw, gotArg, arg, rrc );

		    if  ( res < 0 )
			{ LDEB(res); SDEB(controlWord); return -1;	}
		    }

		res= docRtfFindControl( sis, rrc, &c,
						controlWord, &gotArg, &arg );
		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		break;
	    case RTFfiSTARGROUP:
		res= docRtfReadControlWord( sis, rrc, &c,
					    controlWord, &gotArg, &arg );
		if  ( res )
		    { LDEB(res); return -1;	}
		rcw= docRtfFindWord( controlWord, groupWords, 1 );
		if  ( rcw )
		    { goto groupFound; }

		complainUnknown= rrc->rrcComplainUnknown;

		rrc->rrcComplainUnknown= 0;

		if  ( docRtfReadGroup( sis, rrc->rrcLevel,
			    (RtfControlWord *)0, 0, 0, rrc,
			    docRtfEmptyTable, docRtfEmptyTable,
			    docRtfIgnoreParticule ) )
		    { SDEB(rcw->rcwWord); return -1;	}

		rrc->rrcComplainUnknown= complainUnknown;

		res= docRtfFindControl( sis, rrc, &c,
					    controlWord, &gotArg, &arg );
		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		break;
	    case RTFfiTEXTGROUP:
		if  ( docRtfReadGroup( sis, textLevel,
					(RtfControlWord *)0, 0, 0, rrc,
					contolWords, groupWords,
					addParticule ) )
		    { LDEB(1); return -1;	}

		res= docRtfFindControl( sis, rrc, &c,
					    controlWord, &gotArg, &arg );
		if  ( res < 0 )
		    { LDEB(res); return -1;	}
		break;
	    default:
		LDEB(res); return -1;
	    }
	}
    }

int docRtfIgnoreText(	RtfReadingContext *	rrc,
			const unsigned char *	text,
			int			len	)
    { return 0; }
