<copyright> Stream classes.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1995-1998 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: streams.t,v 1.28 1999/03/09 13:56:42 tiggr Exp $</id>  </copyright>

/******************** Stream ********************/

implementation class
Stream: Conditions, instance (All)

end;

implementation instance
Stream: instance (All)

deferred void
  close;

end;

/******************** InputStream ********************/

implementation class
InputStream: Stream

end;

implementation instance
InputStream

<doc> Discard any unread input.  The default implementation does nothing.
    </doc>
id
  flushInput
{
  = self;
}

<doc> Read a byte from the receiving stream.  This raises a {stream-eos}
    condition upon end-of-stream.  </doc>
deferred byte
  read;

<doc> Read a byte from the receiving stream.  Return -1 for end-of-stream.
    </doc>
deferred int
  read;

<doc> Read at most {num} bytes from the stream into the {buffer}, starting
    to add bytes at its current length.  Return the number of bytes
    successfully read, which is 0 for end-of-stream.  </doc>
int
  readBytes int num
       into MutableByteArray buffer
{
  = [self readRange ([buffer length], num) into buffer];
}

<doc> Read at most {num} bytes from the stream into the {buffer}, by
    writing in it from position {start}.  Return the number of bytes
    successfully read, which is 0 for end-of-stream.  </doc>
deferred int
  readRange (int, int) (start, num)
       into MutableByteArray buffer;

<doc> Read a `\n' terminated sequence of bytes and return them (without
    the `\n' at the end).  Return {nil} upon end of file (if no characters
    have been collected).  </doc>
MutableByteString
  readLine
{
  = MutableByteString ([self readLineInto [MutableByteString new]]);
}

<doc> Read a `\n' terminated sequence of bytes and return them (without
    the `\n' at the end) in {buf}.  Return {nil} upon end of file (if no
    characters have been collected).  If the optional {truncate} is not
    NO, the buffer is truncated before use.  </doc>
MutableByteArray
  readLineInto MutableByteArray buf
     truncate: boolean trunc = YES
{
  int c = [self read];

  if (c == -1)
    return nil;

  if (trunc)
    [buf resize 0];
  for (void; c != -1 && c != '\n'; c = [self read])
    [buf add byte (c)];

  = buf;
}

end;

/******************** OutputStream ********************/

implementation class
OutputStream: Stream
{
  <doc> The buffer used by {print base:...}.  </doc>
  local static MutableByteString print_buffer;
}

end;

implementation instance
OutputStream

<doc> Write out any unwritten (buffered) output.  The default
    implementation does nothing.  </doc>
id
  flushOutput
{
  = self;
}

<doc> Output a new line to the receiving stream.  An interactive stream
    should override this to also flush its output if it desires line based
    buffering.  </doc>
id
  nl
{
  = [["\n" write self] flushOutput];
}

<doc> Return the truncated {print_buffer}, creating it iff necessary.  </doc>
private MutableByteString
  print_buffer
{
  if (!print_buffer)
    print_buffer = [MutableByteString withCapacity 128];
  else
    [print_buffer truncate 0];

  = print_buffer;
}

extern id
  print boolean b;

id
  print byte b
{
  [self write b];

  = self;
}

extern id
  print char c;

extern id
  print int i;

extern id
  print long l;

extern id
  print float f;

extern id
  print double d;

extern id
  print pointer addr;

id
  print All object
{
  if (object == nil)
    = ["*nil*" write self];
  else
    = [object write self];
}

<doc> Send {print} to {self} for each element of the tuple {x}.  </doc>
extern id
  print dynamic x;

// GGG The {value} is an int which should be a long...
// Wed May 22 13:04:04 1996, tiggr@tom.es.ele.tue.nl
<doc> Output the {value} to the receiving stream.

    The optional {base} dictates the base of the representation, which
    defaults to 10.

    If the optional {space} is not 0, it is the number of positions the
    representation must at least occupy.

    If {space} is not 0, the optional {flush} dictates how the
    representation is to be flushed.  A negative value means left, 0 for
    center, and a positive value dictates a right shift.  The absolute
    value of {flush} indicates the amount of whitespace which must be
    available at the other end.

    The optional {signed} should be 0 for unsigned, or 1 for signed.  If
    it is -1 (the default) the value is assumed unsigned, unless {base}
    has its default value, 10.

    The optional {digit_10} sets the value to use for the decimal value 10
    when using a {base} exceeding that value.  </doc>
id
    print int value
    base: int base = 10
   space: int space = 0
   flush: int flush = -1
  signed: int how_signed = -1
   range: char digit_10 = char ('a')
{
  if (!print_buffer)
    print_buffer = [MutableByteString withCapacity 40];
  else
    [print_buffer truncate 0];

  if (how_signed == -1)
    how_signed = base == 10 ? 1 : 0;

  boolean negative = how_signed == 1 && value < 0;

  do
    {
      int digit = int (value % base);
      byte ch;

      /* The % can produce a negative result since VALUE is (by
         definition) signed, even for large (seemingly unsigned)
         hexadecimal values.  */
      if (digit < 0)
	digit = -digit;

      [print_buffer add ((digit < 10)
			 ? '0' + byte (digit)
			 : byte (digit_10) + (byte (digit) - byte (10)))];

      value /= base;
    } while (value != 0);

  if (negative)
    [print_buffer add '-'];

  if (space == 0)
    [print_buffer reverse (0, -1)];
  else
    {
      int n = [print_buffer length];

      if (flush < 0)
	{
	  /* Flush left.  */
	  [print_buffer reverse (0, -1)];

	  while (n < space || flush < 0)
	    {
	      [print_buffer add ' '];
	      n++;
	      flush++;
	    }
	}
      else if (flush > 0)
	{
	  /* Flush right.  */
	  while (n < space || flush > 0)
	    {
	      [print_buffer add ' '];
	      n++;
	      flush--;
	    }

	  [print_buffer reverse (0, -1)];
	}
      else
	{
	  /* Flush center.  */
	  int spaces = space - n;
	  int left;

	  while (left < spaces / 2)
	    {
	      [print_buffer add ' '];
	      left++;
	    }

	  [print_buffer reverse (0, -1)];

	  while (left < spaces)
	    {
	      [print_buffer add ' '];
	      left++;
	    }
	}
    }

  [print_buffer write self];

  = self;
}

<doc> Write the byte {b}, signaling a condition upon eof.  </doc>
deferred void
  write byte b;

<doc> Write the byte {b} and return the number of bytes actually written.
    </doc>
deferred int
  write byte b;
  
<doc> The lowest level multiple-byte writing method: Write the {num} bytes
    from {address} to the stream, and return the number of bytes written.
    </doc>
deferred int
  writeBytes int num
	from pointer address;

int
  writeRange (int, int) (start, length)
        from ByteArray buffer
{
  pointer a;

  (a, length) = [buffer pointerToElements (start, length)];

  = [self writeBytes length from a];
}

int
  writeBytes ByteArray buffer
{
  = [self writeRange (0, -1) from buffer];
}

end;

/******************** InputOutputStream ********************/

implementation class
InputOutputStream: InputStream, OutputStream

end;

implementation instance
InputOutputStream

end;

/******************** SeekableStream ********************/

implementation class
SeekableStream: Stream,
	Constants

end;

implementation instance
SeekableStream

<doc> Return the current position.  </doc>
deferred long
  position;

<doc> Set the position.  Any following operation will operate from the new
    position.  The optional argument {whence} defaults to
    {STREAM_SEEK_SET}, for absolute positioning.  Possible other values
    are {STREAM_SEEK_CUR}, for positioning relative to the current
    position, and {STREAM_SEEK_END}, to work relative to the end.  </doc>
deferred void
       seek long offset
  relative: int whence = STREAM_SEEK_SET;

end;

/******************** StreamStream ********************/

implementation class
StreamStream: Stream, State

instance (id)
  with Stream s
{
  = !s ? nil : [[self alloc] init s];
}

end;

implementation instance
StreamStream
{
  <doc> The stream to which we output and/or from which we input.  </doc>
  public InputOutputStream stream;
}

void
  close
{
  [stream close];
}

id
  init Stream s
{
  stream = InputOutputStream (s);

  = self;
}

end;

/******************** stdio ********************/

implementation class
stdio: instance (All)
{
  <doc> The stream connected to descriptor 0, known as {stdin} in C.  </doc>
  static public InputStream in;

  <doc> The stream connected to descriptor 1, C's {stdout}.  </doc>
  static public OutputStream out;

  <doc> The stream connected to descriptor 2, C's {stderr}.  Like {out},
      this stream is buffered.  </doc>
  static public InputOutputStream err;
}

<doc> Close the {descriptor}.  Raises a {stream-error} on failure.  </doc>
extern void
  close int descriptor;

end;

implementation instance
stdio: instance (All)
end;
