#include "tools.h"

/* ---------------------------------------------------------------------
*
*  -- PBLAS routine (version 1.5) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory,
*     and University of California, Berkeley.
*     March 17, 1995
*
*  ---------------------------------------------------------------------
*/

void pbchkmat( m, mpos0, n, npos0, ia, ja, desc_A, dpos0, iia, jja,
               iarow, iacol, nprow, npcol, myrow, mycol, nra, nca,
               info )
/*
*  .. Scalar Arguments ..
*/
   int         dpos0, * iia, * info, ia, * iacol, * iarow, * jja,
               ja, m, mpos0, myrow, mycol, npcol, nprow, n, * nca,
               npos0, * nra;
/*
*  .. Array Arguments ..
*/
   int         desc_A[];
{
/*
*
*  Purpose
*  =======
*
*  pbmatvect checks the validity of a descriptor vector DESCA, the
*  related global indexes IA, JA. It also computes the starting local
*  indexes (IIA,JJA) corresponding to the submatrix starting globally at
*  the entry pointed by (IA,JA). Moreover, this routine returns the
*  coordinates in the grid of the process owning the global matrix entry
*  of indexes (IA,JA), namely (IAROW,IACOL). The routine prevents from
*  out-of-bound memory access, by performing the adequate MIN operation
*  on IIA and JJA.  Finally, if an inconsitency is found among its
*  parameters ia, ja and desc_A, the routine returns an error code in
*  info.
*
*  Arguments
*  =========
*
*  M       (global input) INTEGER
*          The number or matrix rows of A being operated on.
*
*  MPOS0   (global input) INTEGER
*          Where in the calling routine's parameter list M appears.
*
*  N       (global input) INTEGER
*          The number or matrix columns of A being operated on.
*
*  NPOS0   (global input) INTEGER
*          Where in the calling routine's parameter list N appears.
*
*  IA      (global input) INTEGER
*          A's global row index, which points to the beginning of the
*          submatrix which is to be operated on.
*
*  JA      (global input) INTEGER
*          A's global column index, which points to the beginning of
*          the submatrix which is to be operated on.
*
*  DESCA   (global and local input) INTEGER array of dimension DLEN_.
*          The array descriptor for the distributed matrix A.
*
*  DPOS0   (global input) INTEGER
*          Where in the calling routine's parameter list DESCA
*          appears.  Note that we assume IA and JA are respectively 2
*          and 1 entries behind DESCA.
*
*  IIA     (local output) pointer to INTEGER
*          The local rows starting index of the submatrix.
*
*  JJA     (local output) pointer to INTEGER
*          The local columns starting index of the submatrix.
*
*  IAROW   (global output) pointer to INTEGER
*          The row coordinate of the process that possesses the first
*          row and column of the submatrix.
*
*  IACOL   (global output) pointer to INTEGER
*          The column coordinate of the process that possesses the
*          first row and column of the submatrix.
*
*  NPROW   (global input) INTEGER
*          The total number of process rows over which the distributed
*          matrix is distributed.
*
*  NPCOL   (global input) INTEGER
*          The total number of process columns over which the
*          distributed matrix is distributed.
*
*  MYROW   (local input) INTEGER
*          The row coordinate of the process calling this routine.
*
*  MYCOL   (local input) INTEGER
*          The column coordinate of the process calling this routine.
*
*  NRA     (local output) pointer to INTEGER
*          The maximum between the number of local rows owned by the
*          calling process and 1.
*
*  NCA     (local output) pointer to INTEGER
*          The maximum between the number of local columns owned by the
*          calling process and 1.
*
*  INFO    (local input/local output) INTEGER
*          = 0:  successful exit
*          < 0:  If the i-th argument is an array and the j-entry had
*                an illegal value, then INFO = -(i*100+j), if the i-th
*                argument is a scalar and had an illegal value, then
*                INFO = -i.
*
*  =====================================================================
*
*  .. Parameters ..
*/
#define DESCMULT      100
#define BIGNUM      10000
/* ..
*  .. Local Scalars ..
*/
   int         descpos, ExtraColBlock, ExtraRowBlock, iapos, japos,
               mpos, MyColBlock, MyColDist, MyRowBlock, MyRowDist,
               NColBlock, npos, NRowBlock;
/* ..
*  .. External Functions ..
*/
   F_INTG_FCT  numroc_();
/*
*  .. Executable Statements ..
*/
   if( *info >= 0 )
      *info = BIGNUM;
   else if( *info < -DESCMULT )
      *info = -(*info);
   else
      *info = -(*info) * DESCMULT;
/*
*  Figure where in parameter list each parameter was, factoring in
*  descriptor multiplier
*/
   mpos = mpos0 * DESCMULT;
   npos = npos0 * DESCMULT;
   iapos = ( dpos0 - 2 ) * DESCMULT;
   japos = ( dpos0 - 1 ) * DESCMULT;
   descpos = dpos0 * DESCMULT + 1;
/*
 * Check that we have a legal descriptor type
 */
   if(desc_A[DT_] != BLOCK_CYCLIC_2D) *info = MIN( *info, descpos + DT_ );
/*
*  Check that matrix values make sense from local viewpoint
*/
   if( m < 0 )
      *info = MIN( *info, mpos );
   if( n < 0 )
      *info = MIN( *info, npos );
   else if( ia < 1 )
      *info = MIN( *info, iapos );
   else if( ja < 1 )
      *info = MIN( *info, japos );
   else if( desc_A[MB_] < 1 )
      *info = MIN( *info, descpos + MB_ );
   else if( desc_A[NB_] < 1 )
      *info = MIN( *info, descpos + NB_ );
   else if( ( desc_A[RSRC_] < 0 ) || ( desc_A[RSRC_] >= nprow ) )
      *info = MIN( *info, descpos + RSRC_ );
   else if( ( desc_A[CSRC_] < 0 ) || ( desc_A[CSRC_] >= npcol ) )
      *info = MIN( *info, descpos + CSRC_ );
   else if( desc_A[LLD_] < 1 )
      *info = MIN( *info, descpos + LLD_ );

   if( m == 0 || n == 0 )
   {
/*
*     NULL matrix, relax some checks
*/
      if( desc_A[M_] < 0 )
         *info = MIN( *info, descpos + M_ );
      if( desc_A[N_] < 0 )
         *info = MIN( *info, descpos + N_ );
   }
   else
   {
/*
*     more rigorous checks for non-degenerate matrices
*/
      if( desc_A[M_] < 1 )
         *info = MIN( *info, descpos + M_ );
      else if( desc_A[N_] < 1 )
         *info = MIN( *info, descpos + N_ );
      else
      {
         if( ia > desc_A[M_] )
            *info = MIN( *info, iapos );
         else if( ja > desc_A[N_] )
            *info = MIN( *info, japos );
         else
         {
            if( ia+m-1 > desc_A[M_] )
               *info = MIN( *info, mpos );
            if( ja+n-1 > desc_A[N_] )
               *info = MIN( *info, npos );
         }
      }
   }
/*
*  Retrieve local information for matrix A, and prepare output:
*  set info = 0 if no error, and divide by DESCMULT if error is not
*  in a descriptor entry.
*/
   if( *info == BIGNUM )
   {
      MyRowDist = ( myrow + nprow - desc_A[RSRC_] ) % nprow;
      MyColDist = ( mycol + npcol - desc_A[CSRC_] ) % npcol;
      NRowBlock = desc_A[M_] / desc_A[MB_];
      NColBlock = desc_A[N_] / desc_A[NB_];
      *nra = ( NRowBlock / nprow ) * desc_A[MB_];
      *nca = ( NColBlock / npcol ) * desc_A[NB_];
      ExtraRowBlock = NRowBlock % nprow;
      ExtraColBlock = NColBlock % npcol;

      ia--;
      ja--;
      MyRowBlock = ia / desc_A[MB_];
      MyColBlock = ja / desc_A[NB_];
      *iarow = ( MyRowBlock + desc_A[RSRC_] ) % nprow;
      *iacol = ( MyColBlock + desc_A[CSRC_] ) % npcol;

      *iia = ( MyRowBlock / nprow + 1 ) * desc_A[MB_] + 1;
      *jja = ( MyColBlock / npcol + 1 ) * desc_A[NB_] + 1;

      if( MyRowDist >= ( MyRowBlock % nprow ) )
      {
         if( myrow == *iarow )
            *iia += ia % desc_A[MB_];
         *iia -= desc_A[MB_];
      }
      if( MyRowDist  < ExtraRowBlock )
         *nra += desc_A[MB_];
      else if( MyRowDist == ExtraRowBlock )
         *nra += ( desc_A[M_] % desc_A[MB_] );
      *nra = MAX( 1, *nra );

      if( MyColDist >= ( MyColBlock % npcol ) )
      {
         if( mycol == *iacol )
            *jja += ja % desc_A[NB_];
         *jja -= desc_A[NB_];
      }
      if( MyColDist < ExtraColBlock )
         *nca += desc_A[NB_];
      else if( MyColDist == ExtraColBlock )
         *nca += ( desc_A[N_] % desc_A[NB_] );
      *nca = MAX( 1, *nca );

      *iia = MIN( *iia, *nra );
      *jja = MIN( *jja, *nca );

      if( desc_A[LLD_] < *nra )
      {
         if( numroc_(&desc_A[N_], &desc_A[NB_], &mycol, &desc_A[CSRC_], &npcol) )
            *info = -( descpos + LLD_ );
         else *info = 0;
      }
      else
         *info = 0;
   }
   else if( *info % DESCMULT == 0 )
   {
      *info = -(*info) / DESCMULT;
   }
   else
   {
      *info = -(*info);
   }
}
