(*	$Id: CNDecl.Mod,v 1.3 1998/09/25 20:10:08 acken Exp $	*)
MODULE CNDecl;
(*  Facilities to access declarations in abstract syntax trees.
    Copyright (C) 1998  Michael van Acken

    This file is part of OOC.

    OOC is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.  

    OOC is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details. 

    You should have received a copy of the GNU General Public License
    along with OOC. If not, write to the Free Software Foundation, 59
    Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

(*

This module provides the facilities to identify declarations, access
there properties, identify names, and handle types.  Roughly speaking
it is equivalent to the module SymbolTable of the OOC compiler.

We don't build a separate symbol table to keep track of declarations
or do our searching.  Instead we take the information from the
abstract syntax tree.  For obvious reasons it's all there, even if it
isn't presented in a convenient way.

A declaration is identified by the identifier being defined.  We can't
use the declaration itself, because some declarations (variable and
formal parameter declarations) can define multiple identifiers and are
therefore not unique.  Further properties of a declaration, like kind
and associated type, are extracted from the abstract syntax tree.
Usually it's sufficient to look at the syntactic entity the identifier
is part of.

A type is represented by the subtree of the AST that makes up the
type's type constructor.  That is, a record type is represented by an
instance of AST.Record, an array type by AST.Array, a type alias by an
instance AST.TypeName, etc.  Note that type aliases are not resolved
automatically.  Where appropriate they have to be reduced manually to
their base type.

*)

IMPORT
  Strings, S := CNScanner, AST := CNAST;
  

CONST  (* class ids for uses of identifiers; see procedure Classify *)
  use = 0;          (* using occurence of a name *)
  module = 1;       (* defines name of imported module *)
  const = 2;        (* name of constant definition *)
  type = 3;         (*         type definition *)
  var = 4;          (*         var definition *)
  procedure = 5;    (*         procedure definition *)
  formalParam = 6;  (*         formal parameter *)
  field = 7;        (*         record field definition *)
  redundant = 8;
  (* the class `redundant' encompasses all identifiers that are neither 
     declarations nor uses of declared names: module name, ending names of
     module or procecure, and names of formal parameters of procedure types 
     (but not procedure declarations!) *)

VAR
  currModule*: AST.Module;
  (* pointer to the module name to which all ids, which are generated by 
     DeclarationID or TypeID, are local *)
  modCache: AST.IdentDefList;
  (* cache of imported modules *)


(*PROCEDURE WriteStack (node: AST.Node);
  VAR
    type: Types.Type;
  BEGIN
    Out.String ("Stack: ");
    Out.Ln;
    WHILE (node # NIL) DO
      type := Types.TypeOf (node);
      Out.String ("  ");
      Out.String (type. name^);
      Out.Ln;
      node := node. up
    END
  END WriteStack;*)


PROCEDURE UncacheModule* (name: ARRAY OF CHAR);
  VAR
    ptr: AST.IdentDef;
  BEGIN
    ptr := modCache;
    WHILE (ptr # NIL) & (ptr. name. str^ # name) DO
      ptr := ptr. next
    END;
    IF (ptr # NIL) THEN
      IF (ptr. prev = NIL) THEN
        modCache := ptr. next
      ELSE
        ptr. prev. next := ptr. next
      END;
      IF (ptr. next # NIL) THEN
        ptr. next. prev := ptr. prev
      END
    END
  END UncacheModule;

PROCEDURE CacheModule* (module: AST.Module);
  BEGIN
    UncacheModule (module. name. name. str^);
    module. name. prev := NIL;
    module. name. next := modCache;
    IF (modCache # NIL) THEN
      modCache. prev := module. name
    END;
    modCache := module. name
  END CacheModule;

PROCEDURE FlushCache*;
  BEGIN
    modCache := NIL
  END FlushCache;



PROCEDURE Classify (ident: AST.Ident): SHORTINT;
(* Classify use of identifier `ident'.
   pre: ident. up # NIL  *)
  VAR
    up: AST.Node;
  BEGIN
    up := ident. up;
    WITH up: AST.Import DO
      IF (up. name = ident) THEN
        RETURN module
      END
    | up: AST.ConstDecl DO
      RETURN const
    | up: AST.TypeDecl DO
      RETURN type
    | up: AST.VarDecl DO
      RETURN var
    | up: AST.ProcDecl DO
      IF (up. arrow # NIL) THEN
        RETURN redundant
      ELSE
        RETURN procedure
      END
    | up: AST.FPSection DO
      IF (up. up. up IS AST.ProcDecl) THEN
        RETURN formalParam
      ELSE
        RETURN redundant
      END
    | up: AST.FieldList DO
      RETURN field
      
    | up: AST.Module DO
      RETURN redundant
    | up: AST.Body DO
      RETURN redundant
    ELSE  (* using occurence *)
    END;
    RETURN use
  END Classify;

PROCEDURE IsDeclaration* (ident: AST.Ident): BOOLEAN;
(* Returns TRUE iff `ident' is the name defined by a declaration.
   pre: ident. up # NIL  *)
  VAR
    class: SHORTINT;
  BEGIN
    class := Classify (ident);
    RETURN (class # use) & (class # redundant)
  END IsDeclaration;

PROCEDURE GetDeclSym (start: AST.Node; ident: AST.Symbol; 
                      ignorePos: BOOLEAN): AST.IdentDef;
(* Locates definition refered to by the name `ident' returns its identifier.  
   The node `start' is used as a starting point into the abstract syntax.  
   Unless `ignorePos' is TRUE all declarations following `ident.pos' will be
   ignored.
   Result is NIL if the name is undefined or if it is a predefined entity.  
   Note that the specs of this procs are a little bit muddled because it is
   also called while parsing to identify names of modules, and when searching
   for imported identifiers.
   pre: ~((start IS AST.Module) OR (start IS AST.ProcDecl)) *)
    
  PROCEDURE EnclosingScope (node: AST.Node): AST.Scope;
    BEGIN
      (* move towards root of the syntax tree towards the enclosing procedure
         or module *)
      REPEAT
        node := node. up
      UNTIL (node = NIL) OR (node IS AST.ProcDecl) OR (node IS AST.Module);
      
      IF (node # NIL) THEN
        RETURN node(AST.Scope)
      ELSE
        RETURN NIL
      END
    END EnclosingScope;
  
  PROCEDURE SearchScope (scope: AST.Scope; ident: AST.Symbol): AST.IdentDef;
  (* Searches in the scope of the given module or procedure for a declaration
     that defines the name `ident'.  The relative position of declaration and
     identifier is ignored.  That is, this function may return a declaration
     that appears after `ident' in the source code.  Note that result is 
     NIL if `body' is NIL.  *)
    VAR
      body: AST.Body;
      decl: AST.Decl;
      name: AST.IdentDef;
      fpSection: AST.Decl;
      isList: BOOLEAN;
    
    PROCEDURE GetName (decl: AST.Decl; VAR isList: BOOLEAN): AST.IdentDef;
      BEGIN
        isList := (decl IS AST.VarDecl);
        RETURN decl. name
      END GetName;
    
    PROCEDURE Body (scope: AST.Scope): AST.Body;
      BEGIN
        WITH scope: AST.ProcDecl DO
          RETURN scope. body
        | scope: AST.Module DO
          RETURN scope. body
        END
      END Body;
    
    BEGIN
      IF (scope # NIL) THEN
        (* scan local declarations *)
        body := Body (scope);
        IF (body # NIL) THEN
          decl := body. declSeq;
          WHILE (decl # NIL) DO
            name := GetName (decl, isList);
            IF (name # NIL) THEN
              IF isList THEN
                WHILE (name # NIL) DO
                  IF (name. name. str^ = ident. str^) THEN
                    RETURN name
                  END;
                  name := name. next
                END
              ELSIF (name. name. str^ = ident. str^) THEN
                RETURN name
              END
            END;
            decl := decl.  next
          END
        END;

        IF (scope IS AST.ProcDecl) THEN
          (* check receiver *)
          IF (scope(AST.ProcDecl). receiver # NIL) & 
             (scope(AST.ProcDecl). receiver. fpSections. name. name. str^ = ident. str^) THEN
            RETURN scope(AST.ProcDecl). receiver. fpSections. name
          END;
          
          IF (scope(AST.ProcDecl). fpars # NIL) THEN
            (* scan formal parameters *)
            fpSection := scope(AST.ProcDecl). fpars. fpSections;
            WHILE (fpSection # NIL) DO
              name := fpSection. name;
              WHILE (name # NIL) DO
                IF (name. name. str^ = ident. str^) THEN
                  RETURN name
                END;
                name := name. next
              END;
              fpSection := fpSection. next
            END
          END
          
        ELSIF (scope(AST.Module). importList # NIL) THEN
          (* scan import list *)
          decl := scope(AST.Module). importList. imports;
          WHILE (decl # NIL) DO
            IF (decl(AST.Import). name. name. str^ = ident. str^) THEN
              RETURN decl(AST.Import). name
            END;
            decl := decl. next
          END
        END
      END;
      RETURN NIL
    END SearchScope;
  
  PROCEDURE Identify (scope: AST.Scope; ident: AST.Symbol): AST.IdentDef;
  (* Locates declaration of `ident' that holds at the place in the source text
     where it appears.  Result is NIL if `ident' is undefined, or if it is
     a predefined identifier.  *)
    VAR
      def, upDef: AST.IdentDef;
    
    PROCEDURE IsPtrBaseType (ident: AST.Ident): BOOLEAN;
      BEGIN
        RETURN (ident. up IS AST.Qualident) &
               (ident. up. up IS AST.TypeName) &
               (ident. up. up. up IS AST.Pointer)
      END IsPtrBaseType;
    
    BEGIN
      def := SearchScope (scope, ident);
      IF (def # NIL) THEN
        IF ignorePos OR (def. name. pos < ident. pos) THEN
          RETURN def
        ELSE  (* name is used prior to declaration *)
          upDef := SearchScope (EnclosingScope (scope), ident);
          IF (upDef # NIL) THEN
            RETURN upDef
          ELSIF (ident. up # NIL) & (ident. up IS AST.Ident) &
                IsPtrBaseType (ident. up(AST.Ident)) THEN
            RETURN def
          ELSE
            RETURN NIL
          END
        END
      ELSIF (scope = NIL) THEN  (* no matching declaration in module *)
        RETURN NIL
      ELSE
        RETURN Identify (EnclosingScope (scope), ident)
      END
    END Identify;
  
  BEGIN
    RETURN Identify (EnclosingScope (start), ident)
  END GetDeclSym;

PROCEDURE IsModuleName* (start: AST.Node; ident: S.Symbol): BOOLEAN;
(* Returns TRUE iff `ident' stands for a imported module in the context of
   node `start'.  This function can be called while constructing the 
   syntax tree.  *)
  VAR
    decl: AST.Ident;
  BEGIN
    decl := GetDeclSym (start, ident, FALSE);
    RETURN (decl # NIL) & (Classify (decl) = module)
  END IsModuleName;

PROCEDURE GetDeclaration* (ident: AST.Ident): AST.IdentDef;
  VAR
    modName: AST.Ident;
    module: AST.IdentDef;
    import: AST.Import;
  
  PROCEDURE GetImport (VAR name: ARRAY OF CHAR): AST.Import;
    VAR
      node: AST.Node;
      import: AST.Import;
    BEGIN  (* it is guaranteed that `name' appears in the import statement *)
      node := ident;
      WHILE ~(node IS AST.Module) DO
        node := node. up
      END;
      import := node(AST.Module). importList. imports(AST.Import);
      LOOP
        IF (import. name. name. str^ = name) THEN
          RETURN import
        END;
        import := import. next(AST.Import)
      END
    END GetImport;
  
  BEGIN
    IF (ident. up IS AST.Qualident) &
       (ident. up(AST.Qualident). ident = ident) &
       (ident. up(AST.Qualident). module # NIL) THEN
      import := GetImport (ident. up(AST.Qualident). module. name. str^);
      modName := import. module;
      module := modCache;
      WHILE (module # NIL) & (module. name. str^ # modName. name. str^) DO
        module := module. next
      END;
      IF (module = NIL) THEN  (* no such module *)
        RETURN NIL
      ELSE
        RETURN GetDeclSym (module, ident. name, TRUE)
      END
    ELSE
      RETURN GetDeclSym (ident, ident. name, FALSE)
    END
  END GetDeclaration;
  
PROCEDURE GetType* (decl: AST.Ident): AST.Type;
(* Retrieves the type associated with the given declaration.  
   pre: `ident' refers to a type, variable, or field declaration.  *)
  VAR
    up: AST.Node;
  BEGIN
    up := decl. up;
    WITH up: AST.TypeDecl DO
      RETURN up. type
    | up: AST.VarDecl DO
      RETURN up. type
    | up: AST.FPSection DO
      RETURN up. type
    END
  END GetType;

PROCEDURE BaseRecord* (procDecl: AST.ProcDecl): AST.Record;
(* Given a type-bound procedure, this function returns
   the record type to which the procedure is bound.  *)
   
  PROCEDURE GetBase (type: AST.Type): AST.Record;
    VAR
      decl: AST.Ident;
    BEGIN
      IF (type IS AST.TypeName) THEN
        decl := GetDeclaration (type(AST.TypeName). qualident. ident);
        IF (decl = NIL) THEN
          RETURN NIL
        ELSE
          RETURN GetBase (GetType (decl))
        END
      ELSIF (type IS AST.Pointer) THEN
        RETURN GetBase (type(AST.Pointer). base)
      ELSIF (type IS AST.Record) THEN
        RETURN type(AST.Record)
      ELSE
        RETURN NIL
      END
    END GetBase;
  
  BEGIN
    RETURN GetBase (procDecl. receiver. fpSections. type)
  END BaseRecord;

PROCEDURE StripAlias (type: AST.Type): AST.Type;
  VAR
    decl: AST.Ident;
  BEGIN
    WHILE (type IS AST.TypeName) DO
      decl := GetDeclaration (type(AST.TypeName). qualident. ident);
      IF (decl = NIL) OR ~(decl. up IS AST.TypeDecl) THEN
        RETURN type
      ELSE
        type := GetType (decl)
      END
    END;
    RETURN type
  END StripAlias;

PROCEDURE SearchMember (record: AST.Record; name: ARRAY OF CHAR): AST.IdentDef;
(* Searches record `type' for a record field or type-bound procedure named
   `name'.  Result if NIL if no matching entry is found.  *)
  VAR
    decl: AST.Ident;
    type: AST.Type;
    fieldList: AST.Decl;
    ident: AST.IdentDef;
    node: AST.Node;
    module: AST.Module;
    gdecl: AST.Decl;
  BEGIN
    LOOP
      node := record;
      WHILE ~(node IS AST.Module) DO
        node := node. up
      END;
      module := node(AST.Module);

      (* scan record fields *)
      fieldList := record. fields;
      WHILE (fieldList # NIL) DO
        ident := fieldList. name;
        WHILE (ident # NIL) DO
          IF (ident. name. str^ = name) THEN
            RETURN ident
          END;
          ident := ident. next
        END;
        fieldList := fieldList. next
      END;
      
      (* scan list of procedures for associated type-bound procedures *)
      gdecl := module. body. declSeq;
      WHILE (gdecl # NIL) DO
        WITH gdecl: AST.ProcDecl DO
          IF (gdecl. name. name. str^ = name) &
             (gdecl. arrow = NIL) &
             (gdecl. receiver # NIL) &
             (BaseRecord (gdecl) = record) THEN
            RETURN gdecl. name
          END
        ELSE (* ignore *)
        END;
        gdecl := gdecl. next
      END;
      
      (* continue if there is a base record *)
      IF (record. base = NIL) THEN
        RETURN NIL
      ELSE
        decl := GetDeclaration (record. base(AST.TypeName). qualident. ident);
        IF (decl = NIL) OR ~(decl. up IS AST.TypeDecl) THEN
          RETURN NIL
        ELSE
          type := StripAlias (GetType (decl));
          IF (type IS AST.Record) THEN
            record := type(AST.Record)
          ELSE
            RETURN NIL
          END
        END
      END
    END
  END SearchMember;



PROCEDURE ^ TypeID (type: AST.Type; sep: ARRAY OF CHAR; VAR id: ARRAY OF CHAR);

PROCEDURE DeclarationID* (ident: AST.Ident; sep: ARRAY OF CHAR;
                          VAR id: ARRAY OF CHAR);
(* Returns a name for the definition `ident'.  The returned string is formatted
   like a HTML HREF.  That is, the first (optional) part of the string is the 
   name of a file, followed by "#", followed by a unique identifier within
   the given file.  
   Note that `ident' can be either the name of a declaration, or a using 
   occurence of a name.
   
   pre: `currModule' is NIL, or it is set to the root of the abstract syntax
     tree of the module that is considered local, and for which the "module#"
     prefix should be omitted.  *)
  VAR
    class: SHORTINT;
    procDecl: AST.ProcDecl;
    def, moduleIdent: AST.Ident;
    base: AST.Type;
    
  PROCEDURE Scope (ident: AST.Ident; class: SHORTINT): AST.Ident;
  (* For the given definition, represented by its identifier `ident', find its
     enclosing scope and return its name.  The result is either the name of a
     procedure or that of the module.  *)
    VAR
      body: AST.Body;
    BEGIN
      (* pre: ~(class IN {use, module, field, type-bound procedure} *)
      CASE class OF
      | const, type, var, procedure:
        body := ident. up. up(AST.Body);
        IF (body. up IS AST.Module) THEN
          RETURN body. up(AST.Module). name
        ELSE
          RETURN body. up(AST.ProcDecl). name
        END
      | formalParam:
        RETURN ident. up. up. up(AST.ProcDecl). name
      END
    END Scope;
  
  PROCEDURE GetMemberDecl (member: AST.Member; super: BOOLEAN): AST.Ident;
  (* Locates declaration of the record field or type-bound procedure refered
     to by the selector `member'.  Result is NIL if the declaration can't be
     found.  `super' indicates that an arrow (^) follows the member name, and
     that a type-bound procedure should be taken from the base class.  *)
    VAR
      design: AST.Designator;
      start, decl: AST.Ident;
      select: AST.Selector;
      type: AST.Type;
      with: AST.WithBranch;
      expr: AST.Expr;
    
    PROCEDURE SearchWith (qident: AST.Qualident): AST.WithBranch;
      VAR
        node: AST.Node;
      
      PROCEDURE SameName (q0, q1: AST.Qualident): BOOLEAN;
        BEGIN
          RETURN ((q0. module = NIL) = (q1. module = NIL)) &
                 ((q0. module = NIL) OR
                   (q0. module. name. str^ = q1. module. name. str^)) &
                 (q0. ident. name. str^ = q1. ident. name. str^)
        END SameName;
      
      BEGIN
        node := ident;
        WHILE ~(node IS AST.Body) DO
          IF (node IS AST.WithBranch) & 
             SameName (qident, node(AST.WithBranch). var) THEN
            RETURN node(AST.WithBranch)
          END;
          node := node. up
        END;
        RETURN NIL
      END SearchWith;
    
    BEGIN
      design := member. up(AST.Designator);
      start := GetDeclaration (design. qualident. ident);
      IF (start # NIL) THEN
        (* get type of identifier that starts the designator; if we are in 
           a WITH statement see if the type is modified *)
        IF (design. qualident. module = NIL) THEN
          with := SearchWith (design. qualident)
        ELSE
          with := NIL
        END;
        IF (with = NIL) THEN
          type := StripAlias (GetType (start))
        ELSE
          type := StripAlias (with. type)
        END;
        
        select := design. selectors;
        WHILE (select # member) DO
          IF (type IS AST.Pointer) &
             ((select IS AST.Member) OR (select IS AST.Index)) THEN
            type := StripAlias (type(AST.Pointer). base)  (* auto deref *)
          END;
          
          (* determine type of designator after selector has been applied *)
          WITH select: AST.Member DO
            IF (type IS AST.Record) THEN
              decl := SearchMember(type(AST.Record), select. name. name. str^);
              IF (decl = NIL) OR (decl. up IS AST.ProcDecl) THEN
                RETURN NIL
              ELSE
                type := StripAlias (decl. up(AST.FieldList). type)
              END
            ELSE
              RETURN NIL
            END
            
          | select: AST.Index DO
            expr := select. exprList;
            WHILE (expr # NIL) DO
              IF ~(type IS AST.Array) THEN
                RETURN NIL
              END;
              type := StripAlias (type(AST.Array). type);
              expr := expr. next
            END
            
          | select: AST.Deref DO
            IF (type IS AST.Pointer) THEN
              type := StripAlias (type(AST.Pointer). base)
            ELSE
              RETURN NIL
            END
          
          | select: AST.TypeGuard DO
            type := StripAlias (select. type)
          END;
          
          select := select. next
        END;
        IF (type IS AST.Pointer) THEN  (* auto deref *)
          type := StripAlias (type(AST.Pointer). base)
        END;
        IF (type IS AST.Record) THEN
          decl := SearchMember (type(AST.Record), member. name. name. str^);
          IF super & (decl. up IS AST.ProcDecl) & 
             (type(AST.Record). base # NIL) THEN
            (* super call, return the declaration from the base type *)
            type := StripAlias (type(AST.Record). base);
            WITH type: AST.Record DO
              decl := SearchMember (type, member. name. name. str^)
            ELSE
              decl := NIL
            END
          END;
          RETURN decl
        END
      END;
      RETURN NIL
    END GetMemberDecl;
  
  PROCEDURE DoConcat (VAR str: ARRAY OF CHAR): BOOLEAN;
    VAR
      i: INTEGER;
    BEGIN
      IF (str = "") THEN
        RETURN FALSE
      ELSE
        i := 1;
        WHILE (str[i] # 0X) DO
          INC (i)
        END;
        RETURN (str[i-1] # "#")
      END
    END DoConcat;
  
  BEGIN
    class := Classify (ident);
    CASE class OF
    | module:
      COPY (ident. name. str^, id)
    | const, type, var, formalParam:
      DeclarationID (Scope (ident, class), sep, id);
      IF DoConcat (id) THEN
        Strings.Append (sep, id)
      END;
      Strings.Append (ident. name. str^, id)
    | procedure:
      procDecl := ident. up(AST.ProcDecl);
      IF (procDecl. receiver = NIL) THEN
        DeclarationID (Scope (ident, class), sep, id);
        IF DoConcat (id) THEN
          Strings.Append (sep, id)
        END;
        Strings.Append (ident. name. str^, id)
      ELSE
        base := BaseRecord (procDecl);
        IF (base = NIL) THEN
          COPY ("[undef]", id)
        ELSE
          TypeID (base, sep, id);
          Strings.Append (sep, id);
          Strings.Append (ident. name. str^, id)
        END
      END
    | field:
      TypeID (ident. up. up(AST.Record), sep, id);
      Strings.Append (sep, id);
      Strings.Append (ident. name. str^, id)
    | redundant:
      IF (ident. up IS AST.ProcDecl) THEN  (* forward decl of proc name *)
        DeclarationID (Scope (ident, procedure), sep, id);
        IF DoConcat (id) THEN
          Strings.Append (sep, id)
        END;
        Strings.Append (ident. name. str^, id)
      ELSIF (ident. up IS AST.Module) THEN
        IF ((currModule = NIL) OR
            (ident. name. str^ # currModule. name. name. str^)) THEN
          (* reference to imported module *)
          COPY (ident. name. str^, id);
          Strings.Append (".html#", id)
        ELSE
          COPY ("#", id)
        END
      ELSE
        COPY ("", id)
      END

    ELSE  (* using occurence of a name *)
      IF (ident. up IS AST.Qualident) & 
         (ident. up(AST.Qualident). module # NIL) THEN
        (* reference to imported module *)
        IF (ident. up(AST.Qualident). module. name. str^ = "SYSTEM") THEN
          COPY ("", id)
        ELSE
          def := GetDeclaration (ident. up(AST.Qualident). module);
          IF (def = NIL) THEN
            COPY ("[undef]", id)
          ELSE
            moduleIdent := def. up(AST.Import). module;
            COPY (moduleIdent. name. str^, id);
            Strings.Append (".html#", id);
            Strings.Append (ident. name. str^, id)
          END
        END
      
      ELSIF (ident. up IS AST.Member) THEN
        (* record field or type-bound procedure *)
        def := GetMemberDecl (ident. up(AST.Member), 
                              (ident. up(AST.Member). next # NIL) &
                              (ident. up(AST.Member). next IS AST.Deref));
        IF (def = NIL) THEN
          COPY ("", id)
        ELSE
          DeclarationID (def, sep, id)
        END
        
      ELSE  (* local or predefined object or predefined *)
        def := GetDeclaration (ident);
        IF (def = NIL) THEN
          COPY ("", id)
        ELSE
          DeclarationID (def, sep, id)
        END
      END
    END
  END DeclarationID;

PROCEDURE TypeID (type: AST.Type; sep: ARRAY OF CHAR; VAR id: ARRAY OF CHAR);
  VAR
    decl: AST.Node;
  BEGIN
    WHILE (type. up IS AST.Type) DO
      type := type. up(AST.Type)
    END;
    decl := type. up;
    WITH decl: AST.TypeDecl DO
      DeclarationID (decl. name, sep, id)
    | decl: AST.VarDecl DO
      DeclarationID (decl. name, sep, id)
    | decl: AST.FieldList DO
      TypeID (decl. up(AST.Record), sep, id);
      Strings.Append (sep, id);
      Strings.Append (decl. name. name. str^, id)
    | decl: AST.FPSection DO
      DeclarationID (decl. name, sep, id);
      Strings.Append (sep, id);
      Strings.Append (decl. name. name. str^, id)
    END
  END TypeID;

PROCEDURE ProcSuperDecls* (procDecl: AST.ProcDecl;
                           VAR super, base: AST.IdentDef);
(* Given a declaration of a type-bound procedure, this procedure determines the
   the procedure's super and base definition.  If neither of them exist the 
   returned pointers are NIL, and if the super definition is also the base
   definition `super' will be NIL.
   pre: `procDecl' is type-bound procedure *)
  VAR
    curr, rec, type: AST.Type;
    prospBase: AST.IdentDef;

  PROCEDURE Visible (superDecl: AST.IdentDef): BOOLEAN;
    VAR
      procModule, superModule: AST.Node;
    BEGIN
      IF (superDecl. mark # NIL) THEN  (* exported and therefore visible *)
        RETURN TRUE
      ELSE  (* check if both definition are in the same module *)
        procModule := procDecl;
        WHILE ~(procModule IS AST.Module) DO
          procModule := procModule. up
        END;
        superModule := superDecl;
        WHILE ~(superModule IS AST.Module) DO
          superModule := superModule. up
        END;
        RETURN (procModule = superModule)
      END
    END Visible;
  
  BEGIN
    super := NIL; base := NIL;
    rec := BaseRecord (procDecl);
    IF (rec # NIL) & (rec IS AST.Record) & (rec(AST.Record). base # NIL) THEN
      curr := StripAlias (rec(AST.Record). base);
      WITH curr: AST.Record DO
        super := SearchMember (curr, procDecl. name. name. str^);
        IF (super # NIL) & (super. up IS AST.ProcDecl) & Visible (super) THEN
          base := super;
          type := BaseRecord (base. up(AST.ProcDecl));
          WHILE (type # NIL) & 
                (type IS AST.Record) & (type(AST.Record). base # NIL) DO
            type := StripAlias (type(AST.Record). base);
            WITH type: AST.Record DO
              prospBase := SearchMember (type, procDecl. name. name. str^);
              IF (prospBase = NIL) OR ~(prospBase. up IS AST.ProcDecl) OR
                 ~Visible (prospBase) THEN
                (* no matching definition on this level *)
                type := NIL  (* current contents of `base' are valid *)
              ELSE
                base := prospBase
              END
            ELSE
              type := NIL  (* error, abort *)
            END
          END
        ELSE
          super := NIL
        END;
        
        IF (base = NIL) OR (base = super) THEN
          base := super;
          super := NIL
        END
      ELSE (* illegal base type *)
      END
    END
  END ProcSuperDecls;

PROCEDURE ProcSuperIDs* (procDecl: AST.ProcDecl; sep: ARRAY OF CHAR;
                         VAR super, base: ARRAY OF CHAR);
(* Given a declaration of a type-bound procedure, this procedure determines the
   ids of the procedure's super and base definition.  If neither of them exist
   the returned strings are emtpy, and if the super definition is also the base
   definition `super' will be empty.
   pre: `procDecl' is type-bound procedure *)
  VAR
    superDecl, baseDecl: AST.IdentDef;
  BEGIN
    ProcSuperDecls (procDecl, superDecl, baseDecl);
    IF (superDecl # NIL) THEN
      DeclarationID (superDecl, sep, super)
    ELSE
      COPY ("", super)
    END;
    IF (baseDecl # NIL) THEN
      DeclarationID (baseDecl, sep, base)
    ELSE
      COPY ("", base)
    END
  END ProcSuperIDs;

PROCEDURE RemoveLocal* (module: AST.Module);
(* Removes information from the syntax tree (and the symbol list) that isn't 
   part of the module's public interface.  In particular procedure bodies, 
   comments, and pragma blocks are removed.  *)
  VAR
    decl: AST.Decl;
    s: AST.Symbol;
   
  PROCEDURE RemoveBody (VAR body: AST.Body; keepDecl: BOOLEAN);
    PROCEDURE Start (decl: AST.DeclSeq; alt1, alt2: S.Symbol): S.Symbol;
      BEGIN
        IF (decl # NIL) THEN
          WITH decl: AST.ProcDecl DO
            RETURN decl. procedure
          | decl: AST.VarDecl DO
            RETURN decl. varKW
          | decl: AST.TypeDecl DO
            RETURN decl. typeKW
          | decl: AST.ConstDecl DO
            RETURN decl. constKW
          END
        ELSIF (alt1 # NIL) THEN
          RETURN alt1
        ELSE
          RETURN alt2
        END
      END Start;
    
    PROCEDURE Remove (from, to: S.Symbol; end: AST.Node);
      VAR
        s: S.Symbol;
        node, up: AST.Node;
      BEGIN
        s := from;
        to := to. next;
        WHILE (s # to) DO
          node := s. up;
          WHILE (node # NIL) & (node # end) DO
            up := node. up;
            node. Clear;
            node := up
          END;
          s. up := NIL;
          s := s. next
        END
      END Remove;

    BEGIN
      IF keepDecl THEN
        IF (body. begin # NIL) THEN
          Remove (body. begin, body. end, body)
        END
      ELSE
        Remove (Start (body. declSeq, body. begin, body. end),
                body. end, body);
        body. Clear;
        body := NIL
      END
    END RemoveBody;
  
  BEGIN
    decl := module. body. declSeq;
    WHILE (decl # NIL) DO
      WITH decl: AST.ProcDecl DO
        IF (decl. body # NIL) THEN
          RemoveBody (decl. body, FALSE)
        END
      ELSE
      END;
      decl := decl. next
    END;
    RemoveBody (module. body, TRUE);
    
    s := module. startSymbol. next;
    WHILE (s # NIL) DO
      IF (s. id >= S.comment) THEN
        s. up := NIL
      END;
      s := s. next
    END;
    
    S.RemoveUnusedSymbols (module. startSymbol);
    
    (*s := module. startSymbol. next;
    WHILE (s # NIL) DO
      FOR i := 1 TO s.line DO
        Out.Ln;
      END;
      FOR i := 1 TO s.column DO
        Out.Char (" ");
      END;
      
      Out.String (s. str^);
      s := s. next
    END;*)
  END RemoveLocal;

BEGIN
  modCache := NIL;
  currModule := NIL
END CNDecl.
