/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1999 David Baum.
 * All Rights Reserved.
 */


#include "CallStmt.h"
#include "Bytecode.h"
#include "Program.h"
#include "Symbol.h"
#include "Fragment.h"
#include "CheckState.h"
#include "GosubStmt.h"
#include "InlineStmt.h"
#include "Function.h"
#include "Clause.h"
#include "Mapping.h"
#include "AtomExpr.h"
#include "VarPool.h"
#include "AssignStmt.h"
#include "Error.h"

CallStmt::CallStmt()
{
	fName = 0;
	fStmt = 0;
}

CallStmt::~CallStmt()
{
	for(int i=0; i<fParams.GetLength(); i++)
		delete fParams[i];

	delete fStmt;
}


bool CallStmt::Check(CheckState & state)
{
	if (Fragment *f = gProgram->GetSub(fName))
	{
		// sub call
		if (state.fInSub)
		{
			Error(kErr_NestedSubs).Raise(&fLocation);
			return false;
		}
		
		if (fParams.GetLength() != 0)
		{
			Error(kErr_ParamCount).Raise(&fLocation);
			return false;
		}
		
		fStmt = new GosubStmt(f);
	}
	else if (Function *f = gProgram->GetFunction(fName))
	{
		if (!ExpandFunction(f, state.fVarPool)) return false;
	}
	else
	{
		Error(kErr_UndefinedFunction, fName->GetKey()).Raise(&fLocation);
		return false;
	}

	return fStmt->Check(state);
}


void CallStmt::Emit(Bytecode &b)
{
	if (fStmt)
		fStmt->Emit(b);
}

CallStmt *CallStmt::Clone(Mapping *b) const
{
	CallStmt *c = new CallStmt;
	
	c->fName = fName;
	c->fLocation = fLocation;
	c->fStmt = fStmt ? fStmt->Clone(b) : 0;
	
	for(int i=0; i<fParams.GetLength(); i++)
		c->fParams.Append(fParams[i]->Clone(b));

	return c;
}


bool CallStmt::ExpandFunction(Function *f, VarPool *varPool)
{
	int argCount = f->GetArgCount();
	
	// check number of parameters
	if (argCount != fParams.GetLength())
	{
		Error(kErr_ParamCount).Raise(&fLocation);
		return false;
	}

	// create the inline stmt
	InlineStmt *s = new InlineStmt();
	fStmt = s;

	Mapping mapping(f->GetLocalCount());
	
	// substitute args
	for(int i=0; i<argCount; i++)
	{
		const Expr* arg = fParams[i]->GetExpr();
		int val;
		
		switch(f->GetArgType(i))
		{
			case Function::kConstantArg:
				if (!arg->Evaluate(val))
				{
					Error(kErr_ParamType, "constant").Raise(fParams[i]->GetLoc());
					return false;
				}
				mapping.Set(i, new AtomExpr(kRCX_ConstantType, val));
				break;
			case Function::kIntegerArg:
				if (!varPool)
				{
					Error(kErr_TempsNotAllowed).Raise(fParams[i]->GetLoc());
					return false;
				}
				val = varPool->AllocateLocal();
				if (val == kIllegalVar)
				{
					Error(kErr_NoMoreTemps).Raise(fParams[i]->GetLoc());
					return false;
				}
				mapping.Set(i, new AtomExpr(kRCX_VariableType, val), val);
				s->Add(new AssignStmt(val, kRCX_SetVar , new Clause(arg->Clone(0), *fParams[i]->GetLoc())));
				break;
			case Function::kReferenceArg:
				val = arg->GetLValue();
				if (val == kIllegalVar)
				{
					Error(kErr_ParamType, "variable").Raise(fParams[i]->GetLoc());
					return false;
				}
				mapping.Set(i, new AtomExpr(kRCX_VariableType, val));
				break;
			case Function::kConstRefArg:
				mapping.Set(i, arg->Clone(0));
				break;
			case Function::kSensorArg:
				if (!arg->PromiseSensor())
				{
					Error(kErr_ParamType, "sensor").Raise(fParams[i]->GetLoc());
					return false;
				}
				mapping.Set(i, arg->Clone(0));
				break;
			default:
				Error(kErr_ParamType, "???").Raise(fParams[i]->GetLoc());
				return false;
		}
	}

	// now for local variables other than parameters
	for(int i=argCount; i<f->GetLocalCount(); i++)
	{
		if (!varPool)
		{
			Error(kErr_TempsNotAllowed).Raise(&fLocation);
			return false;
		}
		
		int var = varPool->AllocateLocal();
		if (var == kIllegalVar)
		{
			Error(kErr_NoMoreTemps).Raise(&fLocation);
			return false;
		}
		
		mapping.Set(i, new AtomExpr(kRCX_VariableType, var), var);
	}
	
	// add body of inline
	s->Add(f->GetBody()->Clone(&mapping));
	
	// free all locals
	if (varPool)
	{
		for(int i=0; i<mapping.GetCount(); i++)
			varPool->ReleaseLocal(mapping.GetLocal(i));
	}
	
	return true;
}
