// This is -*- C++ -*-

/* 
 * drawable.h
 *
 * Copyright 1998 Karl E. Nelson
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#ifndef _GDKMM_DRAWABLE_H_
#define _GDKMM_DRAWABLE_H_

#include <gdk/gdk.h>
#include <gdk--/types.h>

//: Gdk Drawable handle.
// A drawable is a base class for all of the objects that accept
// drawing commands.  The available drawables include pixmaps, 
// windows, and bitmaps.  Drawable is an abstract as
// there is no such type on the server side.  
// <p>
// To use a drawable, create a concrete drawable of the type you
// wish to use and a GC (graphics context) for that drawable.
// With the GC you can draw lines, strings, arcs and such. 
// <p>
class Gdk_Drawable: public Gdk_Handle<GdkDrawable>
  {protected:

     virtual void ref();
     virtual void unref();

     Gdk_Drawable ();

   public:
     Gdk_Drawable (GdkDrawable *drawable);
     Gdk_Drawable (const Gdk_Drawable &drawable);

     // Destroy a drawable handle
     ~Gdk_Drawable ();


     /* Provided methods */
     
     //: Draw a point.
     void draw_point      (Gdk_GC       &gc, 
                          gint          x1,
                          gint          y1);

     //: Draw a line.
     void draw_line      (Gdk_GC       &gc, 
                          gint          x1,
                          gint          y1,
                          gint          x2, 
                          gint          y2);

     //: Draw a rectangle.
     // Depending on the GC it may be filled or unfilled.
     void draw_rectangle (Gdk_GC       &gc,
                          gint          filled,
                          gint          x,
                          gint          y,
                          gint          width,
                          gint          height);
  
     //: Draw an arc.
     void draw_arc       (Gdk_GC       &gc,
                          gint          filled,
                          gint          x,
                          gint          y,
                          gint          width,
                          gint          height,
                          gint          angle1,
                          gint          angle2);

     //: Draw a polygon. (not implemented)
     void draw_polygon   (Gdk_GC       &gc,
                          gint          filled,
                          Gdk_Points   &points,
                          gint          npoints);
    

     //: Draw a polygon.
     void draw_polygon    (Gdk_GC        &gc,
			   gint        filled,
			   GdkPoint    *points,
			   gint          npoints);
    

     //: Draw a string.
     // Requires a valid font in addition to a GC.
     void draw_string    (Gdk_Font     &font,
                          Gdk_GC       &gc,
                          gint          x,
                          gint          y,
                          const string &str);
 
     //: Draw text.
     // requires a string, font, and length in addition to a GC.
     void draw_text      (Gdk_Font     &font,
                          Gdk_GC       &gc,
                          gint          x,
                          gint          y,
                          const string &text,
                          gint          text_length);

     //: Draw a pixmap.
     void draw_pixmap     (Gdk_GC       &gc,
                           Gdk_Drawable &src,
                           gint          xsrc,
                           gint          ysrc,
                           gint          xdest,
                           gint          ydest,
                           gint          width = -1,
                           gint          height = -1);

     //: Draw a bitmap. 
     void draw_bitmap    (Gdk_GC        &gc,
                          Gdk_Bitmap   &src,
                          gint          xsrc,
                          gint          ysrc,
                          gint          xdest,
                          gint          ydest,
                          gint          width = -1,
                          gint          height = -1);

     //: Draw an image.
     void draw_image     (Gdk_GC       &gc,
                          Gdk_Image    &image,
                          gint          xsrc,
                          gint          ysrc,
                          gint          xdest,
                          gint          ydest,
                          gint          width = -1,
                          gint          height = -1);


     //: Draw a set of points. (not implemented)
     //  Draws a set of unconnected points.
     void draw_points    (Gdk_GC        &gc,
                          Gdk_Points   &points,
                          gint          npoints);

     //: Draw a set of points. 
     //  Draws a set of unconnected points.
     void draw_points    (Gdk_GC        &gc,
                          GdkPoint    *points,
                          gint          npoints);

     void draw_segments  (Gdk_GC       &gc,
                          Gdk_Segments &segs,
                          gint          nsegs);

     void draw_segments  (Gdk_GC       &gc,
                          GdkSegment  *segs,
                          gint          nsegs);

     //: Draw a set of lines. (not implemented)
     //  Draws a set of connected lines.
     void draw_lines     (Gdk_GC        &gc,
                          Gdk_Points   &points,
                          gint          npoints);

     //: Draw a set of lines.
     //  Draws a set of connected lines.
     void draw_lines     (Gdk_GC        &gc,
                          GdkPoint     *points,
                          gint          npoints);

#if GDK_VERSION_GT(1,0)
     void draw_text_wc   (Gdk_Font      &font,
                          Gdk_GC        &gc,
                          gint          x,
                          gint          y,
                          const GdkWChar *text,
                          gint          text_length);
#endif
  };

#endif // _GDKMM_DRAWABLE_H_
