------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ D B U G                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.26 $                             --
--                                                                          --
--          Copyright (C) 1996-1999 Free Software Foundation, Inc.          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Lib;      use Lib;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Output;   use Output;
with Sem_Eval; use Sem_Eval;
with Sem_Util; use Sem_Util;
with Sinput;   use Sinput;
with Stringt;  use Stringt;
with Urealp;   use Urealp;

package body Exp_Dbug is

   ----------------------
   -- Local Procedures --
   ----------------------

   procedure Add_Uint_To_Buffer (U : Uint);
   --  Add image of universal integer to Name_Buffer, updating Name_Len

   procedure Add_Real_To_Buffer (U : Ureal);
   --  Add nnn_ddd to Name_Buffer, where nnn and ddd are integer values of
   --  the normalized numerator and denominator of the given real value.

   function Bounds_Match_Size (E : Entity_Id) return  Boolean;
   --  Determine whether the bounds of E match the size of the type. This is
   --  used to determine whether encoding is required for a discrete type.

   ------------------------
   -- Add_Real_To_Buffer --
   ------------------------

   procedure Add_Real_To_Buffer (U : Ureal) is
   begin
      Add_Uint_To_Buffer (Norm_Num (U));
      Add_Str_To_Name_Buffer ("_");
      Add_Uint_To_Buffer (Norm_Den (U));
   end Add_Real_To_Buffer;

   ------------------------
   -- Add_Uint_To_Buffer --
   ------------------------

   procedure Add_Uint_To_Buffer (U : Uint) is
   begin
      if U < 0 then
         Add_Uint_To_Buffer (-U);
         Add_Char_To_Name_Buffer ('m');
      else
         UI_Image (U, Decimal);
         Add_Str_To_Name_Buffer (UI_Image_Buffer (1 .. UI_Image_Length));
      end if;
   end Add_Uint_To_Buffer;

   -----------------------
   -- Bounds_Match_Size --
   -----------------------

   function Bounds_Match_Size (E : Entity_Id) return Boolean is
      Siz : Uint;

   begin
      if not Is_OK_Static_Subtype (E) then
         return False;

      elsif Is_Integer_Type (E)
        and then Subtypes_Statically_Match (E, Base_Type (E))
      then
         return True;

      --  Here we check if the static bounds match the natural size, which
      --  is the size passed through with the debugging information. This
      --  is the Esize rounded up to 8, 16, 32 or 64 as appropriate.

      else
         if Esize (E) <= 8 then
            Siz := Uint_8;
         elsif Esize (E) <= 16 then
            Siz := Uint_16;
         elsif Esize (E) <= 32 then
            Siz := Uint_32;
         else
            Siz := Uint_64;
         end if;

         if Is_Modular_Integer_Type (E) or else Is_Enumeration_Type (E) then
            return
              Expr_Rep_Value (Type_Low_Bound (E)) = 0
                and then
              2 ** Siz - Expr_Rep_Value (Type_High_Bound (E)) = 1;

         else
            return
              Expr_Rep_Value (Type_Low_Bound (E)) + 2 ** (Siz - 1) = 0
                and then
              2 ** (Siz - 1) - Expr_Rep_Value (Type_High_Bound (E)) = 1;
         end if;
      end if;
   end Bounds_Match_Size;

   ---------------------------
   -- Get_Encoded_Type_Name --
   ---------------------------

   --  Note: see spec for details on encodings

   function Get_Encoded_Type_Name (E : Entity_Id) return Boolean is
   begin
      Name_Len := 0;

      --  Fixed-point case

      if Is_Fixed_Point_Type (E) then
         Add_Str_To_Name_Buffer ("XF_");
         Add_Real_To_Buffer (Delta_Value (E));

         if Small_Value (E) /= Delta_Value (E) then
            Add_Str_To_Name_Buffer ("_");
            Add_Real_To_Buffer (Small_Value (E));
         end if;

      --  Vax floating-point case

      elsif Vax_Float (E) then
         if Digits_Value (Base_Type (E)) = 6 then
            Add_Str_To_Name_Buffer ("XFF");

         elsif Digits_Value (Base_Type (E)) = 9 then
            Add_Str_To_Name_Buffer ("XFF");

         elsif Digits_Value (Base_Type (E)) = 15 then
            Add_Str_To_Name_Buffer ("XFG");

         else
            pragma Assert (False);
            raise Program_Error;
         end if;

      --  Discrete case where bounds do not match size

      elsif Is_Discrete_Type (E)
        and then not Bounds_Match_Size (E)
      then
         if Has_Biased_Representation (E) then
            Add_Str_To_Name_Buffer ("XB");
         else
            Add_Str_To_Name_Buffer ("XD");
         end if;

         declare
            Lo : constant Node_Id := Type_Low_Bound (E);
            Hi : constant Node_Id := Type_High_Bound (E);

            Lo_Stat : constant Boolean := Is_OK_Static_Expression (Lo);
            Hi_Stat : constant Boolean := Is_OK_Static_Expression (Hi);

            Lo_Discr : constant Boolean :=
                         Nkind (Lo) = N_Identifier
                           and then
                         Ekind (Entity (Lo)) = E_Discriminant;

            Hi_Discr : constant Boolean :=
                         Nkind (Hi) = N_Identifier
                           and then
                         Ekind (Entity (Hi)) = E_Discriminant;

            Lo_Encode : constant Boolean := Lo_Stat or Lo_Discr;
            Hi_Encode : constant Boolean := Hi_Stat or Hi_Discr;

         begin
            if Lo_Encode or Hi_Encode then
               if Lo_Encode then
                  if Hi_Encode then
                     Add_Str_To_Name_Buffer ("LU_");
                  else
                     Add_Str_To_Name_Buffer ("L_");
                  end if;
               else
                  Add_Str_To_Name_Buffer ("U_");
               end if;

               if Lo_Stat then
                  Add_Uint_To_Buffer (Expr_Rep_Value (Lo));
               elsif Lo_Discr then
                  Get_Name_String_And_Append (Chars (Entity (Lo)));
               end if;

               if Lo_Encode and Hi_Encode then
                  Add_Str_To_Name_Buffer ("__");
               end if;

               if Hi_Stat then
                  Add_Uint_To_Buffer (Expr_Rep_Value (Hi));
               elsif Hi_Discr then
                  Get_Name_String_And_Append (Chars (Entity (Hi)));
               end if;
            end if;
         end;

      --  For all other cases, the encoded name is the normal type name

      else
         return False;
      end if;

      --  If we fall through then the Name_Buffer contains the encoded name

      Name_Buffer (Name_Len + 1) := Ascii.Nul;

      if Debug_Flag_B then
         Write_Str ("**** type ");
         Write_Name (Chars (E));
         Write_Str (" is encoded as ");
         Write_Str (Name_Buffer (1 .. Name_Len));
         Write_Eol;
      end if;

      return True;
   end Get_Encoded_Type_Name;

   --------------------------
   -- Get_Variant_Encoding --
   --------------------------

   procedure Get_Variant_Encoding (V : Node_Id) is
      Choice : Node_Id;

      procedure Choice_Val (Typ : Character; Choice : Node_Id);
      --  Output encoded value for a single choice value. Typ is the key
      --  character ('S', 'F', or 'T') that precedes the choice value.

      ----------------
      -- Choice_Val --
      ----------------

      procedure Choice_Val (Typ : Character; Choice : Node_Id) is
      begin
         Add_Char_To_Name_Buffer (Typ);

         if Nkind (Choice) = N_Integer_Literal then
            Add_Uint_To_Buffer (Intval (Choice));

         --  Character literal with no entity present (this is the case
         --  Standard.Character or Standard.Wide_Character as root type)

         elsif Nkind (Choice) = N_Character_Literal
           and then No (Entity (Choice))
         then
            Add_Uint_To_Buffer
              (UI_From_Int (Int (Char_Literal_Value (Choice))));

         else
            declare
               Ent : constant Entity_Id := Entity (Choice);

            begin
               if Ekind (Ent) = E_Enumeration_Literal then
                  Add_Uint_To_Buffer (Enumeration_Rep (Ent));

               else
                  pragma Assert (Ekind (Ent) = E_Constant);
                  Choice_Val (Typ, Constant_Value (Ent));
               end if;
            end;
         end if;
      end Choice_Val;

   --  Start of processing for Get_Variant_Encoding

   begin
      Name_Len := 0;

      Choice := First (Discrete_Choices (V));
      while Present (Choice) loop
         if Nkind (Choice) = N_Others_Choice then
            Add_Char_To_Name_Buffer ('O');

         elsif Nkind (Choice) = N_Range then
            Choice_Val ('R', Low_Bound (Choice));
            Choice_Val ('T', High_Bound (Choice));

         elsif Is_Entity_Name (Choice)
           and then Is_Type (Entity (Choice))
         then
            Choice_Val ('R', Type_Low_Bound (Entity (Choice)));
            Choice_Val ('T', Type_High_Bound (Entity (Choice)));

         elsif Nkind (Choice) = N_Subtype_Indication then
            declare
               Rang : constant Node_Id :=
                        Range_Expression (Constraint (Choice));
            begin
               Choice_Val ('R', Low_Bound (Rang));
               Choice_Val ('T', High_Bound (Rang));
            end;

         else
            Choice_Val ('S', Choice);
         end if;

         Next (Choice);
      end loop;

      Name_Buffer (Name_Len + 1) := Ascii.Nul;

      if Debug_Flag_B then
         declare
            VP : constant Node_Id := Parent (V);    -- Variant_Part
            CL : constant Node_Id := Parent (VP);   -- Component_List
            RD : constant Node_Id := Parent (CL);   -- Record_Definition
            FT : constant Node_Id := Parent (RD);   -- Full_Type_Declaration

         begin
            Write_Str ("**** variant for type ");
            Write_Name (Chars (Defining_Identifier (FT)));
            Write_Str (" is encoded as ");
            Write_Str (Name_Buffer (1 .. Name_Len));
            Write_Eol;
         end;
      end if;
   end Get_Variant_Encoding;

   ---------------------------------
   -- Make_Packed_Array_Type_Name --
   ---------------------------------

   function Make_Packed_Array_Type_Name
     (Typ   : Entity_Id;
      Csize : Uint)
      return  Name_Id
   is
   begin
      Get_Name_String (Chars (Typ));
      Add_Str_To_Name_Buffer ("___XP");
      Add_Uint_To_Buffer (Csize);
      return Name_Find;
   end Make_Packed_Array_Type_Name;

   --------------------------------
   -- Save_Unitname_And_Use_List --
   --------------------------------

   procedure Save_Unitname_And_Use_List
     (Main_Unit_Node : Node_Id;
      Main_Kind      : Node_Kind)
   is
      INITIAL_NAME_LENGTH : constant := 1024;

      Item       : Node_Id;
      Pack_Name  : Node_Id;

      Unit_Spec  : Node_Id := 0;
      Unit_Body  : Node_Id := 0;

      Main_Name : String_Id;
      --  Fully qualified name of Main Unit

      Unit_Name : String_Id;
      --  Name of unit specified in a Use clause

      Spec_Unit_Index : Source_File_Index;
      Spec_File_Name  : File_Name_Type := No_File;

      Body_Unit_Index : Source_File_Index;
      Body_File_Name : File_Name_Type := No_File;

      type String_Ptr is access all String;

      Spec_File_Name_Str : String_Ptr;
      Body_File_Name_Str : String_Ptr;

      type Label is record
        Label_Name  : String_Ptr;
        Name_Length : Integer;
        Pos         : Integer;
      end record;

      Spec_Label : Label;
      Body_Label : Label;

      procedure Initialize  (L : out Label);
      --  Initialize label

      procedure Append      (L : in out Label; Ch : Character);
      --  Append character to label

      procedure Append      (L : in out Label; Str : String);
      --  Append string to label

      procedure Append_Name (L : in out Label; Unit_Name : String_Id);
      --  Append name to label

      function  Sufficient_Space
        (L         : Label;
         Unit_Name : String_Id)
         return      Boolean;
      --  Does sufficient space exist to append another name?

      procedure Initialize (L : out Label) is
      begin
         L.Name_Length := INITIAL_NAME_LENGTH;
         L.Pos := 0;
         L.Label_Name := new String (1 .. L.Name_Length);
      end Initialize;

      procedure Append (L : in out Label; Str : String) is
      begin
         L.Label_Name (L.Pos + 1 .. L.Pos + Str'Length) := Str;
         L.Pos := L.Pos + Str'Length;
      end Append;

      procedure Append (L : in out Label; Ch : Character) is
      begin
         L.Pos := L.Pos + 1;
         L.Label_Name (L.Pos) := Ch;
      end Append;

      procedure Append_Name (L : in out Label; Unit_Name : String_Id) is
         Char         : Char_Code;
         Upper_Offset : constant := Character'Pos ('a') - Character'Pos ('A');

      begin
         for J in 1 .. String_Length (Unit_Name) loop
            Char := Get_String_Char (Unit_Name, J);

            if Character'Val (Char) = '.' then
               Append (L, "__");
            elsif Character'Val (Char) in 'A' .. 'Z' then
               Append (L, Character'Val (Char + Upper_Offset));
            elsif Char /= 0 then
               Append (L, Character'Val (Char));
            end if;
         end loop;
      end Append_Name;

      function  Sufficient_Space
        (L         : Label;
         Unit_Name : String_Id)
         return      Boolean
      is
         Len : Integer := Integer (String_Length (Unit_Name)) + 1;

      begin
         for J in 1 .. String_Length (Unit_Name) loop
            if Character'Val (Get_String_Char (Unit_Name, J)) = '.' then
               Len := Len + 1;
            end if;
         end loop;

         return L.Pos + Len < L.Name_Length;
      end Sufficient_Space;

   --  Start of processing for Save_Unitname_And_Use_List

   begin
      Initialize (Spec_Label);
      Initialize (Body_Label);

      case Main_Kind is
         when N_Package_Declaration =>
            Main_Name := Full_Qualified_Name
              (Defining_Unit_Name (Specification (Unit (Main_Unit_Node))));
            Unit_Spec := Main_Unit_Node;
            Append (Spec_Label, "_LPS__");
            Append (Body_Label, "_LPB__");

         when N_Package_Body =>
            Unit_Spec := Corresponding_Spec (Unit (Main_Unit_Node));
            Unit_Body := Main_Unit_Node;
            Main_Name := Full_Qualified_Name (Unit_Spec);
            Append (Spec_Label, "_LPS__");
            Append (Body_Label, "_LPB__");

         when N_Subprogram_Body =>
            Unit_Body := Main_Unit_Node;

            if Present (Corresponding_Spec (Unit (Main_Unit_Node))) then
               Unit_Spec := Corresponding_Spec (Unit (Main_Unit_Node));
               Main_Name := Full_Qualified_Name
                 (Corresponding_Spec (Unit (Main_Unit_Node)));
            else
               Main_Name := Full_Qualified_Name
                 (Defining_Unit_Name (Specification (Unit (Main_Unit_Node))));
            end if;

            Append (Spec_Label, "_LSS__");
            Append (Body_Label, "_LSB__");

         when others =>
            return;
      end case;

      Append_Name (Spec_Label, Main_Name);
      Append_Name (Body_Label, Main_Name);

      --  If we have a body, process it first

      if Present (Unit_Body) then

         Item := First (Context_Items (Unit_Body));

         while Present (Item) loop
            if Nkind (Item) = N_Use_Package_Clause then
               Pack_Name := First (Names (Item));
               while Present (Pack_Name) loop
                  Unit_Name := Full_Qualified_Name (Entity (Pack_Name));

                  if Sufficient_Space (Body_Label, Unit_Name) then
                     Append (Body_Label, '$');
                     Append_Name (Body_Label, Unit_Name);
                  end if;

                  Pack_Name := Next (Pack_Name);
               end loop;
            end if;

            Item := Next (Item);
         end loop;
      end if;

      while Present (Unit_Spec) and then
        Nkind (Unit_Spec) /= N_Compilation_Unit
      loop
         Unit_Spec := Parent (Unit_Spec);
      end loop;

      if Present (Unit_Spec) then

         Item := First (Context_Items (Unit_Spec));

         while Present (Item) loop
            if Nkind (Item) = N_Use_Package_Clause then
               Pack_Name := First (Names (Item));
               while Present (Pack_Name) loop
                  Unit_Name := Full_Qualified_Name (Entity (Pack_Name));

                  if Sufficient_Space (Spec_Label, Unit_Name) then
                     Append (Spec_Label, '$');
                     Append_Name (Spec_Label, Unit_Name);
                  end if;

                  if Sufficient_Space (Body_Label, Unit_Name) then
                     Append (Body_Label, '$');
                     Append_Name (Body_Label, Unit_Name);
                  end if;

                  Pack_Name := Next (Pack_Name);
               end loop;
            end if;

            Item := Next (Item);
         end loop;
      end if;

      if Present (Unit_Spec) then
         Append (Spec_Label, Character'Val (0));
         Spec_Unit_Index := Source_Index (Get_Cunit_Unit_Number (Unit_Spec));
         Spec_File_Name := Full_File_Name (Spec_Unit_Index);
         Get_Name_String (Spec_File_Name);
         Spec_File_Name_Str := new String (1 .. Name_Len + 1);
         Spec_File_Name_Str (1 .. Name_Len) := Name_Buffer (1 .. Name_Len);
         Spec_File_Name_Str (Name_Len + 1) := Character'Val (0);
         Spec_Filename := Spec_File_Name_Str (1)'Unrestricted_Access;
         Spec_Context_List :=
           Spec_Label.Label_Name.all (1)'Unrestricted_Access;
      end if;

      if Present (Unit_Body) then
         Append (Body_Label, Character'Val (0));
         Body_Unit_Index := Source_Index (Get_Cunit_Unit_Number (Unit_Body));
         Body_File_Name := Full_File_Name (Body_Unit_Index);
         Get_Name_String (Body_File_Name);
         Body_File_Name_Str := new String (1 .. Name_Len + 1);
         Body_File_Name_Str (1 .. Name_Len) := Name_Buffer (1 .. Name_Len);
         Body_File_Name_Str (Name_Len + 1) := Character'Val (0);
         Body_Filename := Body_File_Name_Str (1)'Unrestricted_Access;
         Body_Context_List :=
           Body_Label.Label_Name.all (1)'Unrestricted_Access;
      end if;

   end Save_Unitname_And_Use_List;

end Exp_Dbug;
