------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUN-TIME LIBRARY (GNARL) COMPONENTS              --
--                                                                          --
--                    S Y S T E M . O S _ I N T E R F A C E                 --
--                                                                          --
--                                   S p e c                                --
--                                                                          --
--                             $Revision: 1.11 $
--                                                                          --
--           Copyright (C) 1992-1998 Free Software Foundation, Inc.         --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is the VxWorks version of this package.
--
--  VxWorks does not directly support the needed POSIX routines, but it
--  does have other routines that make it possible to code equivalent
--  POSIX compliant routines.  The approach taken is to provide an
--  FSU threads compliant interface.

--  This package encapsulates all direct interfaces to OS services
--  that are needed by children of System.

--  PLEASE DO NOT add any with-clauses to this package
--  or remove the pragma Elaborate_Body.
--  It is designed to be a bottom-level (leaf) package.

with Interfaces.C;
package System.OS_Interface is
   pragma Preelaborate;

   subtype int            is Interfaces.C.int;
   subtype short          is Interfaces.C.short;
   subtype long           is Interfaces.C.long;
   subtype unsigned       is Interfaces.C.unsigned;
   subtype unsigned_short is Interfaces.C.unsigned_short;
   subtype unsigned_long  is Interfaces.C.unsigned_long;
   subtype unsigned_char  is Interfaces.C.unsigned_char;
   subtype plain_char     is Interfaces.C.plain_char;
   subtype size_t         is Interfaces.C.size_t;
   subtype char           is Interfaces.C.char;

   -----------
   -- Errno --
   -----------

   function errno return int;
   pragma Import (C, errno, "errnoGet");

   --  NAMEs not used are commented-out
   --  NAMEs not supported on this system have __NAME for value

   --  EPERM        : constant := 1;
   --  ENOENT       : constant := 2;
   --  ESRCH        : constant := 3;
   EINTR         : constant := 4;
   --  EIO          : constant := 5;
   --  ENXIO        : constant := 6;
   --  E2BIG        : constant := 7;
   --  ENOEXEC      : constant := 8;
   --  EBADF        : constant := 9;
   --  ECHILD       : constant := 10;
   EAGAIN        : constant := 35;
   ENOMEM        : constant := 12;
   --  EACCES       : constant := 13;
   --  EFAULT       : constant := 14;
   --  ENOTEMPTY    : constant := 15;
   --  EBUSY        : constant := 16;
   --  EEXIST       : constant := 17;
   --  EXDEV        : constant := 18;
   --  ENODEV       : constant := 19;
   --  ENOTDIR      : constant := 20;
   --  EISDIR       : constant := 21;
   EINVAL        : constant := 22;
   --  ENFILE       : constant := 23;
   --  EMFILE       : constant := 24;
   --  ENOTTY       : constant := 25;
   --  ENAMETOOLONG : constant := 26;
   --  EFBIG        : constant := 27;
   --  ENOSPC       : constant := 28;
   --  ESPIPE       : constant := 29;
   --  EROFS        : constant := 30;
   --  EMLINK       : constant := 31;
   --  EPIPE        : constant := 32;
   --  EDEADLK      : constant := 33;
   --  ENOLCK       : constant := 34;
   --  ENOTSUP      : constant := 35;
   --  EMSGSIZE     : constant := 36;

   --  ANSI math software
   --  EDOM         : constant := 37;
   --  ERANGE       : constant := 38;

   ETIMEDOUT    : constant := 60;

   -------------
   -- Signals --
   -------------

   NSIG : constant := 32;
   type Signal is new int range 0 .. Interfaces.C."-" (NSIG, 1);

   --  NAMEs not used are commented-out
   --  NAMEs not supported on this system have __NAME for value

   SIGHUP     : constant := 1; --  hangup
   SIGINT     : constant := 2; --  interrupt (rubout)
   SIGQUIT    : constant := 3; --  quit (ASCD FS)
   SIGILL     : constant := 4; --  illegal instruction (not reset)
   SIGTRAP    : constant := 5; --  trace trap (not reset)

   SIGIOT     : constant := 6;
   SIGABRT    : constant := 6; --  used by abort, replace SIGIOT in the  future

   SIGEMT     : constant := 7; --  EMT instruction
   SIGFPE     : constant := 8; --  floating point exception
   SIGKILL    : constant := 9; --  kill (cannot be caught or ignored)
   SIGBUS     : constant := 10; --  bus error
   SIGSEGV    : constant := 11; --  segmentation violation

   SIGFMT     : constant := 12; --  Stack Format Error (not posix)
   SIGSYS     : constant := 12;

   SIGPIPE    : constant := 13; --  write on a pipe with no one to read it
   SIGALRM    : constant := 14; --  alarm clock
   SIGTERM    : constant := 15; --  software termination signal from kill
   SIGURG     : constant := 16;
   SIGSTOP    : constant := 17; --  stop (cannot be caught or ignored)
   SIGTSTP    : constant := 18; --  user stop requested from tty
   SIGCONT    : constant := 19; --  stopped process has been continued

   SIGCLD     : constant := 20;
   SIGCHLD    : constant := 20; --  child status change

   SIGTTIN    : constant := 21; --  background tty read attempted
   SIGTTOU    : constant := 22; --  background tty write attempted

   SIGRTMIN   : constant := 23; --  Realtime signal min
   SIGIO      : constant := 23;
   SIGPOLL    : constant := 23;
   SIGXCPU    : constant := 24;
   SIGXFSZ    : constant := 25;
   SIGVTALRM  : constant := 26;
   SIGPROF    : constant := 27;
   SIGWINCH   : constant := 28;
   SIGLOST    : constant := 29;
   SIGRTMAX   : constant := 29; --  Realtime signal max

   SIGUSR1    : constant := 30; --  user defined signal 1
   SIGUSR2    : constant := 31; --  user defined signal 2

   -----------------------------------
   -- Signal processing definitions --
   -----------------------------------

   --  The how in sigprocmask().
   SIG_BLOCK   : constant := 1;
   SIG_UNBLOCK : constant := 2;
   SIG_SETMASK : constant := 3;

   --  The sa_flags in struct sigaction.
   --  SA_NOCLDSTOP : constant := 16#0001#    --  not used
   SA_SIGINFO   : constant := 16#0002#;
   SA_ONSTACK   : constant := 16#0004#;
   --  SA_INTERRUPT : constant := 16#0008#;   --  not used
   --  SA_RESETHAND : constant := 16#0010#;   --  not used

   --  ANSI args and returns from signal().
   SIG_DFL : constant := 0;
   SIG_IGN : constant := 1;
   --  SIG_ERR : constant := -1;   --  not used

   --  The si_code returned in siginfo
   SI_SYNC    : constant := 0;  --  (Not posix) generated by hardware
   SI_KILL    : constant := 1;  --  .1 kill() function
   SI_QUEUE   : constant := 2;  --  .4 sigqueue() function
   SI_TIMER   : constant := 3;  --  expiration of a .4 timer
   SI_ASYNCIO : constant := 4;  --  completion of an async I/O
   SI_MESGQ   : constant := 5;  --  arrival of a message

   type sigset_t is private;

   --  This type is really a union in C.
   type sigval_fill_1 is array (4 .. 8) of unsigned_char;
   type union_sigval is record
      sival_int : int;
      fill_1    : sigval_fill_1;
      --  sival_ptr : int;
   end record;
   pragma Convention (C, union_sigval);

   type sigevent_t is record
      sigev_signo  : int;
      sigev_value  : union_sigval;
      sigev_notify : int;
   end record;
   pragma Convention (C, sigevent_t);

   type struct_siginfo is record
      si_signo : int;
      si_code  : int;
      si_value : union_sigval;
   end record;
   pragma Convention (C, struct_siginfo);

   type sc_regs_array is array (1 .. 32) of long;
   type pad_array is array (1 .. 3) of unsigned_long;
   type struct_sigcontext is record
      sc_onstack : long;
      sc_mask : sigset_t;
      sc_pc : long;
      sc_ps : long;
      sc_regs : sc_regs_array;
      sc_ownedfp : long;
      sc_fpregs : sc_regs_array;
      sc_fpcr : unsigned_long;
      sc_fp_control : unsigned_long;
      sc_reserved1 : long;
      sc_reserved2 : long;
      sc_reserved3 : long;
      sc_reserved4 : long;
      sc_traparg_a0 : unsigned_long;
      sc_traparg_a1 : unsigned_long;
      sc_traparg_a2 : unsigned_long;
      sc_fp_trap_pc : unsigned_long;
      sc_fp_trigger_sum : unsigned_long;
      sc_fp_trigger_inst : unsigned_long;

      --  The pad field is only present in vxsim, so the following fields
      --  are dummy's to keep the record compatible with the board runtime.

      pad : pad_array;
      dummy1 : long;
      dummy2 : struct_siginfo;
      dummy3 : System.Address;
--      sc_restart : long;
--      sc_info : struct_siginfo;
--      sc_pregs : System.Address;
   end record;
   pragma Convention (C, struct_sigcontext);

   type struct_sigaction is record
      sa_handler   : System.Address;
      sa_mask      : sigset_t;
      sa_flags     : int;
   end record;
   pragma Convention (C, struct_sigaction);
   type struct_sigaction_ptr is access all struct_sigaction;

   function sigaddset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigaddset, "sigaddset");

   function sigdelset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigdelset, "sigdelset");

   function sigfillset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigfillset, "sigfillset");

   function sigismember
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigismember, "sigismember");

   function sigemptyset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigemptyset, "sigemptyset");

   function sigaction
     (sig  : Signal;
      act  : struct_sigaction_ptr;
      oact : struct_sigaction_ptr)
     return int;
   pragma Import (C, sigaction, "sigaction");

   type isr_address is access procedure (sig : Signal);

   function c_signal (sig : Signal; handler : isr_address) return isr_address;
   pragma Import (C, c_signal, "signal");

   ----------
   -- Time --
   ----------

   type timespec is private;

   type clockid_t is private;

   CLOCK_REALTIME : constant clockid_t;   --  System wide realtime clock

   function To_Duration (TS : timespec) return Duration;
   pragma Inline (To_Duration);

   function To_Timespec (D : Duration) return timespec;
   pragma Inline (To_Timespec);

   function To_Clock_Ticks (D : Duration) return int;
   --  Convert a duration value (in seconds) into clock ticks.

   function clock_gettime
     (clock_id : clockid_t;
      tp       : access timespec)
     return int;
   pragma Import (C, clock_gettime, "clock_gettime");

   -------------------------
   -- Priority Scheduling --
   -------------------------

   --  Use POSIX priority ordering (low to high).
   --
   --  We allow 254 of the 255 priorities in VxWorks saving
   --  the highest priority for the kernel.
   MIN_PRIO : constant := 0;
   MAX_PRIO : constant := 254;

   --  Scheduling policies.
   SCHED_FIFO  : constant := 1;
   SCHED_RR    : constant := 2;
   SCHED_OTHER : constant := 4;

   -------------
   -- Threads --
   -------------

   type Thread_Body is access
     function (arg : System.Address) return System.Address;

   type pthread_t           is private;
   subtype Thread_Id        is pthread_t;

   type pthread_mutex_t     is limited private;
   type pthread_cond_t      is limited private;
   type pthread_attr_t      is limited private;
   type pthread_mutexattr_t is limited private;
   type pthread_condattr_t  is limited private;
   type pthread_key_t       is private;

   PTHREAD_CREATE_DETACHED : constant := 0;
   PTHREAD_CREATE_JOINABLE : constant := 1;

   function kill
     (pid : pthread_t;
      sig : Signal)
     return int;
   pragma Import (C, kill, "kill");

   --  VxWorks doesn't have getpid; taskIdSelf is the equivalent
   --  routine.
   function getpid return pthread_t;
   pragma Import (C, getpid, "taskIdSelf");

   ---------------------------------
   -- Nonstandard Thread Routines --
   ---------------------------------

   --  FSU_THREADS requires pthread_init, which is nonstandard and
   --  this should be invoked during the elaboration of s-taprop.adb

   --  Vxworks requires this for the moment.

   procedure pthread_init;
   pragma Inline (pthread_init);

   --
   --  System.Interrupt_Management needs the following two system
   --  calls to resume a suspended task after a bus error.
   function taskIdSelf return pthread_t;
   pragma Import (C, taskIdSelf, "taskIdSelf");

   function taskSuspend (tid : pthread_t) return int;
   pragma Import (C, taskSuspend, "taskSuspend");

   function taskResume (tid : pthread_t) return int;
   pragma Import (C, taskResume, "taskResume");

   function taskIsSuspended (tid : pthread_t) return int;
   pragma Import (C, taskIsSuspended, "taskIsSuspended");

   function taskVarAdd
     (tid  : pthread_t;
      pVar : access System.Address)
     return  int;
   pragma Import (C, taskVarAdd, "taskVarAdd");

   function taskVarDelete
     (tid  : pthread_t;
      pVar : access System.Address)
     return  int;
   pragma Import (C, taskVarDelete, "taskVarDelete");

   function taskVarSet
     (tid   : pthread_t;
      pVar  : access System.Address;
      value : System.Address)
     return int;
   pragma Import (C, taskVarSet, "taskVarSet");

   function taskVarGet
     (tid   : pthread_t;
      pVar  : access System.Address)
     return int;
   pragma Import (C, taskVarGet, "taskVarGet");

   type TASK_DESC is record
      td_id           : int;      --  task id
      td_name         : Address;  --  name of task
      td_priority     : int;      --  task priority
      td_status       : int;      --  task status
      td_options      : int;      --  task option bits (see below)
      td_entry        : Address;  --  original entry point of task
      td_sp           : Address;  --  saved stack pointer
      td_pStackBase   : Address;  --  the bottom of the stack
      td_pStackLimit  : Address;  --  the effective end of the stack
      td_pStackEnd    : Address;  --  the actual end of the stack
      td_stackSize    : int;      --  size of stack in bytes
      td_stackCurrent : int;      --  current stack usage in bytes
      td_stackHigh    : int;      --  maximum stack usage in bytes
      td_stackMargin  : int;      --  current stack margin in bytes
      td_errorStatus  : int;      --  most recent task error status
      td_delay        : int;      --  delay/timeout ticks
   end record;
   pragma Convention (C, TASK_DESC);

   function taskInfoGet (tid : pthread_t; pTaskDesc : access TASK_DESC)
     return int;
   pragma Import (C, taskInfoGet, "taskInfoGet");

   function taskDelay (ticks : int) return int;
   pragma Import (C, taskDelay, "taskDelay");

   function sysClkRateGet return int;
   pragma Import (C, sysClkRateGet, "sysClkRateGet");

   -------------------------
   -- POSIX.1c  Section 3 --
   -------------------------

   function sigwaitinfo
     (set      : access sigset_t;
      sigvalue : access struct_siginfo)
     return int;
   pragma Import (C, sigwaitinfo, "sigwaitinfo");

   function sigwait
     (set : access sigset_t;
      sig : access Signal)
     return int;
   pragma Inline (sigwait);

   function pthread_kill
     (thread : pthread_t;
      sig    : Signal)
     return   int;
   pragma Import (C, pthread_kill, "kill");

   type sigset_t_ptr is access all sigset_t;

   function pthread_sigmask
     (how  : int;
      set  : sigset_t_ptr;
      oset : sigset_t_ptr)
     return int;
   pragma Import (C, pthread_sigmask, "sigprocmask");

   --------------------------
   -- POSIX.1c  Section 11 --
   --------------------------

   function pthread_mutexattr_init
     (attr : access pthread_mutexattr_t)
     return int;
   pragma Inline (pthread_mutexattr_init);

   function pthread_mutexattr_destroy
     (attr : access pthread_mutexattr_t)
     return int;
   pragma Inline (pthread_mutexattr_destroy);

   function pthread_mutex_init
     (mutex : access pthread_mutex_t;
      attr  : access pthread_mutexattr_t)
     return int;
   pragma Inline (pthread_mutex_init);

   function pthread_mutex_destroy
     (mutex : access pthread_mutex_t)
     return  int;
   pragma Inline (pthread_mutex_destroy);

   function pthread_mutex_lock
     (mutex : access pthread_mutex_t)
     return int;
   pragma Inline (pthread_mutex_lock);

   function pthread_mutex_unlock
     (mutex : access pthread_mutex_t)
     return int;
   pragma Inline (pthread_mutex_unlock);

   function pthread_condattr_init
     (attr : access pthread_condattr_t)
     return int;
   pragma Inline (pthread_condattr_init);

   function pthread_condattr_destroy
     (attr : access pthread_condattr_t)
     return int;
   pragma Inline (pthread_condattr_destroy);

   function pthread_cond_init
     (cond : access pthread_cond_t;
      attr : access pthread_condattr_t)
     return int;
   pragma Inline (pthread_cond_init);

   function pthread_cond_destroy
     (cond : access pthread_cond_t)
     return int;
   pragma Inline (pthread_cond_destroy);

   function pthread_cond_signal
     (cond : access pthread_cond_t)
     return int;
   pragma Inline (pthread_cond_signal);

   function pthread_cond_wait
     (cond  : access pthread_cond_t;
      mutex : access pthread_mutex_t)
     return  int;
   pragma Inline (pthread_cond_wait);

   function pthread_cond_timedwait
     (cond    : access pthread_cond_t;
      mutex   : access pthread_mutex_t;
      abstime : access timespec)
     return    int;
   pragma Inline (pthread_cond_timedwait);

   --------------------------
   -- POSIX.1c  Section 13 --
   --------------------------

   PTHREAD_PRIO_NONE    : constant := 0;
   PTHREAD_PRIO_PROTECT : constant := 2;
   PTHREAD_PRIO_INHERIT : constant := 1;

   function pthread_mutexattr_setprotocol
     (attr     : access pthread_mutexattr_t;
      protocol : int)
     return int;
   pragma Inline (pthread_mutexattr_setprotocol);

   function pthread_mutexattr_getprotocol
     (attr     : access pthread_mutexattr_t;
      protocol : access int)
     return int;
   pragma Inline (pthread_mutexattr_getprotocol);

   function pthread_mutexattr_setprioceiling
     (attr        : access pthread_mutexattr_t;
      prioceiling : int)
     return int;
   pragma Inline (pthread_mutexattr_setprioceiling);

   function pthread_mutexattr_getprioceiling
     (attr        : access pthread_mutexattr_t;
      prioceiling : access int)
     return int;
   pragma Inline (pthread_mutexattr_getprioceiling);

   type struct_sched_param is record
      sched_priority : int;
   end record;

   function pthread_getschedparam
     (thread : pthread_t;
      policy : access int;
      param  : access struct_sched_param)
     return int;
   pragma Inline (pthread_getschedparam);

   function pthread_setschedparam
     (thread : pthread_t;
      policy : int;
      param  : access struct_sched_param)
     return int;
   pragma Inline (pthread_setschedparam);

   function sched_yield return int;
   pragma Inline (sched_yield);

   --  It is not clear where the following belong or what they should
   --  be called to be POSIX compliant???

   function pthread_sched_rr_get_interval return int;
   pragma Inline (pthread_sched_rr_get_interval);

   function pthread_sched_rr_set_interval (usecs : int) return int;
   pragma Inline (pthread_sched_rr_set_interval);

   ---------------------------
   -- P1003.1c - Section 16 --
   ---------------------------

   function pthread_attr_init
     (attr : access pthread_attr_t)
     return int;
   pragma Inline (pthread_attr_init);

   function pthread_attr_destroy
     (attr : access pthread_attr_t)
     return int;
   pragma Inline (pthread_attr_destroy);

   function pthread_attr_getdetachstate
     (attr        : access pthread_attr_t;
      detachstate : access int)
     return int;
   pragma Inline (pthread_attr_getdetachstate);

   function pthread_attr_setdetachstate
     (attr        : access pthread_attr_t;
      detachstate : int)
     return int;
   pragma Inline (pthread_attr_setdetachstate);

   function pthread_attr_getstacksize
     (attr      : access pthread_attr_t;
      stacksize : access size_t)
     return int;
   pragma Inline (pthread_attr_getstacksize);

   function pthread_attr_setstacksize
     (attr      : access pthread_attr_t;
      stacksize : size_t)
     return int;
   pragma Inline (pthread_attr_setstacksize);

   function pthread_attr_setschedparam
     (attr        : access pthread_attr_t;
      sched_param : access struct_sched_param)
     return int;
   pragma Inline (pthread_attr_setschedparam);

   function pthread_attr_getschedparam
     (attr        : access pthread_attr_t;
      sched_param : access struct_sched_param)
     return int;
   pragma Inline (pthread_attr_getschedparam);

   --  In VxWorks tasks, we have a non-portable routine to set the
   --  task name.  This makes it really convenient for debugging.
   function pthread_attr_setname_np
     (attr : access pthread_attr_t;
      name : System.Address)
     return int;
   pragma Inline (pthread_attr_setname_np);

   function pthread_create
     (thread        : access pthread_t;
      attr          : access pthread_attr_t;
      start_routine : Thread_Body;
      arg           : System.Address)
     return int;
   pragma Inline (pthread_create);

   function pthread_detach (thread : pthread_t) return int;
   pragma Inline (pthread_detach);

   procedure pthread_exit (status : System.Address);
   pragma Inline (pthread_exit);

   function pthread_self return pthread_t;
   pragma Inline (pthread_self);

   function pthread_equal (t1 : pthread_t; t2 : pthread_t)
      return int;
   pragma Inline (pthread_equal);
   --  be careful not to use "=" on thread_t!

   ----------------------------
   --  POSIX.1c  Section 17  --
   ----------------------------

   function pthread_setspecific
     (key   : pthread_key_t;
      value : System.Address)
     return  int;
   pragma Inline (pthread_setspecific);

   function pthread_getspecific
     (key : pthread_key_t)
     return System.Address;
   pragma Inline (pthread_getspecific);

   type destructor_pointer is access
      procedure (arg : System.Address);

   function pthread_key_create
     (key        : access pthread_key_t;
      destructor : destructor_pointer)
     return int;
   pragma Inline (pthread_key_create);

private
   --  This interface assumes that "unsigned" and "int" are 32-bit entities.

   type sigset_t is new long;

   type pid_t is new int;

   ERROR_PID : constant pid_t := -1;

   type clockid_t is new int;
   CLOCK_REALTIME : constant clockid_t := 0;

   type time_t is new unsigned_long;

   type timespec is record
      ts_sec       : time_t;
      ts_nsec      : long;
   end record;
   pragma Convention (C, timespec);

   --  The VxWorks semaphore ID is an integer which is really just
   --  a pointer to a semaphore structure.

   type Sem_ID is new long;

   --  Priority ceilings are now implemented in the body of
   --  this package.

   type pthread_mutexattr_t is record
      Flags        : int;   --  mutex semaphore creation flags
      Prio_Ceiling : int;   --  priority ceiling
      Protocol     : int;
   end record;

   type pthread_mutex_t is record
      Mutex               : Sem_ID;
      Protocol            : int;
      Prio_Ceiling        : int;  --  priority ceiling of lock
   end record;

   type pthread_condattr_t is record
      Flags : int;
   end record;

   type pthread_cond_t is record
      Sem      : Sem_ID;   --  VxWorks semaphore ID
      Waiting  : integer;  --  Number of queued tasks waiting
   end record;

   type pthread_attr_t is record
      Stacksize    : size_t;
      Detachstate  : int;
      Priority     : int;
      Taskname     : System.Address;
   end record;

   type pthread_t is new long;

   type pthread_key_t is new int;

   --  These are to store the pthread_keys that are created with
   --  pthread_key_create.  Currently, we only need one key.

   Key_Storage  : array (1 .. 10) of aliased System.Address;
   Keys_Created : Integer;

   Time_Slice : int;

end System.OS_Interface;
