------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUN-TIME LIBRARY (GNARL) COMPONENTS              --
--                                                                          --
--                   S Y S T E M . O S _ I N T E R F A C E                  --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.11 $                            --
--                                                                          --
--          Copyright (C) 1997-1999 Free Software Foundation, Inc.          --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is an Irix (old pthread library) version of this package.

--  This package encapsulates all direct interfaces to OS services
--  that are needed by children of System.

--  PLEASE DO NOT add any with-clauses to this package
--  or remove the pragma Elaborate_Body.
--  It is designed to be a bottom-level (leaf) package.

with Interfaces;
with Interfaces.C;
with Interfaces.C.Strings;

package System.OS_Interface is

   pragma Preelaborate;

   pragma Linker_Options ("-lAtasking");
   pragma Linker_Options ("-lathread");

   subtype int            is Interfaces.C.int;
   subtype short          is Interfaces.C.short;
   subtype long           is Interfaces.C.long;
   subtype unsigned       is Interfaces.C.unsigned;
   subtype unsigned_short is Interfaces.C.unsigned_short;
   subtype unsigned_long  is Interfaces.C.unsigned_long;
   subtype unsigned_char  is Interfaces.C.unsigned_char;
   subtype plain_char     is Interfaces.C.plain_char;
   subtype size_t         is Interfaces.C.size_t;
   subtype chars_ptr      is Interfaces.C.Strings.chars_ptr;

   -----------
   -- Errno --
   -----------

   function errno return int;
   pragma Import (C, errno, "__get_errno");

   EPERM        : constant := 1;  --   Not super-user
   ENOENT       : constant := 2;  --   No such file or directory
   ESRCH        : constant := 3;  --   No such process
   EINTR        : constant := 4;  --   interrupted system call
   EIO          : constant := 5;  --   I/O error
   ENXIO        : constant := 6;  --   No such device or address
   E2BIG        : constant := 7;  --   Arg list too long
   ENOEXEC      : constant := 8;  --   Exec format error
   EBADF        : constant := 9;  --   Bad file number
   ECHILD       : constant := 10; --   No children
   EAGAIN       : constant := 11; --   No more processes
   ENOMEM       : constant := 12; --   Not enough core
   EACCES       : constant := 13; --   Permission denied
   EFAULT       : constant := 14; --   Bad address
   ENOTBLK      : constant := 15; --   Block device required
   EBUSY        : constant := 16; --   Mount device busy
   EEXIST       : constant := 17; --   File exists
   EXDEV        : constant := 18; --   Cross-device link
   ENODEV       : constant := 19; --   No such device
   ENOTDIR      : constant := 20; --   Not a directory
   EISDIR       : constant := 21; --   Is a directory
   EINVAL       : constant := 22; --   Invalid argument
   ENFILE       : constant := 23; --   File table overflow
   EMFILE       : constant := 24; --   Too many open files
   ENOTTY       : constant := 25; --   Not a typewriter
   ETXTBSY      : constant := 26; --   Text file busy
   EFBIG        : constant := 27; --   File too large
   ENOSPC       : constant := 28; --   No space left on device
   ESPIPE       : constant := 29; --   Illegal seek
   EROFS        : constant := 30; --   Read only file system
   EMLINK       : constant := 31; --   Too many links
   EPIPE        : constant := 32; --   Broken pipe
   EDOM         : constant := 33; --   Math arg out of domain of func
   ERANGE       : constant := 34; --   Math result not representable
   ENOMSG       : constant := 35; --   No message of desired type
   EIDRM        : constant := 36; --   Identifier removed
   ECHRNG       : constant := 37; --   Channel number out of range
   EL2NSYNC     : constant := 38; --   Level 2 not synchronized
   EL3HLT       : constant := 39; --   Level 3 halted
   EL3RST       : constant := 40; --   Level 3 reset
   ELNRNG       : constant := 41; --   Link number out of range
   EUNATCH      : constant := 42; --   Protocol driver not attached
   ENOCSI       : constant := 43; --   No CSI structure available
   EL2HLT       : constant := 44; --   Level 2 halted
   EDEADLK      : constant := 45; --   Deadlock condition.
   ENOLCK       : constant := 46; --   No record locks available.

   --  Convergent Error Returns

   EBADE        : constant := 50; --   invalid exchange
   EBADR        : constant := 51; --   invalid request descriptor
   EXFULL       : constant := 52; --   exchange full
   ENOANO       : constant := 53; --   no anode
   EBADRQC      : constant := 54; --   invalid request code
   EBADSLT      : constant := 55; --   invalid slot
   EDEADLOCK    : constant := 56; --   file locking deadlock error

   EBFONT       : constant := 57; --   bad font file fmt

   --  stream problems

   ENOSTR       : constant := 60; --   Device not a stream
   ENODATA      : constant := 61; --   no data (for no delay io)
   ETIME        : constant := 62; --   timer expired
   ENOSR        : constant := 63; --   out of streams resources

   ENONET       : constant := 64; --   Machine is not on the network
   ENOPKG       : constant := 65; --   Package not installed
   EREMOTE      : constant := 66; --   The object is remote
   ENOLINK      : constant := 67; --   the link has been severed
   EADV         : constant := 68; --   advertise error
   ESRMNT       : constant := 69; --   srmount error

   ECOMM        : constant := 70; --   Communication error on send
   EPROTO       : constant := 71; --   Protocol error
   EMULTIHOP    : constant := 74; --   multihop attempted
   EBADMSG      : constant := 77; --   trying to read unreadable message
   ENAMETOOLONG : constant := 78; --   path name is too long
   EOVERFLOW    : constant := 79; --   value too large to be stored in data
   --                                  type
   ENOTUNIQ     : constant := 80; --   given log. name not unique
   EBADFD       : constant := 81; --   f.d. invalid for this operation
   EREMCHG      : constant := 82; --   Remote address changed

   --  shared library problems

   ELIBACC         : constant := 83; --   Can't access a needed shared lib.
   ELIBBAD         : constant := 84; --   Accessing a corrupted shared lib.
   ELIBSCN         : constant := 85; --   .lib section in a.out corrupted.
   ELIBMAX         : constant := 86; --   Attempting to link in too many libs.
   ELIBEXEC        : constant := 87; --   Attempting to exec a shared library.
   EILSEQ          : constant := 88; --   Illegal byte sequence.
   ENOSYS          : constant := 89; --   Unsupported file system operation
   ELOOP           : constant := 90; --   Symbolic link loop
   ERESTART        : constant := 91; --   Restartable system call
   ESTRPIPE        : constant := 92; --   if pipe/FIFO, don't sleep in stream
   --                                     head

   ENOTEMPTY       : constant := 93; --   directory not empty

   EUSERS          : constant := 94; --   Too many users (for UFS)

   --  BSD Networking Software
   --  argument errors

   ENOTSOCK        : constant := 95; --   Socket operation on non-socket
   EDESTADDRREQ    : constant := 96; --   Destination address required
   EMSGSIZE        : constant := 97; --   Message too long
   EPROTOTYPE      : constant := 98; --   Protocol wrong type for socket
   ENOPROTOOPT     : constant := 99; --   Protocol not available
   EPROTONOSUPPORT : constant := 120; --  Protocol not supported
   ESOCKTNOSUPPORT : constant := 121; --  Socket type not supported
   EOPNOTSUPP      : constant := 122; --  Operation not supported on socket
   EPFNOSUPPORT    : constant := 123; --  Protocol family not supported
   EAFNOSUPPORT    : constant := 124; --  Address family not supported by
   --                                     protocol family

   EADDRINUSE      : constant := 125; --   Address already in use
   EADDRNOTAVAIL   : constant := 126; --   Can't assign requested address

   --  operational errors

   ENETDOWN        : constant := 127; --   Network is down
   ENETUNREACH     : constant := 128; --   Network is unreachable
   ENETRESET       : constant := 129; --   Network dropped connection because
   --                                      of reset

   ECONNABORTED    : constant := 130; --   Software caused connection abort
   ECONNRESET      : constant := 131; --   Connection reset by peer
   ENOBUFS         : constant := 132; --   No buffer space available
   EISCONN         : constant := 133; --   Socket is already connected
   ENOTCONN        : constant := 134; --   Socket is not connected

   --  XENIX has 135 - 142

   ESHUTDOWN       : constant := 143; --   Can't send after socket shutdown
   ETOOMANYREFS    : constant := 144; --   Too many references: can't splice
   ETIMEDOUT       : constant := 145; --   Connection timed out
   ECONNREFUSED    : constant := 146; --   Connection refused
   EHOSTDOWN       : constant := 147; --   Host is down
   EHOSTUNREACH    : constant := 148; --   No route to host

   --  Operation would block

   EWOULDBLOCK     : constant := 11;

   EALREADY        : constant := 149; --   operation already in progress
   EINPROGRESS     : constant := 150; --   operation now in progress

   --  SUN Network File System

   ESTALE          : constant := 151; --   Stale NFS file handle

   --  Pyramid's AIO Compatibility - raw disk async I/O

   EIORESID   : constant := 500; --   block not fully transferred

   --  XENIX error numbers

   EUCLEAN    : constant := 135; --   Structure needs cleaning
   ENOTNAM    : constant := 137; --   Not a XENIX named type file
   ENAVAIL    : constant := 138; --   No XENIX semaphores available
   EISNAM     : constant := 139; --   Is a named type file
   EREMOTEIO  : constant := 140; --   Remote I/O error
   EINIT      : constant := 141; --   Reserved for future
   EREMDEV    : constant := 142; --   Error 142

   -------------
   -- Signals --
   -------------

   NSIG : constant := 65;
   type Signal is new int range 0 .. Interfaces.C."-" (NSIG, 1);
   for Signal'Size use int'Size;

   SIGHUP     : constant := 1; --  hangup
   SIGINT     : constant := 2; --  interrupt (rubout)
   SIGQUIT    : constant := 3; --  quit (ASCD FS)
   SIGILL     : constant := 4; --  illegal instruction (not reset)
   SIGTRAP    : constant := 5; --  trace trap (not reset)
   SIGIOT     : constant := 6; --  IOT instruction
   SIGABRT    : constant := 6; --  used by abort, replace SIGIOT in the
   --                              future
   SIGEMT     : constant := 7; --  EMT instruction
   SIGFPE     : constant := 8; --  floating point exception
   SIGKILL    : constant := 9; --  kill (cannot be caught or ignored)
   SIGBUS     : constant := 10; --  bus error
   SIGSEGV    : constant := 11; --  segmentation violation
   SIGSYS     : constant := 12; --  bad argument to system call
   SIGPIPE    : constant := 13; --  write on a pipe with no one to read it
   SIGALRM    : constant := 14; --  alarm clock
   SIGTERM    : constant := 15; --  software termination signal from kill
   SIGUSR1    : constant := 16; --  user defined signal 1
   SIGUSR2    : constant := 17; --  user defined signal 2
   SIGCLD     : constant := 18; --  alias for SIGCHLD
   SIGCHLD    : constant := 18; --  child status change
   SIGPWR     : constant := 19; --  power-fail restart
   SIGWINCH   : constant := 20; --  window size change
   SIGURG     : constant := 21; --  urgent condition on IO channel
   SIGPOLL    : constant := 22; --  pollable event occurred
   SIGIO      : constant := 22; --  I/O possible (Solaris SIGPOLL alias)
   SIGSTOP    : constant := 23; --  stop (cannot be caught or ignored)
   SIGTSTP    : constant := 24; --  user stop requested from tty
   SIGCONT    : constant := 25; --  stopped process has been continued
   SIGTTIN    : constant := 26; --  background tty read attempted
   SIGTTOU    : constant := 27; --  background tty write attempted
   SIGVTALRM  : constant := 28; --  virtual timer expired
   SIGPROF    : constant := 29; --  profiling timer expired
   SIGXCPU    : constant := 30; --  CPU time limit exceeded
   SIGXFSZ    : constant := 31; --  filesize limit exceeded
   SIGK32     : constant := 32; --  reserved for kernel (IRIX)
   SIGCKPT    : constant := 33; --  Checkpoint warning
   SIGRESTART : constant := 34; --  Restart warning
   SIGUME     : constant := 35; --  Uncorrectable memory error
   --  Signals defined for Posix 1003.1c.
   SIGPTINTR    : constant := 47;
   SIGPTRESCHED : constant := 48;
   --  Posix 1003.1b signals
   SIGRTMIN   : constant := 49; --  Posix 1003.1b signals
   SIGRTMAX   : constant := 64; --  Posix 1003.1b signals

   type sigset_t is private;
   type sigset_t_ptr is access all sigset_t;

   Null_sigset_t_ptr : constant sigset_t_ptr := null;

   function sigaddset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigaddset, "sigaddset");

   function sigdelset
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigdelset, "sigdelset");

   function sigfillset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigfillset, "sigfillset");

   function sigismember
     (set : access sigset_t;
      sig : Signal)
     return int;
   pragma Import (C, sigismember, "sigismember");

   function sigemptyset
     (set : access sigset_t)
     return int;
   pragma Import (C, sigemptyset, "sigemptyset");

   type siginfo_t is record
      si_signo     : int;
      si_code      : int;
      si_errno     : int;
      bit_field_substitute_1 : string (1 .. 116);
   end record;
   pragma Convention (C, siginfo_t);

   --  The types mcontext_t and gregset_t are part of the ucontext_t
   --  information, which is specific to Solaris2.4 for SPARC
   --  The ucontext_t info seems to be used by the handler
   --  for SIGSEGV to decide whether it is a Storage_Error (stack overflow) or
   --  a Constraint_Error (bad pointer).  The original code that did this
   --  is suspect, so it is not clear whether we really need this part of
   --  the signal context information, or perhaps something else.
   --  More analysis is needed, after which these declarations may need to
   --  be changed.

   type reg_array is array (integer range 0 .. 31) of Interfaces.Unsigned_64;

   type struct_sigcontext is record
      sc_regmask   : unsigned;
      sc_status    : unsigned;
      sc_pc        : Interfaces.Unsigned_64;
      sc_regs      : reg_array;
      sc_fpregs    : reg_array;
      sc_ownedfp   : unsigned;
      sc_fpc_csr   : unsigned;
      sc_fpc_eir   : unsigned;
      sc_ssflags   : unsigned;
      sc_mdhi      : Interfaces.Unsigned_64;
      sc_mdlo      : Interfaces.Unsigned_64;
      sc_cause     : Interfaces.Unsigned_64;
      sc_badvaddr  : Interfaces.Unsigned_64;
      sc_triggersave  : Interfaces.Unsigned_64;
      sc_sigset    : sigset_t;
      sc_pad       : reg_array;
   end record;
   pragma Convention (C, struct_sigcontext);

   type array_type_2 is array (integer range 0 .. 1) of int;
   type struct_sigaction is record
      sa_flags     : int;
      sa_handler   : System.Address;
      sa_mask      : sigset_t;
      sa_resv      : array_type_2;
   end record;
   pragma Convention (C, struct_sigaction);
   type struct_sigaction_ptr is access all struct_sigaction;

   SIG_BLOCK   : constant := 1;
   SIG_UNBLOCK : constant := 2;
   SIG_SETMASK : constant := 3;

   --  SA_NOCLDSTOP : constant := 131072;
   --  not used
   --  SA_SIGINFO : constant := 8;
   --  not used

   SIG_DFL : constant := 0;
   SIG_IGN : constant := 1;
   --  SIG_ERR : constant := -1;
   --  not used

   function sigaction
     (sig  : Signal;
      act  : struct_sigaction_ptr;
      oact : struct_sigaction_ptr := null)
     return int;
   pragma Import (C, sigaction, "sigaction");

   ----------
   -- Time --
   ----------

   type time_t is new int;

   type timespec is record
      tv_sec  : time_t;
      tv_nsec : long;
   end record;
   pragma Convention (C, timespec);
   type timespec_ptr is access all timespec;

   function To_Duration (TS : timespec) return Duration;
   pragma Inline (To_Duration);

   function To_Timespec (D : Duration) return timespec;
   pragma Inline (To_Timespec);

   function nanosleep
     (rqtp : timespec_ptr;
      rmtp : timespec_ptr := null)
     return int;
   pragma Import (C, nanosleep, "nanosleep");

   type timer_t is new Integer;
   type clockid_t is private;

   CLOCK_REALTIME  : constant clockid_t;
   CLOCK_SGI_FAST  : constant clockid_t;
   CLOCK_SGI_CYCLE : constant clockid_t;

   SGI_CYCLECNTR_SIZE : constant := 165;
   function syssgi (request : Interfaces.C.int) return Interfaces.C.ptrdiff_t;

   pragma Import (C, syssgi, "syssgi");

   function clock_gettime
     (clock_id : clockid_t;
      tp       : access timespec)
     return int;
   pragma Import (C, clock_gettime, "clock_gettime");

   function clock_getres
     (clock_id : clockid_t;
      tp       : access timespec)
     return int;
   pragma Import (C, clock_getres, "clock_getres");

   type struct_timezone is record
      tz_minuteswest  : int;
      tz_dsttime   : int;
   end record;
   pragma Convention (C, struct_timezone);
   type struct_timezone_ptr is access all struct_timezone;

   type struct_timeval is record
      tv_sec       : long;
      tv_usec      : long;
   end record;
   pragma Convention (C, struct_timeval);

   function To_Duration (TV : struct_timeval) return Duration;
   pragma Inline (To_Duration);

   function To_Timeval (D : Duration) return struct_timeval;
   pragma Inline (To_Timeval);

   function gettimeofday
     (tv : access struct_timeval;
      tz : struct_timezone_ptr := null) return int;
   pragma Import (C, gettimeofday, "gettimeofday");

   --  add a hook to locate the Epoch, for use with Calendar????

   --------------
   -- itimers --
   --------------

   type itimerspec is record
      it_interval : timespec; --   timer period
      it_value    : timespec; --   timer expiration
   end record;
   type itimerspec_ptr is access all itimerspec;

   type long_Vec_13 is array (0 .. 12) of Long_Integer;
   type long_Vec_6  is array (0 .. 5) of Long_Integer;

   type notifyinfo is record
      nisigno : Integer;
   end record;

   subtype notifyinfo_t is notifyinfo;

   type sigval is record
      sival_int : Integer;
   end record;

   subtype sigval_t is sigval;

   type sigevent is record
      sigev_notify     : Integer;
      sigev_notifyinfo : notifyinfo_t;
      sigev_value      : sigval_t;
      sigev_reserved   : long_Vec_13;
      sigev_pad        : long_Vec_6;
   end record;

   subtype sigevent_t is sigevent;

   SIGEV_NONE     : constant := 128;
   SIGEV_SIGNAL   : constant := 129;
   SIGEV_CALLBACK : constant := 130;

   function timer_settime
     (timerid : timer_t;
      flags   : Integer;
      value   : access itimerspec;
      ovalue  : itimerspec_ptr := null) return Integer;
   pragma Import (C, timer_settime, "timer_settime");

   -------------------------
   -- Priority Scheduling --
   -------------------------

   MIN_PRIO : constant := 0;
   MAX_PRIO : constant := 0;

   SCHED_FIFO  : constant := 0;
   SCHED_RR    : constant := 0;
   SCHED_OTHER : constant := 0;

   -------------
   -- Process --
   -------------

   type pid_t is private;

   function kill (pid : pid_t; sig : Signal) return int;
   pragma Import (C, kill, "kill");

   function getpid return pid_t;
   pragma Import (C, getpid, "getpid");

   -------------
   -- Threads --
   -------------

   type Thread_Body is access
     function (arg : System.Address) return System.Address;

   type pthread_t           is private; --   thread identifier
   subtype Thread_Id        is pthread_t;

   type pthread_mutex_t     is private; --   mutex identifier
   type pthread_cond_t      is private; --   cond identifier
   type pthread_attr_t      is private; --   pthread attributes
   type pthread_mutexattr_t is private; --   mutex attributes
   type pthread_condattr_t  is private; --   mutex attributes
   type sem_t               is private; --   semaphore identifier
   type pthread_key_t       is private; --   per thread key

   subtype pthread_once_t   is int;     --   dynamic package initialization
   subtype resource_t       is long;    --   sproc. resource info.
   type start_addr is access function (arg : Address) return Address;
   type sproc_start_addr is access function (arg : Address) return int;
   type callout_addr is
     access function (arg : Address; arg1 : Address) return Address;

   --  SGI specific types

   subtype sproc_t      is Address; --   sproc identifier
   subtype sproc_attr_t is Address; --   sproc attributes

   subtype spcb_p is Address;
   subtype ptcb_p is Address;

   --  Pthread Error Types

   FUNC_OK  : constant := 0;
   FUNC_ERR : constant := -1;

   --  pthread run-time initialization data structure

   type pthread_init_struct is record
      conf_initsize       : int; --  shared area size
      max_sproc_count     : int; --  maximum number of sprocs
      sproc_stack_size    : size_t;  --  sproc stack size
      os_default_priority : int; --  default IRIX pri for main process
      os_sched_signal     : int; --  default OS scheduling signal
      guard_pages         : int; --  number of guard pages per stack
      init_sproc_count    : int; --  initial number of sprocs
   end record;

   --
   --  Pthread Attribute Initialize / Destroy
   --

   function pthread_attr_init (attr : access pthread_attr_t) return int;
   pragma Import (C, pthread_attr_init, "pthread_attr_init");

   function pthread_attr_destroy (attr : access pthread_attr_t) return int;
   pragma Import (C, pthread_attr_destroy, "pthread_attr_destroy");

   --
   --  Thread Attributes
   --

   function pthread_attr_setstacksize
     (attr : access pthread_attr_t; stacksize : size_t) return int;
   pragma Import (C, pthread_attr_setstacksize, "pthread_attr_setstacksize");

   function pthread_attr_getstacksize
     (attr : access pthread_attr_t; stacksize : access size_t) return int;
   pragma Import (C, pthread_attr_getstacksize, "pthread_attr_getstacksize");

   function pthread_attr_setdetachstate
     (attr : access pthread_attr_t; detachstate : int) return int;
   pragma Import
     (C, pthread_attr_setdetachstate, "pthread_attr_setdetachstate");

   function pthread_attr_getdetachstate
     (attr        : access pthread_attr_t;
      detachstate : access int) return int;
   pragma Import
     (C, pthread_attr_getdetachstate, "pthread_attr_getdetachstate");

   function pthread_attr_setname
     (attr : access pthread_attr_t; name : chars_ptr) return int;
   pragma Import (C, pthread_attr_setname, "pthread_attr_setname");

   function pthread_attr_getname
     (attr : access pthread_attr_t; name : chars_ptr) return int;
   pragma Import (C, pthread_attr_getname, "pthread_attr_getname");

   --
   --  Thread Scheduling Attributes
   --

   function pthread_attr_setscope
     (attr : access pthread_attr_t; contentionscope : int) return int;
   pragma Import (C, pthread_attr_setscope, "pthread_attr_setscope");

   function pthread_attr_getscope
     (attr            : access pthread_attr_t;
      contentionscope : access int) return int;
   pragma Import (C, pthread_attr_getscope, "pthread_attr_getscope");

   function pthread_attr_setinheritsched
     (attr : access pthread_attr_t; inherit : int) return int;
   pragma Import
     (C, pthread_attr_setinheritsched, "pthread_attr_setinheritsched");

   function  pthread_attr_getinheritsched
     (attr : access pthread_attr_t; inherit : access int) return int;
   pragma Import
     (C, pthread_attr_getinheritsched, "pthread_attr_getinheritsched");

   function pthread_attr_setsched
     (attr : access pthread_attr_t; scheduler : int) return int;
   pragma Import (C, pthread_attr_setsched, "pthread_attr_setsched");

   function pthread_attr_getsched
     (attr      : access pthread_attr_t;
      scheduler : access int) return int;
   pragma Import (C, pthread_attr_getsched, "pthread_attr_getsched");

   function  pthread_attr_setprio
     (attr : access pthread_attr_t; priority : int) return int;
   pragma Import (C, pthread_attr_setprio, "pthread_attr_setprio");

   function  pthread_attr_getprio
     (attr : access pthread_attr_t; priority : access int) return int;
   pragma Import (C, pthread_attr_getprio, "pthread_attr_getprio");

   --
   --  SGI Extensions to Thread Attributes
   --

   --  Bound to sproc attribute values

   PTHREAD_BOUND     : constant := 1;
   PTHREAD_NOT_BOUND : constant := 0;

   function pthread_attr_setresources
     (attr : access pthread_attr_t; resources : resource_t) return int;
   pragma Import (C, pthread_attr_setresources, "pthread_attr_setresources");

   function pthread_attr_getresources
     (attr      : access pthread_attr_t;
      resources : access resource_t) return int;
   pragma Import (C, pthread_attr_getresources, "pthread_attr_getresources");

   function pthread_attr_set_boundtosproc
     (attr : access pthread_attr_t; bound_to_sproc : int) return int;
   pragma Import
     (C, pthread_attr_set_boundtosproc, "pthread_attr_set_boundtosproc");

   function pthread_attr_get_boundtosproc
     (attr           : access pthread_attr_t;
      bound_to_sproc : access int) return int;
   pragma Import
     (C, pthread_attr_get_boundtosproc, "pthread_attr_get_boundtosproc");

   function pthread_attr_set_bsproc
     (attr : access pthread_attr_t; bsproc : spcb_p) return int;
   pragma Import (C, pthread_attr_set_bsproc, "pthread_attr_set_bsproc");

   function pthread_attr_get_bsproc
     (attr : access pthread_attr_t; bsproc : access spcb_p) return int;
   pragma Import (C, pthread_attr_get_bsproc, "pthread_attr_get_bsproc");

   function pthread_attr_set_tslice
     (attr        : access pthread_attr_t;
      ts_interval : access struct_timeval) return int;
   pragma Import (C, pthread_attr_set_tslice, "pthread_attr_set_tslice");

   function pthread_attr_get_tslice
     (attr        : access pthread_attr_t;
      ts_interval : access struct_timeval) return int;
   pragma Import (C, pthread_attr_get_tslice, "pthread_attr_get_tslice");

   --
   --  Thread Creation & Management
   --

   function pthread_create
     (thread        : access pthread_t;
      attr          : access pthread_attr_t;
      start_routine : start_addr;
      arg           : Address) return int;
   pragma Import (C, pthread_create, "pthread_create");

   function pthread_join
      (thread : pthread_t; status : access Address) return int;
   pragma Import (C, pthread_join, "pthread_join");

   function pthread_detach (thread : access pthread_t) return int;
   pragma Import (C, pthread_detach, "pthread_detach");

   procedure pthread_exit (status : Address);
   pragma Import (C, pthread_exit, "pthread_exit");

   procedure pthread_yield (arg : Address := System.Null_Address);
   pragma Import (C, pthread_yield, "pthread_yield");

   function pthread_self  return pthread_t;
   pragma Import (C, pthread_self, "pthread_self");


   function pthread_kill (thread : pthread_t; sig : int) return int;
   pragma Import (C, pthread_kill, "pthread_kill");

   --
   --  SGI Extensions to POSIX thread operations
   --

   function pthread_setprio (thread : pthread_t; priority : int) return int;
   pragma Import (C, pthread_setprio, "pthread_setprio");

   function pthread_getprio
     (thread : pthread_t; priority : access int) return int;
   pragma Import (C, pthread_getprio, "pthread_getprio");

   function pthread_suspend (thread : pthread_t) return int;
   pragma Import (C, pthread_suspend, "pthread_suspend");

   function pthread_resume (thread : pthread_t) return int;
   pragma Import (C, pthread_resume, "pthread_resume");

   function pthread_set_info (index : int; data : int) return int;
   pragma Import (C, pthread_set_info, "pthread_set_info");

   function pthread_get_info (index : int) return int;
   pragma Import (C, pthread_get_info, "pthread_get_info");

   function pthread_get_info_offset (index : int) return int;
   pragma Import (C, pthread_get_info_offset, "pthread_get_info_offset");

   function pthread_get_current_ada_tcb return Address;
   pragma Import
     (C, pthread_get_current_ada_tcb, "pthread_get_current_ada_tcb");

   function pthread_get_ada_tcb (thread : pthread_t) return Address;
   pragma Import (C, pthread_get_ada_tcb, "pthread_get_ada_tcb");

   function pthread_set_ada_tcb
     (thread : pthread_t; data : Address) return int;
   pragma Import (C, pthread_set_ada_tcb, "pthread_set_ada_tcb");

   function pthread_get_jumpbuf_address return Address;
   pragma Import
     (Ada, pthread_get_jumpbuf_address, "pthread_get_jumpbuf_address");

   procedure pthread_set_jumpbuf_address (addr : Address);
   pragma Import
     (Ada, pthread_set_jumpbuf_address, "pthread_set_jumpbuf_address");

   function pthread_get_exception return Address;
   pragma Import (Ada, pthread_get_exception, "pthread_get_exception");

   procedure pthread_set_exception (addr : Address);
   pragma Import (Ada, pthread_set_exception, "pthread_set_exception");

   function pthread_get_sec_stack_addr return Address;
   pragma Import
     (Ada, pthread_get_sec_stack_addr, "pthread_get_sec_stack_addr");

   procedure pthread_set_sec_stack_addr (addr : Address);
   pragma Import
     (Ada, pthread_set_sec_stack_addr, "pthread_set_sec_stack_addr");

   function pthread_get_exc_stack_addr return Address;
   pragma Import
     (Ada, pthread_get_exc_stack_addr, "pthread_get_exc_stack_addr");

   procedure pthread_set_exc_stack_addr (addr : Address);
   pragma Import
     (Ada, pthread_set_exc_stack_addr, "pthread_set_exc_stack_addr");

   --
   --  Thread Scheduling Attributes
   --

   function pthread_getschedattr
     (thread : pthread_t; attr : access pthread_attr_t) return int;
   pragma Import (C, pthread_getschedattr, "pthread_getschedattr");

   function pthread_setschedattr
     (thread : pthread_t; attr : access pthread_attr_t) return int;
   pragma Import (C, pthread_setschedattr, "pthread_setschedattr");

   --
   --  Scheduling lock/unlock/callout
   --

   function pthread_lock_scheduler return int;
   pragma Import (C, pthread_lock_scheduler, "pth_lock_scheduler");

   function pthread_unlock_scheduler return int;
   pragma Import (C, pthread_unlock_scheduler, "pth_unlock_scheduler");

   function pthread_setsched_callout
     (new_call : callout_addr; old_call : access callout_addr) return int;
   pragma Import (C, pthread_setsched_callout, "pth_setsched_callout");

   --  Delay thread for specified time

   function pthread_delay (dtime : access struct_timeval) return int;
   pragma Import (C, pthread_delay, "pth_delay");

   --  Mutex Initialization / Destruction

   function pthread_mutexattr_init
     (attr : access pthread_mutexattr_t) return int;
   pragma Import (C, pthread_mutexattr_init, "pthread_mutexattr_init");

   function  pthread_mutexattr_destroy
     (attr : access pthread_mutexattr_t) return int;
   pragma Import (C, pthread_mutexattr_destroy, "pthread_mutexattr_destroy");

   function pthread_mutexattr_getpshared
     (attr    : access pthread_mutexattr_t;
      pshared : access int) return int;
   pragma Import
     (C, pthread_mutexattr_getpshared, "pthread_mutexattr_getpshared");

   function pthread_mutexattr_setpshared
     (attr : access pthread_mutexattr_t; pshared : int) return int;
   pragma Import
     (C, pthread_mutexattr_setpshared, "pthread_mutexattr_setpshared");

   function pthread_mutexattr_setspin
     (attr : access pthread_mutexattr_t; spincnt : int) return int;
   pragma Import (C, pthread_mutexattr_setspin, "pthread_mutexattr_setspin");

   function pthread_mutexattr_getspin
     (attr    : access pthread_mutexattr_t;
      spincnt : access int) return int;
   pragma Import (C, pthread_mutexattr_getspin, "pthread_mutexattr_getspin");

   function pthread_mutexattr_setqueueorder
     (attr : access pthread_mutexattr_t; order : int) return int;
   pragma Import
     (C, pthread_mutexattr_setqueueorder, "pthread_mutexattr_setqueueorder");

   function pthread_mutexattr_getqueueorder
     (attr  : access pthread_mutexattr_t;
      order : access int) return int;
   pragma Import
     (C, pthread_mutexattr_getqueueorder, "pthread_mutexattr_getqueueorder");

   function pthread_mutexattr_setceilingprio
     (attr : access pthread_mutexattr_t; priority : int) return int;
   pragma Import
     (C, pthread_mutexattr_setceilingprio, "pthread_mutexattr_setceilingprio");

   function pthread_mutexattr_getceilingprio
     (attr     : access pthread_mutexattr_t;
      priority : access int) return int;
   pragma Import
     (C, pthread_mutexattr_getceilingprio, "pthread_mutexattr_getceilingprio");

   --  Mutex Attributes

   --  Threads queueing order

   MUTEX_PRIORITY         : constant := 0; --   wait in priority order
   MUTEX_FIFO             : constant := 1; --   first-in-first-out
   MUTEX_PRIORITY_INHERIT : constant := 2; --   priority inhertance mutex
   MUTEX_PRIORITY_CEILING : constant := 3; --   priority ceiling mutex

   --  Mutex debugging options

   MUTEX_NO_DEBUG  : constant := 0; --   no debugging on mutex
   MUTEX_DEBUG     : constant := 1; --   debugging is on

   --  Mutex spin on lock operations

   MUTEX_NO_SPIN   : constant := 0;  --   no spin, try once only
   MUTEX_SPIN_ONLY : constant := -1; --   spin forever
   --  cnt > 0, limited spin
   --  Mutex sharing attributes

   MUTEX_SHARED    : constant := 0; --   shared between processes
   MUTEX_NOTSHARED : constant := 1; --   not shared between processes

   --  Mutex Operations

   function pthread_mutex_init
     (mutex : access pthread_mutex_t;
      attr  : access pthread_mutexattr_t) return int;
   pragma Import (C, pthread_mutex_init, "pthread_mutex_init");

   function pthread_mutex_destroy
     (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_destroy, "pthread_mutex_destroy");

   function pthread_mutex_lock
     (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_lock, "pthread_mutex_lock");

   function pthread_mutex_trylock
     (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_trylock, "pthread_mutex_trylock");

   function pthread_mutex_unlock
     (mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_mutex_unlock, "pthread_mutex_unlock");

   --  Condition Initialization / Destruction

   function pthread_condattr_init
     (attr : access pthread_condattr_t) return int;
   pragma Import (C, pthread_condattr_init, "pthread_condattr_init");

   function pthread_condattr_destroy
     (attr : access pthread_condattr_t) return int;
   pragma Import (C, pthread_condattr_destroy, "pthread_condattr_destroy");

   function pthread_condattr_getpshared
     (attr    : access pthread_condattr_t;
      pshared : access int) return int;
   pragma Import
     (C, pthread_condattr_getpshared, "pthread_condattr_getpshared");

   function pthread_condattr_setpshared
     (attr : access pthread_condattr_t; pshared : int) return int;
   pragma Import
     (C, pthread_condattr_setpshared, "pthread_condattr_setpshared");

   --  Condition Attributes

   COND_PRIORITY  : constant := 0; --   wait in priority order
   COND_FIFO      : constant := 1; --   first-in-first-out

   --  Condition debugging options

   COND_NO_DEBUG  : constant := 0; --   no debugging on mutex
   COND_DEBUG     : constant := 1; --   debugging is on

   --  Condition sharing attributes

   COND_SHARED    : constant := 0; --   shared between processes
   COND_NOTSHARED : constant := 1; --   not shared between processes

   --  Condition Operations

   function pthread_cond_init
     (cond : access pthread_cond_t;
      attr : access pthread_condattr_t) return int;
   pragma Import (C, pthread_cond_init, "pthread_cond_init");

   function pthread_cond_destroy
     (cond : access pthread_cond_t) return int;
   pragma Import (C, pthread_cond_destroy, "pthread_cond_destroy");

   function pthread_cond_signal (cond : access pthread_cond_t) return int;
   pragma Import (C, pthread_cond_signal, "pthread_cond_signal");

   function pthread_cond_broadcast (cond : access pthread_cond_t) return int;
   pragma Import (C, pthread_cond_broadcast, "pthread_cond_broadcast");

   function pthread_cond_wait
     (cond  : access pthread_cond_t;
      mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_cond_wait, "pthread_cond_wait");

   function pthread_cond_timedwait
     (cond    : access pthread_cond_t;
      mutex   : access pthread_mutex_t;
      abstime : access struct_timeval) return int;
   pragma Import (C, pthread_cond_timedwait, "pthread_cond_timedwait");

   function pthread_cond_reltimedwait
     (cond    : access pthread_cond_t;
      mutex   : access pthread_mutex_t;
      reltime : access struct_timeval) return int;
   pragma Import (C, pthread_cond_reltimedwait, "pthread_cond_reltimedwait");

   function pthread_cond_signal_unlock
     (cond  : access pthread_cond_t;
      mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_cond_signal_unlock, "pthread_cond_signal_unlock");

   function pthread_cond_wait_unlock
     (cond  : access pthread_cond_t;
      mutex : access pthread_mutex_t) return int;
   pragma Import (C, pthread_cond_wait_unlock, "pthread_cond_wait_unlock");

   PTHREAD_ONCE_INIT : constant := 0;  --  set to one after init

   --  Thread-Specific Data

   type foo_h_proc_1 is access procedure (value : Address);

   function pthread_key_create
     (key : access pthread_key_t; destructor : foo_h_proc_1) return int;
   pragma Import (C, pthread_key_create, "pthread_key_create");

   function pthread_setspecific
     (key : pthread_key_t; value : Address) return int;
   pragma Import (C, pthread_setspecific, "pthread_setspecific");

   function pthread_getspecific
     (key : pthread_key_t; value : access Address) return int;
   pragma Import (C, pthread_getspecific, "pthread_getspecific");

   type foo_h_proc_2 is access procedure;

   function pthread_once
     (once_control : access pthread_once_t;
      init_routine : foo_h_proc_2) return int;
   pragma Import (C, pthread_once, "pthread_once");

   function pthread_exec_begin (init : access pthread_init_struct) return int;
   pragma Import (C, pthread_exec_begin, "pthread_exec_begin");

   function sproc_create
     (sproc_id      : access sproc_t;
      attr          : access sproc_attr_t;
      start_routine : sproc_start_addr;
      arg           : Address) return int;
   pragma Import (C, sproc_create, "sproc_create");

   function sproc_self return sproc_t;
   pragma Import (C, sproc_self, "sproc_self");

   --  if equal fast TRUE is returned - common case
   --  if not equal thread resource must NOT be null in order to compare bits

   --
   --  Sproc attribute initialize / destroy
   --

   function sproc_attr_init (attr : access sproc_attr_t) return int;
   pragma Import (C, sproc_attr_init, "sproc_attr_init");

   function sproc_attr_destroy (attr : access sproc_attr_t) return int;
   pragma Import (C, sproc_attr_destroy, "sproc_attr_destroy");

   function sproc_attr_setresources
     (attr : access sproc_attr_t; resources : resource_t) return int;
   pragma Import (C, sproc_attr_setresources, "sproc_attr_setresources");

   function sproc_attr_getresources
     (attr      : access sproc_attr_t;
      resources : access resource_t) return int;
   pragma Import (C, sproc_attr_getresources, "sproc_attr_getresources");

   function sproc_attr_setcpu
     (attr : access sproc_attr_t; cpu_num : int) return int;
   pragma Import (C, sproc_attr_setcpu, "sproc_attr_setcpu");

   function sproc_attr_getcpu
     (attr : access sproc_attr_t; cpu_num : access int) return int;
   pragma Import (C, sproc_attr_getcpu, "sproc_attr_getcpu");

   function sproc_attr_setresident
     (attr : access sproc_attr_t; resident : int) return int;
   pragma Import (C, sproc_attr_setresident, "sproc_attr_setresident");

   function sproc_attr_getresident
     (attr : access sproc_attr_t; resident : access int) return int;
   pragma Import (C, sproc_attr_getresident, "sproc_attr_getresident");

   function sproc_attr_setname
     (attr : access sproc_attr_t; name : chars_ptr) return int;
   pragma Import (C, sproc_attr_setname, "sproc_attr_setname");

   function sproc_attr_getname
     (attr : access sproc_attr_t; name : chars_ptr) return int;
   pragma Import (C, sproc_attr_getname, "sproc_attr_getname");

   function sproc_attr_setstacksize
     (attr : access sproc_attr_t; stacksize : size_t) return int;
   pragma Import (C, sproc_attr_setstacksize, "sproc_attr_setstacksize");

   function sproc_attr_getstacksize
     (attr : access sproc_attr_t; stacksize : access size_t) return int;
   pragma Import (C, sproc_attr_getstacksize, "sproc_attr_getstacksize");

   function sproc_attr_setprio
     (attr : access sproc_attr_t; priority : int) return int;
   pragma Import (C, sproc_attr_setprio, "sproc_attr_setprio");

   function sproc_attr_getprio
     (attr : access sproc_attr_t; priority : access int) return int;
   pragma Import (C, sproc_attr_getprio, "sproc_attr_getprio");

   function sproc_attr_setbthread
     (attr : access sproc_attr_t; bthread : ptcb_p) return int;
   pragma Import (C, sproc_attr_setbthread, "sproc_attr_setbthread");

   function sproc_attr_getbthread
     (attr : access sproc_attr_t; bthread : access ptcb_p) return int;
   pragma Import (C, sproc_attr_getbthread, "sproc_attr_getbthread");

   SPROC_NO_RESOURCES : constant := 0;
   SPROC_ANY_CPU      : constant := -1;
   SPROC_MY_PRIORITY  : constant := -1;
   SPROC_SWAPPED      : constant := 0;
   SPROC_RESIDENT     : constant := 1;

   --  Semaphore Operations

   function sem_init
     (sem     : access sem_t;
      oflag   : int;
      name    : chars_ptr;
      pshared : int;
      value   : int) return int;
   pragma Import (C, sem_init, "ath_sem_init");

   function sem_destroy (sem : access sem_t) return int;
   pragma Import (C, sem_destroy, "ath_sem_destroy");

   function sem_unlink (name : chars_ptr) return int;
   pragma Import (C, sem_unlink, "ath_sem_unlink");

   function sem_lock (sem : access sem_t) return int;
   pragma Import (C, sem_lock, "ath_sem_lock");

   function sem_trylock (sem : access sem_t) return int;
   pragma Import (C, sem_trylock, "ath_sem_trylock");

   function sem_unlock (sem : access sem_t) return int;
   pragma Import (C, sem_unlock, "ath_sem_unlock");

   function sem_check_in_use
     (sem : access sem_t; status : access int) return int;
   pragma Import (C, sem_check_in_use, "ath_sem_check_in_use");

   --  Semaphore Waiting Queue

   SEMA_FIFO       : constant := 0; --   threads blocked in fifo order
   SEMA_PRIORITY   : constant := 1; --   threads blocked in priority order

   --  Semaphore Shared Flag

   SEMA_NOT_SHARED : constant := 0; --   not shared between processes
   SEMA_SHARED     : constant := 1; --   semaphore shared

   type isr_address is access procedure;

   function intr_attach (sig : int; isr : isr_address) return int;
   pragma Import (C, intr_attach, "intr_attach");

   function intr_exchange
     (sig  : int;
      isr  : isr_address;
      oisr : access isr_address) return int;
   pragma Import (C, intr_exchange, "intr_exchange");

   function intr_current_isr
     (sig  : int;
      oisr : access isr_address)
     return int;
   pragma Import (C, intr_current_isr, "intr_current_isr");

   procedure pthread_init;

private

   type clockid_t is new int;

   CLOCK_REALTIME  : constant clockid_t := 1;
   CLOCK_SGI_CYCLE : constant clockid_t := 2;
   CLOCK_SGI_FAST  : constant clockid_t := 3;

   type pthread_t           is new Address; --   thread identifier
   type pthread_mutex_t     is new Address; --   mutex identifier
   type pthread_cond_t      is new Address; --   cond identifier
   type pthread_attr_t      is new Address; --   pthread attributes
   type pthread_mutexattr_t is new Address; --   mutex attributes
   type pthread_condattr_t  is new Address; --   mutex attributes
   type sem_t               is new Address; --   semaphore identifier
   type pthread_key_t       is new Address; --   per thread key

   type sigbits_t is array (integer range 0 .. 3) of unsigned;
   type sigset_t is record
      sigbits      : sigbits_t;
   end record;
   pragma Convention (C, sigset_t);

   type pid_t  is new long;

end System.OS_Interface;
