(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               indtypes.ml                                *)
(****************************************************************************)

open Pp;;
open Std;;
open Names;;
open Vectops;;
open Impuniv;;
open Extraction;;
open Generic;;
open Term;;
open Constrtypes;;
open Reduction;;
open Typing;;
open Termenv;;
open Himsg;;
open Mach;;
open Dischcore;;
open Initial;;
open Environ;;

let empty_evd = Evd.mt_evd();;

let cast_of_judgement = Machops.cast_of_judgement;;

let casted_execute_rec sigma = core_fmachine false (sigma,[]);;
let casted_execute_rec_type sigma = core_fmachine_type false (sigma,[]);;

(* WITH INFORMATION *)
let casted_infmachine nocheck (sigma,metamap) sign fsign constr =
  let constr_metamap = List.map (fun (id,(c,_)) -> (id,c)) metamap
  and inf_metamap = List.map (fun (id,(_,i)) -> (id,i)) metamap in
  let j = core_fmachine nocheck (sigma,constr_metamap) sign constr in
  let inf' = 
    if is_info_judge sigma j then 
      core_infmachine (sigma,(constr_metamap,inf_metamap)) sign fsign
        (j_val_cast j)
    else Logic
  in (j,inf')
;;

let casted_infexecute_rec sigma (sign,fsign) c =
    casted_infmachine false (sigma,[]) sign fsign c;;
let casted_infexecute sigma (sign,fsign) c =
    casted_infmachine false (sigma,[]) (gLOB sign) (gLOB fsign) c;;

let casted_execute sigma sign c =
    core_fmachine false (sigma,[]) (gLOB sign) c;;
(* ou :  fexecute sigma sign c  ?? *)

let extract_of sign fsign c = 
  let (_,infc) = infexecute empty_evd (sign,fsign) c in 
    match infc with
        Logic -> anomaly"extract_of"
      | Inf j  -> Machops.assumption_of_judgement empty_evd (gLOB sign) j;;

(* In this part we can assume the context of existential variables is empty *)

let whd_all = whd_betadeltaiota empty_evd;;

let rec sort_of_arity ar =
   match whd_all ar with
     DOP0(Sort s) as x        -> x
   | DOP2(Prod,a,DLAM(n,c_0)) -> sort_of_arity c_0
   | _                        -> error "Not a product"
;;

let sort_of_arity_and_check name ar =
  try sort_of_arity ar
  with _ ->
    errorlabstrm "Wrong arity" 
      [< 'sTR ("The type of "^(string_of_id name)^
               "is not a product over a sort")>]
;;

(* reduce a term to its hnf form keeping casts *)
let rec decomp_DLAMV_env env = function 
    ([typ],DLAMV(na,lc)) -> (add_rel (na,typ) env,lc)
  | (typ::rest,DLAM(na,lc)) -> 
	decomp_DLAMV_env (add_rel (na,typ) env) (rest,lc)         
  | _ -> error "decomp_DLAMV_env 1";;

(* Le duo decomp_par/hnf_recast est remplac par prods_of_type
   (n est le nb de params  sauter) *)

let rec prods_of_type n env c =
  match whd_all c with 
   DOP2(Prod,DOP2(Cast,t,DOP0 (Sort s)),DLAM(name,c)) -> 
     let var = make_type t s in
     if n=0
       then var::(prods_of_type 0 (add_rel (name,var) env) c)
       else prods_of_type (n-1) (add_rel (name,var) env) c
 | DOP2(Prod,b,DLAM(name,c)) -> 
     let var = execute_rec_type empty_evd env (strong strip_outer_cast b) in 
     if n=0
       then var::(prods_of_type 0 (add_rel (name,var) env) c)
       else prods_of_type (n-1) (add_rel (name,var) env) c
 | x -> [];;


(**** Check a mutually inductive definition is well-formed ***)


(* takes the arity (c) of an inductive definition with the number of parameters
   and compute the number of parameters of the extracted inductive definition
   plus compute the "lamexp" which is of the form 
    [x1]...[xn]((Rel n+1) xi1 .. xik)
   the idea is that if Constr(i):(y1:A1)...(yn:An)B
*)

let inf_minductive sign fsign nparams arity c checksingl =

let sort = sort_of_arity arity.body in

let infexec sign' fsign' t =
  let tj = casted_execute_rec_type empty_evd sign' t in
    (tj,
    if is_info_type empty_evd tj then 
      core_infmachine (empty_evd,([],[])) sign' fsign' (incast_type tj) 
    else Logic) in

let extract_na_ty_pop c_0 =
        (match whd_all c_0 with
         DOP2(Prod,t,DLAM(na,b)) -> (na,t),b
       | _ -> error "infexecute") in

let rec descend e fe n c =
 if n = 0 then 
     let singl = if checksingl & isprop sort then 
        let (_,infC) = infexec e fe c in
        (match whd_all (j_val_only (extract infC)) with 
           DOP2(Prod,a,DLAM(_,x)) -> 
               (match whd_all x with 
                 (Rel _ | DOPN(AppL,_)) ->
                   if noccurn (nparams+1) a then 
                   let tA = unsafe_type_of empty_evd fe a in
		     (* J'admets que le contexte du jugement de a est celui
			du type inductif, il ne contient que les variables
			locales du type inductif considr HH 6/99 *)
                     if conv_x empty_evd tA sort then
		       Some a (* {_VAL=a;_TYPE=tA;_KIND=type_of_type tA}*)
		     else None
                   else None
                | _ -> None)
          | _ -> None)
     else None in (0,[],[],singl)
 else let (na,t),newC = extract_na_ty_pop c in
      let (tj,infT) = infexec e fe t in
      let (inf_nparams,constrlamenv,argsign,singl) =
          descend (add_rel (na, tj) e)
            (add_inf_rel (na,infT) fe) (n-1) newC
      in (match infT with
            Logic -> (inf_nparams, Logic::constrlamenv,argsign,singl)
          | (Inf infTj) ->
            if isprop infTj._TYPE 
            then (inf_nparams,infT::constrlamenv,ERASE::argsign,singl)
            else (inf_nparams+1,infT::constrlamenv,ABSTRACT::argsign,singl)) in
    
let (inf_nparams,constrlamenv,argsign,singl) = 
        descend sign fsign nparams (strong strip_outer_cast c) in

let constrlambody = match singl with 
    None -> applist 
	(Rel (nparams+1),
         List.rev(it_list_i
		    (fun i l m -> match m with 
			 ERASE -> l
                       | ABSTRACT -> (Rel(nparams+1-i))::l)
		    1 [] argsign))
  | Some a -> lambda_create(a,Rel 1) in

let constrlamexp = 
  List.fold_right 
    (fun info b -> match info with
	 Logic ->  (pop b)
       | (Inf j) -> lambda_create(j._VAL,b))
    constrlamenv constrlambody in 

let newsingl = match singl with None -> None 
          | Some a -> 
 let rec put_inf_lam = function 
     (Logic::cle) -> (comp (pop) ((put_inf_lam cle)))
   | (Inf j::cle) -> 
       (function 
	    (ERASE::l) -> pop (put_inf_lam cle l)
          | (ABSTRACT::l) -> 
              lambda_create(j._VAL,put_inf_lam cle l)
	  | _ -> assert false)
   | [] -> (function [] -> a | _ -> assert false)
 in 
   Some(put_inf_lam constrlamenv argsign) 
     
in 
  (inf_nparams,Some constrlamexp,argsign,newsingl);;

let inf_mindterm (sign,fsign) = inf_minductive (gLOB sign) (gLOB fsign);;

(* remove the redondant products corresponding to the given
list of ERASE, ABSTRACT information, keep the outer-most cast unchanged *)

let strip_constr larg k =
  let (c,t) = destCast k in
  let rec striprec (l,c_0) = 
    match (l,whd_all c_0) with 
    	(ERASE::larg,DOP2(Prod,_,DLAM(_,c_0))) -> striprec (larg,pop c_0)
      | (ABSTRACT::larg,DOP2(Prod,t_0,DLAM(na,c_0))) -> 
	  DOP2(Prod,t_0,DLAM(na,striprec(larg,c_0)))
      | [], _                               -> c_0
      | _                                   -> anomaly "strip_constr"
  in DOP2(Cast,striprec (larg,c),t) 
;;


let mapplied_to_params nbtypes nparams indidx c = 
 
 let rec apprec indidx = function
    DOP2(Cast,c,_) -> apprec indidx c
  | DOPN(AppL,cl) ->
    (* Either the arguments of the application have to look like:
     * AppL(Rel indidx+p; Rel (indidx-1);... Rel(indidx-nparams);...)
     * with p<nbtypes
     * or the subterms must themselves meet this test.
     * If we find a bare "Rel" with the value "indidx+p", it's
     * an error.
     *)
       (Array.length cl >= (nparams+1)
       & (match cl.(0) with (Rel p) -> (indidx <= p) & (p < indidx+nbtypes)
                           | _       -> false)
       & for_all_vect_i (fun k v -> (Rel (indidx - k - 1)) = v) 0 
                        (Array.sub cl 1 nparams))
    or (for_all_vect (apprec indidx) cl)
 
  | Rel i -> i < indidx or i >= indidx+nbtypes
  | VAR _ -> true
  | DOP0 _ -> true
  | DOP1(_,c) -> apprec indidx c
  | DOP2(_,c1,c2) -> apprec indidx c1 & apprec indidx c2
  | DOPN(_,cl) -> for_all_vect (apprec indidx) cl
  | DOPL(_,cl) -> List.for_all (apprec indidx) cl
  | DLAM(na,c) -> apprec (indidx+1) c
  | DLAMV(na,cl) -> for_all_vect (apprec (indidx+1)) cl
 in (nparams = 0) or (apprec indidx c)
    
;;
 
let applied_to_params = mapplied_to_params 1;;

let rec decomp_par n c =
  if n <= 0 then c
  else match whd_all c with 
      DOP2(Prod,b,DLAM(na,d)) -> decomp_par (n-1) d
    | _                       -> error "decomp_par";;

(* Functions to determine allowed elimination schemes *)

let is_small_type t = is_small (level_of_type t)

let small_list_constr env nbpar ltypes indlc =
  let indenv,lC = decomp_DLAMV_env env (ltypes,indlc) in
  for_all_vect
  (fun c -> List.for_all is_small_type (prods_of_type nbpar indenv c))
    lC

  (* check that all products are logic *)

let rec logic_constr l =
  List.for_all (fun b -> not (is_info_type empty_evd b)) l

  (* Dans la version d'origine de logic_arity, seul le 1er prod de
     l'arit avait le droit d'tre small (bug ??) *)

let rec logic_arity l =
  List.for_all 
    (fun b -> (not (is_info_type empty_evd b)) or (is_small_type b)) l

let unit_constr env nbpar ltypes arity indlc recargs = 
   List.length ltypes = 1 
  & (not (is_recursive [0] recargs))
  & let (indenv,lC) = decomp_DLAMV_env env (ltypes,indlc) in
    Array.length lC = 1  
  & logic_arity (prods_of_type nbpar indenv arity.body)
  & logic_constr (prods_of_type nbpar indenv lC.(0))

let allowed_msorts issmall isunit s = 
  let sorts = match s with
      DOP0(Sort(Type(_)))   -> [prop;spec;types]
    | DOP0(Sort(Prop(Pos))) -> if issmall then [prop;spec;types]
      else [prop;spec]
    | DOP0(Sort(Prop(Null))) -> if isunit then [prop;spec]
      else [prop]
    | _ -> assert false
in (sorts,sorts);;

let inf_allowed_msorts s = match s with
   DOP0(Sort(Type(_)))   -> [types;spec],[types;spec]
 | DOP0(Sort(Prop(Pos)))     -> [],[types;spec]
 | _ -> assert false
;;

(******************************************************************)
(* check the constructors of the ith- type are well-formed 
   in particular is correctly applied to its arguments and compute
   the vector of recursive arguments

   Added a more precise treatment of errors (4 different kinds of failure) (HH)
*)

type ill_formed_ind =
    NonPos of int   | NotEnoughArgs of int
  | NotConstructor  | NonPar of int * int

exception Ill_formed_ind of ill_formed_ind

let explain_ind_err ntyp lna nbpar c err = 
  let (lpar,c') = decompose_prod_n nbpar c in
  let env = (List.map fst lpar)@lna in
  match err with
    NonPos kt -> error_non_strictly_positive CCI env c' (Rel (kt+nbpar))
  | NotEnoughArgs kt -> error_ill_formed_inductive CCI env c' (Rel (kt+nbpar))
  | NotConstructor ->
      error_ill_formed_constructor CCI env c' (Rel (ntyp+nbpar))
  | NonPar (n,l) ->
      error_bad_ind_parameters CCI env c' n (Rel (nbpar-n+1)) (Rel (l+nbpar))

let failwith_non_pos n nbtyp b =
  for k = n to n + nbtyp - 1 do
    if not (noccurn k b) then raise (Ill_formed_ind (NonPos (k-n+1)))
  done;
  anomaly "failwith_non_pos: some k in [n;n+nbtyp-1] should occur in b";;

let failwith_non_pos_vect n nbtyp v =
  for i = 0 to Array.length v - 1 do
    for k = n to n + nbtyp - 1 do
      if not (noccurn k v.(i)) then raise (Ill_formed_ind (NonPos (k-n+1)))
    done
  done;
  anomaly "failwith_non_pos_vect: some k in [n;n+nbtyp-1] should occur in v";;


(* Check "largs" are correct arguments of a recursive call to one of the
   inductive type mutually defined.

   We must have 
   - largs = [|Rel(n-1);..Rel(n-nbpar);a1;..ak|]
   - Rel p does not occur in ai for n<=p<=n+nbtyp

   nbtyp  = number of inductive types mutually defined
   nbpar  = number of local parameters for the mutual ind types defined
   n      = current lift for de Bruijn indices
   largs  = they are the args of the recursive call checked
   l      = the indice of the recursively applied ind type (used for err msg)
   *)

let check_correct_par nbpar nbtyp n l largs =
  begin
    if Array.length largs < nbpar 
    then raise (Ill_formed_ind (NotEnoughArgs l));
    let (lpar,largs') = chop_vect nbpar largs in
      for k=0 to nbpar -1 do
        if not ((Rel (n-k-1))=(whd_betadeltaiotaeta empty_evd lpar.(k))) then
          raise (Ill_formed_ind (NonPar (k+1,l)))
      done;
      if not (for_all_vect (noccur_bet n nbtyp) largs') 
      then failwith_non_pos_vect n nbtyp largs'
      else true
  end

(* Given a specification of a parameterised inductive defintion
   [X1]...[Xntyp]
     (a1:A1)...(ak:Ak)
       (B1(Xi1(a1,..ak)))->..->Bp(Xip(a1,..ak))->(Xi a1..ak t1..tn), 
   gives its abstract specification as 
   (a1:A1)...(ak:Ak)(B1(Xi1))->..->Bp(Xip))->(Xi t1..tn) *)

let abstract_mind_lc ntyps npars lc = 
  let lC = decomp_DLAMV ntyps lc 
  in
    if npars = 0 then lC else 
      let make_abs = 
        tabulate_list (function i -> lambda_implicit_lift npars (Rel (i+1))) ntyps 
      in Array.map (comp nf_beta (substl make_abs)) lC;;

let listrec_mconstr nbtyp nbpar i indlc =
(* check the inductive types occur positively in C *)
  let rec check_pos n c = 
    match whd_all c with 
        DOP2(Prod,b,DLAM(na,d)) -> 
          if (noccur_bet n nbtyp b) 
          then check_pos (n+1) d
          else raise (Ill_formed_ind (NonPos n))
      | x -> (match ensure_appl x with 
                  DOPN(AppL,cl) -> 
                    let hd = hd_vect cl and largs = tl_vect cl 
                    in (match hd with 
                            Rel k -> 
                              if k >= n & k<n+nbtyp & 
                                (check_correct_par nbpar nbtyp n (k-n+1) largs)
                              then Mrec(n+nbtyp-k-1) 
                              else if noccur_bet n nbtyp x
			      then if (n-nbpar) <= k & k <= (n-1)
                              then Param(n-1-k)
                              else Norec
			      else raise (Ill_formed_ind (NonPos n))
                        | (DOPN(MutInd(sp,i),_) as mi) -> 
                            if (noccur_bet n nbtyp x) 
                            then Norec
                            else Imbr(sp,i,imbr_positive n mi largs)
                        | err -> if noccur_bet n nbtyp x then Norec
		          else raise (Ill_formed_ind (NonPos n)))
                | _         -> anomaly "check_pos")

and imbr_positive n mi largs = 
   let mispeci = mind_specif_of_mind mi in
   let auxnpar = mis_nparams mispeci in
   let (lpar,auxlargs) = chop_vect auxnpar largs in 
     if for_all_vect (noccur_bet n nbtyp) auxlargs 
     then let auxlc = mis_lc mispeci 
          and auxntyp = mis_ntypes mispeci in 
            if auxntyp <> 1 then raise (Ill_formed_ind (NonPos n))
            else let lrecargs = map_vect_list (check_param_pos n) lpar
	    in (* The abstract imbricated inductive type with parameters substituted *)
		 let auxlcvect = abstract_mind_lc auxntyp auxnpar auxlc in
		 let newidx = n + auxntyp in
		 let _ = 
		   (* fails if the inductive type occurs non positively *)
		   (* when substituted *) 
		   Array.map 
		     (function c -> 
			let c' = hnf_prod_appvect empty_evd "is_imbr_pos" c 
				   (Array.map (lift auxntyp) lpar)
			in check_construct false newidx c') 
		     auxlcvect
              in lrecargs
     else raise (Ill_formed_ind (NonPos n))

(* The function check_param_pos is exactly the same as check_pos, but
   with an extra case for traversing abstractions, like in Marseille's
   contribution about bisimulations:

   CoInductive strong_eq:process->process->Prop:=
     str_eq:(p,q:process)((a:action)(p':process)(transition p a p')->
        (Ex [q':process] (transition q a q')/\(strong_eq p' q')))->
           ((a:action)(q':process)(transition q a q')->
        (Ex [p':process] (transition p a p')/\(strong_eq p' q')))->
        (strong_eq p q).


   Abstractions may occur in imbricated recursive ocurrences, but I am
   not sure if they make sense in a form of constructor. This is why I
   chose to duplicated the code.  Eduardo 13/7/99.
*)
  and check_param_pos n c = 
    match whd_all c with 
(* The extra case *)
        DOP2(Lambda,b,DLAM(na,d)) -> 
          if noccur_bet n nbtyp b
          then check_param_pos (n+1) d
          else raise (Ill_formed_ind (NonPos n))
(******************)
      | DOP2(Prod,b,DLAM(na,d)) -> 
          if (noccur_bet n nbtyp b) 
          then check_param_pos (n+1) d
          else raise (Ill_formed_ind (NonPos n))
      | x -> (match ensure_appl x with 
                  DOPN(AppL,cl) -> 
                    let hd = hd_vect cl and largs = tl_vect cl 
                    in (match hd with 
                            Rel k -> 
                              if k >= n & k<n+nbtyp & 
                                (check_correct_par nbpar nbtyp n (k-n+1) largs)
                              then Mrec(n+nbtyp-k-1) 
                              else if noccur_bet n nbtyp x
			      then if (n-nbpar) <= k & k <= (n-1)
                              then Param(n-1-k)
                              else Norec
			      else raise (Ill_formed_ind (NonPos n))
                        | (DOPN(MutInd(sp,i),_) as mi) -> 
                            if (noccur_bet n nbtyp x) 
                            then Norec
                            else Imbr(sp,i,imbr_positive n mi largs)
                        | err -> if noccur_bet n nbtyp x then Norec
		          else raise (Ill_formed_ind (NonPos n)))
                | _         -> anomaly "check_param_pos")
  
(* check the inductive types occur positively in the products of C, if
   checkhd=true, also check the head corresponds to a constructor of
   the ith type *) 

and check_construct check = 
  let rec
    check_constr_rec lrec n c = 
    match whd_all c with |
        DOP2(Prod,b,DLAM(na,d)) -> 
          let recarg = (check_pos n b) 
          in check_constr_rec (recarg::lrec) (n+1) d 
      | x -> (match ensure_appl x with 
                | DOPN(AppL,cl) ->
                    let hd = hd_vect cl 
                    and largs = tl_vect cl in 
                      if check then 
                        (match hd with Rel k -> 
                           if k = n+nbtyp-i & 
                              (check_correct_par nbpar nbtyp n (k-n+1) largs) 
                           then List.rev lrec 
                           else raise (Ill_formed_ind (NonPos n))
                           | _ -> raise (Ill_formed_ind (NonPos n)))
                      else 
                        if for_all_vect (noccur_bet n nbtyp) largs 
                        then List.rev lrec 
                        else raise (Ill_formed_ind (NonPos n)) | _ -> anomaly
                            "ensure_appl should return an AppL") 
  in check_constr_rec []

(* check the inductive types occur positively in c 
let rec check_pos n c = match whd_all c with 
     DOP2(Prod,b,DLAM(na,d)) -> if (noccur_bet n nbtyp b) 
                                then check_pos (n+1) d
                                else failwith_non_pos n nbtyp b
   | x -> (match ensure_appl x with 
           DOPN(AppL,cl) -> let hd = hd_vect cl and largs = tl_vect cl 
           in (match hd with 
                  Rel k ->
		    if k >= n & k<n+nbtyp then
		      (check_correct_par nbpar nbtyp n (k-n+1) largs;
                       Mrec(n-k+nbtyp-1))
                    else failwith_non_pos_vect n nbtyp cl
                | (DOPN(MutInd(_),_) as mi) -> 
                let mispeci = mind_specif_of_mind mi in
                let auxnpar = mis_nparams mispeci in
                let (lpar,auxlargs) = chop_vect auxnpar largs in 
           if for_all_vect (noccur_bet n nbtyp) auxlargs then 
              let auxlc = mis_lc mispeci 
              and auxntyp = mis_ntypes mispeci in 
              if auxntyp <> 1 then failwith_non_pos_vect n nbtyp cl else
              let auxlcvect = abstract_mind_lc auxntyp auxnpar auxlc in
              let newidx = n + auxntyp in
              let lrecvec = Array.map 
                (function c -> 
                   let c' = hnf_prod_appvect empty_evd "check_pos" c 
                      (Array.map (lift auxntyp) lpar)
                   in check_construct false newidx c') auxlcvect
              in Imbr lrecvec
(* Ce n'est pas suffisant il faut prendre en compte les imbriques recursifs *)
           else failwith_non_pos_vect n nbtyp auxlargs
                | _ -> failwith_non_pos_vect n nbtyp cl)
             | _         -> anomaly "check_pos")

(* check the inductive types occur positively in the products of c, 
   if checkhd=true, also check the head corresponds to a constructor of
   the ith type *)

and check_construct check  = 
  let rec check_constr_rec lrec n c = 
    match whd_all c with
     | DOP2(Prod,b,DLAM(na,d)) ->
       if noccur_bet n nbtyp b
       then check_constr_rec (Norec::lrec) (n+1) d
       else let recarg = (check_pos n b) in
         check_constr_rec (recarg::lrec) (n+1) d
     |  x -> (match ensure_appl x with 
		| DOPN(AppL,cl) -> 
		    let hd = hd_vect cl and largs = tl_vect cl in 
		      if check then 
			(match hd with 
			    Rel k when k = n+nbtyp-i ->
			       check_correct_par nbpar nbtyp n (k-n+1) largs;
                               List.rev lrec
			  | _ -> raise (Ill_formed_ind NotConstructor))
		      else if for_all_vect (noccur_bet n nbtyp) largs 
		      then List.rev lrec
                      else failwith_non_pos_vect n nbtyp largs
		| _ -> anomaly "ensure_appl should return an AppL")
  in check_constr_rec []
 >>>>>>> 1.14 *)

in
  let (lna,lC) = decomp_DLAMV_name nbtyp indlc in
    Array.map
      (fun c ->
	try check_construct true (1+nbpar) (decomp_par nbpar c)
        with Ill_formed_ind err -> explain_ind_err (nbtyp-i+1) lna nbpar c err)
      lC;;

let rec strip_outer_DLAM = function DLAM(_,t) -> strip_outer_DLAM t
                                  | DLAMV(_,v) -> v
                                  | _ -> anomaly "strip_outer_DLAM";;


(* check a mutually inductive definition is well-formed and 
   compute the redondant information *)

let recache_mibody_nolamexp mib =

let kind = mib.mINDKIND and
    hyps = mib.mINDHYPS and
    nbpar = mib.mINDNPARAMS and
    nbtyp = mib.mINDNTYPES in

let sign = gLOB hyps in

let ltypes = map_vect_list (fun mip -> mip.mINDARITY) mib.mINDPACKETS in

let recache_mipacket i mip =

  let arity = mip.mINDARITY in

  let s = sort_of_arity arity.body in

  let recargs = listrec_mconstr nbtyp nbpar i mip.mINDLC in

  let (kd,kn) = if kind = CCI then 
              allowed_msorts (small_list_constr sign nbpar ltypes mip.mINDLC)
             (unit_constr sign nbpar ltypes arity mip.mINDLC recargs) s 
              else inf_allowed_msorts s in

   {mINDSTAMP = mip.mINDSTAMP;
    mINDCONSNAMES = mip.mINDCONSNAMES;
    mINDTYPENAME = mip.mINDTYPENAME;
    mINDARITY = mip.mINDARITY;
    mINDLC = mip.mINDLC;
    mINDLAMEXP = mip.mINDLAMEXP;
    mINDKD = kd;
    mINDKN = kn;
    mINDLISTREC = recargs;
    mINDFINITE = mip.mINDFINITE;
    mINDIMPLICITS = if (is_implicit_args()) 
                    then IMPL_AUTO (poly_args_type mip.mINDARITY) else NO_IMPL;
    mINDCONSIMPLICITS = if (is_implicit_args())
                         then Array.map (fun c -> IMPL_AUTO (poly_args c))
                                 (strip_outer_DLAM mip.mINDLC)
                         else Array.create (Array.length mip.mINDCONSNAMES) NO_IMPL
}

in {mINDKIND = mib.mINDKIND;
    mINDNTYPES = nbtyp;
    mINDSINGL = mib.mINDSINGL;
    mINDNPARAMS = nbpar;
    mINDHYPS = hyps;
    mINDPACKETS = map_i_vect recache_mipacket 1 mib.mINDPACKETS
    }
;;

let make_mipacket_lam (stamp,typename,consnamev) (finite,arity,lc,lamexp) = 
(* A quoi sert le hnf_constr, seulement  enlever les cast, ou bien a plus ?
let typ = (match arity with
                 DOP2(Cast,c,t) -> DOP2(Cast,c,hnf_constr empty_evd t)
               | _ -> anomaly "arity") in*)
  let typ=arity in
    {mINDSTAMP = stamp;
     mINDCONSNAMES = consnamev;
     mINDARITY = typ;
     mINDLC = lc;
     mINDLAMEXP = lamexp;
     mINDKD = [];
     mINDKN = [];
     mINDTYPENAME = typename;
     mINDLISTREC = [||];
     mINDFINITE = finite;
     mINDIMPLICITS = NO_IMPL;
     mINDCONSIMPLICITS = Array.create (Array.length consnamev) NO_IMPL
};;

let make_mibody_lam (k,hyps,nparams,lpack,singl) =
    {mINDKIND = k; 
     mINDNPARAMS = nparams;
     mINDSINGL = singl;
     mINDNTYPES = Array.length lpack;
     mINDHYPS = hyps;
     mINDPACKETS = lpack}
;;


let change_constr_mip (newar,newlc,newlamexp) mip = 
    {mINDCONSNAMES = mip.mINDCONSNAMES;
     mINDTYPENAME = mip.mINDTYPENAME;
     mINDSTAMP = mip.mINDSTAMP;
     mINDLC = newlc;
     mINDARITY = newar;
     mINDKD = mip.mINDKD;
     mINDKN = mip.mINDKN;
     mINDLISTREC = mip.mINDLISTREC;
     mINDLAMEXP = newlamexp;
     mINDFINITE = mip.mINDFINITE;
     mINDIMPLICITS = mip.mINDIMPLICITS;
     mINDCONSIMPLICITS = mip.mINDCONSIMPLICITS };;

let compute_implicits_mib mib = 

let compute_implicits_mip mip = 
    {mINDCONSNAMES = mip.mINDCONSNAMES;
     mINDTYPENAME = mip.mINDTYPENAME;
     mINDSTAMP = mip.mINDSTAMP;
     mINDLC = mip.mINDLC;
     mINDARITY = mip.mINDARITY;
     mINDKD = mip.mINDKD;
     mINDKN = mip.mINDKN;
     mINDLISTREC = mip.mINDLISTREC;
     mINDLAMEXP = mip.mINDLAMEXP;
     mINDFINITE = mip.mINDFINITE;
     mINDIMPLICITS = if is_impl_auto mip.mINDIMPLICITS
                     then IMPL_AUTO (poly_args_type mip.mINDARITY)
                     else NO_IMPL;
     mINDCONSIMPLICITS = Array.of_list (List.map
              (fun (i,c) -> if (is_impl_auto i) 
                            then IMPL_AUTO (poly_args c)
                            else NO_IMPL)
                 (combine_vect_list (mip.mINDCONSIMPLICITS,(strip_outer_DLAM mip.mINDLC))))
}
in
if is_implicit_args() then
    {mINDNPARAMS = mib.mINDNPARAMS ;
     mINDKIND = mib.mINDKIND;
     mINDNTYPES = mib.mINDNTYPES;
     mINDSINGL = mib.mINDSINGL;
     mINDHYPS = mib.mINDHYPS ;
     mINDPACKETS = Array.map compute_implicits_mip mib.mINDPACKETS}
else mib
;;

let change_constr_mib (newpar,newhyps,newarlcvec) mib = 
  let fhyps = fsign_of_sign empty_evd newhyps in
  let change_mip (newar,newlc) mip = 
    let newlamexp = (* compute the newlamexp when necessary *)
      if newpar = mib.mINDNPARAMS or (mip.mINDLAMEXP=None)
      then mip.mINDLAMEXP 
      else
	let (_,newl,_,_) =
	  inf_mindterm (newhyps,fhyps) newpar 
	    newar newar.body false
	in newl
    in change_constr_mip (newar,newlc,newlamexp) mip
  in 
    if mib.mINDSINGL = None or newpar = mib.mINDNPARAMS
    then 
      {mINDNPARAMS = newpar;
       mINDKIND = mib.mINDKIND;
       mINDNTYPES = mib.mINDNTYPES;
       mINDSINGL = mib.mINDSINGL;
       mINDHYPS = newhyps;
       mINDPACKETS = map2_vect change_mip newarlcvec mib.mINDPACKETS}
    else   (* case of singleton types *)  
      let (newar,newlc) = newarlcvec.(0) in
      let lC = match newlc with DLAMV (_,lc) -> lc | _ -> assert false in
      let (_,newl,_,newsingl) = 
        let newenv = add_rel (Anonymous,newar) (gLOB newhyps)
        and newinfar = extract_of newhyps fhyps newar.body in 
	let newfenv = add_rel (Anonymous,newinfar) (gLOB fhyps) in
          inf_minductive newenv newfenv newpar newar lC.(0) true
        in  
          {mINDNPARAMS = newpar;
           mINDKIND = mib.mINDKIND;
           mINDNTYPES = mib.mINDNTYPES;
           mINDSINGL = newsingl;
           mINDHYPS = newhyps;
           mINDPACKETS = [|change_constr_mip (newar,newlc,newl) 
			   mib.mINDPACKETS.(0)|]}
    ;;
   
let expmod_mimap modlist mimap =
    List.map (fun (k,mib) ->
     (k,change_constr_mib 
           (mib.mINDNPARAMS, 
            map_sign_typ 
	      (fun {body=c;typ=s} -> {body=expmod_constr modlist c;typ=s})
	      mib.mINDHYPS,
            Array.map 
           (fun mip -> ({body=expmod_constr modlist mip.mINDARITY.body;
			typ=mip.mINDARITY.typ},
(* the under_dlams is essential in order to keep the outermost Cast, 
   necessary for generalisation *)
                  under_dlams (expmod_constr modlist) mip.mINDLC)) mib.mINDPACKETS)
            mib)) mimap
;;

(* On ne garde des variables locales au moment de la df (hyps) que
   les var permanentes de auto_save_variables() et celles
    intervenant (rcursivement) dans le indspec *)

let make_minductive_hyps hyps indspec = 
  let ids = (auto_save_variables()) in
  let globals =
    it_vect (fun gl (ar,lc) -> (global_varsl (global_varsl gl lc) ar.body))
      [] indspec 
  in thin_hyps_glob ids (hyps,globals);;

let make_minductive_fhyps fhyps hyps =
let fhyps' = thin_to_match hyps fhyps in thin_to_type_level fhyps';;

let put_DLAMSV_inf lna vecarinf c =
    
  let rec putfirst = function
      ((na::l), k, c) ->
        if k<0 then anomaly "put_DLAM_inf" 
        else if vecarinf.(k)=Logic then  putfirst (l, k-1, Array.map pop c)
        else (l,k-1,DLAMV(na,c))
    | ([], k, c) -> anomaly"put_DLAM_inf"

  and put (lna,k,c) = match lna with 
        (na::l) -> if k<0 then anomaly "put_DLAM_inf" 
        else if vecarinf.(k)=Logic then  put (l,k-1,pop c)
        else put (l,k-1,DLAM(na,c))
      | []     -> if k = -1 then c else anomaly "put_LAM_inf"

 in put (putfirst (lna, Array.length vecarinf-1, c))
  
  ;;

let enforce_type_construct univ (sign,nparams) c = 
  let exrec = core_fmachine false (empty_evd,[]) in 
  let rec enfrec (sign,nparams,c) = 
    if nparams = 0 then 
      let jC = exrec sign c in
      	(match whd_all jC._TYPE with 
                 (DOP0(Sort(Type(uc)))) -> (enforcegeq univ uc;jC)
                | _ -> error "Type of Constructor not well-formed")
    else let (name,c1,c2) = destProd (whd_all c) in 
         let j = exrec sign c1 in
         let assum = Machops.assumption_of_judgement empty_evd sign j in
         let var = (name,assum) in
         let j' = enfrec (add_rel var sign,(nparams-1),c2) in
	   Machops.gen_rel empty_evd CCI sign name assum j'
  in 
  let judge = 
    (try enfrec (sign,nparams,c)
     with UserError _ -> 
       let pc = pTERMINENV(sign,c) in
       let pe = pENV [< 'sTR" in environment" >] sign in
	 errorlabstrm "enforce_type_construct"
	   [< 'sTR "Type of constructor :" ; pc; 'fNL; 
	      'sTR "not well formed"; pe >])
  in 
    cast_of_judgement judge
;;


let check_prop_construct sign c = 
    let jC = (try  core_fmachine false (empty_evd,[]) sign c 
              with UserError (s,msg) -> 
                let pc = pTERMINENV(sign,c) in
     errorlabstrm "enforce_type_construct"
     [< pc; 'fNL;  
        'sTR "can not be the type of a constructor since it is not well typed:"
         ; 'fNL ;msg >])
    in 
    match whd_all jC._TYPE with 
                 (DOP0(Sort(_))) -> cast_of_judgement jC 
               | _ ->
                   let pc = pTERMINENV(sign,c) in
                     errorlabstrm "indtypes__check_prop_construct" 
[< pc; 'fNL;
   'sTR "can not be the type of a constructor since it is not a type" >];;


let packets_minductive sign nparams namesvec finite arlcvec =
  let ntypes = Array.length arlcvec in
  (* At the moment inductive definitions have an empty stamp *)
  let na = Anonymous in

  let aritj_vec = 
    Array.map (function (ar,_) -> casted_execute empty_evd sign
              (instantiate_universes ar.body)) arlcvec in

  let hyps = (gLOB sign) in
  let ind_sign =
    it_vect 
      (fun isign arj -> 
         (add_rel (na,Machops.assumption_of_judgement empty_evd hyps arj)
            isign))
      hyps aritj_vec in

  let one_ind finite ((_,indid,_) as namesarg) arj (_,lc) =
    let lc = instantiate_universes lc in
    let lna,lC = decomp_DLAMV_name ntypes lc in
    let sort = destSort (sort_of_arity_and_check indid (j_val_only arj)) in
    let ar = {body= arj._VAL; typ=match arj._TYPE with DOP0(Sort s) -> s | _ -> anomaly "Sorte caste"} in
    let lC_0 = match sort with 
        Type(univ) -> 
          Array.map (enforce_type_construct univ (ind_sign,nparams)) lC
      |Prop(_) -> Array.map (check_prop_construct ind_sign) lC in
    let lcind = put_DLAMSV lna lC_0 in
      make_mipacket_lam namesarg (finite,ar,lcind,None),None,None in

  let packs = map3_vect (one_ind finite) namesvec aritj_vec arlcvec in
  let mhyps = make_minductive_hyps sign arlcvec in
    (mhyps,packs);;


let packets_minductive_extract (sign,fsign)  nparams namesvec finite arlcvec =
  let ntypes = Array.length arlcvec in 
  (* At the moment inductive definitions have an empty stamp *)
  let na = Anonymous in
  let aritj_infar_vec = 
    Array.map (function (ar,_) -> casted_infexecute empty_evd (sign,fsign) 
                   (instantiate_universes ar.body)) arlcvec in

  let vecarinf = Array.map snd aritj_infar_vec in

  let hyps = gLOB sign and fhyps = gLOB fsign in

  (* Handles the renaming of parameters *)
  let ind_sign,inf_ind_sign =  
    let top=Array.length  namesvec in 
    let rec build_signs i ((isign,fisign) as res) =
      if i=top then res
      else
        let (arj,infar)=aritj_infar_vec.(i) in
	let (_,id,_)=namesvec.(i) in
        build_signs (i+1)
             (add_rel (Name id,
                       Machops.assumption_of_judgement empty_evd hyps arj)
                isign,
              add_inf_rel ((Name id),infar) fisign)
    in build_signs 0 (hyps,fhyps) in

  let one_ind checksingl finite ((_,indid,_) as namesarg) (arj,infar) (_,lc) = 
    let lc = instantiate_universes lc in
    let sort = destSort (sort_of_arity_and_check indid (j_val_only arj)) in
    let ar = {body= arj._VAL; typ=match arj._TYPE with DOP0(Sort s) -> s | _ -> anomaly "Sorte caste"} in
    let lna,lC = decomp_DLAMV_name ntypes lc in 
    let lC_0 = match sort with 
        Type(univ) -> 
          Array.map (enforce_type_construct univ (ind_sign,nparams)) lC
      |Prop(_) -> Array.map (check_prop_construct ind_sign) lC
    in let lcind = put_DLAMSV lna lC_0 in
      (match infar with 
           Logic -> 
             make_mipacket_lam namesarg (finite,ar,lcind,None),None,None
         | Inf infarj  -> 
             let inflcj =
               Array.map 
                 (function 
		    | DOP2(Cast,c,_) -> 
                   	core_infmachine (empty_evd,([],[]))
			  ind_sign inf_ind_sign c
		    | _ -> assert false) lC_0 
             in if exists_vect (fun inf -> inf = Logic) inflcj 
               then make_mipacket_lam namesarg (finite,ar,lcind,None),None,None
	       else
    (* An inductive definition with only one constructor containing 
       only non-informative arguments should  be considered as non informative *)

             let (infnpar,lamexp,arglist,singl) = 
               let (check,c) =
                 if checksingl & (Array.length lC) = 1 then (true,lC.(0))
                 else false,(DOP2(Cast,ar.body,DOP0(Sort ar.typ)))
               in inf_minductive ind_sign inf_ind_sign nparams ar c check
             in let strip = match sort with Type(_) -> (fun k -> k)
                                           | _      -> (strip_constr arglist)
	     in
               (match singl with 
                    None ->
                      (* compute both the inductive type and its extraction *)
                      let inf_vect =
                        Array.map (comp (comp strip cast_of_judgement) extract)
			  inflcj in
                      let newinflc = put_DLAMSV_inf lna vecarinf inf_vect
                      in (make_mipacket_lam  namesarg (finite,ar,lcind,lamexp),
                          Some(make_mipacket_lam namesarg 
                                 (finite,
    {body= infarj._VAL; typ=match infarj._TYPE with DOP0(Sort s) -> s | _ -> anomaly "Sorte caste"},
                                  newinflc,None),infnpar),
                          None)
                  | _ ->
                      (make_mipacket_lam namesarg
                         (finite,ar,lcind,lamexp),None,singl))) in

  let checksingl = Array.length vecarinf = 1 in 

  let packs =
    map3_vect (one_ind checksingl finite) namesvec aritj_infar_vec arlcvec in

  let mhyps = make_minductive_hyps sign arlcvec in
    (mhyps,packs)
;;

let minductive_decl kind mhyps nparams packets singl = 
  (kind,recache_mibody_nolamexp 
      (make_mibody_lam (kind,mhyps,nparams,packets,singl)));;

let cci_minductive (mhyps,nparams,packs) fsign =

match packs.(0) with 
   (pack,_,(Some _ as singl)) -> 
     [minductive_decl CCI mhyps nparams [|pack|] singl]
  | _ -> let packets = Array.map fst3 packs 
         and infpackets = Array.map (fun (_,x,_) -> x) packs in 
         let mib = minductive_decl CCI mhyps nparams packets None
         and infmib = 
    let infpack,infnpar = 
    it_vect (fun (lpack,n) p -> match p with
		 None -> (lpack,n)
               | Some(pack,npar) -> (pack::lpack,npar)) ([],0)
            infpackets
    in if infpack = [] then []
       else let mfhyps = make_minductive_fhyps fsign mhyps in
       [ minductive_decl FW mfhyps infnpar (Array.of_list (List.rev infpack)) None ]
in mib::infmib;;

let fw_minductive (mhyps,nparams,packs) = 
  let packets = Array.map fst3 packs 
  in [minductive_decl FW mhyps nparams packets None];;

let execute_minductive sign nparams namesvec finite arlcvec = 
  let (mhyps,packs) = 
    packets_minductive sign nparams namesvec finite arlcvec 
  in fw_minductive (mhyps,nparams,packs);; 

let infexecute_minductive (sign,fsign) nparams namesvec finite arlcvec =
  let (mhyps,packs) =
    packets_minductive_extract (sign,fsign) nparams namesvec finite arlcvec 
  in cci_minductive (mhyps,nparams,packs) fsign ;;

(* J'ai supprim un cast sur le c.body (HH) *)
let generalize_type id var c =
(*let c' = mkProd (Name id) (incast_type var) (subst_var id c.body)*)
  let c' = mkProd (Name id) (var.body) (subst_var id c.body) and
    c'ty = sort_of_product var.typ c.typ
  in {body=c';typ=c'ty}
;;

let casted_generalize id var c =
(*let c' = mkProd (Name id) (incast_type var) (subst_var id (cast_term c))*)
 let c' = mkProd (Name id) (var.body) (subst_var id (cast_term c))
 and c'ty = sort_of_product var.typ (destSort (whd_all (cast_type c)))
  in mkCast c' (DOP0 (Sort c'ty))
;;

(* Obsolte? HH 6/99
let casted_abstract id var c =
    let c' = mkLambda (Name id) var (subst_var id c) 
    and c'ty = mkProd (Name id) var (cast_type c)
    in  mkCast c' c'ty
;;
*)

(* The following function computes the new regargs when
   a section is closed. Those Norec positions corresponding
   to the nth abstracted variable are changed into the 
   annotation Param(n).
*)

let change_recargs_mip newlr mip = 
    {mINDCONSNAMES = mip.mINDCONSNAMES;
     mINDTYPENAME = mip.mINDTYPENAME;
     mINDSTAMP = mip.mINDSTAMP;
     mINDLC = mip.mINDLC;
     mINDARITY = mip.mINDARITY;
     mINDKD = mip.mINDKD;
     mINDKN = mip.mINDKN;
     mINDLISTREC = newlr;
     mINDLAMEXP = mip.mINDLAMEXP;
     mINDFINITE = mip.mINDFINITE;
     mINDIMPLICITS = mip.mINDIMPLICITS;
     mINDCONSIMPLICITS = mip.mINDCONSIMPLICITS }
;;


let change_recargs_mib newlrvec mib = 
  {mINDNPARAMS = mib.mINDNPARAMS;
   mINDKIND    = mib.mINDKIND;
   mINDNTYPES  = mib.mINDNTYPES;
   mINDSINGL   = mib.mINDSINGL;
   mINDHYPS    = mib.mINDHYPS;
   mINDPACKETS = map2_vect change_recargs_mip newlrvec mib.mINDPACKETS}
;;

let rec analyse_param_pos nbpar n c = 
  match whd_all c with 
      DOP2(Prod,b,DLAM(na,d)) -> 
        analyse_param_pos nbpar (n+1) d
    | x -> (match ensure_appl x with 
                DOPN(AppL,cl) -> 
                  let hd = hd_vect cl 
                  in (match hd with 
                          Rel k -> 
                            (if (n-nbpar) <= k & k <= (n-1)
                             then Param(n-1-k)
                             else Norec)
                        | _ -> Norec)
              | _         -> anomaly "change_into_param")
;;

let change_Norec_into_Param mib =
  let nbpar    = mib.mINDNPARAMS in 
  let ntypes   = mib.mINDNTYPES  in
  let lrvecvec = Array.map (fun mip -> mip.mINDLISTREC) mib.mINDPACKETS in
  let stripped_lcvec = 
    let rec strip_DLAM c =
      match c with
          (DLAM  (n,c1)) -> strip_DLAM c1 
        | (DLAMV (n,v))  -> v
        | _ -> assert false
    in Array.map 
         (fun mip -> 
            Array.map (decomp_par nbpar) (strip_DLAM mip.mINDLC))
         mib.mINDPACKETS in
  let rec change_into_param n c lrec =
    match (whd_all c) with
      | DOP2(Prod,b,DLAM(na,d)) ->
          (match lrec with
               (Norec::lrec1) -> 
                 let recarg = (analyse_param_pos nbpar n b) 
                 in  recarg::(change_into_param (n+1) d lrec1)
             | (ra::lrec1)    -> 
                 (ra::(change_into_param (n+1) d lrec1))
             | []             -> anomaly "change_into_param_construct")
      | _ -> [] in
  let newlrvec =  
    map2_vect
      (fun lcvec lrvec -> 
         map2_vect (change_into_param (ntypes+nbpar)) lcvec lrvec)
      stripped_lcvec lrvecvec
  in change_recargs_mib newlrvec mib 


(* Abstraction on mutually inductive definitions *)

let abs_var_on_minductive id var arity_lc_vec =
  let ntyp = Array.length arity_lc_vec in 
  let new_refs = tabulate_list (fun k ->applist(Rel (k+2),[Rel 1])) ntyp in 
  let arity_lc_vec' =
    Array.map
      (function (arity,lc) -> 
	 let arity' = generalize_type id var arity in
	 let lc' =
	   under_dlams 
             (fun b -> casted_generalize id var (substl new_refs b)) lc
	 in
           (arity',lc'))
      arity_lc_vec 
  in (arity_lc_vec',ABSTRACT)
;;

let abstract_minductive_once ((hyps,ispec),modl) id =
  if isnull_sign hyps or id <> fst(hd_sign hyps) 
  then ((hyps,ispec),modl)
  else
    let (hyps_ispec,modif) = abs_var_on_minductive id (snd(hd_sign hyps)) ispec
    in ((tl_sign hyps,hyps_ispec),modif::modl)
;;

let abstract_minductive ids_to_abs (hyps,mispec) =
  let ((hyps',ispec'),revmodl) =
    List.fold_left abstract_minductive_once ((hyps,mispec),[]) ids_to_abs in
  let modl = List.rev revmodl
  in ((hyps',ispec'),modl)
;;

let abstract_mibody (osecsp,nsecsp) ids_to_abs mib =
  let miarlc = 
    Array.map (fun mip -> (mip.mINDARITY,mip.mINDLC)) 
      mib.mINDPACKETS in
  let ((hyps',arity_lc_vec'),modl) =
    abstract_minductive ids_to_abs (mib.mINDHYPS,miarlc) in
  let lmodif_one_mind i = 
    let nbc = Array.length ((mind_nth_type_packet mib i).mINDCONSNAMES)
    in (MutInd(osecsp,i),DO_ABSTRACT(MutInd(nsecsp,i),modl))::
       (MutCase(Some (osecsp,i)),DO_ABSTRACT(MutCase(Some (nsecsp,i)),[]))::
       (tabulate_list 
	  (function j -> let j'=j+1 in
	     (MutConstruct((osecsp,i),j'),
	      DO_ABSTRACT(MutConstruct((nsecsp,i),j'),modl)))
          nbc) in
  let modifs = List.flatten (tabulate_list lmodif_one_mind mib.mINDNTYPES) in 
  let changed_constr_body,cmodlist = 
    (change_constr_mib 
       (mib.mINDNPARAMS + (List.length modl), hyps',arity_lc_vec') 
       mib,
       modifs)
  in  change_Norec_into_Param changed_constr_body,cmodlist
;;


let abstract_mimap (osecsp,nsecsp) ids_to_abs cmap =
  let (mods,cmap) =
    List.fold_left 
      (fun (mods,cmap) (k,ib) ->
         match k with
             CCI ->
               let (ib',modifs) = 
		 abstract_mibody (ccisp_of osecsp,ccisp_of nsecsp) 
		   ids_to_abs ib in
                 (modifs@mods,(k,(compute_implicits_mib ib'))::cmap)
           | FW -> 
               let (ib',modifs) = abstract_mibody 
				    (fwsp_of osecsp,fwsp_of nsecsp)
				    ids_to_abs ib in
                 (modifs@mods,(k,ib')::cmap)
	   | _ -> anomaly "abstract_mimap : cci of fw term expected")
      ([],[]) cmap
  in (mods,cmap)
;;

(**** For the discharge operation ******)

let process_mimap (sp,newsp) (ids_to_discard,work_alist) mimap =
  let expmod_mimap = expmod_mimap work_alist mimap
  in abstract_mimap (sp,newsp) ids_to_discard expmod_mimap 
;;

(* $Id: indtypes.ml,v 1.23 1999/11/01 12:41:13 mohring Exp $ *)
