(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                             constraintab.ml                              *)
(****************************************************************************)
open Pp;;
open Names;;
open Libobject;;
open Impuniv;;

type frozen_t = Impuniv.universes;;

(* Universes ensuring the consistency of the context. *)
let glob_univ = ref initial_universes;;

let init () = glob_univ := initial_universes;;
let freeze () = !glob_univ;;
let unfreeze fu = glob_univ := fu;;

Summary.declare_summary "constraints"
{Summary.freeze_function = freeze;
 Summary.unfreeze_function = unfreeze;
 Summary.init_function = init}
;;

let rollback f () =
  let fs = freeze() in
    try f () with e -> (unfreeze fs; raise e)
;;

let current_constraints () = !glob_univ;;

(* Push a local graph in the global one *)
let push_without_rollback u =
  let sp = Lib.make_path OBJ (id_of_string "uni") in
    glob_univ := merge_universes sp !glob_univ u
;;

let push universes =
  rollback (fun () -> push_without_rollback universes) ()
;;

type universes_object = section_path * universes;;

let (inUniverses,outUniverses) =
  declare_object
    ("UNIVERSES",
     {load_function = (fun (_,u) -> push_without_rollback u);
      cache_function = (fun _ -> ());
      specification_function = (fun x -> x) });;

let print_universes_object (sp,uni) =
  [< 'sTR"Universes of " ; 'sTR(string_of_path sp); 'sTR" (";
     'iNT(num_universes uni) ; 'sTR" uni,";
     'iNT(num_edges uni) ; 'sTR" edges)" >]
;;

let add_constraints_object (sp,u) =
  if u != empty_universes & u != initial_universes then
    Library.add_anonymous_object (inUniverses (sp,u))
;;


(* $Id: constraintab.ml,v 1.12 1999/11/07 17:57:13 barras Exp $ *)
